"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? [{ transitionToIa: props.lifecyclePolicy }] : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
                this.mountTargetsAvailable = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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