"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const crypto = require("crypto");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const security_group_1 = require("./security-group");
const volume_1 = require("./volume");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.securityGroups = [];
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tag.add(this, NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                this.node.addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                this.node.addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? volume_1.synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.applyUpdatePolicies(props);
        // Trigger replacement (via new logical ID) on user data change, if specified or cfn-init is being used.
        const originalLogicalId = core_1.Stack.of(this).getLogicalId(this.instance);
        this.instance.overrideLogicalId(core_1.Lazy.stringValue({
            produce: () => {
                var _a;
                let logicalId = originalLogicalId;
                if ((_a = props.userDataCausesReplacement) !== null && _a !== void 0 ? _a : props.initOptions) {
                    const md5 = crypto.createHash('md5');
                    md5.update(this.userData.render());
                    logicalId += md5.digest('hex').substr(0, 16);
                }
                return logicalId;
            },
        }));
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the Instance resource.
     * - Add commands to the instance UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for the `cfn-signal` commands.
     */
    applyCloudFormationInit(init, options = {}) {
        var _a;
        init._attach(this.instance, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
        });
        this.waitForResourceSignal((_a = options.timeout) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5));
    }
    /**
     * Wait for a single additional resource signal
     *
     * Add 1 to the current ResourceSignal Count and add the given timeout to the current timeout.
     *
     * Use this to pause the CloudFormation deployment to wait for the instances
     * in the AutoScalingGroup to report successful startup during
     * creation and updates. The UserData script needs to invoke `cfn-signal`
     * with a success or failure code after it is done setting up the instance.
     */
    waitForResourceSignal(timeout) {
        var _a, _b;
        const oldResourceSignal = (_a = this.instance.cfnOptions.creationPolicy) === null || _a === void 0 ? void 0 : _a.resourceSignal;
        this.instance.cfnOptions.creationPolicy = {
            ...this.instance.cfnOptions.creationPolicy,
            resourceSignal: {
                count: ((_b = oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.count) !== null && _b !== void 0 ? _b : 0) + 1,
                timeout: ((oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout) ? core_1.Duration.parse(oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout).plus(timeout) : timeout).toIsoString(),
            },
        };
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
//# sourceMappingURL=data:application/json;base64,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