"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const fs = require("fs");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * An object that represents reasons to restart an InitService
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *   ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *   ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
/**
 * Base class for all CloudFormation Init elements
 */
class InitElement {
}
exports.InitElement = InitElement;
/**
 * Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 */
class InitCommandWaitDuration {
    /** Wait for a specified duration after a command. */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /** Do not wait for this command. */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /** cfn-init will exit and resume only after a reboot. */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
/**
 * Command to execute on the instance
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * Run a shell command
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand([shellCommand], options);
    }
    /**
     * Run a command from an argv array
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _a, _b;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_b = this.options.waitAfterCompletion) === null || _b === void 0 ? void 0 : _b._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
/**
 * Create files on the EC2 instance.
 */
class InitFile extends InitElement {
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * Use a literal string as the file content
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Write a symlink with the given symlink target
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Read a file from disk and use its contents
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * Download from a URL at instance startup time
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Download a file from an S3 bucket at instance startup time
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * Create an asset from the given file
     *
     * This is appropriate for files that are too large to embed into the template.
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                };
            }
        }(targetFileName, options);
    }
    /**
     * Use a file from an asset at instance startup time
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _a;
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {};
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
/**
 * Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 */
class InitGroup extends InitElement {
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * Create a group from its name, and optionally, group id
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
/**
 * Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 */
class InitUser extends InitElement {
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * Create a user from user name.
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
/**
 * A package to be installed during cfn-init time
 */
class InitPackage extends InitElement {
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * Install an RPM from an HTTP URL or a location on disk
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * Install a package using Yum
     */
    static yum(packageName, options = {}) {
        var _a;
        return new InitPackage('yum', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package from RubyGems
     */
    static rubyGem(gemName, options = {}) {
        var _a;
        return new InitPackage('rubygems', (_a = options.version) !== null && _a !== void 0 ? _a : [], gemName, options.serviceRestartHandles);
    }
    /**
     * Install a package from PyPI
     */
    static python(packageName, options = {}) {
        var _a;
        return new InitPackage('python', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package using APT
     */
    static apt(packageName, options = {}) {
        var _a;
        return new InitPackage('apt', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install an MSI package from an HTTP URL or a location on disk
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
/**
 * A services that be enabled, disabled or restarted when the instance is launched.
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * Enable and start the given service, optionally restarting it
     */
    static enable(serviceName, options = {}) {
        var _a;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_a = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _a !== void 0 ? _a : true,
            ...otherOptions,
        });
    }
    /**
     * Disable and stop the given service
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _a;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_a = this.serviceOptions.serviceRestartHandle) === null || _a === void 0 ? void 0 : _a._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
/**
 * Extract an archive into a directory
 */
class InitSource extends InitElement {
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * Retrieve a URL and extract it into the given directory
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a GitHub branch into a given directory
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * Extract an archive stored in an S3 bucket into the given directory
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Create an InitSource from an asset created from the given path.
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a directory from an existing directory asset.
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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