"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./cross-region-support-stack");
const full_action_descriptor_1 = require("./full-action-descriptor");
const stage_1 = require("./stage");
const validation_1 = require("./validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ name: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            const encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                // remove the key - there is a grace period of a few days before it's gone for good,
                // that should be enough for any emergency access to the bucket artifacts
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: s3.BucketEncryption.KMS,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
            // add an alias to make finding the key in the console easier
            new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                aliasName: this.generateNameForDefaultBucketKeyAlias(),
                targetKey: encryptionKey,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(action);
        // get the role for the given action
        const actionRole = this.getRoleForAction(stage, action, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(action.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = action.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRegion;
        let otherStack;
        const actionResource = action.actionProperties.resource;
        if (actionResource) {
            const pipelineAndActionRegionComparison = core_1.Token.compareStrings(this.env.region, actionResource.env.region);
            const pipelineAndActionInDifferentRegions = pipelineAndActionRegionComparison === core_1.TokenComparison.ONE_UNRESOLVED ||
                pipelineAndActionRegionComparison === core_1.TokenComparison.DIFFERENT;
            if (pipelineAndActionInDifferentRegions) {
                actionRegion = actionResource.env.region;
                // if the resource is from a different stack in another region but the same account,
                // use that stack as home for the cross-region support resources
                const actionResourceStack = core_1.Stack.of(actionResource);
                const actionResourceAndItsStackRegionComparison = core_1.Token.compareStrings(actionResource.env.region, actionResourceStack.region);
                const actionResourceInSameRegionAsItsStack = actionResourceAndItsStackRegionComparison === core_1.TokenComparison.SAME ||
                    actionResourceAndItsStackRegionComparison === core_1.TokenComparison.BOTH_UNRESOLVED;
                const pipelineAndActionResourceStackAccountComparison = core_1.Token.compareStrings(this.env.account, actionResourceStack.account);
                const pipelineAndActionResourceStackInSameAccount = pipelineAndActionResourceStackAccountComparison === core_1.TokenComparison.SAME ||
                    pipelineAndActionResourceStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED;
                if (pipelineAndActionResourceStackInSameAccount && actionResourceInSameRegionAsItsStack) {
                    otherStack = actionResourceStack;
                }
            }
        }
        else {
            actionRegion = action.actionProperties.region;
        }
        // if actionRegion is undefined,
        // it means the action is in the same region as the pipeline -
        // so, just return the artifactBucket
        if (!actionRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // get the region the Pipeline itself is in
        const pipelineRegion = this.requireRegion();
        // if the action is in the same region as the pipeline, nothing to do
        if (actionRegion === pipelineRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        // the stack containing the replication bucket must be deployed before the pipeline
        pipelineStack.addDependency(crossRegionSupport.stack);
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: actionRegion,
        };
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id);
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = this.node.uniqueId;
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        if (!this.artifactBucket.encryptionKey) {
            throw new Error('The Pipeline is being used in a cross-account manner, ' +
                'but its artifact bucket does not have a KMS key defined. ' +
                'A KMS key is required for a cross-account Pipeline. ' +
                'Make sure to pass a Bucket with a Key when creating the Pipeline');
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${this.node.uniqueId}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
//# sourceMappingURL=data:application/json;base64,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