"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubSourceAction = exports.GitHubTrigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * If and how the GitHub source action should be triggered
 */
var GitHubTrigger;
(function (GitHubTrigger) {
    GitHubTrigger["NONE"] = "None";
    GitHubTrigger["POLL"] = "Poll";
    GitHubTrigger["WEBHOOK"] = "WebHook";
})(GitHubTrigger = exports.GitHubTrigger || (exports.GitHubTrigger = {}));
/**
 * Source that is provided by a GitHub repository.
 */
class GitHubSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'ThirdParty',
            provider: 'GitHub',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
            commitUrl: this.variableExpression('CommitUrl'),
        };
    }
    bound(scope, stage, _options) {
        if (!this.props.trigger || this.props.trigger === GitHubTrigger.WEBHOOK) {
            new codepipeline.CfnWebhook(scope, 'WebhookResource', {
                authentication: 'GITHUB_HMAC',
                authenticationConfiguration: {
                    secretToken: this.props.oauthToken.toString(),
                },
                filters: [
                    {
                        jsonPath: '$.ref',
                        matchEquals: 'refs/heads/{Branch}',
                    },
                ],
                targetAction: this.actionProperties.actionName,
                targetPipeline: stage.pipeline.pipelineName,
                targetPipelineVersion: 1,
                registerWithThirdParty: true,
            });
        }
        return {
            configuration: {
                Owner: this.props.owner,
                Repo: this.props.repo,
                Branch: this.props.branch || 'master',
                OAuthToken: this.props.oauthToken.toString(),
                PollForSourceChanges: this.props.trigger === GitHubTrigger.POLL,
            },
        };
    }
}
exports.GitHubSourceAction = GitHubSourceAction;
//# sourceMappingURL=data:application/json;base64,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