"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentConfig = exports.MinimumHealthyHosts = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value,
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value,
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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