"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
        // do nothing
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.anyValue({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _a;
            return ({
                granularity: '1Minute',
                metrics: ((_a = group._metrics) === null || _a === void 0 ? void 0 : _a.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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