"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectType = exports.InterfaceType = exports.Directive = void 0;
const schema_field_1 = require("./schema-field");
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
class Directive {
    constructor(statement) { this.statement = statement; }
    /**
     * Add the @aws_iam directive
     */
    static iam() {
        return new Directive('@aws_iam');
    }
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     * Note: doesn't guarantee functionality
     */
    static custom(statement) {
        return new Directive(statement);
    }
}
exports.Directive = Directive;
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let schemaAddition = `interface ${this.name} {\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            schemaAddition = `${schemaAddition}  ${key}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        super(name, props);
        this.interfaceTypes = props.interfaceTypes;
        this.directives = props.directives;
    }
    /**
     * A method to define Object Types from an interface
     */
    static implementInterface(name, props) {
        if (!props.interfaceTypes || !props.interfaceTypes.length) {
            throw new Error('Static function `implementInterface` requires an interfaceType to implement');
        }
        return new ObjectType(name, {
            interfaceTypes: props.interfaceTypes,
            definition: props.interfaceTypes.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition),
            directives: props.directives,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let title = this.name;
        if (this.interfaceTypes && this.interfaceTypes.length) {
            title = `${title} implements`;
            this.interfaceTypes.map((interfaceType) => {
                title = `${title} ${interfaceType.name},`;
            });
            title = title.slice(0, -1);
        }
        const directives = this.generateDirectives(this.directives);
        let schemaAddition = `type ${title} ${directives}{\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            schemaAddition = `${schemaAddition}  ${key}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
    /**
     * Utility function to generate directives
     *
     * @param directives the directives of a given type
     * @param delimiter the separator betweeen directives
     * @default - ' '
     */
    generateDirectives(directives, delimiter) {
        let schemaAddition = '';
        if (!directives) {
            return schemaAddition;
        }
        directives.map((directive) => {
            schemaAddition = `${schemaAddition}${directive.statement}${delimiter !== null && delimiter !== void 0 ? delimiter : ' '}`;
        });
        return schemaAddition;
    }
}
exports.ObjectType = ObjectType;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hLWludGVybWVkaWF0ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNjaGVtYS1pbnRlcm1lZGlhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsaURBQXFFO0FBQ3JFOzs7Ozs7R0FNRztBQUNILE1BQWEsU0FBUztJQW9CbEIsWUFBb0IsU0FBaUIsSUFBSSxJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUM7SUFuQnRFOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUc7UUFDYixPQUFPLElBQUksU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBaUI7UUFDbEMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNwQyxDQUFDO0NBTUo7QUFyQkQsOEJBcUJDO0FBaUJEOzs7OztHQUtHO0FBQ0gsTUFBYSxhQUFhO0lBV3RCLFlBQW1CLElBQVksRUFBRSxLQUE0QjtRQUN6RCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7SUFDdkMsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxTQUFTLENBQUMsT0FBZ0M7UUFDN0MsT0FBTywwQkFBVyxDQUFDLFlBQVksQ0FBQztZQUM1QixNQUFNLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLE1BQU07WUFDdkIsVUFBVSxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxVQUFVO1lBQy9CLGNBQWMsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsY0FBYztZQUN2QyxnQkFBZ0IsRUFBRSxJQUFJO1NBQ3pCLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7T0FFRztJQUNJLFFBQVE7UUFDWCxJQUFJLGNBQWMsR0FBRyxhQUFhLElBQUksQ0FBQyxJQUFJLE1BQU0sQ0FBQztRQUNsRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUN6QyxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ3ZDLGNBQWMsR0FBRyxHQUFHLGNBQWMsS0FBSyxHQUFHLEtBQUssU0FBUyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUM7UUFDNUUsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLEdBQUcsY0FBYyxHQUFHLENBQUM7SUFDaEMsQ0FBQztDQUNKO0FBMUNELHNDQTBDQztBQXlCRDs7OztHQUlHO0FBQ0gsTUFBYSxVQUFXLFNBQVEsYUFBYTtJQTRCekMsWUFBbUIsSUFBWSxFQUFFLEtBQXNCO1FBQ25ELEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDbkIsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQzNDLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztJQUN2QyxDQUFDO0lBL0JEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLElBQVksRUFBRSxLQUFzQjtRQUNqRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMsNkVBQTZFLENBQUMsQ0FBQztTQUNsRztRQUNELE9BQU8sSUFBSSxVQUFVLENBQUMsSUFBSSxFQUFFO1lBQ3hCLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxVQUFVLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsYUFBYSxFQUFFLEVBQUU7Z0JBQzNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUM1RCxDQUFDLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQztZQUNwQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQWtCRDs7T0FFRztJQUNJLFFBQVE7UUFDWCxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBQ3RCLElBQUksSUFBSSxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRTtZQUNuRCxLQUFLLEdBQUcsR0FBRyxLQUFLLGFBQWEsQ0FBQztZQUM5QixJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFO2dCQUN0QyxLQUFLLEdBQUcsR0FBRyxLQUFLLElBQUksYUFBYSxDQUFDLElBQUksR0FBRyxDQUFDO1lBQzlDLENBQUMsQ0FBQyxDQUFDO1lBQ0gsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDOUI7UUFDRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzVELElBQUksY0FBYyxHQUFHLFFBQVEsS0FBSyxJQUFJLFVBQVUsS0FBSyxDQUFDO1FBQ3RELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQ3pDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDdkMsY0FBYyxHQUFHLEdBQUcsY0FBYyxLQUFLLEdBQUcsS0FBSyxTQUFTLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQztRQUM1RSxDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8sR0FBRyxjQUFjLEdBQUcsQ0FBQztJQUNoQyxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ssa0JBQWtCLENBQUMsVUFBd0IsRUFBRSxTQUFrQjtRQUNuRSxJQUFJLGNBQWMsR0FBRyxFQUFFLENBQUM7UUFDeEIsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNiLE9BQU8sY0FBYyxDQUFDO1NBQ3pCO1FBQ0QsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFO1lBQ3pCLGNBQWMsR0FBRyxHQUFHLGNBQWMsR0FBRyxTQUFTLENBQUMsU0FBUyxHQUFHLFNBQVMsYUFBVCxTQUFTLGNBQVQsU0FBUyxHQUFJLEdBQUcsRUFBRSxDQUFDO1FBQ2xGLENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxjQUFjLENBQUM7SUFDMUIsQ0FBQztDQUNKO0FBdEVELGdDQXNFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEJhc2VHcmFwaHFsVHlwZU9wdGlvbnMsIEdyYXBocWxUeXBlIH0gZnJvbSAnLi9zY2hlbWEtZmllbGQnO1xuLyoqXG4gKiBEaXJlY3RpdmVzIGZvciB0eXBlc1xuICpcbiAqIGkuZS4gQGF3c19pYW0gb3IgQGF3c19zdWJzY3JpYmVcbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBEaXJlY3RpdmUge1xuICAgIC8qKlxuICAgICAqIEFkZCB0aGUgQGF3c19pYW0gZGlyZWN0aXZlXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBpYW0oKTogRGlyZWN0aXZlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBEaXJlY3RpdmUoJ0Bhd3NfaWFtJyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGN1c3RvbSBkaXJlY3RpdmVcbiAgICAgKlxuICAgICAqIEBwYXJhbSBzdGF0ZW1lbnQgLSB0aGUgZGlyZWN0aXZlIHN0YXRlbWVudCB0byBhcHBlbmRcbiAgICAgKiBOb3RlOiBkb2Vzbid0IGd1YXJhbnRlZSBmdW5jdGlvbmFsaXR5XG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBjdXN0b20oc3RhdGVtZW50OiBzdHJpbmcpOiBEaXJlY3RpdmUge1xuICAgICAgICByZXR1cm4gbmV3IERpcmVjdGl2ZShzdGF0ZW1lbnQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiB0aGUgZGlyZWN0aXZlIHN0YXRlbWVudFxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBzdGF0ZW1lbnQ6IHN0cmluZztcbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKHN0YXRlbWVudDogc3RyaW5nKSB7IHRoaXMuc3RhdGVtZW50ID0gc3RhdGVtZW50OyB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNvbmZpZ3VyaW5nIGFuIEludGVybWVkaWF0ZSBUeXBlXG4gKlxuICogQHBhcmFtIGRlZmluaXRpb24gLSB0aGUgdmFyaWFibGVzIGFuZCB0eXBlcyB0aGF0IGRlZmluZSB0aGlzIHR5cGVcbiAqIGkuZS4geyBzdHJpbmc6IEdyYXBocWxUeXBlLCBzdHJpbmc6IEdyYXBocWxUeXBlIH1cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW50ZXJtZWRpYXRlVHlwZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiB0aGUgYXR0cmlidXRlcyBvZiB0aGlzIHR5cGVcbiAgICAgKi9cbiAgICByZWFkb25seSBkZWZpbml0aW9uOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IEdyYXBocWxUeXBlO1xuICAgIH07XG59XG4vKipcbiAqIEludGVyZmFjZSBUeXBlcyBhcmUgYWJzdHJhY3QgdHlwZXMgdGhhdCBpbmNsdWRlcyBhIGNlcnRhaW4gc2V0IG9mIGZpZWxkc1xuICogdGhhdCBvdGhlciB0eXBlcyBtdXN0IGluY2x1ZGUgaWYgdGhleSBpbXBsZW1lbnQgdGhlIGludGVyZmFjZS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBJbnRlcmZhY2VUeXBlIHtcbiAgICAvKipcbiAgICAgKiB0aGUgbmFtZSBvZiB0aGlzIHR5cGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIHRoZSBhdHRyaWJ1dGVzIG9mIHRoaXMgdHlwZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBkZWZpbml0aW9uOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IEdyYXBocWxUeXBlO1xuICAgIH07XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZywgcHJvcHM6IEludGVybWVkaWF0ZVR5cGVQcm9wcykge1xuICAgICAgICB0aGlzLm5hbWUgPSBuYW1lO1xuICAgICAgICB0aGlzLmRlZmluaXRpb24gPSBwcm9wcy5kZWZpbml0aW9uO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDcmVhdGUgYW4gR3JhcGhRTCBUeXBlIHJlcHJlc2VudGluZyB0aGlzIEludGVybWVkaWF0ZSBUeXBlXG4gICAgICpcbiAgICAgKiBAcGFyYW0gb3B0aW9ucyB0aGUgb3B0aW9ucyB0byBjb25maWd1cmUgdGhpcyBhdHRyaWJ1dGVcbiAgICAgKiAtIGlzTGlzdFxuICAgICAqIC0gaXNSZXF1aXJlZFxuICAgICAqIC0gaXNSZXF1aXJlZExpc3RcbiAgICAgKi9cbiAgICBwdWJsaWMgYXR0cmlidXRlKG9wdGlvbnM/OiBCYXNlR3JhcGhxbFR5cGVPcHRpb25zKTogR3JhcGhxbFR5cGUge1xuICAgICAgICByZXR1cm4gR3JhcGhxbFR5cGUuaW50ZXJtZWRpYXRlKHtcbiAgICAgICAgICAgIGlzTGlzdDogb3B0aW9ucz8uaXNMaXN0LFxuICAgICAgICAgICAgaXNSZXF1aXJlZDogb3B0aW9ucz8uaXNSZXF1aXJlZCxcbiAgICAgICAgICAgIGlzUmVxdWlyZWRMaXN0OiBvcHRpb25zPy5pc1JlcXVpcmVkTGlzdCxcbiAgICAgICAgICAgIGludGVybWVkaWF0ZVR5cGU6IHRoaXMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHZW5lcmF0ZSB0aGUgc3RyaW5nIG9mIHRoaXMgb2JqZWN0IHR5cGVcbiAgICAgKi9cbiAgICBwdWJsaWMgdG9TdHJpbmcoKTogc3RyaW5nIHtcbiAgICAgICAgbGV0IHNjaGVtYUFkZGl0aW9uID0gYGludGVyZmFjZSAke3RoaXMubmFtZX0ge1xcbmA7XG4gICAgICAgIE9iamVjdC5rZXlzKHRoaXMuZGVmaW5pdGlvbikuZm9yRWFjaCgoa2V5KSA9PiB7XG4gICAgICAgICAgICBjb25zdCBhdHRyaWJ1dGUgPSB0aGlzLmRlZmluaXRpb25ba2V5XTtcbiAgICAgICAgICAgIHNjaGVtYUFkZGl0aW9uID0gYCR7c2NoZW1hQWRkaXRpb259ICAke2tleX06ICR7YXR0cmlidXRlLnRvU3RyaW5nKCl9XFxuYDtcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiBgJHtzY2hlbWFBZGRpdGlvbn19YDtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNvbmZpZ3VyaW5nIGFuIE9iamVjdCBUeXBlXG4gKlxuICogQHBhcmFtIGRlZmluaXRpb24gLSB0aGUgdmFyaWFibGVzIGFuZCB0eXBlcyB0aGF0IGRlZmluZSB0aGlzIHR5cGVcbiAqIGkuZS4geyBzdHJpbmc6IEdyYXBocWxUeXBlLCBzdHJpbmc6IEdyYXBocWxUeXBlIH1cbiAqIEBwYXJhbSBpbnRlcmZhY2VUeXBlcyAtIHRoZSBpbnRlcmZhY2VzIHRoYXQgdGhpcyBvYmplY3QgdHlwZSBpbXBsZW1lbnRzXG4gKiBAcGFyYW0gZGlyZWN0aXZlcyAtIHRoZSBkaXJlY3RpdmVzIGZvciB0aGlzIG9iamVjdCB0eXBlXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9iamVjdFR5cGVQcm9wcyBleHRlbmRzIEludGVybWVkaWF0ZVR5cGVQcm9wcyB7XG4gICAgLyoqXG4gICAgICogVGhlIEludGVyZmFjZSBUeXBlcyB0aGlzIE9iamVjdCBUeXBlIGltcGxlbWVudHNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gaW50ZXJmYWNlIHR5cGVzXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW50ZXJmYWNlVHlwZXM/OiBJbnRlcmZhY2VUeXBlW107XG4gICAgLyoqXG4gICAgICogdGhlIGRpcmVjdGl2ZXMgZm9yIHRoaXMgb2JqZWN0IHR5cGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gZGlyZWN0aXZlc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGRpcmVjdGl2ZXM/OiBEaXJlY3RpdmVbXTtcbn1cbi8qKlxuICogT2JqZWN0IFR5cGVzIGFyZSB0eXBlcyBkZWNsYXJlZCBieSB5b3UuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgT2JqZWN0VHlwZSBleHRlbmRzIEludGVyZmFjZVR5cGUge1xuICAgIC8qKlxuICAgICAqIEEgbWV0aG9kIHRvIGRlZmluZSBPYmplY3QgVHlwZXMgZnJvbSBhbiBpbnRlcmZhY2VcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGltcGxlbWVudEludGVyZmFjZShuYW1lOiBzdHJpbmcsIHByb3BzOiBPYmplY3RUeXBlUHJvcHMpOiBPYmplY3RUeXBlIHtcbiAgICAgICAgaWYgKCFwcm9wcy5pbnRlcmZhY2VUeXBlcyB8fCAhcHJvcHMuaW50ZXJmYWNlVHlwZXMubGVuZ3RoKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1N0YXRpYyBmdW5jdGlvbiBgaW1wbGVtZW50SW50ZXJmYWNlYCByZXF1aXJlcyBhbiBpbnRlcmZhY2VUeXBlIHRvIGltcGxlbWVudCcpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgT2JqZWN0VHlwZShuYW1lLCB7XG4gICAgICAgICAgICBpbnRlcmZhY2VUeXBlczogcHJvcHMuaW50ZXJmYWNlVHlwZXMsXG4gICAgICAgICAgICBkZWZpbml0aW9uOiBwcm9wcy5pbnRlcmZhY2VUeXBlcy5yZWR1Y2UoKGRlZiwgaW50ZXJmYWNlVHlwZSkgPT4ge1xuICAgICAgICAgICAgICAgIHJldHVybiBPYmplY3QuYXNzaWduKHt9LCBkZWYsIGludGVyZmFjZVR5cGUuZGVmaW5pdGlvbik7XG4gICAgICAgICAgICB9LCBwcm9wcy5kZWZpbml0aW9uKSxcbiAgICAgICAgICAgIGRpcmVjdGl2ZXM6IHByb3BzLmRpcmVjdGl2ZXMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgSW50ZXJmYWNlIFR5cGVzIHRoaXMgT2JqZWN0IFR5cGUgaW1wbGVtZW50c1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBpbnRlcmZhY2UgdHlwZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJmYWNlVHlwZXM/OiBJbnRlcmZhY2VUeXBlW107XG4gICAgLyoqXG4gICAgICogdGhlIGRpcmVjdGl2ZXMgZm9yIHRoaXMgb2JqZWN0IHR5cGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gZGlyZWN0aXZlc1xuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBkaXJlY3RpdmVzPzogRGlyZWN0aXZlW107XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZywgcHJvcHM6IE9iamVjdFR5cGVQcm9wcykge1xuICAgICAgICBzdXBlcihuYW1lLCBwcm9wcyk7XG4gICAgICAgIHRoaXMuaW50ZXJmYWNlVHlwZXMgPSBwcm9wcy5pbnRlcmZhY2VUeXBlcztcbiAgICAgICAgdGhpcy5kaXJlY3RpdmVzID0gcHJvcHMuZGlyZWN0aXZlcztcbiAgICB9XG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgdGhlIHN0cmluZyBvZiB0aGlzIG9iamVjdCB0eXBlXG4gICAgICovXG4gICAgcHVibGljIHRvU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgICAgIGxldCB0aXRsZSA9IHRoaXMubmFtZTtcbiAgICAgICAgaWYgKHRoaXMuaW50ZXJmYWNlVHlwZXMgJiYgdGhpcy5pbnRlcmZhY2VUeXBlcy5sZW5ndGgpIHtcbiAgICAgICAgICAgIHRpdGxlID0gYCR7dGl0bGV9IGltcGxlbWVudHNgO1xuICAgICAgICAgICAgdGhpcy5pbnRlcmZhY2VUeXBlcy5tYXAoKGludGVyZmFjZVR5cGUpID0+IHtcbiAgICAgICAgICAgICAgICB0aXRsZSA9IGAke3RpdGxlfSAke2ludGVyZmFjZVR5cGUubmFtZX0sYDtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgdGl0bGUgPSB0aXRsZS5zbGljZSgwLCAtMSk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgZGlyZWN0aXZlcyA9IHRoaXMuZ2VuZXJhdGVEaXJlY3RpdmVzKHRoaXMuZGlyZWN0aXZlcyk7XG4gICAgICAgIGxldCBzY2hlbWFBZGRpdGlvbiA9IGB0eXBlICR7dGl0bGV9ICR7ZGlyZWN0aXZlc317XFxuYDtcbiAgICAgICAgT2JqZWN0LmtleXModGhpcy5kZWZpbml0aW9uKS5mb3JFYWNoKChrZXkpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGF0dHJpYnV0ZSA9IHRoaXMuZGVmaW5pdGlvbltrZXldO1xuICAgICAgICAgICAgc2NoZW1hQWRkaXRpb24gPSBgJHtzY2hlbWFBZGRpdGlvbn0gICR7a2V5fTogJHthdHRyaWJ1dGUudG9TdHJpbmcoKX1cXG5gO1xuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIGAke3NjaGVtYUFkZGl0aW9ufX1gO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBVdGlsaXR5IGZ1bmN0aW9uIHRvIGdlbmVyYXRlIGRpcmVjdGl2ZXNcbiAgICAgKlxuICAgICAqIEBwYXJhbSBkaXJlY3RpdmVzIHRoZSBkaXJlY3RpdmVzIG9mIGEgZ2l2ZW4gdHlwZVxuICAgICAqIEBwYXJhbSBkZWxpbWl0ZXIgdGhlIHNlcGFyYXRvciBiZXR3ZWVlbiBkaXJlY3RpdmVzXG4gICAgICogQGRlZmF1bHQgLSAnICdcbiAgICAgKi9cbiAgICBwcml2YXRlIGdlbmVyYXRlRGlyZWN0aXZlcyhkaXJlY3RpdmVzPzogRGlyZWN0aXZlW10sIGRlbGltaXRlcj86IHN0cmluZyk6IHN0cmluZyB7XG4gICAgICAgIGxldCBzY2hlbWFBZGRpdGlvbiA9ICcnO1xuICAgICAgICBpZiAoIWRpcmVjdGl2ZXMpIHtcbiAgICAgICAgICAgIHJldHVybiBzY2hlbWFBZGRpdGlvbjtcbiAgICAgICAgfVxuICAgICAgICBkaXJlY3RpdmVzLm1hcCgoZGlyZWN0aXZlKSA9PiB7XG4gICAgICAgICAgICBzY2hlbWFBZGRpdGlvbiA9IGAke3NjaGVtYUFkZGl0aW9ufSR7ZGlyZWN0aXZlLnN0YXRlbWVudH0ke2RlbGltaXRlciA/PyAnICd9YDtcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiBzY2hlbWFBZGRpdGlvbjtcbiAgICB9XG59XG4iXX0=