import { BaseGraphqlTypeOptions, GraphqlType } from './schema-field';
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
export declare class Directive {
    /**
     * Add the @aws_iam directive
     */
    static iam(): Directive;
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     * Note: doesn't guarantee functionality
     */
    static custom(statement: string): Directive;
    /**
     * the directive statement
     */
    readonly statement: string;
    private constructor();
}
/**
 * Properties for configuring an Intermediate Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 *
 * @experimental
 */
export interface IntermediateTypeProps {
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: GraphqlType;
    };
}
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
export declare class InterfaceType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: GraphqlType;
    };
    constructor(name: string, props: IntermediateTypeProps);
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options?: BaseGraphqlTypeOptions): GraphqlType;
    /**
     * Generate the string of this object type
     */
    toString(): string;
}
/**
 * Properties for configuring an Object Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 * @param interfaceTypes - the interfaces that this object type implements
 * @param directives - the directives for this object type
 *
 * @experimental
 */
export interface ObjectTypeProps extends IntermediateTypeProps {
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
}
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
export declare class ObjectType extends InterfaceType {
    /**
     * A method to define Object Types from an interface
     */
    static implementInterface(name: string, props: ObjectTypeProps): ObjectType;
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
    constructor(name: string, props: ObjectTypeProps);
    /**
     * Generate the string of this object type
     */
    toString(): string;
    /**
     * Utility function to generate directives
     *
     * @param directives the directives of a given type
     * @param delimiter the separator betweeen directives
     * @default - ' '
     */
    private generateDirectives;
}
