"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlType = void 0;
const schema_base_1 = require("./schema-base");
/**
 * The GraphQL Types in AppSync's GraphQL. GraphQL Types are the
 * building blocks for object types, queries, mutations, etc. They are
 * types like String, Int, Id or even Object Types you create.
 *
 * i.e. `String`, `String!`, `[String]`, `[String!]`, `[String]!`
 *
 * GraphQL Types are used to define the entirety of schema.
 */
class GraphqlType {
    constructor(type, options) {
        var _a, _b, _c;
        this.type = type;
        this.isList = (_a = options === null || options === void 0 ? void 0 : options.isList) !== null && _a !== void 0 ? _a : false;
        this.isRequired = (_b = options === null || options === void 0 ? void 0 : options.isRequired) !== null && _b !== void 0 ? _b : false;
        this.isRequiredList = (_c = options === null || options === void 0 ? void 0 : options.isRequiredList) !== null && _c !== void 0 ? _c : false;
        this.intermediateType = options === null || options === void 0 ? void 0 : options.intermediateType;
    }
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static id(options) {
        return new GraphqlType(schema_base_1.Type.ID, options);
    }
    /**
     * `String` scalar type is a free-form human-readable text.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static string(options) {
        return new GraphqlType(schema_base_1.Type.STRING, options);
    }
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static int(options) {
        return new GraphqlType(schema_base_1.Type.INT, options);
    }
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static float(options) {
        return new GraphqlType(schema_base_1.Type.FLOAT, options);
    }
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static boolean(options) {
        return new GraphqlType(schema_base_1.Type.BOOLEAN, options);
    }
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDate(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE, options);
    }
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIME, options);
    }
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDateTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE_TIME, options);
    }
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTimestamp(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIMESTAMP, options);
    }
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsEmail(options) {
        return new GraphqlType(schema_base_1.Type.AWS_EMAIL, options);
    }
    /**
     * `AWSJson` scalar type represents a JSON string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsJson(options) {
        return new GraphqlType(schema_base_1.Type.AWS_JSON, options);
    }
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsUrl(options) {
        return new GraphqlType(schema_base_1.Type.AWS_URL, options);
    }
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsPhone(options) {
        return new GraphqlType(schema_base_1.Type.AWS_PHONE, options);
    }
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsIpAddress(options) {
        return new GraphqlType(schema_base_1.Type.AWS_IP_ADDRESS, options);
    }
    /**
     * an intermediate type to be added as an attribute
     * (i.e. an interface or an object type)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     * - intermediateType
     */
    static intermediate(options) {
        if (!(options === null || options === void 0 ? void 0 : options.intermediateType)) {
            throw new Error('GraphQL Type of interface must be configured with corresponding Intermediate Type');
        }
        return new GraphqlType(schema_base_1.Type.INTERMEDIATE, options);
    }
    /**
     * Generate the string for this attribute
     */
    toString() {
        var _a;
        // If an Object Type, we use the name of the Object Type
        let type = this.intermediateType ? (_a = this.intermediateType) === null || _a === void 0 ? void 0 : _a.name : this.type;
        // If configured as required, the GraphQL Type becomes required
        type = this.isRequired ? `${type}!` : type;
        // If configured with isXxxList, the GraphQL Type becomes a list
        type = this.isList || this.isRequiredList ? `[${type}]` : type;
        // If configured with isRequiredList, the list becomes required
        type = this.isRequiredList ? `${type}!` : type;
        return type;
    }
}
exports.GraphqlType = GraphqlType;
//# sourceMappingURL=data:application/json;base64,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