"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.IamResource = exports.SchemaDefinition = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const fs_1 = require("fs");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_intermediate_1 = require("./schema-intermediate");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * Enum containing the different modes of schema definition
 */
var SchemaDefinition;
(function (SchemaDefinition) {
    /**
     * Define schema through functions like addType, addQuery, etc.
     */
    SchemaDefinition["CODE"] = "CODE";
    /**
     * Define schema in a file, i.e. schema.graphql
     */
    SchemaDefinition["FILE"] = "FILE";
})(SchemaDefinition = exports.SchemaDefinition || (exports.SchemaDefinition = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphQLApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schemaMode = props.schemaDefinition;
        if (defaultAuthorizationType === AuthorizationType.API_KEY || ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.some((authMode) => authMode.authorizationType === AuthorizationType.API_KEY))) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this._apiKey = this.createAPIKey(apiKeyConfig);
        }
        this.schema = this.defineSchema(props.schemaDefinitionFile);
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schema);
        return true;
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        return key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
    /**
     * Define schema based on props configuration
     * @param file the file name/s3 location of Schema
     */
    defineSchema(file) {
        let definition;
        if (this.schemaMode === SchemaDefinition.FILE && !file) {
            throw new Error('schemaDefinitionFile must be configured if using FILE definition mode.');
        }
        else if (this.schemaMode === SchemaDefinition.FILE && file) {
            definition = fs_1.readFileSync(file).toString('UTF-8');
        }
        else if (this.schemaMode === SchemaDefinition.CODE && !file) {
            definition = '';
        }
        else if (this.schemaMode === SchemaDefinition.CODE && file) {
            throw new Error('definition mode CODE is incompatible with file definition. Change mode to FILE/S3 or unconfigure schemaDefinitionFile');
        }
        return new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    appendToSchema(addition, delimiter) {
        if (this.schemaMode != SchemaDefinition.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.schema.definition = `${this.schema.definition}${sep}${addition}\n`;
    }
    /**
     * Add an object type to the schema
     *
     * @param name the name of the object type
     * @param props the definition
     *
     * @experimental
     */
    addType(name, props) {
        if (this.schemaMode != SchemaDefinition.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        ;
        const type = new schema_intermediate_1.ObjectType(name, {
            definition: props.definition,
            directives: props.directives,
        });
        this.appendToSchema(type.toString());
        return type;
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,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