import { IUserPool } from "../../aws-cognito";
import { Grant, IGrantable } from "../../aws-iam";
import { CfnResource, Construct, IResolvable } from "../../core";
import { CfnGraphQLSchema } from './appsync.generated';
import { IGraphqlApi, GraphqlApiBase } from './graphqlapi-base';
import { ObjectType, ObjectTypeProps } from './schema-intermediate';
/**
 * enum with all possible values for AppSync authorization type
 */
export declare enum AuthorizationType {
    /**
     * API Key authorization type
     */
    API_KEY = "API_KEY",
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    IAM = "AWS_IAM",
    /**
     * Cognito User Pool authorization type
     */
    USER_POOL = "AMAZON_COGNITO_USER_POOLS",
    /**
     * OpenID Connect authorization type
     */
    OIDC = "OPENID_CONNECT"
}
/**
 * Interface to specify default or additional authorization(s)
 */
export interface AuthorizationMode {
    /**
     * One of possible four values AppSync supports
     *
     * @see https://docs.aws.amazon.com/appsync/latest/devguide/security.html
     *
     * @default - `AuthorizationType.API_KEY`
     */
    readonly authorizationType: AuthorizationType;
    /**
     * If authorizationType is `AuthorizationType.USER_POOL`, this option is required.
     * @default - none
     */
    readonly userPoolConfig?: UserPoolConfig;
    /**
     * If authorizationType is `AuthorizationType.API_KEY`, this option can be configured.
     * @default - name: 'DefaultAPIKey' | description: 'Default API Key created by CDK'
     */
    readonly apiKeyConfig?: ApiKeyConfig;
    /**
     * If authorizationType is `AuthorizationType.OIDC`, this option is required.
     * @default - none
     */
    readonly openIdConnectConfig?: OpenIdConnectConfig;
}
/**
 * enum with all possible values for Cognito user-pool default actions
 */
export declare enum UserPoolDefaultAction {
    /**
     * ALLOW access to API
     */
    ALLOW = "ALLOW",
    /**
     * DENY access to API
     */
    DENY = "DENY"
}
/**
 * Configuration for Cognito user-pools in AppSync
 */
export interface UserPoolConfig {
    /**
     * The Cognito user pool to use as identity source
     */
    readonly userPool: IUserPool;
    /**
     * the optional app id regex
     *
     * @default -  None
     */
    readonly appIdClientRegex?: string;
    /**
     * Default auth action
     *
     * @default ALLOW
     */
    readonly defaultAction?: UserPoolDefaultAction;
}
/**
 * Configuration for API Key authorization in AppSync
 */
export interface ApiKeyConfig {
    /**
     * Unique name of the API Key
     * @default - 'DefaultAPIKey'
     */
    readonly name?: string;
    /**
     * Description of API key
     * @default - 'Default API Key created by CDK'
     */
    readonly description?: string;
    /**
     * The time from creation time after which the API key expires, using RFC3339 representation.
     * It must be a minimum of 1 day and a maximum of 365 days from date of creation.
     * Rounded down to the nearest hour.
     * @default - 7 days from creation time
     */
    readonly expires?: string;
}
/**
 * Configuration for OpenID Connect authorization in AppSync
 */
export interface OpenIdConnectConfig {
    /**
     * The number of milliseconds an OIDC token is valid after being authenticated by OIDC provider.
     * `auth_time` claim in OIDC token is required for this validation to work.
     * @default - no validation
     */
    readonly tokenExpiryFromAuth?: number;
    /**
     * The number of milliseconds an OIDC token is valid after being issued to a user.
     * This validation uses `iat` claim of OIDC token.
     * @default - no validation
     */
    readonly tokenExpiryFromIssue?: number;
    /**
     * The client identifier of the Relying party at the OpenID identity provider.
     * A regular expression can be specified so AppSync can validate against multiple client identifiers at a time.
     * @example - 'ABCD|CDEF' where ABCD and CDEF are two different clientId
     * @default - * (All)
     */
    readonly clientId?: string;
    /**
     * The issuer for the OIDC configuration. The issuer returned by discovery must exactly match the value of `iss` in the OIDC token.
     */
    readonly oidcProvider: string;
}
/**
 * Configuration of the API authorization modes.
 */
export interface AuthorizationConfig {
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly defaultAuthorization?: AuthorizationMode;
    /**
     * Additional authorization modes
     *
     * @default - No other modes
     */
    readonly additionalAuthorizationModes?: AuthorizationMode[];
}
/**
 * log-level for fields in AppSync
 */
export declare enum FieldLogLevel {
    /**
     * No logging
     */
    NONE = "NONE",
    /**
     * Error logging
     */
    ERROR = "ERROR",
    /**
     * All logging
     */
    ALL = "ALL"
}
/**
 * Logging configuration for AppSync
 */
export interface LogConfig {
    /**
     * exclude verbose content
     *
     * @default false
     */
    readonly excludeVerboseContent?: boolean | IResolvable;
    /**
     * log level for fields
     *
     * @default - Use AppSync default
     */
    readonly fieldLogLevel?: FieldLogLevel;
}
/**
 * Enum containing the different modes of schema definition
 */
export declare enum SchemaDefinition {
    /**
     * Define schema through functions like addType, addQuery, etc.
     */
    CODE = "CODE",
    /**
     * Define schema in a file, i.e. schema.graphql
     */
    FILE = "FILE"
}
/**
 * Properties for an AppSync GraphQL API
 */
export interface GraphQLApiProps {
    /**
     * the name of the GraphQL API
     */
    readonly name: string;
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly authorizationConfig?: AuthorizationConfig;
    /**
     * Logging configuration for this api
     *
     * @default - None
     */
    readonly logConfig?: LogConfig;
    /**
     * GraphQL schema definition. Specify how you want to define your schema.
     *
     * SchemaDefinition.CODE allows schema definition through CDK
     * SchemaDefinition.FILE allows schema definition through schema.graphql file
     *
     * @experimental
     */
    readonly schemaDefinition: SchemaDefinition;
    /**
     * File containing the GraphQL schema definition. You have to specify a definition or a file containing one.
     *
     * @default - Use schemaDefinition
     */
    readonly schemaDefinitionFile?: string;
    /**
     * A flag indicating whether or not X-Ray tracing is enabled for the GraphQL API.
     *
     * @default - false
     */
    readonly xrayEnabled?: boolean;
}
/**
 * A class used to generate resource arns for AppSync
 */
export declare class IamResource {
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns: string[]): IamResource;
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type: string, ...fields: string[]): IamResource;
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all(): IamResource;
    private arns;
    private constructor();
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api: GraphQLApi): string[];
}
/**
 * Attributes for GraphQL imports
 */
export interface GraphqlApiAttributes {
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     */
    readonly graphqlApiId: string;
    /**
     * the arn for the GraphQL Api
     * @default - autogenerated arn
     */
    readonly graphqlApiArn?: string;
}
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
export declare class GraphQLApi extends GraphqlApiBase {
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope: Construct, id: string, attrs: GraphqlApiAttributes): IGraphqlApi;
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     */
    readonly apiId: string;
    /**
     * the ARN of the API
     */
    readonly arn: string;
    /**
     * the URL of the endpoint created by AppSync
     *
     * @attribute
     */
    readonly graphQlUrl: string;
    /**
     * the name of the API
     */
    readonly name: string;
    /**
     * underlying CFN schema resource
     */
    readonly schema: CfnGraphQLSchema;
    /**
     * the configured API key, if present
     */
    get apiKey(): string | undefined;
    private schemaMode;
    private api;
    private _apiKey?;
    constructor(scope: Construct, id: string, props: GraphQLApiProps);
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee: IGrantable, resources: IamResource, ...actions: string[]): Grant;
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee: IGrantable, ...fields: string[]): Grant;
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee: IGrantable, ...fields: string[]): Grant;
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee: IGrantable, ...fields: string[]): Grant;
    private validateAuthorizationProps;
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct: CfnResource): boolean;
    private formatOpenIdConnectConfig;
    private formatUserPoolConfig;
    private createAPIKey;
    private formatAdditionalAuthorizationModes;
    private formatAdditionalAuthenticationProviders;
    /**
     * Define schema based on props configuration
     * @param file the file name/s3 location of Schema
     */
    private defineSchema;
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    appendToSchema(addition: string, delimiter?: string): void;
    /**
     * Add an object type to the schema
     *
     * @param name the name of the object type
     * @param props the definition
     *
     * @experimental
     */
    addType(name: string, props: ObjectTypeProps): ObjectType;
}
