"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryEncryption = exports.TagMutability = exports.Repository = exports.RepositoryBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        return this.repositoryUriWithSuffix(tagSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest
     */
    repositoryUriForDigest(digest) {
        const digestSuffix = digest ? `@${digest}` : '';
        return this.repositoryUriWithSuffix(digestSuffix);
    }
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    repositoryUriWithSuffix(suffix) {
        const parts = this.stack.splitArn(this.repositoryArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${suffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_OnCloudTrailImagePushedOptions(options);
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed
     *
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onImageScanCompleted(id, options = {}) {
        var _d;
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_OnImageScanCompletedOptions(options);
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': (_d = options.imageTags) !== null && _d !== void 0 ? _d : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
_a = JSII_RTTI_SYMBOL_1;
RepositoryBase[_a] = { fqn: "@aws-cdk/aws-ecr.RepositoryBase", version: "1.151.0" };
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_RepositoryProps(props);
        Repository.validateRepositoryName(this.physicalName);
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.any({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.any({ produce: () => this.renderLifecyclePolicy() }),
            imageScanningConfiguration: props.imageScanOnPush ? { scanOnPush: true } : { scanOnPush: false },
            imageTagMutability: props.imageTagMutability || undefined,
            encryptionConfiguration: this.parseEncryption(props),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_RepositoryAttributes(attrs);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id, {
            environmentFromArn: repositoryArn,
        });
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    static validateRepositoryName(physicalName) {
        const repositoryName = physicalName;
        if (!repositoryName || core_1.Token.isUnresolved(repositoryName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecr-repository.html
        if (repositoryName.length < 2 || repositoryName.length > 256) {
            errors.push('Repository name must be at least 2 and no more than 256 characters');
        }
        const isPatternMatch = /^(?:[a-z0-9]+(?:[._-][a-z0-9]+)*\/)*[a-z0-9]+(?:[._-][a-z0-9]+)*$/.test(repositoryName);
        if (!isPatternMatch) {
            errors.push('Repository name must follow the specified pattern: (?:[a-z0-9]+(?:[._-][a-z0-9]+)*/)*[a-z0-9]+(?:[._-][a-z0-9]+)*');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid ECR repository name (value: ${repositoryName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policyDocument) === null || _d === void 0 ? void 0 : _d.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_LifecycleRule(rule);
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        var _d;
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: (_d = rule.rulePriority) !== null && _d !== void 0 ? _d : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
    /**
     * Set up key properties and return the Repository encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _d, _e;
        // default based on whether encryptionKey is specified
        const encryptionType = (_d = props.encryption) !== null && _d !== void 0 ? _d : (props.encryptionKey ? RepositoryEncryption.KMS : RepositoryEncryption.AES_256);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== RepositoryEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType.value})`);
        }
        if (encryptionType === RepositoryEncryption.AES_256) {
            return undefined;
        }
        if (encryptionType === RepositoryEncryption.KMS) {
            return {
                encryptionType: 'KMS',
                kmsKey: (_e = props.encryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Repository = Repository;
_b = JSII_RTTI_SYMBOL_1;
Repository[_b] = { fqn: "@aws-cdk/aws-ecr.Repository", version: "1.151.0" };
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    var _d, _e;
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: (_e = (_d = rule.maxImageAge) === null || _d === void 0 ? void 0 : _d.toDays()) !== null && _e !== void 0 ? _e : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
/**
 * The tag mutability setting for your repository.
 */
var TagMutability;
(function (TagMutability) {
    /**
     * allow image tags to be overwritten.
     */
    TagMutability["MUTABLE"] = "MUTABLE";
    /**
     * all image tags within the repository will be immutable which will prevent them from being overwritten.
     */
    TagMutability["IMMUTABLE"] = "IMMUTABLE";
})(TagMutability = exports.TagMutability || (exports.TagMutability = {}));
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class RepositoryEncryption {
    /**
     * @param value the string value of the encryption
     */
    constructor(value) {
        this.value = value;
    }
}
exports.RepositoryEncryption = RepositoryEncryption;
_c = JSII_RTTI_SYMBOL_1;
RepositoryEncryption[_c] = { fqn: "@aws-cdk/aws-ecr.RepositoryEncryption", version: "1.151.0" };
/**
 * 'AES256'
 */
RepositoryEncryption.AES_256 = new RepositoryEncryption('AES256');
/**
 * 'KMS'
 */
RepositoryEncryption.KMS = new RepositoryEncryption('KMS');
//# sourceMappingURL=data:application/json;base64,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