import ctypes
import datetime
import typing
from dataclasses import dataclass

import cbor2

from ._core import VerificationStatus
from ._core import verify as _verify


@dataclass
class EnclaveReport:
    cpu_svn: bytes  # bytes of length 16
    misc_select: ctypes.c_uint32
    reserved1: bytes  # bytes of length 28
    attributes: bytes  # # bytes of length 16
    mr_enclave: bytes  # bytes of length 32
    reserved2: bytes  # bytes of length 32
    mr_signer: bytes  # bytes of length 32
    reserved3: bytes  # bytes of length 96
    isv_prod_id: ctypes.c_uint16
    isv_svn: ctypes.c_uint16
    reserved4: bytes  # bytes of length 60
    report_data: bytes  # bytes of length 64


@dataclass
class VerificationResult:
    ok: bool
    pck_certificate_status: VerificationStatus
    tcb_info_status: VerificationStatus
    qe_identity_status: VerificationStatus
    quote_status: VerificationStatus
    enclave_report: typing.Optional[EnclaveReport]


def convert_enclave_report(enclave_report_dict):
    if enclave_report_dict is None:
        return None
    enclave_report = EnclaveReport(
        cpu_svn=bytes(enclave_report_dict["cpuSvn"]),
        misc_select=ctypes.c_uint32(enclave_report_dict["miscSelect"]),
        reserved1=bytes(enclave_report_dict["reserved1"]),
        attributes=bytes(enclave_report_dict["attributes"]),
        mr_enclave=bytes(enclave_report_dict["mrEnclave"]),
        reserved2=bytes(enclave_report_dict["reserved2"]),
        mr_signer=bytes(enclave_report_dict["mrSigner"]),
        reserved3=bytes(enclave_report_dict["reserved3"]),
        isv_prod_id=ctypes.c_uint16(enclave_report_dict["isvProdID"]),
        isv_svn=ctypes.c_uint16(enclave_report_dict["isvSvn"]),
        reserved4=bytes(enclave_report_dict["reserved4"]),
        report_data=bytes(enclave_report_dict["reportData"]),
    )
    return enclave_report


def int_to_status(i_status):
    status = VerificationStatus(i_status)
    if status not in VerificationStatus.__members__.values():
        raise ValueError(f"Unexpected enum value {i_status} for VerificationStatus")
    return status


def verify(
    trusted_root_ca_certificate: str,
    pck_certificate: str,
    pck_signing_chain: str,
    root_ca_crl: str,
    intermediate_ca_crl: str,
    tcb_info: str,
    tcb_signing_chain: str,
    quote: bytes,
    qe_identity: str,
    expiration_date: datetime.datetime,
) -> VerificationResult:
    """Verify a ECDSA-based quote generated by the Intel provided Quoting
    Enclave.

    The verification is done with the SGX Quote Verification Library
    (QVL) provided by Intel, which is the reference implementation of
    ECDSA-based SGX quote verification. It requires providing Intel-
    issued Provisioning Certification Key (PCK) Certificate Chain,
    Revocation Lists, Trusted Computing Base (TCB) Information and
    Quoting Enclave Identity corresponding to the platform that is
    attested.
    """
    verification_result_dict = cbor2.loads(
        _verify(
            trusted_root_ca_certificate=trusted_root_ca_certificate,
            pck_certificate=pck_certificate,
            pck_signing_chain=pck_signing_chain,
            root_ca_crl=root_ca_crl,
            intermediate_ca_crl=intermediate_ca_crl,
            tcb_info=tcb_info,
            tcb_signing_chain=tcb_signing_chain,
            quote=quote,
            qe_identity=qe_identity,
            expiration_date=expiration_date,
        )
    )

    return VerificationResult(
        ok=verification_result_dict["ok"],
        pck_certificate_status=int_to_status(
            verification_result_dict["pckCertificateStatus"]
        ),
        tcb_info_status=int_to_status(verification_result_dict["tcbInfoStatus"]),
        qe_identity_status=int_to_status(verification_result_dict["qeIdentityStatus"]),
        quote_status=int_to_status(verification_result_dict["quoteStatus"]),
        enclave_report=convert_enclave_report(
            verification_result_dict["enclaveReport"]
        ),
    )
