# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00d_doc_utils.ipynb (unless otherwise specified).

__all__ = ['nbglob', 'hide', 'collapse_cells']

# Internal Cell
# pylint: skip-file
import os
from fastcore.docments import isclass

# Internal Cell
def is_building_docs() -> bool:
    return 'DOCUTILSCONFIG' in os.environ

# Internal Cell
import inspect
from nbdev.imports import get_config
from nbdev.sync import qual_name, get_name
from fastcore.script import Param
from fastcore.utils import IN_NOTEBOOK
from nbdev.showdoc import (
    is_enum,
    _format_enum_doc,
    get_source_link,
    add_doc_links,
    is_source_available,
    _has_docment,
    _arg_prefixes,
    get_nb_source_link,
    _format_annos,
    _handle_delegates,
    _get_docments
)

# Internal Cell
if IN_NOTEBOOK:
    from IPython.display import Markdown, display

# Internal Cell
def doc_link(name, include_bt=True):
    "Create link to documentation for `name`."
    cname = f'`{name}`' if include_bt else name
    try:
        link = get_nb_source_link(name)
        return f'[{cname}]({link})'
    except Exception:
        return cname

# Internal Cell
def type_repr(t):
    "Representation of type `t` (in a type annotation)"
    if (isinstance(t, Param)):
        return f'"{t.help}"'
    if getattr(t, '__args__', None):
        args = t.__args__
        if len(args) == 2 and args[1] == type(None):  # noqa: E721
            return f'`Optional`\[{type_repr(args[0])}\]'  # noqa: W605
        reprs = ', '.join([_format_annos(o, highlight=True) for o in args])
        return f'{doc_link(get_name(t))}\[{reprs}\]'  # noqa: W605
    else:
        return doc_link(_format_annos(t))

# Internal Cell
def format_param(p):
    "Formats function param to `param:Type=val` with font weights: param=bold, val=italic"
    arg_prefix = _arg_prefixes.get(p.kind, '')  # asterisk prefix for *args and **kwargs
    res = f"**{arg_prefix}`{p.name}`**"
    if hasattr(p, 'annotation') and p.annotation != p.empty:
        res += f':{type_repr(p.annotation)}'
    if p.default != p.empty:
        default = getattr(p.default, 'func', p.default)  # For partials
        if hasattr(default, '__name__'):
            default = getattr(default, '__name__')
        else:
            default = repr(default)
        if is_enum(default.__class__):  # Enum have a crappy repr
            res += f'=*`{default.__class__.__name__}.{default.name}`*'
        else:
            res += f'=*`{default}`*'
    return res

# Internal Cell
def _format_func_doc(func, full_name=None):
    "Formatted `func` definition to show in documentation"
    try:
        sig = inspect.signature(func)
        fmt_params = [format_param(param) for name, param
                      in sig.parameters.items() if name not in ('self', 'cls')]
    except Exception as e:
        print(f'exception: {e}')
        fmt_params = []
    name = f'<code>{full_name or func.__name__}</code>'
    arg_str = f"({', '.join(fmt_params)})"
    f_name = f"<code>class</code> {name}" if inspect.isclass(func) else name
    return f'{f_name}', f'{name}{arg_str}'

# Internal Cell
def _format_cls_doc(cls, full_name):
    "Formatted `cls` definition to show in documentation"
    parent_class = inspect.getclasstree([cls])[-1][0][1][0]
    name, args = _format_func_doc(cls, full_name)
    if parent_class != object:
        args += f' :: {doc_link(get_name(parent_class))}'
    return name, args

# Internal Cell
def show_doc(
    elt,
    doc_string: bool = True,
    name=None,
    title_level=None,
    disp=True,
    default_cls_level=2,
    show_all_docments=False,
    verbose=False
):
    """
    Show documentation for element `elt` with potential input
    documentation. Supported types: class, function, and enum.
    """
    elt = getattr(elt, '__func__', elt)
    qname = name or qual_name(elt)
    is_class = '.' in qname or inspect.isclass
    if inspect.isclass(elt):
        if is_enum(elt):
            name, args = _format_enum_doc(elt, qname)
        else:
            name, args = _format_cls_doc(elt, qname)
    elif callable(elt):
        name, args = _format_func_doc(elt, qname)
    else:
        name, args = f"<code>{qname}</code>", ''
    link = get_source_link(elt)
    source_link = f'<a href="{link}" class="source_link" style="float:right">[source]</a>'
    title_level = title_level or (default_cls_level if inspect.isclass(elt) else 4)
    doc = f'<h{title_level} id="{qname}" class="doc_header">{name}{source_link}</h{title_level}>'
    doc += f'\n\n> {args}\n\n' if len(args) > 0 else '\n\n'
    s = ''
    try:
        monospace = get_config().d.getboolean('monospace_docstrings', False)
    except FileNotFoundError:
        monospace = False

    if doc_string and inspect.getdoc(elt):
        s = inspect.getdoc(elt)  # type: ignore
        # doc links don't work inside markdown pre/code blocks
        s = f'```\n{s}\n```' if monospace else add_doc_links(s, elt)
        doc += s

    if len(args) > 0:
        if hasattr(elt, '__init__') and isclass(elt):
            elt = elt.__init__
        if is_source_available(elt):
            if show_all_docments or _has_docment(elt):
                if hasattr(elt, "__delwrap__"):
                    arg_dict, kwargs = _handle_delegates(elt)
                    doc += _get_docments(
                        elt,
                        ment_dict=arg_dict,
                        with_return=True,
                        kwargs=kwargs,
                        monospace=monospace,
                        is_class=is_class
                    )
                else:
                    doc += _get_docments(elt, monospace=monospace, is_class=is_class)
            elif verbose:
                print(f"""Warning: `docments` annotations will not work for
                built-in modules, classes, functions, and `enums` and are
                unavailable for {qual_name(elt)}. They will not be shown""")
    if disp:
        display(Markdown(doc))
    else:
        return doc

# Internal Cell
import glob
from fastcore.all import L, compose, Path
from nbdev.export2html import _mk_flag_re, _re_cell_to_collapse_output, check_re
from nbdev.export import check_re_multi
import nbformat as nbf

# Cell
def nbglob(fname='.', recursive=False, extension='.ipynb') -> L:
    """Find all files in a directory matching an extension.
    Ignores hidden directories and filenames starting with `_`"""
    fname = Path(fname)
    if fname.is_dir():
        abs_name = fname.absolute()
        rec_path = f'{abs_name}/**/*{extension}'
        non_rec_path = f'{abs_name}/*{extension}'
        fname = rec_path if recursive else non_rec_path
    fls = L(
        glob.glob(str(fname), recursive=recursive)
    ).filter(
        lambda x: '/.' not in x
    ).map(Path)
    return fls.filter(lambda x: not x.name.startswith('_') and x.name.endswith(extension))

# Internal Cell
def upd_metadata(cell, tag):
    cell_tags = list(set(cell.get('metadata', {}).get('tags', [])))
    if tag not in cell_tags:
        cell_tags.append(tag)
    cell['metadata']['tags'] = cell_tags

# Cell
def hide(cell):
    """Hide inputs of `cell` that need to be hidden
    if check_re_multi(cell, [_re_show_doc, *_re_hide_input]): upd_metadata(cell, 'remove-input')
    elif check_re(cell, _re_hide_output): upd_metadata(cell, 'remove-output')
    """
    regexes = ['#(.+|)hide', '%%ipytest']
    if check_re_multi(cell, regexes):
        upd_metadata(cell, 'remove-cell')

    regexes = ['#(.+|)exporti']
    if check_re_multi(cell, regexes):
        upd_metadata(cell, 'remove-input')

    return cell


_re_cell_to_collapse_input = _mk_flag_re(
    '(collapse_input|collapse-input)', 0, "Cell with #collapse_input")


def collapse_cells(cell):
    "Add a collapse button to inputs or outputs of `cell` in either the open or closed position"
    if check_re(cell, _re_cell_to_collapse_input):
        upd_metadata(cell, 'hide-input')
    elif check_re(cell, _re_cell_to_collapse_output):
        upd_metadata(cell, 'hide-output')
    return cell

# Internal Cell
if __name__ == '__main__':

    _func = compose(hide, collapse_cells)
    files = nbglob('nbs/')

    for file in files:
        nb = nbf.read(file, nbf.NO_CONVERT)
        for c in nb.cells:
            _func(c)
        nbf.write(nb, file)