/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.util;

import mt.Matrix;
import mt.MatrixEntry;

/**
 * Matrix utility functions
 */
public final class MatrixUtil {

    private MatrixUtil() {
        // No need to instantiate
    }

    /**
     * Converts the matrix into an array
     */
    public static double[][] toArray(Matrix A) {
        double[][] values = new double[A.numRows()][A.numColumns()];
        for (MatrixEntry e : A)
            values[e.row()][e.column()] = e.get();
        return values;
    }

    /**
     * Finds the bandwidth of the matrix. Useful for converting a general matrix
     * into a banded matrix
     * 
     * @return An int[2] array. First component is the number of diagonals below
     *         the main diagonal, while the second component is the number of
     *         diagonal above the main diagonal
     */
    public static int[] getBounds(Matrix A) {
        int kl = 0, ku = 0;

        for (MatrixEntry e : A) {
            kl = Math.max(kl, e.row() - e.column());
            ku = Math.max(ku, e.column() - e.row());
        }

        return new int[] { kl, ku };
    }

    /**
     * Finds the number of non-zero entries on each row
     */
    public static int[] rowBandwidth(Matrix A) {
        int[] nz = new int[A.numRows()];

        for (MatrixEntry e : A)
            nz[e.row()]++;

        return nz;
    }

    /**
     * Finds the number of repeated entries
     * 
     * @param num
     *            Maximum index value
     * @param ind
     *            Indices to check for repetitions
     * @return Array of length <code>num</code> with the number of repeated
     *         indices of <code>ind</code>
     */
    public static int[] bandwidth(int num, int[] ind) {
        int[] nz = new int[num];

        for (int i = 0; i < ind.length; ++i)
            nz[ind[i]]++;

        return nz;
    }

    /**
     * Finds the number of non-zero entries on each column
     */
    public static int[] columnBandwidth(Matrix A) {
        int[] nz = new int[A.numColumns()];

        for (MatrixEntry e : A)
            nz[e.column()]++;

        return nz;
    }

    /**
     * Sets the selected rows of <code>A</code> equal zero, and puts
     * <code>diagonal</code> on the diagonal of those rows
     */
    public static void zeroRows(Matrix A, int[] row, double diagonal) {
        // Sort the rows
        int[] rowS = row.clone();
        java.util.Arrays.sort(rowS);

        for (MatrixEntry e : A) {
            int j = java.util.Arrays.binarySearch(rowS, e.row());
            if (j >= 0) { // Found
                if (e.row() == e.column()) // Diagonal
                    e.set(diagonal);
                else
                    // Off diagonal
                    e.set(0);
            }
        }

        // Ensure the diagonal is set. This is necessary in case of missing
        // rows
        if (diagonal != 0)
            for (int rowI : row)
                A.set(rowI, rowI, diagonal);
    }

    /**
     * Sets the selected columns of <code>A</code> equal zero, and puts
     * <code>diagonal</code> on the diagonal of those columns
     */
    public static void zeroColumns(Matrix A, int[] column, double diagonal) {
        // Sort the rows
        int[] columnS = column.clone();
        java.util.Arrays.sort(columnS);

        for (MatrixEntry e : A) {
            int j = java.util.Arrays.binarySearch(columnS, e.column());
            if (j >= 0) { // Found
                if (e.row() == e.column()) // Diagonal
                    e.set(diagonal);
                else
                    // Off diagonal
                    e.set(0);
            }
        }

        // Ensure the diagonal is set. This is necessary in case of missing
        // columns
        if (diagonal != 0)
            for (int columnI : column)
                A.set(columnI, columnI, diagonal);
    }

}
