/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.Matrix;
import mt.Vector;
import mt.Vector.Norm;

/**
 * Power iteration eigenvalue solver. Using
 * {@link smt.iter.eig.NoEigenvalueTransformation NoEigenvalueTransformation}
 * gives the Power method,
 * {@link smt.iter.eig.LockShiftInvertEigenvalueTransformation LockShiftInvertEigenvalueTransformation}
 * will produce the Inverse iteration, and
 * {@link smt.iter.eig.ShiftInvertEigenvalueTransformation ShiftInvertEigenvalueTransformation}
 * gives the Rayleigh quotient iteration.
 */
public class PowerIteration extends AbstractIterativeEigenvalueSolver {

    /**
     * Temporary work-vectors
     */
    private Vector r, Ax;

    /**
     * Constructor for PowerIteration
     * 
     * @param template
     *            Template for creating work-vectors. Must have the same size as
     *            the eigenvector
     */
    public PowerIteration(Vector template) {
        r = template.copy().zero();
        Ax = template.copy().zero();
    }

    protected double solveI(Matrix A, double v, Vector x)
            throws IterativeEigenvalueSolverNotConvergedException {

        randomize(r);

        for (iter.setFirst(); !iter.converged(r, v, x); iter.next()) {

            // Improve on the eigenvector, and compute the eigenvalue
            x.scale(1. / x.norm(Norm.Two));
            et.setShift(v);
            et.apply(A, x, Ax);
            v = x.dot(Ax);

            // Compute residual, and cycle vectors
            r.set(Ax, -v, x);
            x.set(Ax);
        }

        // Untransform the eigenvalue
        return et.eigenvalue(v);
    }

}
