/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter;

import mt.AbstractMatrix;
import mt.Matrix;
import mt.Vector;
import smt.iter.prec.IdentityPreconditioner;
import smt.iter.prec.Preconditioner;

/**
 * Partial implementation of IterativeSolver
 */
public abstract class AbstractIterativeSolver implements IterativeSolver {

    /**
     * Preconditioner to use. In support of transpose operations, this is not
     * the real preconditioner, merely a wrapper
     */
    protected Preconditioner M;

    /**
     * The actual preconditioner. Not used in solvers except thru a wrapper
     */
    protected Preconditioner Mp;

    /**
     * Iteration monitor
     */
    protected IterationMonitor iter;

    /**
     * Constructor for AbstractIterativeSolver. Does not use preconditioning,
     * and uses the default linear iteration object.
     */
    public AbstractIterativeSolver() {
        Mp = new IdentityPreconditioner();
        iter = new DefaultIterationMonitor();
    }

    public void setPreconditioner(Preconditioner M) {
        this.Mp = M;
    }

    public Preconditioner getPreconditioner() {
        return Mp;
    }

    public IterationMonitor getIterationMonitor() {
        return iter;
    }

    public void setIterationMonitor(IterationMonitor iter) {
        this.iter = iter;
    }

    public Vector solve(Matrix A, Vector b, Vector x)
            throws IterativeSolverNotConvergedException {
        checkSizes(A, b, x);
        M = Mp;
        solveI(A, b, x);
        return x;
    }

    public Vector transSolve(Matrix A, Vector b, Vector x)
            throws IterativeSolverNotConvergedException {
        checkSizes(A, b, x);
        M = new TransposedPreconditioner(Mp);
        solveI(new TransposedMatrix(A), b, x);
        return x;
    }

    /**
     * Solves the (non-transposed) system
     */
    protected abstract void solveI(Matrix A, Vector b, Vector x)
            throws IterativeSolverNotConvergedException;

    /**
     * Checks sizes of input data for <code>solve</code>. Throws an exception
     * if the sizes does not match.
     */
    private void checkSizes(Matrix A, Vector b, Vector x) {
        if (!A.isSquare())
            throw new IllegalArgumentException("!A.isSquare()");
        if (b.size() != A.numRows())
            throw new IllegalArgumentException("b.size() != A.numRows()");
        if (b.size() != x.size())
            throw new IllegalArgumentException("b.size() != x.size()");
    }

    /**
     * Changes matrix/vector products such that the matrix appears to be
     * transposed to an iterative solver
     */
    private class TransposedMatrix extends AbstractMatrix {

        private static final long serialVersionUID = 3258411724876166200L;

        private Matrix A;

        public TransposedMatrix(Matrix A) {
            super(A);
            this.A = A;
        }

        public Vector multAdd(double alpha, Vector x, double beta, Vector y,
                Vector z) {
            return A.transMultAdd(alpha, x, beta, y, z);
        }

        public Vector transMultAdd(double alpha, Vector x, double beta,
                Vector y, Vector z) {
            return A.multAdd(alpha, x, beta, y, z);
        }

    }

    /**
     * Changes a preconditioner to work on a transposed matrix transparently
     */
    class TransposedPreconditioner implements Preconditioner {

        private Preconditioner m;

        public TransposedPreconditioner(Preconditioner M) {
            this.m = M;
        }

        public Vector apply(Vector b, Vector x) {
            return m.transApply(b, x);
        }

        public Vector transApply(Vector b, Vector x) {
            return m.apply(b, x);
        }

        public void setMatrix(Matrix A) {
            m.setMatrix(A);
        }

    }
}
