/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import java.io.IOException;
import java.util.Iterator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import mt.AbstractMatrix;
import mt.Matrix;
import mt.MatrixEntry;
import mt.Vector;
import mt.VectorEntry;
import smt.util.DaemonThreadFactory;
import smt.util.SuperIterator;
import smt.util.SuperIterator.SuperIteratorEntry;

/**
 * Matrix stored row-wise into vectors
 */
abstract class AbstractRowMatrix extends AbstractMatrix {

    /**
     * Matrix data
     */
    Vector[] rowD;

    /**
     * Partition of rows for parallel operations
     */
    private final int[] part;

    /**
     * For shared memory parallel operations
     */
    private transient ExecutorService executor;

    /**
     * Number of threads to use in parallel operations
     */
    private int numThreads;

    /**
     * Constructor for AbstractRowMatrix
     * 
     * @param numRows
     *            Number of rows
     * @param numColumns
     *            Number of column
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     * @param numThreads
     *            Number of threads to use in parallelization
     */
    AbstractRowMatrix(int numRows, int numColumns, Vector template,
            int numThreads) {
        super(numRows, numColumns);
        construct(numThreads);
        part = smt.util.Arrays.partition(0, numRows, numThreads);

        for (int i = 0; i < numRows; ++i)
            rowD[i] = template.copy().zero();
    }

    /**
     * Constructor for AbstractRowMatrix
     * 
     * @param A
     *            Matrix to copy contents from
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     * @param numThreads
     *            Number of threads to use in parallelization
     * @param deep
     *            True for a deep copy, false for a reference copy. A reference
     *            copy can only be made of an <code>AbstractRowMatrix</code>
     */
    AbstractRowMatrix(Matrix A, Vector template, int numThreads, boolean deep) {
        super(A);
        construct(numThreads);
        part = smt.util.Arrays.partition(0, numRows, numThreads);

        if (deep) {
            for (int i = 0; i < numRows; ++i)
                rowD[i] = template.copy().zero();
            set(A);
        } else {
            AbstractRowMatrix Ar = (AbstractRowMatrix) A;
            for (int i = 0; i < numRows; ++i)
                rowD[i] = Ar.getRow(i);
        }
    }

    /**
     * Constructor for AbstractRowMatrix
     * 
     * @param A
     *            Matrix to copy contents from. The copy will be deep
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     * @param numThreads
     *            Number of threads to use in parallelization
     */
    AbstractRowMatrix(Matrix A, Vector template, int numThreads) {
        this(A, template, numThreads, true);
    }

    /**
     * Constructor for AbstractRowMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param numRows
     *            Number of rows
     * @param numColumns
     *            Number of column
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     */
    AbstractRowMatrix(int numRows, int numColumns, Vector template) {
        this(numRows, numColumns, template, Runtime.getRuntime()
                .availableProcessors());
    }

    /**
     * Constructor for AbstractRowMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param A
     *            Matrix to copy contents from. The copy will be deep
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     */
    AbstractRowMatrix(Matrix A, Vector template) {
        this(A, template, Runtime.getRuntime().availableProcessors());
    }

    /**
     * Constructor for AbstractRowMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param A
     *            Matrix to copy contents from
     * @param template
     *            Vector template. Uses to create the other vectors by copying
     * @param deep
     *            True for a deep copy, false for a reference copy. A reference
     *            copy can only be made of an <code>AbstractRowMatrix</code>
     */
    AbstractRowMatrix(Matrix A, Vector template, boolean deep) {
        this(A, template, Runtime.getRuntime().availableProcessors(), deep);
    }

    private void readObject(java.io.ObjectInputStream in) throws IOException,
            ClassNotFoundException {
        in.defaultReadObject();
        if (numThreads > 1)
            executor = Executors.newFixedThreadPool(numThreads,
                    new DaemonThreadFactory());
    }

    /**
     * Performs some common setup chores
     */
    private void construct(int numThreads) {
        rowD = new Vector[numRows];
        this.numThreads = numThreads;
        if (numThreads > 1)
            executor = Executors.newFixedThreadPool(numThreads,
                    new DaemonThreadFactory());
    }

    /**
     * Returns the given row
     */
    public Vector getRow(int i) {
        return rowD[i];
    }

    public Vector multAdd(final double alpha, final Vector x,
            final double beta, final Vector y, final Vector z) {
        checkMultAdd(x, y, z);

        if (numThreads == 1)
            multI(alpha, x, beta, y, z, 0, numRows);
        else {

            // Create the tasks
            Runnable[] threads = new Runnable[numThreads];
            for (int i = 0; i < numThreads; ++i) {
                final int j = i;
                threads[i] = new Runnable() {
                    public void run() {
                        multI(alpha, x, beta, y, z, part[j], part[j + 1]);
                    }
                };
            }

            // Run them
            Future[] future = new Future[numThreads];
            for (int i = 0; i < numThreads; ++i)
                future[i] = executor.submit(threads[i]);

            // Await completion
            try {
                for (Future f : future)
                    f.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        return z;
    }

    void multI(double alpha, Vector x, double beta, Vector y, Vector z,
            int start, int stop) {
        for (int i = start; i < stop; ++i)
            z.set(i, alpha * rowD[i].dot(x) + beta * y.get(i));
    }

    public void add(int row, int column, double value) {
        rowD[row].add(column, value);
    }

    public void add(int[] row, int[] column, double[][] values) {
        check(row, column, values);

        for (int i = 0; i < row.length; ++i)
            rowD[row[i]].add(column, values[i]);
    }

    public Matrix copy() {
        return new FlexCompRowMatrix(this);
    }

    public double get(int row, int column) {
        return rowD[row].get(column);
    }

    public double[][] get(int[] row, int[] column, double[][] values) {
        check(row, column, values);

        for (int i = 0; i < row.length; ++i)
            values[i] = rowD[row[i]].get(column, values[i]);
        return values;
    }

    public Iterator<MatrixEntry> iterator() {
        return new RowMatrixIterator();
    }

    public void set(int row, int column, double value) {
        rowD[row].set(column, value);
    }

    public void set(int[] row, int[] column, double[][] values) {
        check(row, column, values);

        for (int i = 0; i < row.length; ++i)
            rowD[row[i]].set(column, values[i]);
    }

    public AbstractRowMatrix zero() {
        for (int i = 0; i < numRows; ++i)
            rowD[i].zero();
        return this;
    }

    /**
     * Iterator over a matrix stored vectorwise by rows
     */
    private class RowMatrixIterator implements Iterator<MatrixEntry> {

        /**
         * This does most of the work
         */
        private SuperIterator iterator;

        /**
         * Entry returned
         */
        private RowMatrixEntry e;

        public RowMatrixIterator() {
            iterator = new SuperIterator(rowD);
            e = new RowMatrixEntry();
        }

        public boolean hasNext() {
            return iterator.hasNext();
        }

        public MatrixEntry next() {
            SuperIteratorEntry se = iterator.next();
            e.update(se.getIndex(), (VectorEntry) se.getObject());
            return e;
        }

        public void remove() {
            iterator.remove();
        }

    }

    /**
     * Entry of this row-built matrix
     */
    private class RowMatrixEntry extends RefMatrixEntry {

        private VectorEntry entry;

        public void update(int row, VectorEntry entry) {
            super.update(row, entry.index(), entry.get());
            this.entry = entry;
        }

        public void set(double value) {
            this.value = value;
            entry.set(value);
        }

    }

}
