/*
 * Copyright (C) 2003-2005 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import mt.ll.BLASkernel.Diag;
import mt.ll.BLASkernel.Side;
import mt.ll.BLASkernel.Transpose;
import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;

/**
 * Partial implementation of a triangular, dense matrix
 */
abstract class AbstractTriangDenseMatrix extends AbstractDenseMatrix {

    /**
     * If the matrix is upper triangular
     */
    private UpLo uplo;

    /**
     * If the matrix is unit diagonal or not unit
     */
    private Diag diag;

    /**
     * Leading dimension of the matrix
     */
    int ld;

    /**
     * Constructor for AbstractTriangDenseMatrix
     * 
     * @param n
     *            Size of the matrix. Since the matrix must be square, this
     *            equals both the number of rows and columns
     */
    AbstractTriangDenseMatrix(int n, UpLo uplo, Diag diag) {
        super(n, n);
        ld = n;
        this.uplo = uplo;
        this.diag = diag;
    }

    /**
     * Constructor for AbstractTriangDenseMatrix
     * 
     * @param A
     *            Matrix to copy from
     */
    AbstractTriangDenseMatrix(Matrix A, UpLo uplo, Diag diag) {
        this(A, Math.min(A.numRows(), A.numColumns()), uplo, diag);
    }

    /**
     * Constructor for AbstractTriangDenseMatrix
     * 
     * @param A
     *            Matrix to copy from
     * @param deep
     *            If true, <code>A</code> is copied, else a shallow copy is
     *            made and the matrices share underlying storage. For this,
     *            <code>A</code> must be a dense matrix
     */
    AbstractTriangDenseMatrix(Matrix A, boolean deep, UpLo uplo, Diag diag) {
        this(A, Math.min(A.numRows(), A.numColumns()), deep, uplo, diag);
    }

    /**
     * Constructor for AbstractTriangDenseMatrix
     * 
     * @param A
     *            Matrix to copy from
     * @param k
     *            Size of matrix to refer.
     *            <code>k&lt;min(numRows,numColumns)</code>
     */
    AbstractTriangDenseMatrix(Matrix A, int k, UpLo uplo, Diag diag) {
        this(A, k, true, uplo, diag);
    }

    /**
     * Constructor for AbstractTriangDenseMatrix
     * 
     * @param A
     *            Matrix to copy from
     * @param k
     *            Size of matrix to refer.
     *            <code>k&lt;min(numRows,numColumns)</code>
     * @param deep
     *            If true, <code>A</code> is copied, else a shallow copy is
     *            made and the matrices share underlying storage. For this,
     *            <code>A</code> must be a dense matrix
     */
    AbstractTriangDenseMatrix(Matrix A, int k, boolean deep, UpLo uplo,
            Diag diag) {
        super(A, deep);

        if (deep && !A.isSquare())
            throw new IllegalArgumentException("deep && !A.isSquare()");

        ld = A.numRows();
        numRows = numColumns = k;
        this.uplo = uplo;
        this.diag = diag;
    }

    public Vector multAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(z instanceof DenseVector))
            return super.multAdd(alpha, x, beta, y, z);

        checkMultAdd(x, y, z);

        double[] zd = ((DenseVector) z).getData();

        // z = alpha*x
        z.set(alpha, x);

        // z = A*z
        Interface.blas().trmv(uplo, Transpose.NoTranspose, diag, numRows, data,
                Math.max(1, ld), zd);

        // z = z + beta*y
        return z.add(beta, y);
    }

    public Vector transMultAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(z instanceof DenseVector))
            return super.transMultAdd(alpha, x, beta, y, z);

        checkTransMultAdd(x, y, z);

        double[] zd = ((DenseVector) z).getData();

        // z = alpha*x
        z.set(alpha, x);

        // z = A'*z
        Interface.blas().trmv(uplo, Transpose.Transpose, diag, numRows, data,
                Math.max(1, ld), zd);

        // z = z + beta*y
        return z.add(beta, y);
    }

    public Matrix multAdd(double alpha, Matrix B, double beta, Matrix C,
            Matrix D) {
        if (!(D instanceof DenseMatrix))
            return super.multAdd(alpha, B, beta, C, D);

        checkMultAdd(B, C, D);

        double[] Dd = ((DenseMatrix) D).getData();

        // D = B
        D.set(B);

        // D = alpha*A*D
        Interface.blas().trmm(Side.Left, uplo, Transpose.NoTranspose, diag,
                D.numRows(), D.numColumns(), alpha, data, Math.max(1, ld), Dd,
                Math.max(1, D.numRows()));

        // D = D + beta*C
        return D.add(beta, C);
    }

    public Matrix transAmultAdd(double alpha, Matrix B, double beta, Matrix C,
            Matrix D) {
        if (!(D instanceof DenseMatrix))
            return super.transAmultAdd(alpha, B, beta, C, D);

        checkTransAmultAdd(B, C, D);

        double[] Dd = ((DenseMatrix) D).getData();

        // D = B
        D.set(B);

        // D = alpha*A'*D
        Interface.blas().trmm(Side.Left, uplo, Transpose.Transpose, diag,
                D.numRows(), D.numColumns(), alpha, data, Math.max(1, ld), Dd,
                Math.max(1, D.numRows()));

        // D = D + beta*C
        return D.add(beta, C);
    }

    public Matrix solve(Matrix B, Matrix X) {
        return solve(B, X, Transpose.NoTranspose);
    }

    public Vector solve(Vector b, Vector x) {
        DenseMatrix B = new DenseMatrix(b, false), X = new DenseMatrix(x, false);
        solve(B, X);
        return x;
    }

    public Matrix transSolve(Matrix B, Matrix X) {
        return solve(B, X, Transpose.Transpose);
    }

    public Vector transSolve(Vector b, Vector x) {
        DenseMatrix B = new DenseMatrix(b, false), X = new DenseMatrix(x, false);
        transSolve(B, X);
        return x;
    }

    Matrix solve(Matrix B, Matrix X, Transpose trans) {
        if (!(X instanceof DenseMatrix))
            throw new UnsupportedOperationException("X must be a DenseMatrix");

        // Different argument checking to support Hessenberg type matrices for
        // solvers such as GMRES
        if (B.numRows() < numRows)
            throw new IllegalArgumentException("B.numRows() < A.numRows()");
        if (B.numColumns() != X.numColumns())
            throw new IllegalArgumentException(
                    "B.numColumns() != X.numColumns()");
        if (X.numRows() < numRows)
            throw new IllegalArgumentException("X.numRows() < A.numRows()");

        double[] Xd = ((DenseMatrix) X).getData();

        X.set(B);

        int info = Interface.lapack().trtrs(uplo, trans, diag, numRows,
                X.numColumns(), data, Math.max(1, ld), Xd);

        if (info > 0)
            throw new MatrixSingularException();
        else if (info < 0)
            throw new IllegalArgumentException();

        return X;
    }

    int getIndex(int row, int column) {
        check(row, column);
        return row + column * Math.max(ld, numRows);
    }

}
