/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of DMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package dmt;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import mpp.Communicator;
import mpp.Reductions;
import mt.Matrix;
import mt.MatrixEntry;
import mt.Vector;

/**
 * Distributed matrix with column major blocks
 */
public class DistColMatrix extends DistMatrix {

    private static final long serialVersionUID = 3618421523053359673L;

    /**
     * Constructor for DistColMatrix
     * 
     * @param numRows
     *            Global number of rows
     * @param numColumns
     *            Global number of columns
     * @param comm
     *            Communicator to use
     * @param A
     *            Block diagonal matrix. The sum of the local row sizes of
     *            <code>A</code> must equal the global number, and likewise
     *            with the column sizes.
     * @param B
     *            Off-diagonal matrix part. Its number of rows must equal the
     *            global number of rows, and its number of columns must equal
     *            that of <code>A</code>
     */
    public DistColMatrix(int numRows, int numColumns, Communicator comm,
            Matrix A, Matrix B) {
        super(numRows, numColumns, comm, A, B);

        if (A.numColumns() != B.numColumns())
            throw new IllegalArgumentException(
                    "A.numColumns() != B.numColumns()");
        if (B.numRows() != numRows)
            throw new IllegalArgumentException("B.numRows() != numRows");
    }

    public void add(int row, int column, double value) {
        check(row, column);

        if (inA(row, column))
            A.add(row - n[rank], column - m[rank], value);
        else if (local(row, column))
            B.add(row, column - m[rank], value);
        else {
            if (!compatible(StashMode.Add))
                throw new IllegalArgumentException("Incompatible assembly mode");
            stashMode = StashMode.Add;
            stash.add(new Entry(row, column, value, getRank(column)));
        }
    }

    public void set(int row, int column, double value) {
        check(row, column);

        if (inA(row, column))
            A.set(row - n[rank], column - m[rank], value);
        else if (local(row, column))
            B.set(row, column - m[rank], value);
        else {
            if (!compatible(StashMode.Set))
                throw new IllegalArgumentException("Incompatible assembly mode");
            stashMode = StashMode.Set;
            stash.add(new Entry(row, column, value, getRank(column)));
        }
    }

    public double get(int row, int column) {
        check(row, column);
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");

        if (inA(row, column))
            return A.get(row - n[rank], column - m[rank]);
        else if (local(row, column))
            return B.get(row, column - m[rank]);
        else
            throw new IndexOutOfBoundsException("Entry not available locally");
    }

    public DistColMatrix copy() {
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");
        return new DistColMatrix(numRows, numColumns, comm, A.copy(), B.copy());
    }

    public Matrix transpose() {
        checkTranspose();
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");

        for (int i = 0; i < numRows; ++i)
            for (int j = m[rank]; j < m[rank + 1]; ++j) {
                double value = get(i, j);
                set(i, j, get(j, i));
                set(j, i, value);
            }

        try {
            finalizeAssembly();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return this;
    }

    public Iterator<MatrixEntry> iterator() {
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");
        return new DistMatrixIterator(n[rank], m[rank], 0, m[rank]);
    }

    public Vector multAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(x instanceof DistVector && z instanceof DistVector))
            throw new IllegalArgumentException("Vectors must be DistVectors");

        checkMultAdd(x, y, z);

        // z = beta/alpha * y
        z.set(beta / alpha, y);

        // z = A*x + z = A*x + beta/alpha * y

        DistVector xd = (DistVector) x, zd = (DistVector) z;

        // Global part
        B.mult(xd.getLocal(), locR);

        // Send it to the others
        scatter.startGather(locR, zd);

        // Local part
        A.multAdd(xd.getLocal(), zd.getLocal());

        // Finish communications, concluding the matrix product
        scatter.endAddGather(locR, zd);

        // z = alpha*z = alpha * A*x + beta*y
        return z.scale(alpha);
    }

    public Vector transMultAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(x instanceof DistVector && y instanceof DistVector && z instanceof DistVector))
            throw new IllegalArgumentException("Vectors must be DistVectors");

        checkTransMultAdd(x, y, z);

        DistVector xd = (DistVector) x, yd = (DistVector) y, zd = (DistVector) z;

        // Recieve the needed components of the global x into the local vector
        scatter.startScatter(xd, locC);

        // Local part
        A
                .transMultAdd(alpha, xd.getLocal(), beta, yd.getLocal(), zd
                        .getLocal());

        // Finish communications
        scatter.endSetScatter(xd, locC);

        // Global part
        B.transMultAdd(alpha, locC, zd.getLocal());

        return z;
    }

    public boolean local(int row, int column) {
        return column >= m[rank] && column < m[rank + 1];
    }

    int getRank(int column) {
        int i = 1;
        for (; i < m.length; ++i)
            if (column < m[i])
                break;
        return i - 1;
    }

    int[] getDelimiter() {
        return m;
    }

    int[] getCommIndices() {
        // Get the unique row indices from B
        Collection<Integer> set = new HashSet<Integer>();
        for (MatrixEntry e : B)
            set.add(e.row());

        // Get an array representation
        int[] indices = new int[set.size()];
        int j = 0;
        for (Integer i : set)
            indices[j++] = i;
        return indices;
    }

    protected double norm1() {
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");

        // Compute as much locally as possible
        double[] rowSum = new double[numRows];
        for (MatrixEntry e : this)
            rowSum[e.row()] += Math.abs(e.get());

        // Sum in the rest from the other ranks
        double[] recv = new double[numRows];
        try {
            comm.allReduce(rowSum, recv, Reductions.sum());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        // The global maximum
        return max(recv);
    }

    protected double normInf() {
        if (!assembled())
            throw new IllegalArgumentException("Matrix is not assembled");

        // Compute local norm
        double norm = super.normInf();

        // Find the maximum
        double[] recv = new double[1];
        try {
            comm.allReduce(new double[] { norm }, recv, Reductions.max());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return recv[0];
    }

    public DistColMatrix addDiagonal(double shift) {
        super.addDiagonal(shift);
        return this;
    }

    public DistColMatrix zero() {
        super.zero();
        return this;
    }

}
