"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NewRelicStack = exports.EndpointUrlMetrics = exports.EndpointUrlLogs = exports.EndpointType = exports.NEW_RELIC_AWS_ACCOUNT_ID = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const firehose = require("aws-cdk-lib/aws-kinesisfirehose");
const s3 = require("aws-cdk-lib/aws-s3");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const utils_1 = require("../../common/utils");
const base_1 = require("../base");
exports.NEW_RELIC_AWS_ACCOUNT_ID = '754728514883';
var EndpointType;
(function (EndpointType) {
    EndpointType["METRICS"] = "metrics";
    EndpointType["LOGS"] = "logs";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
var EndpointUrlLogs;
(function (EndpointUrlLogs) {
    EndpointUrlLogs["EU_LOGS"] = "https://aws-api.eu.newrelic.com/firehose/v1";
    EndpointUrlLogs["US_LOGS"] = "https://aws-api.newrelic.com/firehose/v1";
})(EndpointUrlLogs = exports.EndpointUrlLogs || (exports.EndpointUrlLogs = {}));
var EndpointUrlMetrics;
(function (EndpointUrlMetrics) {
    EndpointUrlMetrics["EU_METRICS"] = "https://aws-api.eu01.nr-data.net/cloudwatch-metrics/v1";
    EndpointUrlMetrics["US_METRICS"] = "https://aws-api.newrelic.com/cloudwatch-metrics/v1";
})(EndpointUrlMetrics = exports.EndpointUrlMetrics || (exports.EndpointUrlMetrics = {}));
class NewRelicStack extends base_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.newRelicIntegrationRole = this.createNewRelicRole(props.newRelicAccountId);
        this.newRelicSecret = this.createSecrets(props.newRelicAccountId, props.newRelicLicenseKey);
        this.newRelicBucket = this.createFirehoseBucket(props.newRelicBucketName);
        this.newRelicFirehoseRole = this.createFirehoseRole(this.newRelicBucket);
        if (props.newRelicApiUrlLogs) {
            this.newRelicFirehoseLogs = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.LOGS, props.newRelicApiUrlLogs, props.newRelicLicenseKey);
            this.newRelicCloudwatchLogsStreamRole = this.createCloudwatchLogsStreamRole();
        }
        if (props.newRelicApiUrlMetrics) {
            this.newRelicFirehoseMetrics = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.METRICS, props.newRelicApiUrlMetrics, props.newRelicLicenseKey);
        }
    }
    createCloudwatchMetricStream(roleArn, firehoseArn) {
        return new cloudwatch.CfnMetricStream(this, 'newrelic-cloudwatch-stream-metrics', {
            firehoseArn: firehoseArn,
            outputFormat: 'opentelemetry0.7',
            roleArn: roleArn,
            name: 'newelic-stream-metrics',
        });
    }
    createCloudwatchLogsStreamRole() {
        let role = new iam.Role(this, 'newrelic-logstream-role', {
            roleName: 'NewRelicInfrastructure-CloudwatchStream',
            assumedBy: new iam.ServicePrincipal('logs.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecord',
                'firehose:PutRecordBatch',
                'kinesis:PutRecords',
                'kinesis:PutRecord',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        return role;
    }
    createNewRelicRole(newRelicAccountId) {
        let role = new iam.Role(this, 'newrelic-role', {
            roleName: 'NewRelicInfrastructure-Integrations',
            assumedBy: new iam.AccountPrincipal(exports.NEW_RELIC_AWS_ACCOUNT_ID),
            externalIds: [
                newRelicAccountId,
            ],
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'budgets:ViewBudget',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        new aws_cdk_lib_1.CfnOutput(this, 'newrelic-role-output', {
            value: role.roleArn,
            description: 'New Relic role arn',
            exportName: 'newRelicRole',
        });
        return role;
    }
    createFirehoseStream(newRelicBucket, role, endpointType, endpointUrl, newRelicLicenseLey) {
        if (this.stage == 'production') {
            // Minute in one day: 1440
            // Interval: 5min
            // Sends per day: 1440/5 = 288
            // Usage per day: 288*5mb = 1.5gb
            var bufferingHints = {
                intervalInSeconds: 300,
                sizeInMBs: 5,
            };
        }
        else {
            // Usage per day: 144*3mb = 0.432gb
            var bufferingHints = {
                intervalInSeconds: 600,
                sizeInMBs: 3,
            };
        }
        let httpEndpointMetrics = {
            bufferingHints: bufferingHints,
            endpointConfiguration: {
                url: endpointUrl,
                accessKey: newRelicLicenseLey,
                name: endpointType,
            },
            s3Configuration: {
                bucketArn: newRelicBucket.bucketArn,
                roleArn: role.roleArn,
            },
            requestConfiguration: {
                contentEncoding: 'GZIP',
            },
            roleArn: role.roleArn,
        };
        let firehoseStream = new firehose.CfnDeliveryStream(this, `newrelic-firehose-${endpointType}`, {
            deliveryStreamName: `NewRelic-stream-${endpointType}`,
            deliveryStreamType: 'DirectPut',
            httpEndpointDestinationConfiguration: httpEndpointMetrics,
        });
        utils_1.addBaseTags(firehoseStream);
        return firehoseStream;
    }
    createSecrets(newRelicAccountId, newRelicLicenseLey) {
        let secret = new secretsmanager.Secret(this, 'newrelic-secret', {
            secretName: 'NEW_RELIC_LICENSE_KEY',
            secretObjectValue: {
                NrAccountId: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicAccountId),
                LicenseKey: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicLicenseLey),
            },
        });
        utils_1.addBaseTags(secret);
        return secret;
    }
    createFirehoseBucket(newRelicBucketName) {
        let bucket = new s3.Bucket(this, 'newrelic-bucket', {
            bucketName: newRelicBucketName,
            versioned: true,
            lifecycleRules: [
                {
                    id: 'raw-bucket-lifecycle-rule',
                    enabled: true,
                    noncurrentVersionExpiration: aws_cdk_lib_1.Duration.days(30),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                    ],
                },
            ],
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        utils_1.addBaseTags(bucket);
        return bucket;
    }
    createFirehoseRole(newRelicFirehoseBucket) {
        let role = new iam.Role(this, 'newrelic-firehose-role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        utils_1.addBaseTags(role);
        // TODO: create more restrictive policy
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'glue:GetTable',
                'glue:GetTableVersion',
                'glue:GetTableVersions',
                'glue:GetSchemaByDefinition',
                'glue:GetSchemaVersion',
                'logs:PutLogEvents',
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
                'kinesis:ListShards',
                'kms:Decrypt',
                'kms:GenerateDataKey',
                'kms:Decrypt',
                'kafka:GetBootstrapBrokers',
                'kafka:DescribeCluster',
                'kafka:DescribeClusterV2',
                'kafka-cluster:Connect',
                'kafka-cluster:DescribeTopic',
                'kafka-cluster:DescribeTopicDynamicConfiguration',
                'kafka-cluster:ReadData',
                'kafka-cluster:DescribeGroup',
                'lambda:InvokeFunction',
                'lambda:GetFunctionConfiguration',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'S3:PutObjectAcl',
            ],
            resources: [
                `${newRelicFirehoseBucket.bucketArn}`,
                `${newRelicFirehoseBucket.bucketArn}/*`,
            ],
        }));
        return role;
    }
}
exports.NewRelicStack = NewRelicStack;
_a = JSII_RTTI_SYMBOL_1;
NewRelicStack[_a] = { fqn: "neulabs-cdk-constructs.stacks.monitoring.newrelic.NewRelicStack", version: "0.1.7" };
//# sourceMappingURL=data:application/json;base64,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