"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricSet = exports.allMetricsGraphJson = void 0;
const drop_empty_object_at_the_end_of_an_array_token_1 = require("./drop-empty-object-at-the-end-of-an-array-token");
const env_tokens_1 = require("./env-tokens");
const metric_util_1 = require("./metric-util");
const object_1 = require("./object");
/**
 * Return the JSON structure which represents these metrics in a graph.
 *
 * Depending on the metric type (stat or expression), one `Metric` object
 * can render to multiple time series.
 *
 * - Top-level metrics will be rendered visibly, additionally added metrics will
 *   be rendered invisibly.
 * - IDs used in math expressions need to be either globally unique, or refer to the same
 *   metric object.
 *
 * This will be called by GraphWidget, no need for clients to call this.
 */
function allMetricsGraphJson(left, right) {
    // Add metrics to a set which will automatically expand them recursively,
    // making sure to retain conflicting the visible one on conflicting metrics objects.
    const mset = new MetricSet();
    mset.addTopLevel('left', ...left);
    mset.addTopLevel('right', ...right);
    // Render all metrics from the set.
    return mset.entries.map(entry => new drop_empty_object_at_the_end_of_an_array_token_1.DropEmptyObjectAtTheEndOfAnArray(metricGraphJson(entry.metric, entry.tag, entry.id)));
}
exports.allMetricsGraphJson = allMetricsGraphJson;
function metricGraphJson(metric, yAxis, id) {
    const config = metric.toMetricConfig();
    const ret = [];
    const options = { ...config.renderingProperties };
    metric_util_1.dispatchMetric(metric, {
        withStat(stat) {
            ret.push(stat.namespace, stat.metricName);
            // Dimensions
            for (const dim of (stat.dimensions || [])) {
                ret.push(dim.name, dim.value);
            }
            // Metric attributes that are rendered to graph options
            if (stat.account) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(stat.account);
            }
            if (stat.region) {
                options.region = env_tokens_1.regionIfDifferentFromStack(stat.region);
            }
            if (stat.period && stat.period.toSeconds() !== 300) {
                options.period = stat.period.toSeconds();
            }
            if (stat.statistic && stat.statistic !== 'Average') {
                options.stat = stat.statistic;
            }
        },
        withExpression(expr) {
            options.expression = expr.expression;
        },
    });
    // Options
    if (!yAxis) {
        options.visible = false;
    }
    if (yAxis !== 'left') {
        options.yAxis = yAxis;
    }
    if (id) {
        options.id = id;
    }
    // If math expressions don't have a label (or an ID), they'll render with an unelegant
    // autogenerated id ("metric_alias0"). Our ids may in the future also be autogenerated,
    // so if an ME doesn't have a label, use its toString() as the label (renders the expression).
    if (options.visible !== false && options.expression && !options.label) {
        options.label = metric.toString();
    }
    const renderedOpts = object_1.dropUndefined(options);
    if (Object.keys(renderedOpts).length !== 0) {
        ret.push(renderedOpts);
    }
    return ret;
}
/**
 * Contain a set of metrics, expanding math expressions
 *
 * "Primary" metrics (added via a top-level call) can be tagged with an additional value.
 */
class MetricSet {
    constructor() {
        this.metrics = new Array();
        this.metricById = new Map();
        this.metricByKey = new Map();
    }
    /**
     * Add the given set of metrics to this set
     */
    addTopLevel(tag, ...metrics) {
        for (const metric of metrics) {
            this.addOne(metric, tag);
        }
    }
    /**
     * Access all the accumulated timeseries entries
     */
    get entries() {
        return this.metrics;
    }
    /**
     * Add a metric into the set
     *
     * The id may not be the same as a previous metric added, unless it's the same metric.
     *
     * It can be made visible, in which case the new "metric" object replaces the old
     * one (and the new ones "renderingPropertieS" will be honored instead of the old
     * one's).
     */
    addOne(metric, tag, id) {
        const key = metric_util_1.metricKey(metric);
        let existingEntry;
        // Try lookup existing by id if we have one
        if (id) {
            existingEntry = this.metricById.get(id);
            if (existingEntry && metric_util_1.metricKey(existingEntry.metric) !== key) {
                throw new Error(`Cannot have two different metrics share the same id ('${id}') in one Alarm or Graph. Rename one of them.`);
            }
        }
        if (!existingEntry) {
            // Try lookup by metric if we didn't find one by id
            existingEntry = this.metricByKey.get(key);
            // If the one we found already has an id, it must be different from the id
            // we're trying to add and we want to add a new metric. Pretend we didn't
            // find one.
            if ((existingEntry === null || existingEntry === void 0 ? void 0 : existingEntry.id) && id) {
                existingEntry = undefined;
            }
        }
        // Create a new entry if we didn't find one so far
        let entry;
        if (existingEntry) {
            entry = existingEntry;
        }
        else {
            entry = { metric };
            this.metrics.push(entry);
            this.metricByKey.set(key, entry);
        }
        // If it didn't have an id but now we do, add one
        if (!entry.id && id) {
            entry.id = id;
            this.metricById.set(id, entry);
        }
        // If it didn't have a tag but now we do, add one
        if (!entry.tag && tag) {
            entry.tag = tag;
        }
        // Recurse and add children
        const conf = metric.toMetricConfig();
        if (conf.mathExpression) {
            for (const [subId, subMetric] of Object.entries(conf.mathExpression.usingMetrics)) {
                this.addOne(subMetric, undefined, subId);
            }
        }
    }
}
exports.MetricSet = MetricSet;
//# sourceMappingURL=data:application/json;base64,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