"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Spacer = exports.Column = exports.Row = void 0;
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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