"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const access_1 = require("./access");
const delete_runner_function_1 = require("./lambdas/delete-runner-function");
const setup_function_1 = require("./lambdas/setup-function");
const status_function_1 = require("./lambdas/status-function");
const token_retriever_function_1 = require("./lambdas/token-retriever-function");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.extraLambdaEnv = {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props?.vpc,
            vpcSubnets: this.props?.vpcSubnets,
            allowPublicSubnet: this.props?.allowPublicSubnet,
            securityGroups: this.props?.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props?.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props?.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props?.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunnerProvider(this, 'CodeBuild'),
                new lambda_1.LambdaRunnerProvider(this, 'Lambda'),
                new fargate_1.FargateRunnerProvider(this, 'Fargate'),
            ];
        }
        this.checkIntersectingLabels();
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
            access: this.props?.webhookAccess ?? access_1.LambdaAccess.lambdaUrl(),
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteRunnerFunction = this.deleteRunner();
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: false,
            }),
        });
        deleteRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const waitForIdleRunner = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, 'Wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.duration(props?.idleTimeout ?? cdk.Duration.minutes(10)),
        });
        const deleteIdleRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Idle Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: true,
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(...provider.labels.map(label => aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${label.toLowerCase()}`))), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .branch(waitForIdleRunner.next(deleteIdleRunnerTask))
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        let logOptions;
        if (this.props?.logOptions) {
            this.stateMachineLogGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                logGroupName: props?.logOptions?.logGroupName,
                retention: props?.logOptions?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            logOptions = {
                destination: this.stateMachineLogGroup,
                includeExecutionData: props?.logOptions?.includeExecutionData ?? true,
                level: props?.logOptions?.level ?? aws_cdk_lib_1.aws_stepfunctions.LogLevel.ALL,
            };
        }
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: work,
            logs: logOptions,
        });
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new token_retriever_function_1.TokenRetrieverFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new delete_runner_function_1.DeleteRunnerFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new status_function_1.StatusFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                STEP_FUNCTION_LOG_GROUP: this.stateMachineLogGroup?.logGroupName ?? '',
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
        const access = this.props?.statusAccess ?? access_1.LambdaAccess.noAccess();
        const url = access._bind(this, 'status access', statusFunction);
        if (url !== '') {
            new cdk.CfnOutput(this, 'status url', {
                value: url,
            });
        }
    }
    setupFunction() {
        const setupFunction = new setup_function_1.SetupFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        const access = this.props?.setupAccess ?? access_1.LambdaAccess.lambdaUrl();
        return access._bind(this, 'setup access', setupFunction);
    }
    checkIntersectingLabels() {
        // this "algorithm" is very inefficient, but good enough for the tiny datasets we expect
        for (const p1 of this.providers) {
            for (const p2 of this.providers) {
                if (p1 == p2) {
                    continue;
                }
                if (p1.labels.every(l => p2.labels.includes(l))) {
                    if (p2.labels.every(l => p1.labels.includes(l))) {
                        throw new Error(`Both ${p1.node.path} and ${p2.node.path} use the same labels [${p1.labels.join(', ')}]`);
                    }
                    aws_cdk_lib_1.Annotations.of(p1).addWarning(`Labels [${p1.labels.join(', ')}] intersect with another provider (${p2.node.path} -- [${p2.labels.join(', ')}]). If a workflow specifies the labels [${p1.labels.join(', ')}], it is not guaranteed which provider will be used. It is recommended you do not use intersecting labels`);
                }
            }
        }
    }
    /**
     * Metric for the number of GitHub Actions jobs completed. It has `ProviderLabels` and `Status` dimensions. The status can be one of "Succeeded", "SucceededWithIssues", "Failed", "Canceled", "Skipped", or "Abandoned".
     *
     * **WARNING:** this method creates a metric filter for each provider. Each metric has a status dimension with six possible values. These resources may incur cost.
     */
    metricJobCompleted(props) {
        if (!this.jobsCompletedMetricFilters) {
            // we can't use logs.FilterPattern.spaceDelimited() because it has no support for ||
            // status list taken from https://github.com/actions/runner/blob/be9632302ceef50bfb36ea998cea9c94c75e5d4d/src/Sdk/DTWebApi/WebApi/TaskResult.cs
            // we need "..." for Lambda that prefixes some extra data to log lines
            const pattern = aws_cdk_lib_1.aws_logs.FilterPattern.literal('[..., marker = "CDKGHA", job = "JOB", done = "DONE", labels, status = "Succeeded" || status = "SucceededWithIssues" || status = "Failed" || status = "Canceled" || status = "Skipped" || status = "Abandoned"]');
            this.jobsCompletedMetricFilters = this.providers.map(p => p.logGroup.addMetricFilter(`${p.logGroup.node.id} filter`, {
                metricNamespace: 'GitHubRunners',
                metricName: 'JobCompleted',
                filterPattern: pattern,
                metricValue: '1',
                // can't with dimensions -- defaultValue: 0,
                dimensions: {
                    ProviderLabels: '$labels',
                    Status: '$status',
                },
            }));
            for (const metricFilter of this.jobsCompletedMetricFilters) {
                if (metricFilter.node.defaultChild instanceof aws_cdk_lib_1.aws_logs.CfnMetricFilter) {
                    metricFilter.node.defaultChild.addPropertyOverride('MetricTransformations.0.Unit', 'Count');
                }
                else {
                    aws_cdk_lib_1.Annotations.of(metricFilter).addWarning('Unable to set metric filter Unit to Count');
                }
            }
        }
        return new aws_cdk_lib_1.aws_cloudwatch.Metric({
            namespace: 'GitHubRunners',
            metricName: 'JobsCompleted',
            unit: aws_cdk_lib_1.aws_cloudwatch.Unit.COUNT,
            statistic: aws_cdk_lib_1.aws_cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for successful executions.
     *
     * A successful execution doesn't always mean a runner was started. It can be successful even without any label matches.
     *
     * A successful runner doesn't mean the job it executed was successful. For that, see {@link metricJobCompleted}.
     */
    metricSucceeded(props) {
        return this.orchestrator.metricSucceeded(props);
    }
    /**
     * Metric for failed runner executions.
     *
     * A failed runner usually means the runner failed to start and so a job was never executed. It doesn't necessarily mean the job was executed and failed. For that, see {@link metricJobCompleted}.
     */
    metricFailed(props) {
        return this.orchestrator.metricFailed(props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes. This time may be longer than the time the runner took.
     */
    metricTime(props) {
        return this.orchestrator.metricTime(props);
    }
}
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.9.3" };
exports.GitHubRunners = GitHubRunners;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicnVubmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3J1bm5lci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLG1DQUFtQztBQUNuQyw2Q0FVcUI7QUFDckIsMkNBQXVDO0FBQ3ZDLHFDQUF3QztBQUN4Qyw2RUFBd0U7QUFDeEUsNkRBQXlEO0FBQ3pELCtEQUEyRDtBQUMzRCxpRkFBNEU7QUFDNUUscURBQWdFO0FBRWhFLGlEQUE0RDtBQUM1RCwrQ0FBMEQ7QUFDMUQsdUNBQW9DO0FBQ3BDLHVDQUFpRDtBQXVJakQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXVDRztBQUNILE1BQWEsYUFBYyxTQUFRLHNCQUFTO0lBbUIxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFXLEtBQTBCO1FBQzNFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFEZ0MsVUFBSyxHQUFMLEtBQUssQ0FBcUI7UUFMNUQsbUJBQWMsR0FBMEIsRUFBRSxDQUFDO1FBUTFELElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsZ0JBQWdCLEdBQUc7WUFDdEIsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRztZQUNwQixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxVQUFVO1lBQ2xDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsaUJBQWlCO1lBQ2hELGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xGLE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksd0JBQU0sQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO29CQUMzRixXQUFXLEVBQUUsOEVBQThFO29CQUMzRixJQUFJLEVBQUUsd0JBQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7aUJBQzFELENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ2hCLENBQUM7UUFDRixJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFDakMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsR0FBRyxnQkFBZ0IsQ0FBQztTQUM1RDtRQUVELElBQUksSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDekIsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztTQUN2QzthQUFNO1lBQ0wsSUFBSSxDQUFDLFNBQVMsR0FBRztnQkFDZixJQUFJLG1DQUF1QixDQUFDLElBQUksRUFBRSxXQUFXLENBQUM7Z0JBQzlDLElBQUksNkJBQW9CLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQztnQkFDeEMsSUFBSSwrQkFBcUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDO2FBQzNDLENBQUM7U0FDSDtRQUVELElBQUksQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO1FBRS9CLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksOEJBQW9CLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQy9ELFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsYUFBYSxJQUFJLHFCQUFZLENBQUMsU0FBUyxFQUFFO1NBQzlELENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQ3JDLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUN4QixDQUFDO0lBRU8sWUFBWSxDQUFDLEtBQTBCO1FBQzdDLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxxQ0FBbUIsQ0FBQyxZQUFZLENBQzdELElBQUksRUFDSixrQkFBa0IsRUFDbEI7WUFDRSxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUNyQyxtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLFVBQVUsRUFBRSxVQUFVO1NBQ3ZCLENBQ0YsQ0FBQztRQUVGLElBQUksb0JBQW9CLEdBQUcsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDO1FBQy9DLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxxQ0FBbUIsQ0FBQyxZQUFZLENBQzNELElBQUksRUFDSixlQUFlLEVBQ2Y7WUFDRSxjQUFjLEVBQUUsb0JBQW9CO1lBQ3BDLG1CQUFtQixFQUFFLElBQUk7WUFDekIsVUFBVSxFQUFFLFVBQVU7WUFDdEIsT0FBTyxFQUFFLCtCQUFhLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztnQkFDMUMsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDaEUsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUM7Z0JBQ2pELElBQUksRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDO2dCQUMvQyxLQUFLLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQztnQkFDakQsY0FBYyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDbkUsUUFBUSxFQUFFLEtBQUs7YUFDaEIsQ0FBQztTQUNILENBQ0YsQ0FBQztRQUNGLGdCQUFnQixDQUFDLFFBQVEsQ0FBQztZQUN4QixNQUFNLEVBQUU7Z0JBQ04sWUFBWTthQUNiO1lBQ0QsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNqQyxXQUFXLEVBQUUsQ0FBQztZQUNkLFdBQVcsRUFBRSxFQUFFO1NBQ2hCLENBQUMsQ0FBQztRQUVILE1BQU0saUJBQWlCLEdBQUcsSUFBSSwrQkFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQzdELElBQUksRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFdBQVcsSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUN0RixDQUFDLENBQUM7UUFDSCxNQUFNLG9CQUFvQixHQUFHLElBQUkscUNBQW1CLENBQUMsWUFBWSxDQUMvRCxJQUFJLEVBQ0osb0JBQW9CLEVBQ3BCO1lBQ0UsY0FBYyxFQUFFLG9CQUFvQjtZQUNwQyxtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLFVBQVUsRUFBRSxVQUFVO1lBQ3RCLE9BQU8sRUFBRSwrQkFBYSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUM7Z0JBQ2hFLEtBQUssRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDO2dCQUNqRCxJQUFJLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQztnQkFDL0MsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUM7Z0JBQ2pELGNBQWMsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUM7Z0JBQ25FLFFBQVEsRUFBRSxJQUFJO2FBQ2YsQ0FBQztTQUNILENBQ0YsQ0FBQztRQUVGLE1BQU0sZUFBZSxHQUFHLElBQUksK0JBQWEsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDMUUsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ3JDLE1BQU0sWUFBWSxHQUFHLFFBQVEsQ0FBQyxtQkFBbUIsQ0FDL0M7Z0JBQ0UsZUFBZSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDbEUsY0FBYyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDcEUsZ0JBQWdCLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDO2dCQUNwRSxTQUFTLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQztnQkFDckQsUUFBUSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUM7YUFDcEQsQ0FDRixDQUFDO1lBQ0YsZUFBZSxDQUFDLElBQUksQ0FDbEIsK0JBQWEsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUN6QixHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUNwQixLQUFLLENBQUMsRUFBRSxDQUFDLCtCQUFhLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxZQUFZLEtBQUssQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQzlFLENBQ0YsRUFDRCxZQUFZLENBQ2IsQ0FBQztTQUNIO1FBRUQsZUFBZSxDQUFDLFNBQVMsQ0FBQyxJQUFJLCtCQUFhLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDO1FBRTVFLE1BQU0sSUFBSSxHQUFHLGtCQUFrQixDQUFDLElBQUksQ0FDbEMsSUFBSSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxDQUFDO2FBQzFFLE1BQU0sQ0FBQyxlQUFlLENBQUM7YUFDdkIsTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO2FBQ3BELFFBQVEsQ0FDUCxnQkFBZ0I7YUFDYixJQUFJLENBQUMsSUFBSSwrQkFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUMsRUFDdEQ7WUFDRSxVQUFVLEVBQUUsU0FBUztTQUN0QixDQUNGLENBQ0osQ0FBQztRQUVGLElBQUksVUFBd0QsQ0FBQztRQUM3RCxJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQzFCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLHNCQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7Z0JBQzFELFlBQVksRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLFlBQVk7Z0JBQzdDLFNBQVMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLFlBQVksSUFBSSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO2dCQUMxRSxhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO2FBQ3JDLENBQUMsQ0FBQztZQUVILFVBQVUsR0FBRztnQkFDWCxXQUFXLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtnQkFDdEMsb0JBQW9CLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxvQkFBb0IsSUFBSSxJQUFJO2dCQUNyRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLElBQUksK0JBQWEsQ0FBQyxRQUFRLENBQUMsR0FBRzthQUM5RCxDQUFDO1NBQ0g7UUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLCtCQUFhLENBQUMsWUFBWSxDQUNqRCxJQUFJLEVBQ0oscUJBQXFCLEVBQ3JCO1lBQ0UsVUFBVSxFQUFFLElBQUk7WUFDaEIsSUFBSSxFQUFFLFVBQVU7U0FDakIsQ0FDRixDQUFDO1FBRUYsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ3JDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUMxQztRQUVELE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFFTyxjQUFjO1FBQ3BCLE1BQU0sSUFBSSxHQUFHLElBQUksaURBQXNCLENBQ3JDLElBQUksRUFDSixpQkFBaUIsRUFDakI7WUFDRSxXQUFXLEVBQUUsb0VBQW9FO1lBQ2pGLFdBQVcsRUFBRTtnQkFDWCxpQkFBaUIsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTO2dCQUNoRCw2QkFBNkIsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLFNBQVM7Z0JBQ3RFLEdBQUcsSUFBSSxDQUFDLGNBQWM7YUFDdkI7WUFDRCxPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQ2pDLFlBQVksRUFBRSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1lBQzFDLEdBQUcsSUFBSSxDQUFDLGdCQUFnQjtTQUN6QixDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sWUFBWTtRQUNsQixNQUFNLElBQUksR0FBRyxJQUFJLDZDQUFvQixDQUNuQyxJQUFJLEVBQ0osZUFBZSxFQUNmO1lBQ0UsV0FBVyxFQUFFLHVDQUF1QztZQUNwRCxXQUFXLEVBQUU7Z0JBQ1gsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUztnQkFDaEQsNkJBQTZCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTO2dCQUN0RSxHQUFHLElBQUksQ0FBQyxjQUFjO2FBQ3ZCO1lBQ0QsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNqQyxZQUFZLEVBQUUsc0JBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztZQUMxQyxHQUFHLElBQUksQ0FBQyxnQkFBZ0I7U0FDekIsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVPLGNBQWM7UUFDcEIsTUFBTSxjQUFjLEdBQUcsSUFBSSxnQ0FBYyxDQUN2QyxJQUFJLEVBQ0osUUFBUSxFQUNSO1lBQ0UsV0FBVyxFQUFFLG1FQUFtRTtZQUNoRixXQUFXLEVBQUU7Z0JBQ1gsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUztnQkFDbEQsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUztnQkFDaEQsNkJBQTZCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTO2dCQUN0RSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTO2dCQUM5QyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHO2dCQUM3QixtQkFBbUIsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsV0FBVztnQkFDbkUsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlO2dCQUNwRCx1QkFBdUIsRUFBRSxJQUFJLENBQUMsb0JBQW9CLEVBQUUsWUFBWSxJQUFJLEVBQUU7Z0JBQ3RFLGtCQUFrQixFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUNqQyxHQUFHLElBQUksQ0FBQyxjQUFjO2FBQ3ZCO1lBQ0QsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNoQyxZQUFZLEVBQUUsc0JBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztZQUMxQyxHQUFHLElBQUksQ0FBQyxnQkFBZ0I7U0FDekIsQ0FDRixDQUFDO1FBRUYsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7UUFFbEYsc0ZBQXNGO1FBQ3RGLHVGQUF1RjtRQUN2RixNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQyxNQUFNLENBQUMsR0FBSSxjQUFjLENBQUMsSUFBSSxDQUFDLFlBQW1DLENBQUM7UUFDbkUsQ0FBQyxDQUFDLG1CQUFtQixDQUFDLGtDQUFrQyxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN2RSxDQUFDLENBQUMsbUJBQW1CLENBQUMsa0NBQWtDLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzNFLENBQUMsQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3RDLGNBQWMsQ0FBQyxlQUFlLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxPQUFPLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQztZQUNqRCxTQUFTLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO1NBQzNCLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFNUMsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUNmLElBQUksRUFDSixnQkFBZ0IsRUFDaEI7WUFDRSxLQUFLLEVBQUUsZ0JBQWdCLEtBQUssQ0FBQyxNQUFNLGtDQUFrQyxjQUFjLENBQUMsWUFBWSxjQUFjO1NBQy9HLENBQ0YsQ0FBQztRQUVGLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLEVBQUUsWUFBWSxJQUFJLHFCQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDbkUsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRWhFLElBQUksR0FBRyxLQUFLLEVBQUUsRUFBRTtZQUNkLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FDZixJQUFJLEVBQ0osWUFBWSxFQUNaO2dCQUNFLEtBQUssRUFBRSxHQUFHO2FBQ1gsQ0FDRixDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRU8sYUFBYTtRQUNuQixNQUFNLGFBQWEsR0FBRyxJQUFJLDhCQUFhLENBQ3JDLElBQUksRUFDSixPQUFPLEVBQ1A7WUFDRSxXQUFXLEVBQUUsMkRBQTJEO1lBQ3hFLFdBQVcsRUFBRTtnQkFDWCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTO2dCQUM5QyxrQkFBa0IsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTO2dCQUNsRCxpQkFBaUIsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTO2dCQUNoRCw2QkFBNkIsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLFNBQVM7Z0JBQ3RFLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUc7Z0JBQzdCLEdBQUcsSUFBSSxDQUFDLGNBQWM7YUFDdkI7WUFDRCxPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLFlBQVksRUFBRSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1lBQzFDLEdBQUcsSUFBSSxDQUFDLGdCQUFnQjtTQUN6QixDQUNGLENBQUM7UUFFRixpREFBaUQ7UUFDakQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDOUMsMERBQTBEO1FBQzFELElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFN0MsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLEtBQUssRUFBRSxXQUFXLElBQUkscUJBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNuRSxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRU8sdUJBQXVCO1FBQzdCLHdGQUF3RjtRQUN4RixLQUFLLE1BQU0sRUFBRSxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDL0IsS0FBSyxNQUFNLEVBQUUsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUMvQixJQUFJLEVBQUUsSUFBSSxFQUFFLEVBQUU7b0JBQ1osU0FBUztpQkFDVjtnQkFDRCxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtvQkFDL0MsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7d0JBQy9DLE1BQU0sSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksUUFBUSxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUkseUJBQXlCLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDM0c7b0JBQ0QseUJBQVcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksUUFBUSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsMkNBQTJDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQywyR0FBMkcsQ0FBQyxDQUFDO2lCQUN4VDthQUNGO1NBQ0Y7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGtCQUFrQixDQUFDLEtBQThCO1FBQ3RELElBQUksQ0FBQyxJQUFJLENBQUMsMEJBQTBCLEVBQUU7WUFDcEMsb0ZBQW9GO1lBQ3BGLCtJQUErSTtZQUMvSSxzRUFBc0U7WUFDdEUsTUFBTSxPQUFPLEdBQUcsc0JBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLGdOQUFnTixDQUFDLENBQUM7WUFFN1AsSUFBSSxDQUFDLDBCQUEwQixHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQ3ZELENBQUMsQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxTQUFTLEVBQUU7Z0JBQ3pELGVBQWUsRUFBRSxlQUFlO2dCQUNoQyxVQUFVLEVBQUUsY0FBYztnQkFDMUIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFdBQVcsRUFBRSxHQUFHO2dCQUNoQiw0Q0FBNEM7Z0JBQzVDLFVBQVUsRUFBRTtvQkFDVixjQUFjLEVBQUUsU0FBUztvQkFDekIsTUFBTSxFQUFFLFNBQVM7aUJBQ2xCO2FBQ0YsQ0FBQyxDQUNILENBQUM7WUFFRixLQUFLLE1BQU0sWUFBWSxJQUFJLElBQUksQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUQsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLFlBQVksWUFBWSxzQkFBSSxDQUFDLGVBQWUsRUFBRTtvQkFDbEUsWUFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsOEJBQThCLEVBQUUsT0FBTyxDQUFDLENBQUM7aUJBQzdGO3FCQUFNO29CQUNMLHlCQUFXLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDLFVBQVUsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO2lCQUN0RjthQUNGO1NBQ0Y7UUFFRCxPQUFPLElBQUksNEJBQVUsQ0FBQyxNQUFNLENBQUM7WUFDM0IsU0FBUyxFQUFFLGVBQWU7WUFDMUIsVUFBVSxFQUFFLGVBQWU7WUFDM0IsSUFBSSxFQUFFLDRCQUFVLENBQUMsSUFBSSxDQUFDLEtBQUs7WUFDM0IsU0FBUyxFQUFFLDRCQUFVLENBQUMsU0FBUyxDQUFDLEdBQUc7WUFDbkMsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZSxDQUFDLEtBQThCO1FBQ25ELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsS0FBOEI7UUFDaEQsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxVQUFVLENBQUMsS0FBOEI7UUFDOUMsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM3QyxDQUFDOzs7O0FBamFVLHNDQUFhIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7XG4gIEFubm90YXRpb25zLFxuICBhd3NfY2xvdWR3YXRjaCBhcyBjbG91ZHdhdGNoLFxuICBhd3NfZWMyIGFzIGVjMixcbiAgYXdzX2lhbSBhcyBpYW0sXG4gIGF3c19sYW1iZGEgYXMgbGFtYmRhLFxuICBhd3NfbG9ncyBhcyBsb2dzLFxuICBhd3Nfc3RlcGZ1bmN0aW9ucyBhcyBzdGVwZnVuY3Rpb25zLFxuICBhd3Nfc3RlcGZ1bmN0aW9uc190YXNrcyBhcyBzdGVwZnVuY3Rpb25zX3Rhc2tzLFxuICBSZW1vdmFsUG9saWN5LFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IExhbWJkYUFjY2VzcyB9IGZyb20gJy4vYWNjZXNzJztcbmltcG9ydCB7IERlbGV0ZVJ1bm5lckZ1bmN0aW9uIH0gZnJvbSAnLi9sYW1iZGFzL2RlbGV0ZS1ydW5uZXItZnVuY3Rpb24nO1xuaW1wb3J0IHsgU2V0dXBGdW5jdGlvbiB9IGZyb20gJy4vbGFtYmRhcy9zZXR1cC1mdW5jdGlvbic7XG5pbXBvcnQgeyBTdGF0dXNGdW5jdGlvbiB9IGZyb20gJy4vbGFtYmRhcy9zdGF0dXMtZnVuY3Rpb24nO1xuaW1wb3J0IHsgVG9rZW5SZXRyaWV2ZXJGdW5jdGlvbiB9IGZyb20gJy4vbGFtYmRhcy90b2tlbi1yZXRyaWV2ZXItZnVuY3Rpb24nO1xuaW1wb3J0IHsgQ29kZUJ1aWxkUnVubmVyUHJvdmlkZXIgfSBmcm9tICcuL3Byb3ZpZGVycy9jb2RlYnVpbGQnO1xuaW1wb3J0IHsgSVJ1bm5lclByb3ZpZGVyIH0gZnJvbSAnLi9wcm92aWRlcnMvY29tbW9uJztcbmltcG9ydCB7IEZhcmdhdGVSdW5uZXJQcm92aWRlciB9IGZyb20gJy4vcHJvdmlkZXJzL2ZhcmdhdGUnO1xuaW1wb3J0IHsgTGFtYmRhUnVubmVyUHJvdmlkZXIgfSBmcm9tICcuL3Byb3ZpZGVycy9sYW1iZGEnO1xuaW1wb3J0IHsgU2VjcmV0cyB9IGZyb20gJy4vc2VjcmV0cyc7XG5pbXBvcnQgeyBHaXRodWJXZWJob29rSGFuZGxlciB9IGZyb20gJy4vd2ViaG9vayc7XG5cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBHaXRIdWJSdW5uZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR2l0SHViUnVubmVyc1Byb3BzIHtcbiAgLyoqXG4gICAqIExpc3Qgb2YgcnVubmVyIHByb3ZpZGVycyB0byB1c2UuIEF0IGxlYXN0IG9uZSBwcm92aWRlciBpcyByZXF1aXJlZC4gUHJvdmlkZXIgd2lsbCBiZSBzZWxlY3RlZCB3aGVuIGl0cyBsYWJlbCBtYXRjaGVzIHRoZSBsYWJlbHMgcmVxdWVzdGVkIGJ5IHRoZSB3b3JrZmxvdyBqb2IuXG4gICAqXG4gICAqIEBkZWZhdWx0IENvZGVCdWlsZCwgTGFtYmRhIGFuZCBGYXJnYXRlIHJ1bm5lcnMgd2l0aCBhbGwgdGhlIGRlZmF1bHRzIChubyBWUEMgb3IgZGVmYXVsdCBhY2NvdW50IFZQQylcbiAgICovXG4gIHJlYWRvbmx5IHByb3ZpZGVycz86IElSdW5uZXJQcm92aWRlcltdO1xuXG4gIC8qKlxuICAgKiBWUEMgdXNlZCBmb3IgYWxsIG1hbmFnZW1lbnQgZnVuY3Rpb25zLiBVc2UgdGhpcyB3aXRoIEdpdEh1YiBFbnRlcnByaXNlIFNlcnZlciBob3N0ZWQgdGhhdCdzIGluYWNjZXNzaWJsZSBmcm9tIG91dHNpZGUgdGhlIFZQQy5cbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBWUEMgc3VibmV0cyB1c2VkIGZvciBhbGwgbWFuYWdlbWVudCBmdW5jdGlvbnMuIFVzZSB0aGlzIHdpdGggR2l0SHViIEVudGVycHJpc2UgU2VydmVyIGhvc3RlZCB0aGF0J3MgaW5hY2Nlc3NpYmxlIGZyb20gb3V0c2lkZSB0aGUgVlBDLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIEFsbG93IG1hbmFnZW1lbnQgZnVuY3Rpb25zIHRvIHJ1biBpbiBwdWJsaWMgc3VibmV0cy4gTGFtYmRhIEZ1bmN0aW9ucyBpbiBhIHB1YmxpYyBzdWJuZXQgY2FuIE5PVCBhY2Nlc3MgdGhlIGludGVybmV0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb3dQdWJsaWNTdWJuZXQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTZWN1cml0eSBncm91cCBhdHRhY2hlZCB0byBhbGwgbWFuYWdlbWVudCBmdW5jdGlvbnMuIFVzZSB0aGlzIHdpdGggdG8gcHJvdmlkZSBhY2Nlc3MgdG8gR2l0SHViIEVudGVycHJpc2UgU2VydmVyIGhvc3RlZCBpbnNpZGUgYSBWUEMuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3VwPzogZWMyLklTZWN1cml0eUdyb3VwO1xuXG4gIC8qKlxuICAgKiBQYXRoIHRvIGEgZGlyZWN0b3J5IGNvbnRhaW5pbmcgYSBmaWxlIG5hbWVkIGNlcnRzLnBlbSBjb250YWluaW5nIGFueSBhZGRpdGlvbmFsIGNlcnRpZmljYXRlcyByZXF1aXJlZCB0byB0cnVzdCBHaXRIdWIgRW50ZXJwcmlzZSBTZXJ2ZXIuIFVzZSB0aGlzIHdoZW4gR2l0SHViIEVudGVycHJpc2UgU2VydmVyIGNlcnRpZmljYXRlcyBhcmUgc2VsZi1zaWduZWQuXG4gICAqXG4gICAqIFlvdSBtYXkgYWxzbyB3YW50IHRvIHVzZSBjdXN0b20gaW1hZ2VzIGZvciB5b3VyIHJ1bm5lciBwcm92aWRlcnMgdGhhdCBjb250YWluIHRoZSBzYW1lIGNlcnRpZmljYXRlcy4gU2VlIHtAbGluayBDb2RlQnVpbGRJbWFnZUJ1aWxkZXIuYWRkQ2VydGlmaWNhdGVzfS5cbiAgICpcbiAgICogYGBgdHlwZXNjcmlwdFxuICAgKiBjb25zdCBpbWFnZUJ1aWxkZXIgPSBDb2RlQnVpbGRSdW5uZXJQcm92aWRlci5pbWFnZUJ1aWxkZXIodGhpcywgJ0ltYWdlIEJ1aWxkZXIgd2l0aCBDZXJ0cycpO1xuICAgKiBpbWFnZUJ1aWxkZXIuYWRkQ29tcG9uZW50KFJ1bm5lckltYWdlQ29tcG9uZW50LmV4dHJhQ2VydGlmaWNhdGVzKCdwYXRoLXRvLW15LWV4dHJhLWNlcnRzLWZvbGRlci9jZXJ0cy5wZW0nLCAncHJpdmF0ZS1jYScpO1xuICAgKlxuICAgKiBjb25zdCBwcm92aWRlciA9IG5ldyBDb2RlQnVpbGRSdW5uZXJQcm92aWRlcih0aGlzLCAnQ29kZUJ1aWxkJywge1xuICAgKiAgICAgaW1hZ2VCdWlsZGVyOiBpbWFnZUJ1aWxkZXIsXG4gICAqIH0pO1xuICAgKlxuICAgKiBuZXcgR2l0SHViUnVubmVycyhcbiAgICogICB0aGlzLFxuICAgKiAgICdydW5uZXJzJyxcbiAgICogICB7XG4gICAqICAgICBwcm92aWRlcnM6IFtwcm92aWRlcl0sXG4gICAqICAgICBleHRyYUNlcnRpZmljYXRlczogJ3BhdGgtdG8tbXktZXh0cmEtY2VydHMtZm9sZGVyJyxcbiAgICogICB9XG4gICAqICk7XG4gICAqIGBgYFxuICAgKi9cbiAgcmVhZG9ubHkgZXh0cmFDZXJ0aWZpY2F0ZXM/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRpbWUgdG8gd2FpdCBiZWZvcmUgc3RvcHBpbmcgYSBydW5uZXIgdGhhdCByZW1haW5zIGlkbGUuIElmIHRoZSB1c2VyIGNhbmNlbGxlZCB0aGUgam9iLCBvciBpZiBhbm90aGVyIHJ1bm5lciBzdG9sZSBpdCwgdGhpcyBzdG9wcyB0aGUgcnVubmVyIHRvIGF2b2lkIHdhc3RpbmcgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxMCBtaW51dGVzXG4gICAqL1xuICByZWFkb25seSBpZGxlVGltZW91dD86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogTG9nZ2luZyBvcHRpb25zIGZvciB0aGUgc3RhdGUgbWFjaGluZSB0aGF0IG1hbmFnZXMgdGhlIHJ1bm5lcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIGxvZ3NcbiAgICovXG4gIHJlYWRvbmx5IGxvZ09wdGlvbnM/OiBMb2dPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBY2Nlc3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIHNldHVwIGZ1bmN0aW9uLiBPbmNlIHlvdSBmaW5pc2ggdGhlIHNldHVwIHByb2Nlc3MsIHlvdSBjYW4gc2V0IHRoaXMgdG8gYExhbWJkYUFjY2Vzcy5ub0FjY2VzcygpYCB0byByZW1vdmUgYWNjZXNzIHRvIHRoZSBzZXR1cCBmdW5jdGlvbi4gWW91IGNhbiBhbHNvIHVzZSBgTGFtYmRhQWNjZXNzLmFwaUdhdGV3YXkoeyBhbGxvd2VkSXBzOiBbJ215LWlwLzAnXX0pYCB0byBsaW1pdCBhY2Nlc3MgdG8geW91ciBJUCBvbmx5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBMYW1iZGFBY2Nlc3MubGFtYmRhVXJsKClcbiAgICovXG4gIHJlYWRvbmx5IHNldHVwQWNjZXNzPzogTGFtYmRhQWNjZXNzO1xuXG5cbiAgLyoqXG4gICAqIEFjY2VzcyBjb25maWd1cmF0aW9uIGZvciB0aGUgd2ViaG9vayBmdW5jdGlvbi4gVGhpcyBmdW5jdGlvbiBpcyBjYWxsZWQgYnkgR2l0SHViIHdoZW4gYSBuZXcgd29ya2Zsb3cgam9iIGlzIHNjaGVkdWxlZC4gRm9yIGFuIGV4dHJhIGxheWVyIG9mIHNlY3VyaXR5LCB5b3UgY2FuIHNldCB0aGlzIHRvIGBMYW1iZGFBY2Nlc3MuYXBpR2F0ZXdheSh7IGFsbG93ZWRJcHM6IExhbWJkYUFjY2Vzcy5naXRodWJXZWJob29rSXBzKCkgfSlgLlxuICAgKlxuICAgKiBZb3UgY2FuIGFsc28gc2V0IHRoaXMgdG8gYExhbWJkYUFjY2Vzcy5wcml2YXRlQXBpR2F0ZXdheSgpYCBpZiB5b3VyIEdpdEh1YiBFbnRlcnByaXNlIFNlcnZlciBpcyBob3N0ZWQgaW4gYSBWUEMuIFRoaXMgd2lsbCBjcmVhdGUgYW4gQVBJIEdhdGV3YXkgZW5kcG9pbnQgdGhhdCdzIG9ubHkgYWNjZXNzaWJsZSBmcm9tIHdpdGhpbiB0aGUgVlBDLlxuICAgKlxuICAgKiAqV0FSTklORyo6IGNoYW5naW5nIGFjY2VzcyB0eXBlIG1heSBjaGFuZ2UgdGhlIFVSTC4gV2hlbiB0aGUgVVJMIGNoYW5nZXMsIHlvdSBtdXN0IHVwZGF0ZSBHaXRIdWIgYXMgd2VsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgTGFtYmRhQWNjZXNzLmxhbWJkYVVybCgpXG4gICAqL1xuICByZWFkb25seSB3ZWJob29rQWNjZXNzPzogTGFtYmRhQWNjZXNzO1xuXG4gIC8qKlxuICAgKiBBY2Nlc3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIHN0YXR1cyBmdW5jdGlvbi4gVGhpcyBmdW5jdGlvbiByZXR1cm5zIGEgbG90IG9mIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgcnVubmVyLCBzbyB5b3Ugc2hvdWxkIG9ubHkgYWxsb3cgYWNjZXNzIHRvIGl0IGZyb20gdHJ1c3RlZCBJUHMsIGlmIGF0IGFsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgTGFtYmRhQWNjZXNzLm5vQWNjZXNzKClcbiAgICovXG4gIHJlYWRvbmx5IHN0YXR1c0FjY2Vzcz86IExhbWJkYUFjY2Vzcztcbn1cblxuLyoqXG4gKiBEZWZpbmVzIHdoYXQgZXhlY3V0aW9uIGhpc3RvcnkgZXZlbnRzIGFyZSBsb2dnZWQgYW5kIHdoZXJlIHRoZXkgYXJlIGxvZ2dlZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBsb2cgZ3JvdXAgd2hlcmUgdGhlIGV4ZWN1dGlvbiBoaXN0b3J5IGV2ZW50cyB3aWxsIGJlIGxvZ2dlZC5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ0dyb3VwTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB3aGV0aGVyIGV4ZWN1dGlvbiBkYXRhIGlzIGluY2x1ZGVkIGluIHlvdXIgbG9nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5jbHVkZUV4ZWN1dGlvbkRhdGE/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHdoaWNoIGNhdGVnb3J5IG9mIGV4ZWN1dGlvbiBoaXN0b3J5IGV2ZW50cyBhcmUgbG9nZ2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBFUlJPUlxuICAgKi9cbiAgcmVhZG9ubHkgbGV2ZWw/OiBzdGVwZnVuY3Rpb25zLkxvZ0xldmVsO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGRheXMgbG9nIGV2ZW50cyBhcmUga2VwdCBpbiBDbG91ZFdhdGNoIExvZ3MuIFdoZW4gdXBkYXRpbmdcbiAgICogdGhpcyBwcm9wZXJ0eSwgdW5zZXR0aW5nIGl0IGRvZXNuJ3QgcmVtb3ZlIHRoZSBsb2cgcmV0ZW50aW9uIHBvbGljeS4gVG9cbiAgICogcmVtb3ZlIHRoZSByZXRlbnRpb24gcG9saWN5LCBzZXQgdGhlIHZhbHVlIHRvIGBJTkZJTklURWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEhcbiAgICovXG4gIHJlYWRvbmx5IGxvZ1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cztcbn1cblxuLyoqXG4gKiBDcmVhdGUgYWxsIHRoZSByZXF1aXJlZCBpbmZyYXN0cnVjdHVyZSB0byBwcm92aWRlIHNlbGYtaG9zdGVkIEdpdEh1YiBydW5uZXJzLiBJdCBjcmVhdGVzIGEgd2ViaG9vaywgc2VjcmV0cywgYW5kIGEgc3RlcCBmdW5jdGlvbiB0byBvcmNoZXN0cmF0ZSBhbGwgcnVucy4gU2VjcmV0cyBhcmUgbm90IGF1dG9tYXRpY2FsbHkgZmlsbGVkLiBTZWUgUkVBRE1FLm1kIGZvciBpbnN0cnVjdGlvbnMgb24gaG93IHRvIHNldHVwIEdpdEh1YiBpbnRlZ3JhdGlvbi5cbiAqXG4gKiBCeSBkZWZhdWx0LCB0aGlzIHdpbGwgY3JlYXRlIGEgcnVubmVyIHByb3ZpZGVyIG9mIGVhY2ggYXZhaWxhYmxlIHR5cGUgd2l0aCB0aGUgZGVmYXVsdHMuIFRoaXMgaXMgZ29vZCBlbm91Z2ggZm9yIHRoZSBpbml0aWFsIHNldHVwIHN0YWdlIHdoZW4geW91IGp1c3Qgd2FudCB0byBnZXQgR2l0SHViIGludGVncmF0aW9uIHdvcmtpbmcuXG4gKlxuICogYGBgdHlwZXNjcmlwdFxuICogbmV3IEdpdEh1YlJ1bm5lcnModGhpcywgJ3J1bm5lcnMnKTtcbiAqIGBgYFxuICpcbiAqIFVzdWFsbHkgeW91J2Qgd2FudCB0byBjb25maWd1cmUgdGhlIHJ1bm5lciBwcm92aWRlcnMgc28gdGhlIHJ1bm5lcnMgY2FuIHJ1biBpbiBhIGNlcnRhaW4gVlBDIG9yIGhhdmUgY2VydGFpbiBwZXJtaXNzaW9ucy5cbiAqXG4gKiBgYGB0eXBlc2NyaXB0XG4gKiBjb25zdCB2cGMgPSBlYzIuVnBjLmZyb21Mb29rdXAodGhpcywgJ3ZwYycsIHsgdnBjSWQ6ICd2cGMtMTIzNDU2NycgfSk7XG4gKiBjb25zdCBydW5uZXJTZyA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cCh0aGlzLCAncnVubmVyIHNlY3VyaXR5IGdyb3VwJywgeyB2cGM6IHZwYyB9KTtcbiAqIGNvbnN0IGRiU2cgPSBlYzIuU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHRoaXMsICdkYXRhYmFzZSBzZWN1cml0eSBncm91cCcsICdzZy0xMjM0NTY3Jyk7XG4gKiBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdydW5uZXIgYnVja2V0Jyk7XG4gKlxuICogLy8gY3JlYXRlIGEgY3VzdG9tIENvZGVCdWlsZCBwcm92aWRlclxuICogY29uc3QgbXlQcm92aWRlciA9IG5ldyBDb2RlQnVpbGRSdW5uZXIoXG4gKiAgIHRoaXMsICdjb2RlYnVpbGQgcnVubmVyJyxcbiAqICAge1xuICogICAgICBsYWJlbDogJ215LWNvZGVidWlsZCcsXG4gKiAgICAgIHZwYzogdnBjLFxuICogICAgICBzZWN1cml0eUdyb3VwOiBydW5uZXJTZyxcbiAqICAgfSxcbiAqICk7XG4gKiAvLyBncmFudCBzb21lIHBlcm1pc3Npb25zIHRvIHRoZSBwcm92aWRlclxuICogYnVja2V0LmdyYW50UmVhZFdyaXRlKG15UHJvdmlkZXIpO1xuICogZGJTZy5jb25uZWN0aW9ucy5hbGxvd0Zyb20ocnVubmVyU2csIGVjMi5Qb3J0LnRjcCgzMzA2KSwgJ2FsbG93IHJ1bm5lcnMgdG8gY29ubmVjdCB0byBNeVNRTCBkYXRhYmFzZScpO1xuICpcbiAqIC8vIGNyZWF0ZSB0aGUgcnVubmVyIGluZnJhc3RydWN0dXJlXG4gKiBuZXcgR2l0SHViUnVubmVycyhcbiAqICAgdGhpcyxcbiAqICAgJ3J1bm5lcnMnLFxuICogICB7XG4gKiAgICAgcHJvdmlkZXJzOiBbbXlQcm92aWRlcl0sXG4gKiAgIH1cbiAqICk7XG4gKiBgYGBcbiAqL1xuZXhwb3J0IGNsYXNzIEdpdEh1YlJ1bm5lcnMgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogQ29uZmlndXJlZCBydW5uZXIgcHJvdmlkZXJzLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJvdmlkZXJzOiBJUnVubmVyUHJvdmlkZXJbXTtcblxuICAvKipcbiAgICogU2VjcmV0cyBmb3IgR2l0SHViIGNvbW11bmljYXRpb24gaW5jbHVkaW5nIHdlYmhvb2sgc2VjcmV0IGFuZCBydW5uZXIgYXV0aGVudGljYXRpb24uXG4gICAqL1xuICByZWFkb25seSBzZWNyZXRzOiBTZWNyZXRzO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgd2ViaG9vazogR2l0aHViV2ViaG9va0hhbmRsZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgb3JjaGVzdHJhdG9yOiBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZTtcbiAgcHJpdmF0ZSByZWFkb25seSBzZXR1cFVybDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGV4dHJhTGFtYmRhRW52OiB7W3A6IHN0cmluZ106IHN0cmluZ30gPSB7fTtcbiAgcHJpdmF0ZSByZWFkb25seSBleHRyYUxhbWJkYVByb3BzOiBsYW1iZGEuRnVuY3Rpb25PcHRpb25zO1xuICBwcml2YXRlIHN0YXRlTWFjaGluZUxvZ0dyb3VwPzogbG9ncy5Mb2dHcm91cDtcbiAgcHJpdmF0ZSBqb2JzQ29tcGxldGVkTWV0cmljRmlsdGVycz86IGxvZ3MuTWV0cmljRmlsdGVyW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcmVhZG9ubHkgcHJvcHM/OiBHaXRIdWJSdW5uZXJzUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5zZWNyZXRzID0gbmV3IFNlY3JldHModGhpcywgJ1NlY3JldHMnKTtcbiAgICB0aGlzLmV4dHJhTGFtYmRhUHJvcHMgPSB7XG4gICAgICB2cGM6IHRoaXMucHJvcHM/LnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHRoaXMucHJvcHM/LnZwY1N1Ym5ldHMsXG4gICAgICBhbGxvd1B1YmxpY1N1Ym5ldDogdGhpcy5wcm9wcz8uYWxsb3dQdWJsaWNTdWJuZXQsXG4gICAgICBzZWN1cml0eUdyb3VwczogdGhpcy5wcm9wcz8uc2VjdXJpdHlHcm91cCA/IFt0aGlzLnByb3BzLnNlY3VyaXR5R3JvdXBdIDogdW5kZWZpbmVkLFxuICAgICAgbGF5ZXJzOiB0aGlzLnByb3BzPy5leHRyYUNlcnRpZmljYXRlcyA/IFtuZXcgbGFtYmRhLkxheWVyVmVyc2lvbihzY29wZSwgJ0NlcnRpZmljYXRlIExheWVyJywge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ0xheWVyIGNvbnRhaW5pbmcgR2l0SHViIEVudGVycHJpc2UgU2VydmVyIGNlcnRpZmljYXRlIGZvciBjZGstZ2l0aHViLXJ1bm5lcnMnLFxuICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQodGhpcy5wcm9wcy5leHRyYUNlcnRpZmljYXRlcyksXG4gICAgICB9KV0gOiB1bmRlZmluZWQsXG4gICAgfTtcbiAgICBpZiAodGhpcy5wcm9wcz8uZXh0cmFDZXJ0aWZpY2F0ZXMpIHtcbiAgICAgIHRoaXMuZXh0cmFMYW1iZGFFbnYuTk9ERV9FWFRSQV9DQV9DRVJUUyA9ICcvb3B0L2NlcnRzLnBlbSc7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucHJvcHM/LnByb3ZpZGVycykge1xuICAgICAgdGhpcy5wcm92aWRlcnMgPSB0aGlzLnByb3BzLnByb3ZpZGVycztcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5wcm92aWRlcnMgPSBbXG4gICAgICAgIG5ldyBDb2RlQnVpbGRSdW5uZXJQcm92aWRlcih0aGlzLCAnQ29kZUJ1aWxkJyksXG4gICAgICAgIG5ldyBMYW1iZGFSdW5uZXJQcm92aWRlcih0aGlzLCAnTGFtYmRhJyksXG4gICAgICAgIG5ldyBGYXJnYXRlUnVubmVyUHJvdmlkZXIodGhpcywgJ0ZhcmdhdGUnKSxcbiAgICAgIF07XG4gICAgfVxuXG4gICAgdGhpcy5jaGVja0ludGVyc2VjdGluZ0xhYmVscygpO1xuXG4gICAgdGhpcy5vcmNoZXN0cmF0b3IgPSB0aGlzLnN0YXRlTWFjaGluZShwcm9wcyk7XG4gICAgdGhpcy53ZWJob29rID0gbmV3IEdpdGh1YldlYmhvb2tIYW5kbGVyKHRoaXMsICdXZWJob29rIEhhbmRsZXInLCB7XG4gICAgICBvcmNoZXN0cmF0b3I6IHRoaXMub3JjaGVzdHJhdG9yLFxuICAgICAgc2VjcmV0czogdGhpcy5zZWNyZXRzLFxuICAgICAgYWNjZXNzOiB0aGlzLnByb3BzPy53ZWJob29rQWNjZXNzID8/IExhbWJkYUFjY2Vzcy5sYW1iZGFVcmwoKSxcbiAgICB9KTtcblxuICAgIHRoaXMuc2V0dXBVcmwgPSB0aGlzLnNldHVwRnVuY3Rpb24oKTtcbiAgICB0aGlzLnN0YXR1c0Z1bmN0aW9uKCk7XG4gIH1cblxuICBwcml2YXRlIHN0YXRlTWFjaGluZShwcm9wcz86IEdpdEh1YlJ1bm5lcnNQcm9wcykge1xuICAgIGNvbnN0IHRva2VuUmV0cmlldmVyVGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zX3Rhc2tzLkxhbWJkYUludm9rZShcbiAgICAgIHRoaXMsXG4gICAgICAnR2V0IFJ1bm5lciBUb2tlbicsXG4gICAgICB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uOiB0aGlzLnRva2VuUmV0cmlldmVyKCksXG4gICAgICAgIHBheWxvYWRSZXNwb25zZU9ubHk6IHRydWUsXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLnJ1bm5lcicsXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBsZXQgZGVsZXRlUnVubmVyRnVuY3Rpb24gPSB0aGlzLmRlbGV0ZVJ1bm5lcigpO1xuICAgIGNvbnN0IGRlbGV0ZVJ1bm5lclRhc2sgPSBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5MYW1iZGFJbnZva2UoXG4gICAgICB0aGlzLFxuICAgICAgJ0RlbGV0ZSBSdW5uZXInLFxuICAgICAge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbjogZGVsZXRlUnVubmVyRnVuY3Rpb24sXG4gICAgICAgIHBheWxvYWRSZXNwb25zZU9ubHk6IHRydWUsXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLmRlbGV0ZScsXG4gICAgICAgIHBheWxvYWQ6IHN0ZXBmdW5jdGlvbnMuVGFza0lucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgIHJ1bm5lck5hbWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQkLkV4ZWN1dGlvbi5OYW1lJyksXG4gICAgICAgICAgb3duZXI6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQub3duZXInKSxcbiAgICAgICAgICByZXBvOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnJlcG8nKSxcbiAgICAgICAgICBydW5JZDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5ydW5JZCcpLFxuICAgICAgICAgIGluc3RhbGxhdGlvbklkOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLmluc3RhbGxhdGlvbklkJyksXG4gICAgICAgICAgaWRsZU9ubHk6IGZhbHNlLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgKTtcbiAgICBkZWxldGVSdW5uZXJUYXNrLmFkZFJldHJ5KHtcbiAgICAgIGVycm9yczogW1xuICAgICAgICAnUnVubmVyQnVzeScsXG4gICAgICBdLFxuICAgICAgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgYmFja29mZlJhdGU6IDEsXG4gICAgICBtYXhBdHRlbXB0czogNjAsXG4gICAgfSk7XG5cbiAgICBjb25zdCB3YWl0Rm9ySWRsZVJ1bm5lciA9IG5ldyBzdGVwZnVuY3Rpb25zLldhaXQodGhpcywgJ1dhaXQnLCB7XG4gICAgICB0aW1lOiBzdGVwZnVuY3Rpb25zLldhaXRUaW1lLmR1cmF0aW9uKHByb3BzPy5pZGxlVGltZW91dCA/PyBjZGsuRHVyYXRpb24ubWludXRlcygxMCkpLFxuICAgIH0pO1xuICAgIGNvbnN0IGRlbGV0ZUlkbGVSdW5uZXJUYXNrID0gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuTGFtYmRhSW52b2tlKFxuICAgICAgdGhpcyxcbiAgICAgICdEZWxldGUgSWRsZSBSdW5uZXInLFxuICAgICAge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbjogZGVsZXRlUnVubmVyRnVuY3Rpb24sXG4gICAgICAgIHBheWxvYWRSZXNwb25zZU9ubHk6IHRydWUsXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLmRlbGV0ZScsXG4gICAgICAgIHBheWxvYWQ6IHN0ZXBmdW5jdGlvbnMuVGFza0lucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgIHJ1bm5lck5hbWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQkLkV4ZWN1dGlvbi5OYW1lJyksXG4gICAgICAgICAgb3duZXI6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQub3duZXInKSxcbiAgICAgICAgICByZXBvOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnJlcG8nKSxcbiAgICAgICAgICBydW5JZDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5ydW5JZCcpLFxuICAgICAgICAgIGluc3RhbGxhdGlvbklkOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLmluc3RhbGxhdGlvbklkJyksXG4gICAgICAgICAgaWRsZU9ubHk6IHRydWUsXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgcHJvdmlkZXJDaG9vc2VyID0gbmV3IHN0ZXBmdW5jdGlvbnMuQ2hvaWNlKHRoaXMsICdDaG9vc2UgcHJvdmlkZXInKTtcbiAgICBmb3IgKGNvbnN0IHByb3ZpZGVyIG9mIHRoaXMucHJvdmlkZXJzKSB7XG4gICAgICBjb25zdCBwcm92aWRlclRhc2sgPSBwcm92aWRlci5nZXRTdGVwRnVuY3Rpb25UYXNrKFxuICAgICAgICB7XG4gICAgICAgICAgcnVubmVyVG9rZW5QYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnJ1bm5lci50b2tlbicpLFxuICAgICAgICAgIHJ1bm5lck5hbWVQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckJC5FeGVjdXRpb24uTmFtZScpLFxuICAgICAgICAgIGdpdGh1YkRvbWFpblBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQucnVubmVyLmRvbWFpbicpLFxuICAgICAgICAgIG93bmVyUGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5vd25lcicpLFxuICAgICAgICAgIHJlcG9QYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnJlcG8nKSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgICBwcm92aWRlckNob29zZXIud2hlbihcbiAgICAgICAgc3RlcGZ1bmN0aW9ucy5Db25kaXRpb24uYW5kKFxuICAgICAgICAgIC4uLnByb3ZpZGVyLmxhYmVscy5tYXAoXG4gICAgICAgICAgICBsYWJlbCA9PiBzdGVwZnVuY3Rpb25zLkNvbmRpdGlvbi5pc1ByZXNlbnQoYCQubGFiZWxzLiR7bGFiZWwudG9Mb3dlckNhc2UoKX1gKSxcbiAgICAgICAgICApLFxuICAgICAgICApLFxuICAgICAgICBwcm92aWRlclRhc2ssXG4gICAgICApO1xuICAgIH1cblxuICAgIHByb3ZpZGVyQ2hvb3Nlci5vdGhlcndpc2UobmV3IHN0ZXBmdW5jdGlvbnMuU3VjY2VlZCh0aGlzLCAnVW5rbm93biBsYWJlbCcpKTtcblxuICAgIGNvbnN0IHdvcmsgPSB0b2tlblJldHJpZXZlclRhc2submV4dChcbiAgICAgIG5ldyBzdGVwZnVuY3Rpb25zLlBhcmFsbGVsKHRoaXMsICdFcnJvciBDYXRjaGVyJywgeyByZXN1bHRQYXRoOiAnJC5yZXN1bHQnIH0pXG4gICAgICAgIC5icmFuY2gocHJvdmlkZXJDaG9vc2VyKVxuICAgICAgICAuYnJhbmNoKHdhaXRGb3JJZGxlUnVubmVyLm5leHQoZGVsZXRlSWRsZVJ1bm5lclRhc2spKVxuICAgICAgICAuYWRkQ2F0Y2goXG4gICAgICAgICAgZGVsZXRlUnVubmVyVGFza1xuICAgICAgICAgICAgLm5leHQobmV3IHN0ZXBmdW5jdGlvbnMuRmFpbCh0aGlzLCAnUnVubmVyIEZhaWxlZCcpKSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICByZXN1bHRQYXRoOiAnJC5lcnJvcicsXG4gICAgICAgICAgfSxcbiAgICAgICAgKSxcbiAgICApO1xuXG4gICAgbGV0IGxvZ09wdGlvbnM6IGNkay5hd3Nfc3RlcGZ1bmN0aW9ucy5Mb2dPcHRpb25zIHwgdW5kZWZpbmVkO1xuICAgIGlmICh0aGlzLnByb3BzPy5sb2dPcHRpb25zKSB7XG4gICAgICB0aGlzLnN0YXRlTWFjaGluZUxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ0xvZ3MnLCB7XG4gICAgICAgIGxvZ0dyb3VwTmFtZTogcHJvcHM/LmxvZ09wdGlvbnM/LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgcmV0ZW50aW9uOiBwcm9wcz8ubG9nT3B0aW9ucz8ubG9nUmV0ZW50aW9uID8/IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0pO1xuXG4gICAgICBsb2dPcHRpb25zID0ge1xuICAgICAgICBkZXN0aW5hdGlvbjogdGhpcy5zdGF0ZU1hY2hpbmVMb2dHcm91cCxcbiAgICAgICAgaW5jbHVkZUV4ZWN1dGlvbkRhdGE6IHByb3BzPy5sb2dPcHRpb25zPy5pbmNsdWRlRXhlY3V0aW9uRGF0YSA/PyB0cnVlLFxuICAgICAgICBsZXZlbDogcHJvcHM/LmxvZ09wdGlvbnM/LmxldmVsID8/IHN0ZXBmdW5jdGlvbnMuTG9nTGV2ZWwuQUxMLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoXG4gICAgICB0aGlzLFxuICAgICAgJ1J1bm5lciBPcmNoZXN0cmF0b3InLFxuICAgICAge1xuICAgICAgICBkZWZpbml0aW9uOiB3b3JrLFxuICAgICAgICBsb2dzOiBsb2dPcHRpb25zLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgZm9yIChjb25zdCBwcm92aWRlciBvZiB0aGlzLnByb3ZpZGVycykge1xuICAgICAgcHJvdmlkZXIuZ3JhbnRTdGF0ZU1hY2hpbmUoc3RhdGVNYWNoaW5lKTtcbiAgICB9XG5cbiAgICByZXR1cm4gc3RhdGVNYWNoaW5lO1xuICB9XG5cbiAgcHJpdmF0ZSB0b2tlblJldHJpZXZlcigpIHtcbiAgICBjb25zdCBmdW5jID0gbmV3IFRva2VuUmV0cmlldmVyRnVuY3Rpb24oXG4gICAgICB0aGlzLFxuICAgICAgJ3Rva2VuLXJldHJpZXZlcicsXG4gICAgICB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiAnR2V0IHRva2VuIGZyb20gR2l0SHViIEFjdGlvbnMgdXNlZCB0byBzdGFydCBuZXcgc2VsZi1ob3N0ZWQgcnVubmVyJyxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBHSVRIVUJfU0VDUkVUX0FSTjogdGhpcy5zZWNyZXRzLmdpdGh1Yi5zZWNyZXRBcm4sXG4gICAgICAgICAgR0lUSFVCX1BSSVZBVEVfS0VZX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy5naXRodWJQcml2YXRlS2V5LnNlY3JldEFybixcbiAgICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhRW52LFxuICAgICAgICB9LFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgLi4udGhpcy5leHRyYUxhbWJkYVByb3BzLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5zZWNyZXRzLmdpdGh1Yi5ncmFudFJlYWQoZnVuYyk7XG4gICAgdGhpcy5zZWNyZXRzLmdpdGh1YlByaXZhdGVLZXkuZ3JhbnRSZWFkKGZ1bmMpO1xuXG4gICAgcmV0dXJuIGZ1bmM7XG4gIH1cblxuICBwcml2YXRlIGRlbGV0ZVJ1bm5lcigpIHtcbiAgICBjb25zdCBmdW5jID0gbmV3IERlbGV0ZVJ1bm5lckZ1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdkZWxldGUtcnVubmVyJyxcbiAgICAgIHtcbiAgICAgICAgZGVzY3JpcHRpb246ICdEZWxldGUgR2l0SHViIEFjdGlvbnMgcnVubmVyIG9uIGVycm9yJyxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBHSVRIVUJfU0VDUkVUX0FSTjogdGhpcy5zZWNyZXRzLmdpdGh1Yi5zZWNyZXRBcm4sXG4gICAgICAgICAgR0lUSFVCX1BSSVZBVEVfS0VZX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy5naXRodWJQcml2YXRlS2V5LnNlY3JldEFybixcbiAgICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhRW52LFxuICAgICAgICB9LFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgLi4udGhpcy5leHRyYUxhbWJkYVByb3BzLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5zZWNyZXRzLmdpdGh1Yi5ncmFudFJlYWQoZnVuYyk7XG4gICAgdGhpcy5zZWNyZXRzLmdpdGh1YlByaXZhdGVLZXkuZ3JhbnRSZWFkKGZ1bmMpO1xuXG4gICAgcmV0dXJuIGZ1bmM7XG4gIH1cblxuICBwcml2YXRlIHN0YXR1c0Z1bmN0aW9uKCkge1xuICAgIGNvbnN0IHN0YXR1c0Z1bmN0aW9uID0gbmV3IFN0YXR1c0Z1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdzdGF0dXMnLFxuICAgICAge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ1Byb3ZpZGUgdXNlciB3aXRoIHN0YXR1cyBhYm91dCBzZWxmLWhvc3RlZCBHaXRIdWIgQWN0aW9ucyBydW5uZXJzJyxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBXRUJIT09LX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy53ZWJob29rLnNlY3JldEFybixcbiAgICAgICAgICBHSVRIVUJfU0VDUkVUX0FSTjogdGhpcy5zZWNyZXRzLmdpdGh1Yi5zZWNyZXRBcm4sXG4gICAgICAgICAgR0lUSFVCX1BSSVZBVEVfS0VZX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy5naXRodWJQcml2YXRlS2V5LnNlY3JldEFybixcbiAgICAgICAgICBTRVRVUF9TRUNSRVRfQVJOOiB0aGlzLnNlY3JldHMuc2V0dXAuc2VjcmV0QXJuLFxuICAgICAgICAgIFdFQkhPT0tfVVJMOiB0aGlzLndlYmhvb2sudXJsLFxuICAgICAgICAgIFdFQkhPT0tfSEFORExFUl9BUk46IHRoaXMud2ViaG9vay5oYW5kbGVyLmxhdGVzdFZlcnNpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgICAgU1RFUF9GVU5DVElPTl9BUk46IHRoaXMub3JjaGVzdHJhdG9yLnN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICBTVEVQX0ZVTkNUSU9OX0xPR19HUk9VUDogdGhpcy5zdGF0ZU1hY2hpbmVMb2dHcm91cD8ubG9nR3JvdXBOYW1lID8/ICcnLFxuICAgICAgICAgIFNFVFVQX0ZVTkNUSU9OX1VSTDogdGhpcy5zZXR1cFVybCxcbiAgICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhRW52LFxuICAgICAgICB9LFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygzKSxcbiAgICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhUHJvcHMsXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBjb25zdCBwcm92aWRlcnMgPSB0aGlzLnByb3ZpZGVycy5tYXAocHJvdmlkZXIgPT4gcHJvdmlkZXIuc3RhdHVzKHN0YXR1c0Z1bmN0aW9uKSk7XG5cbiAgICAvLyBleHBvc2UgcHJvdmlkZXJzIGFzIHN0YWNrIG1ldGFkYXRhIGFzIGl0J3MgdG9vIGJpZyBmb3IgTGFtYmRhIGVudmlyb25tZW50IHZhcmlhYmxlc1xuICAgIC8vIHNwZWNpZmljYWxseSBpbnRlZ3JhdGlvbiB0ZXN0aW5nIGdvdCBhbiBlcnJvciBiZWNhdXNlIGxhbWJkYSB1cGRhdGUgcmVxdWVzdCB3YXMgPjVrYlxuICAgIGNvbnN0IHN0YWNrID0gY2RrLlN0YWNrLm9mKHRoaXMpO1xuICAgIGNvbnN0IGYgPSAoc3RhdHVzRnVuY3Rpb24ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgbGFtYmRhLkNmbkZ1bmN0aW9uKTtcbiAgICBmLmFkZFByb3BlcnR5T3ZlcnJpZGUoJ0Vudmlyb25tZW50LlZhcmlhYmxlcy5MT0dJQ0FMX0lEJywgZi5sb2dpY2FsSWQpO1xuICAgIGYuYWRkUHJvcGVydHlPdmVycmlkZSgnRW52aXJvbm1lbnQuVmFyaWFibGVzLlNUQUNLX05BTUUnLCBzdGFjay5zdGFja05hbWUpO1xuICAgIGYuYWRkTWV0YWRhdGEoJ3Byb3ZpZGVycycsIHByb3ZpZGVycyk7XG4gICAgc3RhdHVzRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja1Jlc291cmNlJ10sXG4gICAgICByZXNvdXJjZXM6IFtzdGFjay5zdGFja0lkXSxcbiAgICB9KSk7XG5cbiAgICB0aGlzLnNlY3JldHMud2ViaG9vay5ncmFudFJlYWQoc3RhdHVzRnVuY3Rpb24pO1xuICAgIHRoaXMuc2VjcmV0cy5naXRodWIuZ3JhbnRSZWFkKHN0YXR1c0Z1bmN0aW9uKTtcbiAgICB0aGlzLnNlY3JldHMuZ2l0aHViUHJpdmF0ZUtleS5ncmFudFJlYWQoc3RhdHVzRnVuY3Rpb24pO1xuICAgIHRoaXMuc2VjcmV0cy5zZXR1cC5ncmFudFJlYWQoc3RhdHVzRnVuY3Rpb24pO1xuICAgIHRoaXMub3JjaGVzdHJhdG9yLmdyYW50UmVhZChzdGF0dXNGdW5jdGlvbik7XG5cbiAgICBuZXcgY2RrLkNmbk91dHB1dChcbiAgICAgIHRoaXMsXG4gICAgICAnc3RhdHVzIGNvbW1hbmQnLFxuICAgICAge1xuICAgICAgICB2YWx1ZTogYGF3cyAtLXJlZ2lvbiAke3N0YWNrLnJlZ2lvbn0gbGFtYmRhIGludm9rZSAtLWZ1bmN0aW9uLW5hbWUgJHtzdGF0dXNGdW5jdGlvbi5mdW5jdGlvbk5hbWV9IHN0YXR1cy5qc29uYCxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGNvbnN0IGFjY2VzcyA9IHRoaXMucHJvcHM/LnN0YXR1c0FjY2VzcyA/PyBMYW1iZGFBY2Nlc3Mubm9BY2Nlc3MoKTtcbiAgICBjb25zdCB1cmwgPSBhY2Nlc3MuX2JpbmQodGhpcywgJ3N0YXR1cyBhY2Nlc3MnLCBzdGF0dXNGdW5jdGlvbik7XG5cbiAgICBpZiAodXJsICE9PSAnJykge1xuICAgICAgbmV3IGNkay5DZm5PdXRwdXQoXG4gICAgICAgIHRoaXMsXG4gICAgICAgICdzdGF0dXMgdXJsJyxcbiAgICAgICAge1xuICAgICAgICAgIHZhbHVlOiB1cmwsXG4gICAgICAgIH0sXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBGdW5jdGlvbigpOiBzdHJpbmcge1xuICAgIGNvbnN0IHNldHVwRnVuY3Rpb24gPSBuZXcgU2V0dXBGdW5jdGlvbihcbiAgICAgIHRoaXMsXG4gICAgICAnc2V0dXAnLFxuICAgICAge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ1NldHVwIEdpdEh1YiBBY3Rpb25zIGludGVncmF0aW9uIHdpdGggc2VsZi1ob3N0ZWQgcnVubmVycycsXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgU0VUVVBfU0VDUkVUX0FSTjogdGhpcy5zZWNyZXRzLnNldHVwLnNlY3JldEFybixcbiAgICAgICAgICBXRUJIT09LX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy53ZWJob29rLnNlY3JldEFybixcbiAgICAgICAgICBHSVRIVUJfU0VDUkVUX0FSTjogdGhpcy5zZWNyZXRzLmdpdGh1Yi5zZWNyZXRBcm4sXG4gICAgICAgICAgR0lUSFVCX1BSSVZBVEVfS0VZX1NFQ1JFVF9BUk46IHRoaXMuc2VjcmV0cy5naXRodWJQcml2YXRlS2V5LnNlY3JldEFybixcbiAgICAgICAgICBXRUJIT09LX1VSTDogdGhpcy53ZWJob29rLnVybCxcbiAgICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhRW52LFxuICAgICAgICB9LFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygzKSxcbiAgICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICAuLi50aGlzLmV4dHJhTGFtYmRhUHJvcHMsXG4gICAgICB9LFxuICAgICk7XG5cbiAgICAvLyB0aGlzLnNlY3JldHMud2ViaG9vay5ncmFudFJlYWQoc2V0dXBGdW5jdGlvbik7XG4gICAgdGhpcy5zZWNyZXRzLndlYmhvb2suZ3JhbnRXcml0ZShzZXR1cEZ1bmN0aW9uKTtcbiAgICB0aGlzLnNlY3JldHMuZ2l0aHViLmdyYW50UmVhZChzZXR1cEZ1bmN0aW9uKTtcbiAgICB0aGlzLnNlY3JldHMuZ2l0aHViLmdyYW50V3JpdGUoc2V0dXBGdW5jdGlvbik7XG4gICAgLy8gdGhpcy5zZWNyZXRzLmdpdGh1YlByaXZhdGVLZXkuZ3JhbnRSZWFkKHNldHVwRnVuY3Rpb24pO1xuICAgIHRoaXMuc2VjcmV0cy5naXRodWJQcml2YXRlS2V5LmdyYW50V3JpdGUoc2V0dXBGdW5jdGlvbik7XG4gICAgdGhpcy5zZWNyZXRzLnNldHVwLmdyYW50UmVhZChzZXR1cEZ1bmN0aW9uKTtcbiAgICB0aGlzLnNlY3JldHMuc2V0dXAuZ3JhbnRXcml0ZShzZXR1cEZ1bmN0aW9uKTtcblxuICAgIGNvbnN0IGFjY2VzcyA9IHRoaXMucHJvcHM/LnNldHVwQWNjZXNzID8/IExhbWJkYUFjY2Vzcy5sYW1iZGFVcmwoKTtcbiAgICByZXR1cm4gYWNjZXNzLl9iaW5kKHRoaXMsICdzZXR1cCBhY2Nlc3MnLCBzZXR1cEZ1bmN0aW9uKTtcbiAgfVxuXG4gIHByaXZhdGUgY2hlY2tJbnRlcnNlY3RpbmdMYWJlbHMoKSB7XG4gICAgLy8gdGhpcyBcImFsZ29yaXRobVwiIGlzIHZlcnkgaW5lZmZpY2llbnQsIGJ1dCBnb29kIGVub3VnaCBmb3IgdGhlIHRpbnkgZGF0YXNldHMgd2UgZXhwZWN0XG4gICAgZm9yIChjb25zdCBwMSBvZiB0aGlzLnByb3ZpZGVycykge1xuICAgICAgZm9yIChjb25zdCBwMiBvZiB0aGlzLnByb3ZpZGVycykge1xuICAgICAgICBpZiAocDEgPT0gcDIpIHtcbiAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocDEubGFiZWxzLmV2ZXJ5KGwgPT4gcDIubGFiZWxzLmluY2x1ZGVzKGwpKSkge1xuICAgICAgICAgIGlmIChwMi5sYWJlbHMuZXZlcnkobCA9PiBwMS5sYWJlbHMuaW5jbHVkZXMobCkpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEJvdGggJHtwMS5ub2RlLnBhdGh9IGFuZCAke3AyLm5vZGUucGF0aH0gdXNlIHRoZSBzYW1lIGxhYmVscyBbJHtwMS5sYWJlbHMuam9pbignLCAnKX1dYCk7XG4gICAgICAgICAgfVxuICAgICAgICAgIEFubm90YXRpb25zLm9mKHAxKS5hZGRXYXJuaW5nKGBMYWJlbHMgWyR7cDEubGFiZWxzLmpvaW4oJywgJyl9XSBpbnRlcnNlY3Qgd2l0aCBhbm90aGVyIHByb3ZpZGVyICgke3AyLm5vZGUucGF0aH0gLS0gWyR7cDIubGFiZWxzLmpvaW4oJywgJyl9XSkuIElmIGEgd29ya2Zsb3cgc3BlY2lmaWVzIHRoZSBsYWJlbHMgWyR7cDEubGFiZWxzLmpvaW4oJywgJyl9XSwgaXQgaXMgbm90IGd1YXJhbnRlZWQgd2hpY2ggcHJvdmlkZXIgd2lsbCBiZSB1c2VkLiBJdCBpcyByZWNvbW1lbmRlZCB5b3UgZG8gbm90IHVzZSBpbnRlcnNlY3RpbmcgbGFiZWxzYCk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIEdpdEh1YiBBY3Rpb25zIGpvYnMgY29tcGxldGVkLiBJdCBoYXMgYFByb3ZpZGVyTGFiZWxzYCBhbmQgYFN0YXR1c2AgZGltZW5zaW9ucy4gVGhlIHN0YXR1cyBjYW4gYmUgb25lIG9mIFwiU3VjY2VlZGVkXCIsIFwiU3VjY2VlZGVkV2l0aElzc3Vlc1wiLCBcIkZhaWxlZFwiLCBcIkNhbmNlbGVkXCIsIFwiU2tpcHBlZFwiLCBvciBcIkFiYW5kb25lZFwiLlxuICAgKlxuICAgKiAqKldBUk5JTkc6KiogdGhpcyBtZXRob2QgY3JlYXRlcyBhIG1ldHJpYyBmaWx0ZXIgZm9yIGVhY2ggcHJvdmlkZXIuIEVhY2ggbWV0cmljIGhhcyBhIHN0YXR1cyBkaW1lbnNpb24gd2l0aCBzaXggcG9zc2libGUgdmFsdWVzLiBUaGVzZSByZXNvdXJjZXMgbWF5IGluY3VyIGNvc3QuXG4gICAqL1xuICBwdWJsaWMgbWV0cmljSm9iQ29tcGxldGVkKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNQcm9wcyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICBpZiAoIXRoaXMuam9ic0NvbXBsZXRlZE1ldHJpY0ZpbHRlcnMpIHtcbiAgICAgIC8vIHdlIGNhbid0IHVzZSBsb2dzLkZpbHRlclBhdHRlcm4uc3BhY2VEZWxpbWl0ZWQoKSBiZWNhdXNlIGl0IGhhcyBubyBzdXBwb3J0IGZvciB8fFxuICAgICAgLy8gc3RhdHVzIGxpc3QgdGFrZW4gZnJvbSBodHRwczovL2dpdGh1Yi5jb20vYWN0aW9ucy9ydW5uZXIvYmxvYi9iZTk2MzIzMDJjZWVmNTBiZmIzNmVhOTk4Y2VhOWM5NGM3NWU1ZDRkL3NyYy9TZGsvRFRXZWJBcGkvV2ViQXBpL1Rhc2tSZXN1bHQuY3NcbiAgICAgIC8vIHdlIG5lZWQgXCIuLi5cIiBmb3IgTGFtYmRhIHRoYXQgcHJlZml4ZXMgc29tZSBleHRyYSBkYXRhIHRvIGxvZyBsaW5lc1xuICAgICAgY29uc3QgcGF0dGVybiA9IGxvZ3MuRmlsdGVyUGF0dGVybi5saXRlcmFsKCdbLi4uLCBtYXJrZXIgPSBcIkNES0dIQVwiLCBqb2IgPSBcIkpPQlwiLCBkb25lID0gXCJET05FXCIsIGxhYmVscywgc3RhdHVzID0gXCJTdWNjZWVkZWRcIiB8fCBzdGF0dXMgPSBcIlN1Y2NlZWRlZFdpdGhJc3N1ZXNcIiB8fCBzdGF0dXMgPSBcIkZhaWxlZFwiIHx8IHN0YXR1cyA9IFwiQ2FuY2VsZWRcIiB8fCBzdGF0dXMgPSBcIlNraXBwZWRcIiB8fCBzdGF0dXMgPSBcIkFiYW5kb25lZFwiXScpO1xuXG4gICAgICB0aGlzLmpvYnNDb21wbGV0ZWRNZXRyaWNGaWx0ZXJzID0gdGhpcy5wcm92aWRlcnMubWFwKHAgPT5cbiAgICAgICAgcC5sb2dHcm91cC5hZGRNZXRyaWNGaWx0ZXIoYCR7cC5sb2dHcm91cC5ub2RlLmlkfSBmaWx0ZXJgLCB7XG4gICAgICAgICAgbWV0cmljTmFtZXNwYWNlOiAnR2l0SHViUnVubmVycycsXG4gICAgICAgICAgbWV0cmljTmFtZTogJ0pvYkNvbXBsZXRlZCcsXG4gICAgICAgICAgZmlsdGVyUGF0dGVybjogcGF0dGVybixcbiAgICAgICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgICAgICAgIC8vIGNhbid0IHdpdGggZGltZW5zaW9ucyAtLSBkZWZhdWx0VmFsdWU6IDAsXG4gICAgICAgICAgZGltZW5zaW9uczoge1xuICAgICAgICAgICAgUHJvdmlkZXJMYWJlbHM6ICckbGFiZWxzJyxcbiAgICAgICAgICAgIFN0YXR1czogJyRzdGF0dXMnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcblxuICAgICAgZm9yIChjb25zdCBtZXRyaWNGaWx0ZXIgb2YgdGhpcy5qb2JzQ29tcGxldGVkTWV0cmljRmlsdGVycykge1xuICAgICAgICBpZiAobWV0cmljRmlsdGVyLm5vZGUuZGVmYXVsdENoaWxkIGluc3RhbmNlb2YgbG9ncy5DZm5NZXRyaWNGaWx0ZXIpIHtcbiAgICAgICAgICBtZXRyaWNGaWx0ZXIubm9kZS5kZWZhdWx0Q2hpbGQuYWRkUHJvcGVydHlPdmVycmlkZSgnTWV0cmljVHJhbnNmb3JtYXRpb25zLjAuVW5pdCcsICdDb3VudCcpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIEFubm90YXRpb25zLm9mKG1ldHJpY0ZpbHRlcikuYWRkV2FybmluZygnVW5hYmxlIHRvIHNldCBtZXRyaWMgZmlsdGVyIFVuaXQgdG8gQ291bnQnKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnR2l0SHViUnVubmVycycsXG4gICAgICBtZXRyaWNOYW1lOiAnSm9ic0NvbXBsZXRlZCcsXG4gICAgICB1bml0OiBjbG91ZHdhdGNoLlVuaXQuQ09VTlQsXG4gICAgICBzdGF0aXN0aWM6IGNsb3Vkd2F0Y2guU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3Igc3VjY2Vzc2Z1bCBleGVjdXRpb25zLlxuICAgKlxuICAgKiBBIHN1Y2Nlc3NmdWwgZXhlY3V0aW9uIGRvZXNuJ3QgYWx3YXlzIG1lYW4gYSBydW5uZXIgd2FzIHN0YXJ0ZWQuIEl0IGNhbiBiZSBzdWNjZXNzZnVsIGV2ZW4gd2l0aG91dCBhbnkgbGFiZWwgbWF0Y2hlcy5cbiAgICpcbiAgICogQSBzdWNjZXNzZnVsIHJ1bm5lciBkb2Vzbid0IG1lYW4gdGhlIGpvYiBpdCBleGVjdXRlZCB3YXMgc3VjY2Vzc2Z1bC4gRm9yIHRoYXQsIHNlZSB7QGxpbmsgbWV0cmljSm9iQ29tcGxldGVkfS5cbiAgICovXG4gIHB1YmxpYyBtZXRyaWNTdWNjZWVkZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY1Byb3BzKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm9yY2hlc3RyYXRvci5tZXRyaWNTdWNjZWVkZWQocHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgZmFpbGVkIHJ1bm5lciBleGVjdXRpb25zLlxuICAgKlxuICAgKiBBIGZhaWxlZCBydW5uZXIgdXN1YWxseSBtZWFucyB0aGUgcnVubmVyIGZhaWxlZCB0byBzdGFydCBhbmQgc28gYSBqb2Igd2FzIG5ldmVyIGV4ZWN1dGVkLiBJdCBkb2Vzbid0IG5lY2Vzc2FyaWx5IG1lYW4gdGhlIGpvYiB3YXMgZXhlY3V0ZWQgYW5kIGZhaWxlZC4gRm9yIHRoYXQsIHNlZSB7QGxpbmsgbWV0cmljSm9iQ29tcGxldGVkfS5cbiAgICovXG4gIHB1YmxpYyBtZXRyaWNGYWlsZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY1Byb3BzKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm9yY2hlc3RyYXRvci5tZXRyaWNGYWlsZWQocHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIGludGVydmFsLCBpbiBtaWxsaXNlY29uZHMsIGJldHdlZW4gdGhlIHRpbWUgdGhlIGV4ZWN1dGlvbiBzdGFydHMgYW5kIHRoZSB0aW1lIGl0IGNsb3Nlcy4gVGhpcyB0aW1lIG1heSBiZSBsb25nZXIgdGhhbiB0aGUgdGltZSB0aGUgcnVubmVyIHRvb2suXG4gICAqL1xuICBwdWJsaWMgbWV0cmljVGltZShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljUHJvcHMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMub3JjaGVzdHJhdG9yLm1ldHJpY1RpbWUocHJvcHMpO1xuICB9XG59XG4iXX0=