"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildRunnerImageBuilder = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_image_builder_1 = require("./aws-image-builder");
const common_1 = require("./common");
const build_image_function_1 = require("../../lambdas/build-image-function");
const utils_1 = require("../../utils");
const common_2 = require("../common");
/**
 * @internal
 */
class CodeBuildRunnerImageBuilder extends common_1.RunnerImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props?.awsImageBuilderOptions) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('awsImageBuilderOptions are ignored when using CodeBuild runner image builder.');
        }
        this.os = props?.os ?? common_2.Os.LINUX_UBUNTU;
        this.architecture = props?.architecture ?? common_2.Architecture.X86_64;
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        this.logRetention = props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        this.vpc = props?.vpc;
        this.securityGroups = props?.securityGroups;
        this.subnetSelection = props?.subnetSelection;
        this.timeout = props?.codeBuildOptions?.timeout ?? aws_cdk_lib_1.Duration.hours(1);
        this.computeType = props?.codeBuildOptions?.computeType ?? aws_codebuild_1.ComputeType.SMALL;
        this.baseImage = props?.baseDockerImage ?? (0, aws_image_builder_1.defaultBaseDockerImage)(this.os);
        this.buildImage = props?.codeBuildOptions?.buildImage ?? this.getDefaultBuildImage();
        // warn against isolated networks
        if (props?.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // create service role for CodeBuild
        this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
    }
    bindAmi() {
        throw new Error('CodeBuild image builder cannot be used to build AMI');
    }
    bindDockerImage() {
        if (this.boundDockerImage) {
            return this.boundDockerImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const buildSpec = this.getBuildSpec(this.repository, logGroup);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec,
            vpc: this.vpc,
            securityGroups: this.securityGroups,
            subnetSelection: this.subnetSelection,
            role: this.role,
            timeout: this.timeout,
            environment: {
                buildImage: this.buildImage,
                computeType: this.computeType,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        // call CodeBuild during deployment and delete all images from repository during destruction
        const cr = this.customResource(project, buildSpec.toBuildSpec());
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.rebuildInterval);
        // return the image
        this.boundDockerImage = {
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryAttributes(this, 'Dependable Image', {
                // There are simpler ways to get name and ARN, but we want an image object that depends on the custom resource.
                // We want whoever is using this image to automatically wait for CodeBuild to start and finish through the custom resource.
                repositoryName: cr.getAttString('Name'),
                repositoryArn: cr.ref,
            }),
            imageTag: 'latest',
            architecture: this.architecture,
            os: this.os,
            logGroup,
            runnerVersion: common_2.RunnerVersion.specific('unknown'),
        };
        return this.boundDockerImage;
    }
    getDefaultBuildImage() {
        if (this.os.is(common_2.Os.LINUX_UBUNTU) || this.os.is(common_2.Os.LINUX_AMAZON_2) || this.os.is(common_2.Os.LINUX)) {
            // CodeBuild just runs `docker build` so its OS doesn't really matter
            if (this.architecture.is(common_2.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0;
            }
            else if (this.architecture.is(common_2.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
            }
        }
        if (this.os.is(common_2.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getDockerfileGenerationCommands() {
        let commands = [];
        let dockerfile = `FROM ${this.baseImage}\nVOLUME /var/lib/docker\n`;
        for (let i = 0; i < this.components.length; i++) {
            const componentName = this.components[i].name;
            const assetDescriptors = this.components[i].getAssets(this.os, this.architecture);
            for (let j = 0; j < assetDescriptors.length; j++) {
                if (this.os.is(common_2.Os.WINDOWS)) {
                    throw new Error("Can't add asset as we can't build Windows Docker images on CodeBuild");
                }
                const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, `Component ${i} ${componentName} Asset ${j}`, {
                    path: assetDescriptors[j].source,
                });
                if (asset.isFile) {
                    commands.push(`aws s3 cp ${asset.s3ObjectUrl} asset${i}-${componentName}-${j}`);
                }
                else if (asset.isZipArchive) {
                    commands.push(`aws s3 cp ${asset.s3ObjectUrl} asset${i}-${componentName}-${j}.zip`);
                    commands.push(`unzip asset${i}-${componentName}-${j}.zip -d "asset${i}-${componentName}-${j}"`);
                }
                else {
                    throw new Error(`Unknown asset type: ${asset}`);
                }
                dockerfile += `COPY asset${i}-${componentName}-${j} ${assetDescriptors[j].target}\n`;
                asset.grantRead(this);
            }
            const componentCommands = this.components[i].getCommands(this.os, this.architecture);
            const script = '#!/bin/bash\nset -exuo pipefail\n' + componentCommands.join('\n');
            commands.push(`cat > component${i}-${componentName}.sh <<'EOFGITHUBRUNNERSDOCKERFILE'\n${script}\nEOFGITHUBRUNNERSDOCKERFILE`);
            commands.push(`chmod +x component${i}-${componentName}.sh`);
            dockerfile += `COPY component${i}-${componentName}.sh /tmp\n`;
            dockerfile += `RUN /tmp/component${i}-${componentName}.sh\n`;
            dockerfile += this.components[i].getDockerCommands(this.os, this.architecture).join('\n') + '\n';
        }
        commands.push(`cat > Dockerfile <<'EOFGITHUBRUNNERSDOCKERFILE'\n${dockerfile}\nEOFGITHUBRUNNERSDOCKERFILE`);
        return commands;
    }
    getBuildSpec(repository, logGroup) {
        const thisStack = cdk.Stack.of(this);
        return aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
            version: '0.2',
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    STACK_ID: 'unspecified',
                    REQUEST_ID: 'unspecified',
                    LOGICAL_RESOURCE_ID: 'unspecified',
                    RESPONSE_URL: 'unspecified',
                },
            },
            phases: {
                pre_build: {
                    commands: [
                        `aws ecr get-login-password --region "$AWS_DEFAULT_REGION" | docker login --username AWS --password-stdin ${thisStack.account}.dkr.ecr.${thisStack.region}.amazonaws.com`,
                    ],
                },
                build: {
                    commands: this.getDockerfileGenerationCommands().concat([
                        'docker build . -t "$REPO_URI"',
                        'docker push "$REPO_URI"',
                    ]),
                },
                post_build: {
                    commands: [
                        'STATUS="SUCCESS"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILED"; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "StackId": "$STACK_ID",\n' +
                            '  "RequestId": "$REQUEST_ID",\n' +
                            '  "LogicalResourceId": "$LOGICAL_RESOURCE_ID",\n' +
                            '  "PhysicalResourceId": "$REPO_ARN",\n' +
                            '  "Status": "$STATUS",\n' +
                            `  "Reason": "See logs in ${logGroup.logGroupName}/$CODEBUILD_LOG_PATH (deploy again with \'cdk deploy -R\' or logRemovalPolicy=RemovalPolicy.RETAIN if they are already deleted)",\n` +
                            `  "Data": {"Name": "${repository.repositoryName}"}\n` +
                            '}\n' +
                            'EOF',
                        'if [ "$RESPONSE_URL" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$RESPONSE_URL"; fi',
                    ],
                },
            },
        });
    }
    customResource(project, buildSpec) {
        const crHandler = (0, utils_1.singletonLambda)(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
            ],
        });
        crHandler.role.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                // We include the full buildSpec so the image is built immediately on changes, and we don't have to wait for its scheduled build.
                // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
                BuildSpec: buildSpec,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(this.role);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler.role);
        cr.node.addDependency(crHandler);
        return cr;
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.securityGroups,
        });
    }
    get grantPrincipal() {
        return this.role;
    }
}
exports.CodeBuildRunnerImageBuilder = CodeBuildRunnerImageBuilder;
//# sourceMappingURL=data:application/json;base64,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