"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsImageBuilderRunnerImageBuilder = exports.ImageBuilderComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const ami_1 = require("./ami");
const common_1 = require("./common");
const container_1 = require("./container");
const build_image_function_1 = require("../../../lambdas/build-image-function");
const delete_ami_function_1 = require("../../../lambdas/delete-ami-function");
const utils_1 = require("../../../utils");
const common_2 = require("../../common");
const common_3 = require("../common");
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 *
 * @deprecated Use `RunnerImageComponent` instead as this be internal soon.
 */
class ImageBuilderComponent extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: this.prefixCommandsWithErrorHandling(props.platform, extractCommands),
                    },
                });
            }
        }
        if (props.commands.length > 0) {
            steps.push({
                name: 'Run',
                action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                inputs: {
                    commands: this.prefixCommandsWithErrorHandling(props.platform, props.commands),
                },
            });
        }
        const data = {
            name: props.displayName,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = (0, common_3.uniqueImageBuilderName)(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            description: props.description,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
                description: props.description,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
    prefixCommandsWithErrorHandling(platform, commands) {
        if (platform == 'Windows') {
            return [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
            ].concat(commands);
        }
        else {
            return [
                'set -ex',
            ].concat(commands);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.9.3" };
exports.ImageBuilderComponent = ImageBuilderComponent;
/**
 * @internal
 */
class AwsImageBuilderRunnerImageBuilder extends common_3.RunnerImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.boundComponents = [];
        if (props?.codeBuildOptions) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('codeBuildOptions are ignored when using AWS Image Builder to build runner images.');
        }
        this.os = props?.os ?? common_2.Os.LINUX_UBUNTU;
        this.architecture = props?.architecture ?? common_2.Architecture.X86_64;
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        this.logRetention = props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'VPC', { isDefault: true });
        this.securityGroups = props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
        this.subnetSelection = props?.subnetSelection;
        this.baseImage = props?.baseDockerImage ?? (0, container_1.defaultBaseDockerImage)(this.os);
        this.baseAmi = props?.baseAmi ?? (0, ami_1.defaultBaseAmi)(this.os, this.architecture).getImage(this).imageId;
        this.instanceType = props?.awsImageBuilderOptions?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        // confirm instance type
        if (!this.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`Builder architecture (${this.architecture.name}) doesn't match selected instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        // warn against isolated networks
        if (props?.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // role to be used by AWS Image Builder
        this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
    }
    platform() {
        if (this.os.is(common_2.Os.WINDOWS)) {
            return 'Windows';
        }
        if (this.os.is(common_2.Os.LINUX_AMAZON_2) || this.os.is(common_2.Os.LINUX_UBUNTU)) {
            return 'Linux';
        }
        throw new Error(`OS ${this.os.name} is not supported by AWS Image Builder`);
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bindDockerImage() {
        if (this.boundDockerImage) {
            return this.boundDockerImage;
        }
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Docker Distribution', {
            name: (0, common_3.uniqueImageBuilderName)(this),
            // description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        let dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
        if (this.boundComponents.length == 0) {
            this.boundComponents.push(...this.components.map((c, i) => c._asAwsImageBuilderComponent(this, `Component ${i}`, this.os, this.architecture)));
        }
        for (const c of this.components) {
            const commands = c.getDockerCommands(this.os, this.architecture);
            if (commands.length > 0) {
                dockerfileTemplate += '\n' + commands.join('\n') + '\n';
            }
        }
        const recipe = new container_1.ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform(),
            components: this.boundComponents,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate,
            parentImage: this.baseImage,
        });
        const log = this.createLog('Docker Log', recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
        ]);
        const image = this.createImage(infra, dist, log, undefined, recipe.arn);
        this.createPipeline(infra, dist, log, undefined, recipe.arn);
        this.imageCleaner(image, recipe.name);
        this.boundDockerImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
            runnerVersion: common_2.RunnerVersion.specific('unknown'),
        };
        return this.boundDockerImage;
    }
    imageCleaner(image, recipeName) {
        const crHandler = (0, utils_1.singletonLambda)(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'], // Image Builder doesn't support scoping this :(
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
    createLog(id, recipeName) {
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, id, {
            logGroupName: `/aws/imagebuilder/${recipeName}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
    }
    createInfrastructure(managedPolicies) {
        if (this.infrastructure) {
            return this.infrastructure;
        }
        for (const managedPolicy of managedPolicies) {
            this.role.addManagedPolicy(managedPolicy);
        }
        for (const component of this.boundComponents) {
            component.grantAssetsRead(this.role);
        }
        this.infrastructure = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: (0, common_3.uniqueImageBuilderName)(this),
            // description: this.description,
            subnetId: this.vpc?.selectSubnets(this.subnetSelection).subnetIds[0],
            securityGroupIds: this.securityGroups?.map(sg => sg.securityGroupId),
            instanceTypes: [this.instanceType.toString()],
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    this.role.roleName,
                ],
            }).ref,
        });
        return this.infrastructure;
    }
    createImage(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, this.amiOrContainerId('Image', imageRecipeArn, containerRecipeArn), {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        image.node.addDependency(infra);
        image.node.addDependency(log);
        return image;
    }
    amiOrContainerId(baseId, imageRecipeArn, containerRecipeArn) {
        if (imageRecipeArn) {
            return `AMI ${baseId}`;
        }
        if (containerRecipeArn) {
            return `Docker ${baseId}`;
        }
        throw new Error('Either imageRecipeArn or containerRecipeArn must be defined');
    }
    createPipeline(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, this.amiOrContainerId('Pipeline', imageRecipeArn, containerRecipeArn), {
            name: (0, common_3.uniqueImageBuilderName)(this),
            // description: this.description,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            schedule: scheduleOptions,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        pipeline.node.addDependency(infra);
        pipeline.node.addDependency(log);
        return pipeline;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
    get grantPrincipal() {
        return this.role;
    }
    bindAmi() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'AMI Distribution', {
            name: (0, common_3.uniqueImageBuilderName)(this),
            // description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new ami_1.AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform(),
            components: this.bindComponents(),
            architecture: this.architecture,
            baseAmi: this.baseAmi,
        });
        const log = this.createLog('Ami Log', recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: common_2.RunnerVersion.specific('unknown'),
        };
        this.amiCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    amiCleaner(launchTemplate, stackName, builderName) {
        const deleter = (0, utils_1.singletonLambda)(delete_ami_function_1.DeleteAmiFunction, this, 'delete-ami', {
            description: 'Delete old GitHub Runner AMIs',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
    bindComponents() {
        if (this.boundComponents.length == 0) {
            this.boundComponents.push(...this.components.map((c, i) => c._asAwsImageBuilderComponent(this, `Component ${i} ${c.name}`, this.os, this.architecture)));
        }
        return this.boundComponents;
    }
}
exports.AwsImageBuilderRunnerImageBuilder = AwsImageBuilderRunnerImageBuilder;
//# sourceMappingURL=data:application/json;base64,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