"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseProvider = exports.Os = exports.Architecture = exports.RunnerVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Defines desired GitHub Actions runner version.
 */
class RunnerVersion {
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest() {
        return new RunnerVersion('latest');
    }
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version) {
        return new RunnerVersion(version);
    }
    constructor(version) {
        this.version = version;
    }
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other) {
        return this.version == other.version;
    }
}
_a = JSII_RTTI_SYMBOL_1;
RunnerVersion[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerVersion", version: "0.9.3" };
exports.RunnerVersion = RunnerVersion;
/**
 * CPU architecture enum for an image.
 */
class Architecture {
    static of(architecture) {
        return new Architecture(architecture);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch) {
        return arch.name == this.name;
    }
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches) {
        for (const arch of arches) {
            if (this.is(arch)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType) {
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.X86_64) {
            return this.is(Architecture.X86_64);
        }
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.ARM_64) {
            return this.is(Architecture.ARM64);
        }
        throw new Error('Unknown instance type architecture');
    }
}
_b = JSII_RTTI_SYMBOL_1;
Architecture[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Architecture", version: "0.9.3" };
/**
 * ARM64
 */
Architecture.ARM64 = Architecture.of('ARM64');
/**
 * X86_64
 */
Architecture.X86_64 = Architecture.of('X86_64');
exports.Architecture = Architecture;
/**
 * OS enum for an image.
 */
class Os {
    static of(os) {
        return new Os(os);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os) {
        return os.name == this.name;
    }
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses) {
        for (const os of oses) {
            if (this.is(os)) {
                return true;
            }
        }
        return false;
    }
}
_c = JSII_RTTI_SYMBOL_1;
Os[_c] = { fqn: "@cloudsnorkel/cdk-github-runners.Os", version: "0.9.3" };
/**
* Linux
*
* @deprecated use {@link LINUX_UBUNTU} or {@link LINUX_AMAZON_2}
*/
Os.LINUX = Os.of('Linux');
/**
 * Ubuntu Linux
 */
Os.LINUX_UBUNTU = Os.of('Ubuntu Linux');
/**
 * Amazon Linux 2
 */
Os.LINUX_AMAZON_2 = Os.of('Amazon Linux 2');
/**
* Windows
*/
Os.WINDOWS = Os.of('Windows');
exports.Os = Os;
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
class BaseProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.retryOptions = props?.retryOptions;
    }
    labelsFromProperties(defaultLabel, propsLabel, propsLabels) {
        if (propsLabels && propsLabel) {
            throw new Error('Must supply either `label` or `labels` in runner properties, but not both. Try removing the `label` property.');
        }
        if (propsLabels) {
            return propsLabels;
        }
        if (propsLabel) {
            return [propsLabel];
        }
        return [defaultLabel];
    }
    addRetry(task, errors) {
        if (this.retryOptions?.retry ?? true) {
            task.addRetry({
                errors,
                interval: this.retryOptions?.interval ?? aws_cdk_lib_1.Duration.minutes(1),
                maxAttempts: this.retryOptions?.maxAttempts ?? 10,
                backoffRate: this.retryOptions?.backoffRate ?? 1.3,
            });
        }
    }
}
exports.BaseProvider = BaseProvider;
//# sourceMappingURL=data:application/json;base64,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