"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const auth_app_1 = require("@octokit/auth-app");
const core_1 = require("@octokit/core");
const AWS = require("aws-sdk");
const github_1 = require("./github");
const helpers_1 = require("./helpers");
/* eslint-disable-next-line import/no-extraneous-dependencies,import/no-unresolved */
const cfn = new AWS.CloudFormation();
const ec2 = new AWS.EC2();
const ecr = new AWS.ECR();
const sf = new AWS.StepFunctions();
function secretArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:secretsmanager:us-east-1:12345678:secret:secret-name-REVISION
    const region = parts[3];
    const fullName = parts[6];
    const name = fullName.slice(0, fullName.lastIndexOf('-'));
    return `https://${region}.console.aws.amazon.com/secretsmanager/home?region=${region}#!/secret?name=${name}`;
}
function lambdaArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:lambda:us-east-1:12345678:function:name-XYZ
    const region = parts[3];
    const name = parts[6];
    return `https://${region}.console.aws.amazon.com/lambda/home?region=${region}#/functions/${name}?tab=monitoring`;
}
function stepFunctionArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:states:us-east-1:12345678:stateMachine:name-XYZ
    const region = parts[3];
    return `https://${region}.console.aws.amazon.com/states/home?region=${region}#/statemachines/view/${arn}`;
}
async function generateProvidersStatus(stack, logicalId) {
    const resource = await cfn.describeStackResource({ StackName: stack, LogicalResourceId: logicalId }).promise();
    const providers = JSON.parse(resource.StackResourceDetail?.Metadata ?? '{}').providers;
    if (!providers) {
        return {};
    }
    return Promise.all(providers.map(async (p) => {
        // add ECR data, if image is from ECR
        if (p.image?.imageRepository?.match(/[0-9]+\.dkr\.ecr\.[a-z0-9\-]+\.amazonaws\.com\/.+/)) {
            const tags = await ecr.describeImages({
                repositoryName: p.image.imageRepository.split('/')[1],
                filter: {
                    tagStatus: 'TAGGED',
                },
                maxResults: 1,
            }).promise();
            if (tags.imageDetails && tags.imageDetails?.length >= 1) {
                p.image.latestImage = {
                    tags: tags.imageDetails[0].imageTags,
                    digest: tags.imageDetails[0].imageDigest,
                    date: tags.imageDetails[0].imagePushedAt,
                };
            }
        }
        // add AMI data, if image is AMI
        if (p.ami?.launchTemplate) {
            const versions = await ec2.describeLaunchTemplateVersions({
                LaunchTemplateId: p.ami.launchTemplate,
                Versions: ['$Default'],
            }).promise();
            if (versions.LaunchTemplateVersions && versions.LaunchTemplateVersions.length >= 1) {
                p.ami.latestAmi = versions.LaunchTemplateVersions[0].LaunchTemplateData?.ImageId;
            }
        }
        return p;
    }));
}
function safeReturnValue(event, status) {
    if (event.path) {
        return {
            statusCode: 200,
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(status),
        };
    }
    return status;
}
exports.handler = async function (event) {
    // confirm required environment variables
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.GITHUB_SECRET_ARN || !process.env.GITHUB_PRIVATE_KEY_SECRET_ARN || !process.env.LOGICAL_ID ||
        !process.env.WEBHOOK_HANDLER_ARN || !process.env.STEP_FUNCTION_ARN || !process.env.SETUP_SECRET_ARN || !process.env.SETUP_FUNCTION_URL ||
        !process.env.STACK_NAME) {
        throw new Error('Missing environment variables');
    }
    // base status
    const status = {
        github: {
            setup: {
                status: 'Unknown',
                url: '',
                secretArn: process.env.SETUP_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.SETUP_SECRET_ARN),
            },
            domain: 'Unknown',
            webhook: {
                url: process.env.WEBHOOK_URL,
                status: 'Unable to check',
                secretArn: process.env.WEBHOOK_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.WEBHOOK_SECRET_ARN),
            },
            auth: {
                type: 'Unknown',
                status: 'Unknown',
                secretArn: process.env.GITHUB_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.GITHUB_SECRET_ARN),
                privateKeySecretArn: process.env.GITHUB_PRIVATE_KEY_SECRET_ARN,
                privateKeySecretUrl: secretArnToUrl(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN),
                app: {
                    id: '',
                    url: '',
                    installations: [],
                },
                personalAuthToken: '',
            },
        },
        providers: await generateProvidersStatus(process.env.STACK_NAME, process.env.LOGICAL_ID),
        troubleshooting: {
            webhookHandlerArn: process.env.WEBHOOK_HANDLER_ARN,
            webhookHandlerUrl: lambdaArnToUrl(process.env.WEBHOOK_HANDLER_ARN),
            stepFunctionArn: process.env.STEP_FUNCTION_ARN,
            stepFunctionUrl: stepFunctionArnToUrl(process.env.STEP_FUNCTION_ARN),
            stepFunctionLogGroup: process.env.STEP_FUNCTION_LOG_GROUP,
            recentRuns: [],
        },
    };
    // setup url
    const setupToken = (await (0, helpers_1.getSecretJsonValue)(process.env.SETUP_SECRET_ARN)).token;
    if (setupToken) {
        status.github.setup.status = 'Pending';
        status.github.setup.url = `${process.env.SETUP_FUNCTION_URL}?token=${setupToken}`;
    }
    else {
        status.github.setup.status = 'Complete';
    }
    // list last 10 executions and their status
    try {
        const executions = await sf.listExecutions({
            stateMachineArn: process.env.STEP_FUNCTION_ARN,
            maxResults: 10,
        }).promise();
        for (const execution of executions.executions) {
            const executionDetails = await sf.describeExecution({
                executionArn: execution.executionArn,
            }).promise();
            const input = JSON.parse(executionDetails.input || '{}');
            status.troubleshooting.recentRuns.push({
                executionArn: execution.executionArn,
                status: execution.status,
                owner: input.owner,
                repo: input.repo,
                runId: input.runId,
            });
        }
    }
    catch (e) {
        status.troubleshooting.recentRuns.push({ status: `Error getting executions: ${e}` });
    }
    // get secrets
    let githubSecrets;
    try {
        githubSecrets = await (0, helpers_1.getSecretJsonValue)(process.env.GITHUB_SECRET_ARN);
    }
    catch (e) {
        status.github.auth.status = `Unable to read secret: ${e}`;
        return safeReturnValue(event, status);
    }
    let privateKey;
    try {
        privateKey = await (0, helpers_1.getSecretValue)(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN);
    }
    catch (e) {
        status.github.auth.status = `Unable to read private key secret: ${e}`;
        return safeReturnValue(event, status);
    }
    // calculate base url
    let baseUrl = (0, github_1.baseUrlFromDomain)(githubSecrets.domain);
    status.github.domain = githubSecrets.domain;
    if (githubSecrets.personalAuthToken) {
        // try authenticating with personal authentication token
        status.github.auth.type = 'Personal Auth Token';
        status.github.auth.personalAuthToken = '*redacted*';
        let octokit;
        try {
            octokit = new core_1.Octokit({ baseUrl, auth: githubSecrets.personalAuthToken });
        }
        catch (e) {
            status.github.auth.status = `Unable to authenticate using personal auth token: ${e}`;
            return safeReturnValue(event, status);
        }
        try {
            const user = await octokit.request('GET /user');
            status.github.auth.personalAuthToken = `username: ${user.data.login}`;
        }
        catch (e) {
            status.github.auth.status = `Unable to call /user with personal auth token: ${e}`;
            return safeReturnValue(event, status);
        }
        status.github.auth.status = 'OK';
        status.github.webhook.status = 'Unable to verify automatically';
    }
    else {
        // try authenticating with GitHub app
        status.github.auth.type = 'GitHub App';
        status.github.auth.app.id = githubSecrets.appId;
        let appOctokit;
        try {
            appOctokit = new core_1.Octokit({
                baseUrl,
                authStrategy: auth_app_1.createAppAuth,
                auth: {
                    appId: githubSecrets.appId,
                    privateKey: privateKey,
                },
            });
        }
        catch (e) {
            status.github.auth.status = `Unable to authenticate app: ${e}`;
            return safeReturnValue(event, status);
        }
        // get app url
        try {
            const app = (await appOctokit.request('GET /app')).data;
            status.github.auth.app.url = app.html_url;
        }
        catch (e) {
            status.github.auth.status = `Unable to get app details: ${e}`;
            return safeReturnValue(event, status);
        }
        // list all app installations
        try {
            const installations = (await appOctokit.request('GET /app/installations')).data;
            for (const installation of installations) {
                let installationDetails = {
                    id: installation.id,
                    url: installation.html_url,
                    status: 'Unable to query',
                    repositories: [],
                };
                let token;
                try {
                    token = (await appOctokit.auth({
                        type: 'installation',
                        installationId: installation.id,
                    })).token;
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate app installation: ${e}`;
                    continue;
                }
                let octokit;
                try {
                    octokit = new core_1.Octokit({ baseUrl, auth: token });
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate using app: ${e}`;
                    continue;
                }
                try {
                    const repositories = (await octokit.request('GET /installation/repositories')).data.repositories;
                    for (const repo of repositories) {
                        installationDetails.repositories.push(repo.full_name);
                    }
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate using installation token: ${e}`;
                    continue;
                }
                installationDetails.status = 'OK';
                status.github.auth.app.installations.push(installationDetails);
            }
        }
        catch (e) {
            status.github.auth.status = 'Unable to list app installations';
            return safeReturnValue(event, status);
        }
        status.github.auth.status = 'OK';
        // check webhook config
        try {
            const response = await appOctokit.request('GET /app/hook/config', {});
            if (response.data.url !== process.env.WEBHOOK_URL) {
                status.github.webhook.status = 'GitHub has wrong webhook URL configured';
            }
            else {
                // TODO check secret by doing a dummy delivery? force apply secret?
                status.github.webhook.status = 'OK (note that secret cannot be checked automatically)';
            }
        }
        catch (e) {
            status.github.webhook.status = `Unable to check app configuration: ${e}`;
            return safeReturnValue(event, status);
        }
    }
    return safeReturnValue(event, status);
};
//# sourceMappingURL=data:application/json;base64,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