"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaAccess = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
/**
 * Access configuration options for Lambda functions like setup and webhook function. Use this to limit access to these functions.
 */
class LambdaAccess {
    /**
     * Disables access to the configured Lambda function. This is useful for the setup function after setup is done.
     */
    static noAccess() {
        return new NoAccess();
    }
    /**
     * Provide access using Lambda URL. This is the default and simplest option. It puts no limits on the requester, but the Lambda functions themselves authenticate every request.
     */
    static lambdaUrl() {
        return new LambdaUrl();
    }
    /**
     * Provide access using API Gateway. This is the most secure option, but requires additional configuration. It allows you to limit access to specific IP addresses and even to a specific VPC.
     *
     * To limit access to GitHub.com use:
     *
     * ```
     * LambdaAccess.apiGateway({
     *   allowedIps: LambdaAccess.githubWebhookIps(),
     * });
     * ```
     *
     * Alternatively, get and manually update the list manually with:
     *
     * ```
     * curl https://api.github.com/meta | jq .hooks
     * ```
     */
    static apiGateway(props) {
        return new ApiGateway(props);
    }
    /**
     * Downloads the list of IP addresses used by GitHub.com for webhooks.
     *
     * Note that downloading dynamic data during deployment is not recommended in CDK. This is a workaround for the lack of a better solution.
     */
    static githubWebhookIps() {
        const githubMeta = (0, child_process_1.execFileSync)('curl', ['-fsSL', 'https://api.github.com/meta']).toString();
        const githubMetaJson = JSON.parse(githubMeta);
        return githubMetaJson.hooks;
    }
}
_a = JSII_RTTI_SYMBOL_1;
LambdaAccess[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LambdaAccess", version: "0.9.3" };
exports.LambdaAccess = LambdaAccess;
/**
 * @internal
 */
class NoAccess extends LambdaAccess {
    _bind(_construct, _id, _lambdaFunction) {
        return '';
    }
}
/**
 * @internal
 */
class LambdaUrl extends LambdaAccess {
    _bind(_construct, _id, lambdaFunction) {
        return lambdaFunction.addFunctionUrl({
            authType: aws_lambda_1.FunctionUrlAuthType.NONE,
        }).url;
    }
}
/**
 * @internal
 */
class ApiGateway {
    constructor(props) {
        this.props = props;
    }
    _bind(scope, id, lambdaFunction) {
        let policy;
        let endpointConfig = undefined;
        if (this.props?.allowedVpc) {
            // private api gateway
            const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(scope, `${id}/SG`, {
                vpc: this.props.allowedVpc,
                allowAllOutbound: true,
            });
            for (const otherSg of this.props?.allowedSecurityGroups ?? []) {
                sg.connections.allowFrom(otherSg, aws_cdk_lib_1.aws_ec2.Port.tcp(443));
            }
            const vpcEndpoint = new aws_cdk_lib_1.aws_ec2.InterfaceVpcEndpoint(scope, `${id}/VpcEndpoint`, {
                vpc: this.props.allowedVpc,
                service: aws_cdk_lib_1.aws_ec2.InterfaceVpcEndpointAwsService.APIGATEWAY,
                privateDnsEnabled: true,
                securityGroups: [sg],
                open: false,
            });
            endpointConfig = {
                types: [aws_cdk_lib_1.aws_apigateway.EndpointType.PRIVATE],
                vpcEndpoints: [vpcEndpoint],
            };
            policy = aws_iam_1.PolicyDocument.fromJson({
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Principal: '*',
                        Action: 'execute-api:Invoke',
                        Resource: 'execute-api:/*/*/*',
                        Condition: {
                            StringEquals: {
                                'aws:SourceVpce': vpcEndpoint.vpcEndpointId,
                            },
                        },
                    },
                ],
            });
        }
        else {
            // public api gateway
            if (this.props?.allowedSecurityGroups) {
                cdk.Annotations.of(scope).addWarning('allowedSecurityGroups is ignored when allowedVpc is not specified.');
            }
            policy = aws_iam_1.PolicyDocument.fromJson({
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Principal: '*',
                        Action: 'execute-api:Invoke',
                        Resource: 'execute-api:/*/*/*',
                        Condition: {
                            IpAddress: {
                                'aws:SourceIp': this.props?.allowedIps ?? ['0.0.0.0/0'],
                            },
                        },
                    },
                ],
            });
        }
        const api = new aws_cdk_lib_1.aws_apigateway.LambdaRestApi(scope, id, {
            handler: lambdaFunction,
            proxy: true,
            cloudWatchRole: false,
            endpointConfiguration: endpointConfig,
            policy,
        });
        // remove CfnOutput
        api.node.tryRemoveChild('Endpoint');
        return api.url;
    }
}
//# sourceMappingURL=data:application/json;base64,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