# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyask']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyask',
    'version': '0.1.9',
    'description': 'Library for terminal programs to ask users questions.',
    'long_description': '# Table of Contents\n\n* [Table of Contents](#table-of-contents)\n* [pyask](#pyask)\n* [Brief Overview](#brief-overview)\n* [Installation](#installation)\n* [Use](#use)\n  * [ask()](#ask)\n  * [process()](#process)\n* [Convenience functions](#convenience-functions)\n* [Number convenience functions in pyask\\.numeric](#number-convenience-functions-in-pyasknumeric)\n  * [decimal()](#decimal)\n  * [integer()](#integer)\n  * [number()](#number)\n  * [number\\_between()](#number_between)\n  * [number\\_eq\\_to()](#number_eq_to)\n  * [number\\_greater\\_than()](#number_greater_than)\n  * [number\\_greater\\_than\\_or\\_eq()](#number_greater_than_or_eq)\n  * [number\\_less\\_than()](#number_less_than)\n  * [number\\_less\\_than\\_or\\_eq()](#number_less_than_or_eq)\n  * [seconds()](#seconds)\n* [Filepath convenience functions in pyask\\.path](#filepath-convenience-functions-in-pyaskpath)\n  * [dirname()](#dirname)\n  * [filename()](#filename)\n  * [pathname()](#pathname)\n* [Selection convenience functions in pyask\\.selection](#selection-convenience-functions-in-pyaskselection)\n  * [which()](#which)\n  * [which\\_items()](#which_items)\n* [String convenience functions in pyask\\.string](#string-convenience-functions-in-pyaskstring)\n  * [any\\_string()](#any_string)\n  * [char()](#char)\n  * [string\\_regex()](#string_regex)\n  * [yes\\_no()](#yes_no)\n\n# `pyask`\n\nA python library for terminal programs to ask users questions\nand validate their responses.\nIt is pure Python (3.6+)\nand requires no dependencies beyond the standard library.\n\n# Brief Overview\n\nDo you wish you could easily ask a user a question\nand keep on asking it until they give a valid response?\nNow you can! Here\'s an example of asking the user for a\ntime duration in seconds:\n\n```\n>>> answer = pyask.seconds("How long is the game?")\nHow long is the game? (enter time in HH:MM:SS, MM:SS, or SS format) [] -10\nInvalid respose.\nHow long is the game? (enter time in HH:MM:SS, MM:SS, or SS format) [] 10:61\nInvalid respose.\nHow long is the game? (enter time in HH:MM:SS, MM:SS, or SS format) [] 10:59\n>>> answer\n659\n>>> type(answer)\n<class \'int\'>\n>\n```\n\nNote that it checked to make sure the answer was a valid timestamp\nand continued to ask the user until they gave a valid response.\nIt then returned an integer so you can immediately put their response to use in your program.\n\nHere\'s another example, this time asking a user to select the filenames for videos to concatenate together.\nNote that videos are not named nicely, so sorting them alphabetically and putting them together would scramble the video!\nAlso note that there are non-video files you would not want to put together in the directory:\n\n```\n>>> pyask.which_items(os.listdir(), "What videos, in order, should be stitched together?")\nAvailable choices are:\n(0) video1.mp4\n(1) video3.mp4\n(2) usb.txt\n(3) video2.mp4\n(4) video_4.mp4\nWhat videos, in order, should be stitched together? (space/comma separated number(s), repeats ok) [] 0 3 1 4\n[\'video1.mp4\', \'video2.mp4\', \'video3.mp4\', \'video_4.mp4\']\n```\n\nIt returns a nice list in the order the user specified.\nThey can also use commas and any variable number of spaces to separate items.\n\nThe package can ask many types of questions about numbers,\nstrings, selecting multiple items, and filenames.\nPlease see below for more information.\n\n# Installation\n\n`pyask` is available on the Python Package Index (PyPI) and\ncan be installed with `pip`:\n\n```\n$ pip install pyask\n```\n\n# Use\n\n## `ask()`\n\n`pyask`\'s main functionality stems from the function `ask()`:\n\n```\nask(question, aid="", default="", allow_empty=False, process_func=whatever, choices=None)\n```\n\n`ask()` takes the following arguments:\n\n1. `question`: a question in the form of a string.\n    Including a question mark is optional, so `"What is your favorite color?" and\n    "What is your favorite color" are equivalent.\n2. `aid`: Optional and defaults to an empty string.\n    Provides user with aid to answer the question. So setting\n    `aid="primary color"` to the above question will ask the user\\\n    `What is your favorite color? (primary color)`.\n3. `default`: Optional and defaults to an empty string. Whatever you\n   provide will be the assumed answer from the user if they simply hit\n   `Enter` at the prompt rather than providing an answer.\n   Providing a default response will prevent the user from entering a blank response.\n   Providing a default of "blue" to the above question would prompt:\\\n   `What is your favorite color? (primary color) [blue]`\n4. `allow_empty`: Optional and defaults to `False`.\n    Setting it to `True` will prevent the user from entering a blank response\n    (remember, if you provide a `default` then the user will not be able to enter a blank response).\n5. `process_func`: Defaults to `whatever`,\n    a function that keeps the user\'s answer and returns it as is.\n    It is easy to build an appropriate `process_func` using `pyask.utils.process()`.\n    See documentation below for full explanation, but in brief,\n    a `process_func` will first process the user input into a certain form\n    (be it a number, list of characters, etc) then\n    validate that it meets the needed requirements (e.g., is positive for example).\n6. `choices`: Optional and defaults to `None`.\n    Will provide the user with a list of choices to select from the sequence provided.\n    Options are numbered with the index (0-based) of the sequence.\n\n`ask()` then returns the user\'s answer. If the user\'s answer is not a valid one,\nit will inform the user it is not valid then redo the prompt.\nThis can continue indefinitely until the user provides a valid response.\n\nEmpty responses return `None`.\n\n## `process()`\n\n```\nprocess(func, valid, s)\n```\n\nTakes `func` and `valid` functions with a string, `s`,\nand returns `s` processed by `func` if `valid(func(s))` is `True`.\nOtherwise, it raises a `ValueError`. Both `func` and `valid` are functions.\nBest used with `partial` from Python\'s standard library `functools`, like:\n\n```\npartial(process, int, lambda x: x > 0)\n```\n\nWhich would create a function that takes a string,\ncasts it as an integer, then returns it if it\'s positive.\n\n# Convenience functions\nHaving to build a `process_func` for `ask()` would be cumbersome.\nI created various convenience functions which ask users common questions.\nAll these convenience functions take the same arguments as `ask()`,\nbut some may require additional arguments (see below).\nAll the convenience functions are available at the top level of the package\n(as in, can be called with `pyask.decimal` rather than `pyask.numeric.decimal`.\n\n# Number convenience functions in `pyask.numeric`\n\n## `decimal()`\n\n```\ndecimal(question, aid=\'enter a decimal\', **kwargs)\n```\n\nAsks user for a decimal (aka float). Example would be calling:\n\n```\n>>> pyask.decimal("How tall are you", aid="in cm", default=5)\nHow tall are you? (in cm) [5]\n5.0\n\n>>> pyask.decimal("How tall are you", aid="in cm", default=5)\nHow tall are you? (in cm) [5] 7.8\n7.8\n```\n\n## `integer()`\n\n```\ninteger(question, aid=\'enter an integer\', **kwargs)\n```\n\nAsks user for an integer (a float/decimal response is illegal and not accepted).\n\n## `number()`\n\n```\nnumber(question, aid=\'enter a number\', **kwargs)\n```\n\nAsks user for any number, and returns an integer if an integer was given\nor a float if a float was given.\n\n## `number_between()`\n\n```\nnumber_between(l, r, question, aid=\'\', only_int=False, **kwargs)\n```\nAsks user for a number between l and r, inclusive.\n`only_int` allows you to accept only an integer response\nif `True` or accept decimals/floats if `False` (default).\n\n## `number_eq_to()`\n\n```\nnumber_eq_to(n, question, aid=\'enter the correct number\', **kwargs)\n```\nAsks user for number equal to n.\n\n## `number_greater_than()`\n\n```\nnumber_greater_than(l, question, aid=\'\', only_int=False, **kwargs)\n```\n\nAsks user for number greater than l.\n`only_int` allows you to accept only an integer response\nif `True` or accept decimals/floats if `False` (default).\n\n## `number_greater_than_or_eq()`\n\n```\nnumber_greater_than_or_eq(l, question, aid=\'\', only_int=False, **kwargs)\n```\n\nAsks user for number greater than or equal to l.\n`only_int` allows you to accept only an integer response\nif `True` or accept decimals/floats if `False` (default).\n\n## `number_less_than()`\n\n```\nnumber_less_than(r, question, aid=\'\', only_int=False, **kwargs)\n```\n\nAsks user for number less than r.\n`only_int` allows you to accept only an integer response\nif `True` or accept decimals/floats if `False` (default).\n\n## `number_less_than_or_eq()`\n\n```\nnumber_less_than_or_eq(r, question, aid=\'\', only_int=False, **kwargs)\n```\n\nAsks user for number less than or equal to r.\n\n## `seconds()`\n\n```\nseconds(question, aid=\'enter time in HH:MM:SS, MM:SS, or SS format\', **kwargs)\n```\n\nAsks user for time. Time can be in HH:MM:SS, MM:SS, or SS format.\nSS format takes any number of positive seconds.\nMM:SS and HH:MM:SS must have MM and SS between 0 and 59 (inclusive).\nHH can be any positive number or zero.\nIt returns the time (seconds).\n\n# Filepath convenience functions in `pyask.path`\n\n## `dirname()`\n\n```\ndirname(question, aid=\'directory name\', absolute=False, must_exist=False, **kwargs)\n```\n\nAsks user for directory name, returns path.\n`absolute` when `True` will return an absolute path rather than a relative one.\n`must_exist` when `True` will only let the user enter in the path of an existing directory.\n\n## `filename()`\n\n```\nfilename(question, aid=\'filename\', absolute=False, must_exist=False, **kwargs)\n```\n\nAsks user for filename, returns path.\n`absolute` when `True` will return an absolute path rather than a relative one.\n`must_exist` when `True` will only let the user enter in the path of an existing file.\n\n## `pathname()`\n\n```\npathname(question, aid=\'pathname\', absolute=False, must_exist=False, **kwargs)\n```\n\nAsks user for pathname (either file or directory), returns path.\n`absolute` when `True` will return an absolute path rather than a relative one.\n`must_exist` when `True` will only let the user enter in the path of an existing file or directory.\n\n# Selection convenience functions in `pyask.selection`\n\nContains functions that prompt a user to select from a list of questions.\n\n## `which()`\n\n```\nwhich(xs, question, aid=\'pick a number\', **kwargs)\n```\n\nAsk user to select an item from the sequence `xs` and returns that item.\n\nExample, with a default answer of 2 and the user hitting `Enter` to use the default answer:\n\n```\n>>> pyask.which([\'a\', \'b\', \'c\'], "Which letter?", default="2")\nAvailable choices are:\n(0) a\n(1) b\n(2) c\nWhich letter? (pick a number) [2]\n\'c\'\n```\n\n## `which_items()`\n\n```\nwhich_items(xs, question, aid=\'space/comma separated number(s)\', allow_repeats=True, **kwargs)\n```\n\nAsk user to select multiple items from the sequence `xs`,\nby specifying a comma or space delimited response.\nIf `allow_repeats` is `True`, they can specify the same item multiple times.\n\nExample:\n\n```\n>>> pyask.which_items(os.listdir(), "What videos, in order, should be stitched together?")\nAvailable choices are:\n(0) video1.mp4\n(1) video3.mp4\n(2) usb.txt\n(3) video2.mp4\n(4) video_4.mp4\nWhat videos, in order, should be stitched together? (space/comma separated number(s), repeats ok) [] 0, 3, 1, 4\n[\'video1.mp4\', \'video2.mp4\', \'video3.mp4\', \'video_4.mp4\']\n```\n\n# String convenience functions in `pyask.string`\n\nContains functions that prompt a user to select from a list of questions.\n\n## `any_string()`\n\n```\nany_string(question, aid=\'any string\', **kwargs)\n```\n\nAsks user for a string response.\n\n## `char()`\n\n```\nchar(question, aid=\'single character\', **kwargs)\n```\n\nAsks user for a single character response.\n\n## `string_regex()`\n\n```\nstring_regex(regex, question, case_insensitive=False, **kwargs)\n```\n\nAsks user for a string response that must completely match the regex.\nThe regex must be in standard\n[Python regex](https://docs.python.org/3/library/re.html?highlight=re#module-re).\nThis allows you to easily ask any possible variant of a question\nand validate it with any regex you can think of.\nThe example below checks for a string that ends in "mp4", "avi", or "mov" and is case-insensitive:\n\n```\n>>> pyask.string_regex(r"(?i).+mp4|avi|mov$", "What is a valid video filename")\nWhat is a valid video filename? [] myvideo.docx\nInvalid respose.\nWhat is a valid video filename? [] myvideo.MP4\n\'myvideo.MP4\'\n>>> pyask.string_regex(r"(?i).+mp4|avi|mov$", "What is a valid video filename")\nWhat is a valid video filename? [] myvideo.mp4\n\'myvideo.mp4\'\n```\n\n## `yes_no()`\n\n```\nyes_no(question, aid=\'yes or no\', **kwargs)\n```\n\nAsks user a yes or no question.\nAccepts "yes", "y", "no", or "n" in any case (so "YES" or "YeS" works).\nReturns a boolean. Nice to use in control-flow logic. \n\nExample:\n\n```\n>>> if pyask.yes_no("Are you hungry"):\n...     print("Here\'s a snack.")\n... else:\n...     print("You don\'t need a snack.")\n...\nAre you hungry? (yes or no) [] yes\nHere\'s a snack.\n>>>\n```\n\n Contributions/suggestions\n\nContributions and suggestions are welcome.\nFeel free to raise an [issue](https://github.com/tmward/pyask/issues)\nfor a feature request or\nto submit a [pull request](https://github.com/tmward/pyask/pulls)\nif you have implemented features you would like to see in `pyask`.\n',
    'author': 'Thomas M Ward',
    'author_email': 'thomas@thomasward.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tmward/pyask',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
