"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
const DEFAULT_NAG_PACKS = [
    new cdk_nag_1.AwsSolutionsChecks({
        verbose: true,
        reports: true,
    }),
];
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
        this.nagPacks = props?.nagPacks ?? DEFAULT_NAG_PACKS;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNagApp", version: "0.14.17" };
class PDKNagAspect {
    constructor(app) {
        this.app = app;
    }
    visit(node) {
        this.app.nagPacks.forEach((nagPack) => nagPack.visit(node));
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNag", version: "0.14.17" };
//# sourceMappingURL=data:application/json;base64,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