from fractions import Fraction
import pickle
import copy
from math import log, exp
import pandas as pd
import matplotlib.pyplot as plt
import numpy as np

general = {
    "det()": ["matrix M", "Returns the determinant of the given matrix.", "det"],
    "transpose()": ["matrix M", "Returns the transpose of the given matrix.", "transpose"],
    "zeroMatrix()": ["rows m, columns n=m", "Returns a zero matrix of the given size.", "zeroMatrix"],
    "identityMatrix()": ["order m", "Returns the identity matrix of the given order.", "identityMatrix"],
    "matrify()": ["vector v, format='col'", "Converts a 1d list array vector into a 2d list array vector, of specified format.", "matrify"],
    "dot()": ["vector v1, vector v2", "Returns the dot product of two vectors.", "dot"],
    "multiplyMatrices()": ["matrix A, matrix B", "Multiplies two matrices and returns the resulting matrix.", "multiplyMatrices"],
    "norm()": ["vector v, norm n=2", "Returns the nth norm of the given vector.", "norm"],
    "scale()": ["matrix M, num alpha", "Scales the given matrix by some constant alpha and returns the result.", "scale"],
    "addMatrices()": ["matrix A, matrix B", "Adds two matrices of equal size.", "addMatrices"],
    "axpy()": ["matrix x, matrix y, num alpha", "Runs the AXPY operation, returning the vector or matrix resulting from alpha*x+y.", "axpy"],
    "linComb()": ["matrix x, matrix y, num alpha, num beta", "Performs a linear combination, returning the vector or matrix resulting from alpha*x+beta*y.", "linComb"],
    "rowSwap()": ["matrix M, idx row1, idx row2", "Performs the row swap operation row1<->row2, and returns the resulting matrix.", "rowSwap"],
    "rowAddition()": ["matrix M, idx row1, idx row2", "Performs the row addition operation row1+row2->row1, and returns the resulting matrix.", "rowAddition"],
    "rowScale()": ["matrix M, idx row1, num alpha", "Performs the row scale operation alpha*row1->row1, and returns the resulting matrix.", "rowScale"],
    "scaledRowAddition()": ["matrix M, idx row1, idx row2, num alpha", "Performs the scaled row addition operation row1+alpha*row2->row1, and returns the resulting matrix.", "scaledRowAddition"],
    "hAppend()": ["matrix A, matrix B", "Appends B to the right of A, returning the resulting matrix.", "hAppend"],
    "trace()": ["matrix M", "Returns the trace of the given matrix.", "trace"],
    "mPrint()": ["matrix M, num s=1, num l=0", "Prints the values of a matrix in a unique format, with s spaces between entries and l lines between rows.", "mPrint"],
    "ratMPrint()": ["matrix M, num s=1, num l=0", "Prints the values of a matrix in a unique format, with s spaces between entries and l lines between rows. Entries will be in rational (fractional) format.", "ratMPrint"],
    "line()": ["num n=1", "Prints out n blank lines for readability purposes.", "line"],
    "cutLeftSquare()": ["matrix M", "Cuts a non-square matrix into a square one, keeping the upper left side.", "cutLeftSquare"],
    "cutRightSquare()": ["matrix M", "Cuts a non-square matrix into a square one, keeping the upper right side.", "cutRightSquare"],
    "cofactor()": ["matrix M", "Returns the cofactor matrix associated with the given matrix.", "cofactor"],
    "adjugate()": ["matrix M", "Returns the adjugate (adjoint) matrix associated with the given one, which is the transpose of its cofactor.", "adjugate"],
    "inverse()": ["matrix M", "Returns the inverse of a matrix, computed using the determinant and adjugate.", "inverse"],
    "invSystem()": ["matrix A, vector x, vector b, num r=1000", "Computes the solutions to the system Ax=b, using the inverse-multiplication method. A must be a square matrix of coefficients, x and b should be column vectors in string and numerical formats respectively. r is the rational-denominator limit. The output will be a list, including both the solution matrix and a series of solution strings.", "invSystem"],
    "solutionPrint()": ["list S", "Prints the solutions provided in a list generated by a function such as invSystem.", "solutionPrint"],
    "sysSign()": ["num n", "Returns a + for positive numbers and - for negative numbers.", "sysSign"],
    "unityElim()": ["num n", "Essentially deletes the number from print if it is a 1. Used only for dependency purposes.", "unityElim"],
    "inputSystem()": ["num r=1000, num suppress=0, num readback=1", "A program for user-friendly input in order to solve and output solutions to a system of linear equations. r is the rational-denominator limit. Suppress will cancel solution output if ==1, and readback will cancel equation readback output if ==0.", "inputSystem"],
    "help()": ["num repress", "The help function for the matrix module.", "modulehelp"],
    "create()": ["num s=0, num p=1", "A program accepting user input to create and store a matrix for use while in the command prompt. String input will only be allowed if s==1. p will change the matrix print to rational if ==2, and will suppress printing if ==0.", "create"],
    "remove()": ["str key", "Removes the matrix stored under the given key from the matrices index."],
    "addIndex()": ["matrix M, str key", "Adds a new matrix to the matrices index, under a given key."],
    "listMatrices()": ["num p=1", "Prints all matrices stored in the matrices index. Print defaults to rational, but if p is changed it will be in decimal form.", "listMatrices"],
    "saveMatrices()": ["str filename='matrices.txt'", "A function which saves the matrices index into a text file.", "saveMatrices"],
    "loadMatrices()": ["str filename='matrices.txt'", "A function which returns an index from a text file. To update the current matrix index, use MatrixModule.index=MatrixModule.loadMatrices(filename).", "loadMatrices"],
    "rref()": ["matrix M", "Returns the Reduced Row Echelon Form (RREF) of the given matrix, computed using Gauss-Jordan elimination."],
    "rInverse()": ["matrix M", "Computes the inverse of a matrix by putting an augmented matrix into RREF using Gauss-Jordan elimination, and returns this inverse."],
    "elimSystem()": ["matrix A, vector x, vector b, num r=1000", "Computes the solutions to the system Ax=b, using Gauss-Jordan elimination. A must be a square matrix of coefficients, x and b should be column vectors in string and numerical formats respectively. r is the rational-denominator limit. The output will be a list, including both the solution matrix and a series of solution strings.", "elimSystem"],
    "isMatrix()": ["obj A", "Returns True if the passed object is a matrix, otherwise returns False."],
    "regression()": ["matrix M, str model='lin', str degree='2', str type='ls', num rcomp=1, num restrict=0", "Uses a regression model to compute a best-fitting curve to a given dataset. Includes linear, exponential, power, polynomial (nth-degree), and conic regression. All but polynomial and conic can be given a regression type, either least-squares (ls) or Theil-Sen (robust). Unless rcomp==0, an R^2 coefficient of determination value and an R_a absolute-residual coefficient will be calculated. Set restrict==0 to print at the end of calculations. Output is a list [[coeffs],function,[R^2,R_a]], where function = output[1] = a function formatted to work with the matplotlib graphing utilities (except for conics!), and the output[2] list will not exist if rcomp!=1."],
    "regPlot()": ["str src, str filetype='CSV', str model='lin', str type='ls', int deg=2, str xlab='X Axis', str ylab='Y Axis', int prec=10", "Takes an input data file (src) using pandas and plots both the data and a curve of best fit. Type should be the file type, such as csv. xlab and ylab label plot axes. prec sets the precision - higher numbers allow for higher precision plotting, but longer computation time."]
    }

def det(mat):
    matrix = copy.deepcopy(mat)
    # Returns the determinant of the given matrix, so long as the input matrix is square.
    m = len(matrix)
    n = len(matrix[0])
    if (m != n):  # Makes sure input is square
        print("Failed: Input matrix must be square.")
        return
    if(m == 1):  # If 1x1 then return inner value
        return matrix[0][0]
    output = 0  # Initialize output
    for k in range(0, n):  # Iterate through each item in the first row
        inner = []
        for i in range(1, m):  # Iterate through each row other than the first
            inRow = []
            for j in range(0, n):  # Iterate through each value in the current row
                if (j != k):  # Set aside all values not in the same column as the current row 1 value for recursive calculation
                    inRow.append(matrix[i][j])
            inner.append(inRow)
        # We will multiply each first row value by the determinant of its corresponding inner matrix and add/subtract these new values. Starting at 0, even valued indexes are added and odds are subtracted.
        if(k % 2 == 0):
            output += matrix[0][k] * det(inner)
        else:
            output -= matrix[0][k] * det(inner)
    return output


def transpose(mat):
    matrix = copy.deepcopy(mat)
    # Returns the transpose of a given matrix.
    m = len(matrix)
    n = len(matrix[0])
    output = []
    for i in range(0, n):  # Iterates through each column
        row = []
        for j in range(0, m):  # Converts current column into a row by iterating through each value
            row.append(matrix[j][i])
        output.append(row)  # Puts these new rows into the output
    return output


def zeroMatrix(num1, num2=0):
    m, n = copy.copy(num1), copy.copy(num2)
    # Returns a zero matrix of specified size.
    if(n == 0):
        # If number of columns is not specified then make the matrix square.
        n = m
    output = []
    for i in range(0, m):  # Create m rows
        row = []
        for j in range(0, n):  # Each row should have n values/columns
            row.append(0)  # Append necessary 0 values
        output.append(row)
    return output


def identityMatrix(ordr):
    order = copy.copy(ordr)
    # Returns the identity matrix of specified size.
    output = zeroMatrix(order)  # Begin with a zero matrix
    for i in range(0, order):
        output[i][i] = 1  # Puts values of 1 along the main diagonal.
    return output


def matrify(vec, format="col"):
    vector = copy.deepcopy(vec)
    # Convert a vector in list (1d list array) format into a matrix (2d list array) for the purposes of vector algebra functions which assume matrix-format. Specify either row (1xn) or column (mx1) vector format for conversion; column is default.
    s = len(vector)
    if (format == "row"):
        output = [[]]  # Initializes output as 1xn
        for i in range(0, s):  # Iterate through values to create matrix adaptation in row vector format
            output[0].append(vector[i])
        return output
    elif(format == "col" or format == "column"):
        output = []  # Initializes output as mx1
        # Iterate through values to create matrix adaptation in column vector format
        for i in range(0, s):
            output.append([vector[i]])
        return output
    else:
        # Catches formats which are not either row or column.
        return(format + " is not a valid format.")


def dot(vc1, vc2):
    vec1, vec2 = copy.deepcopy(vc1), copy.deepcopy(vc2)
    # Returns the dot product of two vectors. Input vectors should both be of the same orientation (row or column) and the same size, and in matrix format.
    m1, n1, m2, n2 = len(vec1), len(vec1[0]), len(vec2), len(vec2[0])
    if (not (m1 == m2 and n1 == n2) or not(m1 == 1 or n1 == 1)):  # Test for size and formatting
        return "Vectors must be of equal size and properly formatted."
    output = 0  # Initialize constant output value
    if(m1 == 1):  # If the vectors are row vectors:
        for i in range(0, n1):  # Iterate through values and multiply correspondingly
            output += (vec1[0][i]*vec2[0][i])
        return output
    if(n1 == 1):  # If the vectors are column vectors:
        for i in range(0, m1):  # Iterate through values and multiply correspondingly
            output += (vec1[i][0]*vec2[i][0])
        return output


def multiplyMatrices(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Multiplies two matrices and returns the resulting matrix.
    m1, n1, m2, n2 = len(matrix1), len(
        matrix1[0]), len(matrix2), len(matrix2[0])
    # Confirms that the number of columns in the first matrix is equal to the number of rows in the second.
    if (n1 != m2):
        return "Matrices are not of compatible sizes."
    # Initializes output matrix as a blank/zero matrix, with the number of rows from the first matrix and columns from the second.
    output = zeroMatrix(m1, n2)
    for i in range(0, m1):  # Iterate through each row of the first matrix.
        rowV = matrix1[i]
        # Iterate through each column of the second matrix.
        for j in range(0, n2):
            colV = []
            for k in range(0, m2):  # Set aside all values of a column into a vector
                colV.append(matrix2[k][j])
            # Take the dot product of these vectors, converted into corresponding matrix formats, and put it into the proper position in the resulting matrix.
            output[i][j] = dot(matrify(rowV), matrify(colV))
    return output


def norm(vec, nr=2):
    vector, nrm = copy.deepcopy(vec), copy.copy(nr)
    # Returns the nth norm of a vector. If norm is not specified, 2 is used as default, which is the same as the vector's magnitude. The vector must be in matrix list format.
    m = len(vector)
    n = len(vector[0])
    # Make sure the vector is of proper size and format.
    if not (m == 1 or n == 1):
        return "Input must be a vector."
    vec = []  # Initialize a list which is easy to iterate through.
    if(m == 1):  # If it's a row vector then just use the one row as the list.
        vec = vector[0]
    else:  # If it's a column vector then iterate to create the list.
        for i in range(0, m):
            vec.append(vector[i][0])
    value = 0
    if (nrm == "inf"):  # Support for inf-norm
        for j in range(0, len(vec)):  # Iterate to find max value.
            if (vec[j] > value):
                value = vec[j]
        return value
    else:  # For regular n-norms, calculate the norm value through iteration.
        for j in range(0, len(vec)):
            value += (vec[j] ** nrm)
        return (value**(1/nrm))


def scale(mat, a):
    matrix, alpha = copy.deepcopy(mat), copy.copy(a)
    # Returns the matrix scaled by the given constant value.
    m = len(matrix)
    n = len(matrix[0])
    output = matrix
    for i in range(0, m):  # Iterate through rows
        for j in range(0, n):  # Iterate through values in these rows
            output[i][j] *= alpha  # Perform individual scalar multiplication
    return output


def addMatrices(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Adds two matrices and returns the resulting matrix.
    m1, n1, m2, n2 = len(matrix1), len(
        matrix1[0]), len(matrix2), len(matrix2[0])
    # Confirms that the matrices are the same size.
    if not (m1 == m2 and n1 == n2):
        return "Matrices must be the same sizes."
    output = matrix1
    for i in range(0, m1):  # Iterate through rows
        for j in range(0, n1):  # Iterate through values in these rows
            output[i][j] += matrix2[i][j]  # Add corresponding values.
    return output


def axpy(mx, my, alpha):
    x, y, a = copy.deepcopy(mx), copy.deepcopy(my), copy.copy(alpha)
    # Performs the AXPY operation, or single-scaled addition, on two matrices; returning the resulting matrix.
    return addMatrices(scale(x, a), y)


def linComb(mx, my, alpha, beta):
    x, y, a, b = copy.deepcopy(mx), copy.deepcopy(
        my), copy.copy(alpha), copy.copy(beta)
    # Returns the linear combination of two scaled matrices.
    return addMatrices(scale(x, a), scale(y, b))


def rowSwap(mat, rw1, rw2):
    matrix, r1, r2 = copy.deepcopy(mat), copy.copy(rw1), copy.copy(rw2)
    # Performs the operation R1<->R2. Inputs should be the 0-indices of the desired rows to swap. Returns resulting matrix.
    output = matrix
    output[r1], output[r2] = output[r2], output[r1]
    return output


def rowAddition(mat, rw1, rw2):
    matrix, r1, r2 = copy.deepcopy(mat), copy.copy(rw1), copy.copy(rw2)
    # Performs the operation R1+R2->R1. Inputs should be the 0-indices of the desired rows to add. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[r1])):  # Iterative addition
        output[r1][i] += output[r2][i]
    return output


def rowScale(mat, rw, alph):
    matrix, row, alpha = copy.deepcopy(mat), copy.copy(rw), copy.copy(alph)
    # Performs the operation AR1->R1. Input the 0-index of the desired row and the scalar Alpha. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[row])):  # Iterative scaling
        output[row][i] *= alpha
    return output


def scaledRowAddition(mat, rw1, rw2, alph):
    matrix, r1, r2, alpha = copy.deepcopy(mat), copy.copy(
        rw1), copy.copy(rw2), copy.copy(alph)
    # Performs the operation R1+AR2->R1. Inputs should be the 0-indices of the desired rows to add, as well as the scalar Alpha. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[r1])):  # Iterative scaled addition
        output[r1][i] += alpha*output[r2][i]
    return output


def hAppend(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Appends matrix2 to the right of matrix, and returns the resulting matrix.
    m1 = len(matrix1)
    m2 = len(matrix2)
    # Confirms that the matrices have the same number of rows.
    if not (m1 == m2):
        return "Matrices must have the same number of rows."
    aC = len(matrix2[0])
    output = matrix1
    for i in range(0, m1):  # Iterates through each row
        # Appends each value of this row from the second matrix into the first.
        for j in range(0, aC):
            output[i].append(matrix2[i][j])
    return output


def trace(mat):
    matrix = copy.deepcopy(mat)
    # Returns the trace of a square matrix (the sum of elements along the main diagonal).
    m, n = len(matrix), len(matrix[0])
    if (m != n):  # Confiems that matrix is square.
        return "Input matrix must be square."
    output = 0
    for i in range(0, m):  # Iterates through and sums the main diagonal.
        output += matrix[i][i]
    return output


def mPrint(mat, s=1, l=0):
    matrix = copy.deepcopy(mat)
    # Prints the values of a matrix or vector in a specific format designed for readability.
    # There will be s spaces between each entry of a row, and l lines between rows.
    m, n = len(matrix), len(matrix[0])
    for i in range(0, m):
        str = ""
        spaces = ""
        for a in range(0, s):
            spaces += " "
        for j in range(0, n):
            str += ("%s%s") % (spaces, matrix[i][j])
        print(str)
        for k in range(0, l):
            print
    return


def ratMPrint(mat, s=1, l=0):
    matrix = copy.deepcopy(mat)
    # Prints the values of a matrix or vector in a specific format designed for readability, in rational (fractional) format rather than decimal.
    # There will be s spaces between each entry of a row, and l lines between rows.
    m, n = len(matrix), len(matrix[0])
    for i in range(0, m):
        str = ""
        spaces = ""
        for a in range(0, s):
            spaces += " "
        for j in range(0, n):
            str += ("%s%s") % (spaces, Fraction(matrix[i][j]).limit_denominator())
        print(str)
        for k in range(0, l):
            print(" ")
    return


def line(n=1):
    # Prints n blank lines.
    for i in range(0, n):
        print(" ")
    return


def cutLeftSquare(mat):
    matrix = copy.deepcopy(mat)
    # Cuts a non square matrix into a square, focused on the upper left side, by removing rows or columns from the bottom or right as needed.
    m, n = len(matrix), len(matrix[0])
    output = []
    if (m > n):  # If there are too many rows, just recreate the matrix with fewer.
        for i in range(0, n):
            output.append(matrix[i])
    else:  # Otherwise, iterate through each row and only allow the proper number of values in
        for i in range(0, m):
            row = []
            for j in range(0, m):
                row.append(matrix[i][j])
            output.append(row)
    return output


def cutRightSquare(mat):
    matrix = copy.deepcopy(mat)
    # Cuts a non square matrix into a square, focused on the upper right side, by removing rows or columns from the bottom or left as needed.
    m, n = len(matrix), len(matrix[0])
    output = []
    if (m > n):  # If there are too many rows, just recreate the matrix with fewer.
        for i in range(0, n):
            output.append(matrix[i])
    else:  # Otherwise, iterate through each row and only allow the proper values in.
        for i in range(0, m):
            row = []
            # Make sure to iterate only on right-side values.
            for j in range(n-m, m+1):
                row.append(matrix[i][j])
            output.append(row)
    return output


def cofactor(mat):
    matrix = copy.deepcopy(mat)
    # Returns the cofactor matrix associated with the given matrix.
    m, n = len(matrix), len(matrix[0])
    if (m != n):
        return("Matrix must be square.")
    output = zeroMatrix(m)  # Initialize the output matrix
    for i in range(0, m):  # iterate through all rows
        for j in range(0, n):  # iterate through all entries in a row
            subMtx = []  # Initialize the associated submatrix for each entry
            for r in range(0, m):
                if (r != i):
                    subRow = []
                    for c in range(0, n):
                        if (c != j):  # Append the proper entries to the submatrix
                            subRow.append(matrix[r][c])
                    subMtx.append(subRow)
            # Calculate the minor and then the cofactor for the entry
            output[i][j] = ((-1)**(i+j) * det(subMtx))
    return output


def adjugate(mat):
    matrix = copy.deepcopy(mat)
    # Returns the adjugate (adjoint) matrix associated with the given matrix, which is the transpose of the associated cofactor matrix.
    return transpose(cofactor(matrix))


def inverse(mat):
    matrix = copy.deepcopy(mat)
    # Returns the inverse of a matrix, calculated using the determinant and adjugate of that matrix.
    return scale(adjugate(matrix), 1/det(matrix))


def invSystem(mA, vx, vb, rat=1000):
    A, x, b = copy.deepcopy(mA), copy.deepcopy(vx), copy.deepcopy(vb)
    # Calculates the solutions to the system Ax=b, using the method of inverse multiplication.
    # Returns a list, beginning with the solution matrix, followed by a series of strings expressing these solutions.
    # Only works on square systems, where there are as many coefficients in a row of A as there are unknowns in x.
    # A should be an mxm matrix. x and b both should be mx1 column vectors in matrix format.
    # A should be the matrix of coefficients, as numerical values. x should be the vector of unknowns, as string values. b should be the vector of constants, as numerical values.
    output = []  # Initialize the output list
    if not(len(A) == len(A[0]) == len(x) == len(b)):  # Confirms compatiblity
        output.append("Matrices are not of compatible sizes.")
    else:
        m = len(x)
        # Perform the actual calculation
        solMtx = multiplyMatrices(inverse(A), b)
        output.append(solMtx)  # Primary element of list inserted
        strList = []
        # Will provide the solution strings in rational fraction format, unless passed a 'rat' value of 0.
        for i in range(0, m):  # Iterates through each unknown
            if(rat != 0):
                # Creates readable string in rational fraction form for the solution to this unknown
                st = "%s = %s" % (x[i][0], Fraction(
                    solMtx[i][0]).limit_denominator(rat))
            else:
                # Creates readable string for the solution to this unknown
                st = "%s = %s" % (x[i][0], solMtx[i][0])
            strList.append(st)  # Appends this solution string into the list
        # Appends the solution string list into the output
        output.append(strList)
    return output


def solutionPrint(sols):
    solutions = copy.deepcopy(sols)
    # Prints the solutions to a system of equations generated by another function in a readable way.
    m = len(solutions[1])
    for i in range(0, m):
        print(solutions[1][i])
    return


def sysSign(n):
    num = copy.copy(n)
    if(num >= 0):
        return "+"
    elif(num < 0):
        return "-"
    else:
        return ""


def unityElim(n):
    num = copy.copy(n)
    if(num == 1):
        return ""
    else:
        return num


def help(rep=0):
    modulehelp(rep)
    return


def modulehelp(rep=0):
    if(rep == 0):
        print("This is the Matrix Module V0.1.4, a package created by Elliott Walker. Type 'funcs' for a list of available functions, 'details' for a more detailed list including parameters, or 'exit' to exit this help function.")
    else:
        print("Available commands: 'funcs', 'details', 'exit'")
    inp = input()
    if(inp == "func" or inp == "functions" or inp == "list" or inp == "funcs"):
        print("Available functions under Matrix Module:")
        for key in general.keys():
            print(key)
    elif(inp == "details" or inp == "det" or inp == "dets" or inp == "more"):
        print("Available functions under Matrix Module (with parameters and descriptions):")
        line()
        for key in general.keys():
            print("%s  Parameters: %s;  Description: %s" %
                  (key, general[key][0], general[key][1]))
            line()
    elif(inp == "exit"):
        return
    help(1)
    return


index={}

def create(string=0, p=1):
    pr=copy.copy(p)
    print("Enter size of matrix, e.g. '3x3':")
    sizeIn=input()
    sizeArr=sizeIn.split("x")
    if (len(sizeArr) != 2):
        print("Program failure: Improper formatting of input.")
        return
    m=int(sizeArr[0])
    n=int(sizeArr[1])
    print("Enter a key to store this %d x %d matrix in." % (m, n))
    key=input()
    if(key in index.keys()):
        print("Program failure: Key already exists in index.")
        return
    index[key]=zeroMatrix(m, n)
    print("Matrix stored in index['%s']." % (key))
    for i in range(m):
        for j in range(n):
            if(string == 1):
                index[key][i][j]=str(
                    input("Input item at position %d, %d: " % (i+1, j+1)))
            else:
                index[key][i][j]=int(
                    input("Input value at position %d, %d: " % (i+1, j+1)))
    print("Matrix created.")
    if pr == 1:
        mPrint(index[key])
    elif pr == 2:
        ratMPrint(index[key])
    return


def remove(key):
    if key in index:
        del index[key]
    return


def addIndex(matrix, key):
    if (matrix.isMatrix == True):
        if (key in index.keys()):
            print("Program failure: Key already exists in index.")
        else:
            index[key]=matrix
    else:
        print("Program failure: Object passed is not a matrix.")
    return


def listMatrices(rat=1):
    print("All stored matrices:")
    for key in index.keys():
        print("")
        print(" At index['%s']: " % (key))
        if rat == 1:
            ratMPrint(index[key])
        else:
            mPrint(index[key])
    print("")
    print("Access a stored matrix using MatrixModule.index['key'].")
    return


def saveMatrices(filename="matrices.txt"):
    # Saves the matrices index to a file using the pickle dump method.
    file=open(filename, "wb")
    file.truncate(0)
    pickle.dump(index, file)
    file.close()
    return


def loadMatrices(filename="matrices.txt"):
    # Loads a matrices index from a pickle-binary text file.
    file=open(filename, "rb")
    index=pickle.load(file)
    file.close()
    return index


def rref(mat):
    matrix=copy.deepcopy(mat)
    # Through Gauss-Jordan elimination, reduces a matrix to reduced row echelon form (RREF) and returns the resulting matrix. Only works if the matrix is square or fat: tall matrices (m>n) will not work with the given algorithm.
    m, n=len(matrix), len(matrix[0])
    if (m > n):
        print("Failure: tall matrices (m>n) cannot reduce through given method.")
        return
    output=matrix
    # Row Reduction
    for i in range(m):  # Iterate through each row.
        # If the row's lead value isn't 1, then scale the row by dividing by that value so that it is.
        if (output[i][i] != 1):
            output=rowScale(output, i, 1/output[i][i])
        # Now iterate through all of the other rows, and perform scaled addition to make the values in the column of the current leading 1 become zeroes.
        for j in range(i+1, m):
            output=scaledRowAddition(output, j, i, -output[j][i])
    # Back Substitution
    # Iterate through the rows, excluding the last one, backwards.
    for i in range(m-1, -1, -1):
        # For each element to the right of the diagonal, subtract the row with a leading 1 in that column (scaled by the value of the aforementioned element) from the current row.
        for j in range(m-1, i, -1):
            output=scaledRowAddition(output, i, j, -output[i][j])
    return output


def rInverse(mat):
    matrix=copy.deepcopy(mat)
    # Computes the inverse of a matrix using Gauss-Jordan elimination on an augmented matrix, and returns this resulting inverse matrix.
    if (det(matrix) == 0 or len(matrix) != len(matrix[0])):
        return("Matrix must be invertible.")
    return cutRightSquare(rref(hAppend(matrix, identityMatrix(len(matrix)))))


def elimSystem(mA, vx, vb, rat=1000):
    A, x, b=copy.deepcopy(mA), copy.deepcopy(vx), copy.deepcopy(vb)
    # Calculates the solutions to the system Ax=b, using the method of Gaussian elimination.
    # Returns a list, beginning with the solution matrix, followed by a series of strings expressing these solutions.
    # Only works on square systems, where there are as many coefficients in a row of A as there are unknowns in x.
    # A should be an mxm matrix. x and b both should be mx1 column vectors in matrix format.
    # A should be the matrix of coefficients, as numerical values. x should be the vector of unknowns, as string values. b should be the vector of constants, as numerical values.
    output=[]
    if not(len(A) == len(A[0]) == len(x) == len(b)):  # Confirms compatiblity
        output.append("Matrices are not of compatible sizes.")
    else:
        m=len(x)
        # Perform the actual calculation
        solMtx=rref(hAppend(A, b))
        output.append(solMtx)  # Primary element of list inserted
        strList=[]
        # Will provide the solution strings in rational fraction format, unless passed a 'rat' value of 0.
        for i in range(0, m):  # Iterates through each unknown
            if(rat != 0):
                # Creates readable string in rational fraction form for the solution to this unknown
                st="%s = %s" % (x[i][0], Fraction(
                    solMtx[i][len(x)]).limit_denominator(rat))
            else:
                # Creates readable string for the solution to this unknown
                st="%s = %s" % (x[i][0], solMtx[i][len(x)])
            strList.append(st)  # Appends this solution string into the list
        # Appends the solution string list into the output
        output.append(strList)
    return output


def inputSystem(rat=1000, suppress=0, readback=1):
    # Accepts input for a much more user-friendly method of solving a system of linear equations.
    print("Would you like to solve using Gauss-Jordan elimination (1), or computational inverse (2)?")
    methodIn=input()
    if (int(methodIn) == 2):
        method="Inv"
    else:
        method="GJ"
    print("How many unknowns?")
    m=int(input())
    x=zeroMatrix(m, 1)
    for i in range(0, m):
        print("What is the unknown at index %d?" % (i))
        x[i][0]=input()
    print("When inputting coefficients separate them with spaces. For example, input 4x+3y=9 as '4 3 9'.")
    A=zeroMatrix(m)
    b=zeroMatrix(m, 1)
    for j in range(0, m):
        print("What are the coefficients in the equation of index %d?" % (j))
        B=input().split()
        if (len(B) != (m+1)):
            print("Program failure: Incorrect number of coefficients entered.")
            return
        for k in range(0, m+1):
            if (k == m):
                b[j][0]=int(B[k])
            else:
                A[j][k]=int(B[k])
    if (readback == 1):
        line()
        print("Your input:")
        for row in range(0, m):
            equ=""
            for val in range(0, m+1):
                if (val == m):
                    equ += ("= %d" % (b[row][0]))
                elif (val == 0):
                    equ += ("%s%s " % (unityElim(A[row][val]), x[val][0]))
                else:
                    equ += ("%s %s%s " %
                            (sysSign(A[row][val]), unityElim(abs(A[row][val])), x[val][0]))
            print(equ)
    if (det(A) == 0):
        print("System dependent - method failed.")
        return
    if (method == "Inv"):
        output=invSystem(A, x, b, rat)
    else:
        output=elimSystem(A, x, b, rat)
    if (suppress == 0):
        line()
        print("Solution:")
        solutionPrint(output)
        line()
        input("Press return to exit.")
    return(output)


def isMatrix(obj):
    if (type(obj)==list):
        for i in range(len(obj)):
            if (obj[i].type() != list):
                return False
        return True
    else:
        return False


def regression(inmat,model="lin",degree=2,type="ls",rcomp=1, restrict=1):
    # Combine the many different types of regression into one separate function (no dependency) that requires matrix input. This will be more compatible with raw datasets, and have a generalized output compatible with matplotlib.
    # Note: do NOT use the Fraction().limit_denominator() function to create limited-precision rational coefficients. This can cause zero-rounding error.
    output, A, b = [], [], [] # Initialize output matrix and intermediate matrices A and b.
    m = len(inmat)
    func = lambda x: x # Initialize a function. This will be redefined at the end of each regression model.
    eq = ""
    if (model == "lin" or model == "linear"): # y=mx+b linear regression
        slope, intercept = 0, 0
        if(type=="leastsquares" or type=="ls"): # Least squares regression - based in matrix multiplication
            for i in range(m):
                A.append([1,inmat[i][0]]) # Append [1,x] rows to matrix A
                b.append([inmat[i][1]]) # Append all y values as independent rows in matrified vector b
            coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(A),A)),transpose(A)),b) # Perform multiplication. Based in least-squares projection: A x = b --> A^T A x_h = A^T b --> x_h = (A^T A)^-1 A^T b
            slope = coeffs[1][0]
            intercept = coeffs[0][0]
        elif(type=="robust" or type=="theilsen"): # Theil-Sen estimator regression - based in medians. Very high breakdown point, making it great for large and fairly noisy datasets.
            for i in range(m):
                for j in range(m):
                    if(i < j): # For all possible pairs of different points... (using < to make sure pairs are not double-counted)
                        if(inmat[i][0] != inmat[j][0]): # where the slope is not undefined...
                            A.append((inmat[i][1]-inmat[j][1])/(inmat[i][0]-inmat[j][0])) # Append the slope of these points into the matrix A
            A.sort() # Sort the list of slopes to find the median slope between points.
            slope = A[int(round(len(A)/2))] # Take median
            for i in range(m):
                b.append(inmat[i][1] - (slope*inmat[i][0])) # Find the median intercept b provided by iterating through all datapoints with the equation b=y-ax
            b.sort()
            intercept = b[int(round(len(b)/2))] # Take median
        func = lambda x : slope*x+intercept
        eq = "%s*x+%s" % (slope, intercept) 
        output.append([slope,intercept])        
    elif (model == "pwr" or model == "power"): # Power function regression, in the form of y=ax^p. Exploits logarithm properties in order to use a linear regression technique (either least-squares or Theil-Sen) to form a power-function fit.
        a, p = 0, 0
        if(type=="leastsquares" or type=="ls"):
            for i in range(m):  # Take all datapoint inputs (log(x),log(y))
                A.append([1,log(inmat[i][0])])
                b.append([log(inmat[i][1])])
            coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(A),A)),transpose(A)),b)
            a = exp(coeffs[0][0]) # Need to exponentiate a, because it is still in its logarithmic form.
            p = coeffs[1][0]
        elif(type=="robust" or type=="theilsen"):
            for i in range(m):
                for j in range(m):
                    if(i < j):
                        if(inmat[i][0] != inmat[j][0]):
                            A.append((log(inmat[i][1])-log(inmat[j][1]))/(log(inmat[i][0])-log(inmat[j][0])))
            A.sort()
            p = A[int(round(len(A)/2))]
            for i in range(m):
                b.append(log(inmat[i][1]) - p*log(inmat[i][0]))
            b.sort()
            a = exp(b[int(round(len(b)/2))])
        func = lambda x : a * (x**p)
        eq = "%s*x**(%s)" % (a,p)
        output.append([a,p])
    elif (model == "nth" or model == "ndegree" or model == "poly" or model == "polynomial" or model == "quadratic"): # Polynomial regression to the nth degree. Doesn't work very well if n is greater than 6 or 7, as the matrix equation gets exponentially more expensive to compute.
        if model=="quadratic": degree=2 # Default to 2nd degree regresion if quadratic is requested.
        for i in range(m):
            rowapp = []
            for j in range(degree+1):
                rowapp.append(inmat[i][0]**j) # Rows of form 1, x, x^2, ..., x^n where n is the degree
            A.append(rowapp)
            b.append([inmat[i][1]])
        coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(A),A)),transpose(A)),b)
        eq = "%s" % (coeffs[0][0])
        for i in range(degree):
            eq += ("+%s*x**%s" % (coeffs[i+1][0],i+1))
        def func(x): # The function for r-computation can't be trivially defined with lambda anymore because an indeterminate number of terms need to be summed.
            out = 0
            for i in range(degree+1):
                out += coeffs[i][0]*(x**i)
            return out
        output.append(coeffs)
    elif (model == "exp" or model == "exponential"): # Exponential function regression, in the form of y=ap^x. Exploits logarithm properties in order to use a linear regression technique (either least-squares or Theil-Sen) to form a power-function fit.
        a, p = 0, 0
        if(type=="leastsquares" or type=="ls"):
            for i in range(m):  # Take all datapoint inputs (x,log(y))
                A.append([1,inmat[i][0]])
                b.append([log(inmat[i][1])])
            coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(A),A)),transpose(A)),b)
            a = exp(coeffs[0][0]) # Need to exponentiate both a and p, because they are both in logarithmic form here.
            p = exp(coeffs[1][0])
        elif(type=="robust" or type=="theilsen"):
            for i in range(m):
                for j in range(m):
                    if(i < j):
                        if(inmat[i][0] != inmat[j][0]):
                            A.append((log(inmat[i][1])-log(inmat[j][1]))/(inmat[i][0]-inmat[j][0])) # Take inputs (x,log(y)) for the Theil-Sen slope estimation
            A.sort()
            p = exp(A[int(round(len(A)/2))])
            for i in range(m):
                b.append(log(inmat[i][1]) - p*log(inmat[i][0]))
            b.sort()
            a = exp(b[int(round(len(b)/2))])
        func = lambda x : a * (p**x)
        eq = "%s*(%s)**x" % (a,p)
        output.append([a,p])
    elif (model == "con" or model == "conic"): # General conic form is Ax^2 + Bxy + Cy^2 + Dx + Ey + F = 0.
        rcomp=0 # Can't compute R^2 or R_a by normal method using this
        for i in range(m):
            A.append([inmat[i][0]**2,inmat[i][0]*inmat[i][1],inmat[i][1]**2,inmat[i][0],inmat[i][1]])
            b.append([1]) # I can't set the LHS equal to 0, so instead I set F = -1 and obtain Ax^2 + Bxy + Cy^2 + Dx +Ey = 1. This gives a nonzero b-vector to multiply by.
        coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(A),A)),transpose(A)),b)
        eq = "%s*x**2+%s*xy+%s*y**2+%s*x+%s*y+%s=0" % (coeffs[0][0],coeffs[1][0],coeffs[2][0],coeffs[3][0],coeffs[4][0],-1)
        func = lambda x : "NaF: Conic" # The conic is defined by an implicit equation of x and y, not by an explicit function.
        coeffs.append([-1])
        output.append(coeffs)  
    output.append(eq)
    if(rcomp==1):
        RSS,RSA,TSS,TSA,mu = 0,0,0,0,0
        for i in range(m):
            mu += inmat[i][1] # Sum total for mean (mu) computation
            RSS += (inmat[i][1] - func(inmat[i][0]))**2 # Residual sum of squared errors
            RSA += abs(inmat[i][1] - func(inmat[i][0])) # Residual sum of absolute errors
        mu/=m
        for i in range(m):
            TSS += (inmat[i][1] - mu)**2 # Total sum of squares (of distances from mean)
            TSA += abs(inmat[i][1] - mu) # Total sum of absolute distances from mean
        R2 = 1-RSS/TSS # R^2 coefficient of determination
        RA = 1-RSA/TSA # R_a residual absolute error coefficient
        if(restrict != 1):
            print("R^2 = %s, R_a = %s" % (R2, RA))
        output.append([R2,RA])
    if(restrict != 1):
        if(model != "conic" and model != "con"):
            print("y=" + eq)
        else:           
            print(eq)
    return output


def regPlot(src,filetype="CSV",model="lin",type="ls",deg=2,xlab="X Axis",ylab="Y Axis",prec=10):
    df=[]
    if filetype in ["CSV", "csv", ".csv"]: df = pd.read_csv(src,names=[xlab,ylab])
    if filetype in ["json", "JSON"]: df = pd.read_json(src,names=[xlab,ylab])
    if filetype in ["excel", "Excel", "xlsx", "XLSX"]: df = pd.read_excel(src,names=[xlab,ylab])
    if filetype in ["pickle", "Pickle"]: df = pd.read_pickle(src,names=[xlab,ylab])
    arr = np.array(df)
    minx,maxx=arr[0][0],arr[0][0]
    for i in range(len(arr)):
        if (arr[i][0] > maxx): maxx = arr[i][0]
        if (arr[i][0] < minx): minx = arr[i][0]
    rangex = maxx - minx
    x=np.linspace(int(minx-(rangex/100)),int(maxx+(rangex/100)),int(rangex*prec))
    fig,ax=plt.subplots()
    plt.xlabel(xlab)
    plt.ylabel(ylab)
    y=eval(regression(arr,model=model,type=type,degree=deg,rcomp=0)[1])
    plt.plot(x,y,'r')
    xd,yd=arr.T
    plt.scatter(xd,yd,s=5)
    plt.show()
    return