c file:                  taucloud.f
c
c external routines:     taucloud
c
c internal routines:     cloudpar, icepar, usrcloud 
c
c internal data block:   cloudblk
c
c=======================================================================
      subroutine taucloud(nz,ncldz,wl,lcld,lwp,tcloud,nre,taucld,wcld,
     &                    imomc,nmom,pmom)
c
c purpose:  compute cloud radiometric properties
c
c input: 
c
c  nz      number of atmospheric layers
c  ncldz   maximum number of cloud layers
c  wl      wavelength
c  lcld    cloud layer indecies
c  lwp     liquid water path of each cloud layer
c  tcloud  optical depth of each cloud layer
c  nre     effective radius within each cloud layer
c
c  imomc   selects a cloud phase function
c          1   isotropic scattering
c          2   rayleigh scattering
c          3   henyey_greenstein(g(re))
c          4   haze L as specified by garcia/siewert
C          5   cloud c.1 as specified by garcia/siewert
c 
c output:
c
c  taucld  increment of cloud optical depth at each vertical grid point
c  wcld    cloud single scatter albedo at each vertical grid point
c  pmom    phase function moments multiplied by taucld*wcld
c
      use params, only: mxly,nstrms,maxmom,wl55,kr

      implicit none
c                            
      real(kr), parameter :: rhoice=.917   ! bulk density of ice (g/cm3):

      real(kr) :: pmcld(0:maxmom), pmom(0:maxmom,*),
     &     lwp(*), tcloud(*), nre(*), lwpth, wcld(*), taucld(*),
     &     qcld(mxly), q550(mxly), reff, tcld, wt, wl, qc, wc, gc,
     &     qc550, wc550, gc550

      integer :: i, j, k, nz, lcld(*), icnt(mxly), ncmom=0, 
     &     ncldz, nmom, nmomc, imomc, lbot, ltop, ifirst=1

      save q550
     
      do j=1,nz
        qcld(j)=0.
        wcld(j)=0.
        taucld(j)=0.
        icnt(j)=0
      enddo

      do 20 i=1,ncldz
        call levrng(ncldz,lcld,i,lbot,ltop)
        if(lbot.eq.0) goto 20
        if(tcloud(i).eq.0..and.lwp(i).eq.0.) goto 20
        do 10 j=ltop,lbot
          if(ltop.eq.lbot) then
            reff=nre(i)
            tcld=tcloud(i)
            lwpth=lwp(i)
          else
            wt=float(j-ltop)/(lbot-ltop)
            reff=nre(i+1)*(nre(i)/nre(i+1))**wt

c           if tcloud(i+1) is zero, spread the opacity uniformly over
c           the affected layers.  if tcloud(i+1) is nonzero then the
c           opacity is distributed such that it linearly increases
c           (or decreases) from the bottom to the top layer.  The rate
c           of increase is controled by tcloud(i+1) which sets the ratio
c           of the the opacity in the top layer to that of the bottom
c           layer.  The total opacity over all layers adds up to
c           tcloud(i)

            if(tcloud(i+1).eq.0.) then
              tcld=tcloud(i)/(lbot-ltop+1)
            else
              tcld=2*tcloud(i)/((lbot-ltop+1)*(1.+tcloud(i+1)))
              tcld=tcld+(lbot-j)*tcld*(tcloud(i+1)-1.)/(lbot-ltop)
            endif
            
            if(lwp(i+1).eq.0.) then
              lwpth=lwp(i)/(lbot-ltop+1)
            else
              lwpth=2*lwp(i)/((lbot-ltop+1)*(1.+lwp(i+1)))
              lwpth=lwpth+(lbot-j)*lwpth*(lwp(i+1)-1.)/(lbot-ltop)
            endif
          endif

          ncmom=nmom
          call cloudpar(wl,reff,qc,wc,gc)
          call getmom(imomc,gc,nmom,pmcld)

          do k=1,min(nmom,ncmom)
            pmom(k,j)=pmcld(k)+pmom(k,j)
          enddo
          qcld(j)=qc+qcld(j)
          wcld(j)=wc+wcld(j)
          icnt(j)=1+icnt(j)
          if(tcloud(i).ne.0.) then
            if(ifirst.eq.1) then
              nmomc=nmom
              call cloudpar(wl55,reff,qc550,wc550,gc550)
              q550(j)=qc550
            endif
            taucld(j)=tcld*qc/q550(j)+taucld(j)
          else
            if(lwpth.ne.0.) then
              if(reff.lt.0.) then
                taucld(j)=-.75*qc*lwpth/reff/rhoice+taucld(j)
              else
                taucld(j)=.75*qc*lwpth/reff+taucld(j)
              endif
            endif
          endif
 10     continue
 20   continue

      do 30 j=1,nz
        if(icnt(j).ne.0) then
          qcld(j)=qcld(j)/icnt(j)
          wcld(j)=wcld(j)/icnt(j)
          do i=1,min(nmom,ncmom)
            pmom(i,j)=taucld(j)*wcld(j)*pmom(i,j)/icnt(j)
          enddo
        endif
        
 30   continue

      ifirst=0
      return
      end
c=======================================================================
      subroutine usrcloud(nz,wl,p,taucld,wcld,imomc,nmom,pmom,idb)
c
c purpose:  read user specified cloud properties
c           this routine is called when tcloud(1) < 0
c
c input:   
c   wl      wavelength
c   p       pressure array (mb), pressure at surface=p(1)
c   imomc   phase function type passed to getmom
c   nmom    number of phase function moments required
c   idb     if 1 then print diagnostics
c
c output:
c
c  taucld   increment of cloud optical depth at each vertical grid point
c  wcld     cloud single scatter albedo at each vertical grid point
c  pmom     phase function moments multiplied by taucld*wcld
c
c usrcloud format (read on first call only):
c
c read(13,*) (lwp(i),reff(i),fwp(i),reice(i),cldfrac(i)),i=1,nzz)
cc
c   lwp     liquid water path within layer (g/m2)          (default=0)
c   reff    effective radius of water droplets (microns).  (default=8.)
c   fwp     frozen water path within layer (g/m2)          (default=0)
c           negative fwp => ccm3 model parameters
c   reice   effective radius of ice particles (microns)    (default=-1)
c           reice.le.0 .and. fwp.gt.0 => ccm3 effective 
c           radius is used (see icepar)
c   cldfrac fraction of cloud within layer.  to simulate   (default=1)
c           the effect of partial cloud cover, optical depth is
c           reduced by cldfrac**1.5, an empirical result from
c           the NCAR Community Climate Model (CCM)
c 
c   DISCUSSION:

c           Cloud records are read from file usrcld.dat starting with
c           the lowest layer in the atmosphere. The input file need not
c           specify values above the altitude where clouds are present.
c           In addition the input record may be terminated before
c           reading all the input variables by inserting a forward slash
c           in place of the first variable to be ignord.  If left
c           unspecified, input quantities retain their default of
c           lwp=0. reff=10, fwp=0, reice=-1, and cldfrac=1.  The radiative
c           properties of ice are computed from a CCM3 model
c           parameterization contained in subroutine icepar

c
      use params, only: maxmom,mxly,kr
      implicit none

      integer, parameter :: rhoice=.917 ! the bulk density of ice (g/cm3)

      real(kr) :: p(*), pmom(0:maxmom,*), lwp(mxly), reff(mxly),
     &     fwp(mxly), reice(mxly), cldfrac(mxly), wcld(*),taucld(*),
     &     ww, gw, qw, wi, gi, qi, qcld(mxly), wl, tauw, prat, ri,
     &     taui, gcld

      integer :: i, j, nz, imomc, idb, nmom, ifirst=1
     
      data lwp/mxly*0./
      data reff/mxly*8./
      data fwp/mxly*0./
      data reice/mxly*-1./
      data cldfrac/mxly*1./

      save lwp,reff,fwp,cldfrac

      if(ifirst.eq.1) then 
        open(unit=13,file='usrcld.dat',form='formatted',status='old')
        do i=nz,1,-1
          read(13,*,end=100) lwp(i),reff(i),fwp(i),reice(i),cldfrac(i)
        enddo
        close(unit=13)
 100    continue
        ifirst=0
      endif

      if(imomc.lt.0) then
        print *,'imomc < 0 not allowed with usrcld.dat option'
        stop
      endif

      if(idb.gt.0) print '(a5,10a11)',
     &   'i','p','taucld','tauw','taui','qw','qi','ww','wi','gw','gi'

      do i=1,nz

        if(lwp(i).gt.0.) then 
          call cloudpar(wl,reff(i),qw,ww,gw)
          tauw=.75*qw*lwp(i)/reff(i)
        else
          qw=0.
          ww=0.
          gw=0.
          tauw=0.
        endif

        if(fwp(i).ne.0.) then
          prat=p(nz-i+1)/p(1)
          ri=reice(i)
          if(fwp(i).lt.0) then 
            call icepar(wl,prat,qi,wi,gi,ri)
          else 
            if(ri.le.0) call icepar(wl,prat,qi,wi,gi,ri)
            ri=-abs(ri)
            call cloudpar(wl,ri,qi,wi,gi)
          endif
          taui=.75*qi*abs(fwp(i))/(rhoice*abs(ri))
        else
          qi=0.
          wi=0.
          gi=0.
          taui=0.
        endif

        taucld(i)=tauw+taui
        if(taucld(i).ne.0.) then
          wcld(i)=(tauw*ww+taui*wi)/taucld(i)
          gcld=(tauw*gw+taui*gi)/taucld(i)
          call getmom(imomc,gcld,nmom,pmom(0,i))
        endif
        taucld(i)=taucld(i)*cldfrac(i)**1.5
        do j=1,nmom
          pmom(j,i)=taucld(i)*wcld(i)*pmom(j,i)
        enddo
        if(idb.gt.0) print '(i5,10es11.3)',
     &       i,p(nz-i+1),taucld(i),tauw,taui,qw,qi,ww,wi,gw,gi
      enddo

      return
      end
c_end
c=======================================================================
      subroutine icepar(wl,prat,qice,wice,gice,rei)
c
c purpose: compute ice radiative parameters using CCM3 ice coefficients.
c
c input:
c   wl        wavelength (um)
c   prat      pressure ratio p(z)/p_surface
c
c input/output:
c   rei       effective radius.  if rei.le.0 then replace with
c             effective radius from ccm3 ice model 
c             
c
c output:
c   qice      extinction coefficient
c   wice      single scattering albedo of ice
c   gice      asymmetry factor of ice 
c   rei       effective radius of ice (um)
c
      use params, only: kr

      implicit none

      real(kr) :: abari(4), bbari(4), cbari(4), dbari(4), ebari(4),
     &     fbari(4), pllow, plhgh, reimax, reimin, wl, rei, prat,
     &     qice, wice, gice

      integer :: iceindx
 
      data abari /3.448e-03,3.448e-03,3.448e-03,3.448e-03/
      data bbari /2.431    ,2.431    ,2.431    ,2.431    /
      data cbari /1.00e-05 ,1.10e-04 ,1.861e-02,.46658   /
      data dbari /0.0      ,1.405e-05,8.328e-04,2.05e-05 /
      data ebari /0.7661   ,0.7730   ,0.794    ,0.9595   /
      data fbari /5.851e-04,5.665e-04,7.267e-04,1.076e-04/
      data pllow  /.0/
      data plhgh  /.4/
      data reimax /30./
      data reimin /10./
 
c     CCM3 code - partitions solar spectrum to 4 bands.
 
      if(wl.lt.0.69) then
        iceindx=1
      else if(wl.lt.1.19) then
        iceindx=2
      else if(wl.lt.2.38) then
        iceindx=3
      else
        iceindx=4
      endif
 
      if(rei.le.0.) then 
        if(prat.gt.plhgh) then
          rei=reimin
        else
          rei=reimax-((reimax-reimin)*((prat-plhgh)/(plhgh-pllow)))
        endif
      endif

      gice=ebari(iceindx)+fbari(iceindx)*rei
      wice=1.-cbari(iceindx)-dbari(iceindx)*rei
      qice=(abari(iceindx)+(bbari(iceindx)/rei))

      return
      end
c=======================================================================

      SUBROUTINE cloudpar(wl,re,qc,wc,gc)
c
c input:
c   wl      wavelength in microns
c   re      effective cloud drop radius (microns)
c output:
c   qc      mean extinction efficiency (unitless)
c   wc      single scatter albedo      (unitless)
c   gc      asymmetry factor           (unitless)
c     

c the parameters contained in this subroutine were computed using 
c Wiscombe Mie scattering code, MIEV0.  The radius integrations
c were performed over a modified gamma distribution
c f(x)=(x/xm)^(p-1)*exp(-x/xm) where p=7.

c NOTE:

c re < 0 selects scattering parameters for ice spheres
c Mie scattering parameters for water and ice clouds
c
c Wavelength range is  0.29 to  333.33 microns 
c
c Wavelengths steps are in equal increments of log(wavelength)
c Hence, the i'th table entry is at wavelenth
c
c          wl = 0.29 * (333.33/0.29)**((i-1)/(mxwv-1))
c
c Included in the table are entries for effective radii of
c
c          re = 2**((j+1)/2) for j=1,13
c
c This Mie database was created by 
c    /home/paul/mie/wiscombe/{water,ice}/pmom.cmd 
c and formated with
c    /home/paul/mie/wiscombe/{water,ice}/mkpmom.pro
 
      use params, only: one, kr
      implicit none

      real(kr), parameter :: wlmin=0.29, wlmax=333.33, eps=.000001

      integer, parameter :: mxwv=400, mre=13
      integer ::  icall, i, iw, ir

      real(kr) :: qq(mxwv,mre),ww(mxwv,mre),gg(mxwv,mre),
     &        qqi(mxwv,mre),wwi(mxwv,mre),ggi(mxwv,mre),
     &        fr, fw, gc, qc, re, wc, wl, wmax, wmin, wstep

      data icall,wmin,wmax,wstep/0,0.,0.,0./

c 
c        ---------------- extinction efficiency ----------------

      data (qq(i,1),i=1,mxwv)/
     &   2.17502666, 2.17705488, 2.17945981, 2.18155527, 2.18386197,
     &   2.18641090, 2.18857670, 2.19129729, 2.19366312, 2.19596457,
     &   2.19853497, 2.20104814, 2.20371795, 2.20627928, 2.20893717,
     &   2.21180654, 2.21439219, 2.21722698, 2.22005177, 2.22292018,
     &   2.22576642, 2.22874284, 2.23176241, 2.23473716, 2.23774362,
     &   2.24082780, 2.24402070, 2.24720716, 2.25038552, 2.25361443,
     &   2.25692534, 2.26020098, 2.26354837, 2.26690340, 2.27032566,
     &   2.27374268, 2.27719760, 2.28062248, 2.28411269, 2.28760719,
     &   2.29113030, 2.29466820, 2.29824281, 2.30179715, 2.30538130,
     &   2.30899572, 2.31262636, 2.31630254, 2.32001519, 2.32378149,
     &   2.32762146, 2.33154297, 2.33557439, 2.33973503, 2.34405398,
     &   2.34856272, 2.35329628, 2.35829353, 2.36359835, 2.36925769,
     &   2.37532187, 2.38184500, 2.38888407, 2.39649868, 2.40475059,
     &   2.41370153, 2.42341542, 2.43395329, 2.44537401, 2.45773554,
     &   2.47109199, 2.48549771, 2.50098848, 2.51760030, 2.53535986,
     &   2.55428433, 2.57437634, 2.59563327, 2.61801910, 2.64149570,
     &   2.66603732, 2.69156599, 2.71801019, 2.74528360, 2.77326369,
     &   2.80183864, 2.83086061, 2.86014557, 2.88957477, 2.91879225,
     &   2.94668722, 2.97444415, 3.00203633, 3.02939057, 3.05576015,
     &   3.08073235, 3.10413814, 3.12579226, 3.14543080, 3.16288662,
     &   3.17790985, 3.19019771, 3.19952631, 3.20562434, 3.20852327,
     &   3.20691943, 3.19620252, 3.18033361, 3.17178464, 3.16243815,
     &   3.14695644, 3.12438679, 3.09612036, 3.06194472, 3.02178931,
     &   2.97530246, 2.92187381, 2.86102247, 2.79152226, 2.71235824,
     &   2.62156391, 2.51724195, 2.39780188, 2.25397897, 2.06965280,
     &   1.81314278, 1.47268379, 1.15169585, 1.28466785, 1.53216481,
     &   1.85487890, 2.12174034, 2.31310582, 2.47859597, 2.64122748,
     &   2.79979491, 2.90341711, 2.95882654, 2.94658995, 2.86584187,
     &   2.77218103, 2.66592813, 2.55328059, 2.43995881, 2.33058119,
     &   2.22600079, 2.12766719, 2.03446722, 1.94598961, 1.86261523,
     &   1.78330398, 1.70802879, 1.63683224, 1.57049978, 1.50879097,
     &   1.45302904, 1.40313184, 1.36117184, 1.31957674, 1.27231741,
     &   1.21919715, 1.16273332, 1.10242403, 1.03997958, 0.97503036,
     &   0.91024303, 0.84782869, 0.78598338, 0.72648561, 0.67700797,
     &   0.68004543, 0.80958587, 1.06148815, 1.21916318, 1.06732845,
     &   0.92990327, 0.84266120, 0.77429891, 0.71917790, 0.67489249,
     &   0.63671166, 0.60254693, 0.57224268, 0.54376388, 0.51730049,
     &   0.49202278, 0.46848267, 0.44672537, 0.42581266, 0.40601236,
     &   0.38732231, 0.36951771, 0.35282731, 0.33719108, 0.32282090,
     &   0.30936635, 0.29687786, 0.28551263, 0.27485919, 0.26570359,
     &   0.25775653, 0.25239041, 0.25175655, 0.25901312, 0.27342898,
     &   0.29718980, 0.32283768, 0.35670570, 0.39677656, 0.44318610,
     &   0.48895615, 0.54749864, 0.60162365, 0.65336853, 0.69634545,
     &   0.73471516, 0.76891905, 0.79815042, 0.81967932, 0.83824831,
     &   0.85156107, 0.85920340, 0.86628318, 0.86854428, 0.86856580,
     &   0.86789280, 0.86537731, 0.86100680, 0.85260862, 0.84360075,
     &   0.83239943, 0.81652278, 0.80135447, 0.78186607, 0.76419902,
     &   0.74566597, 0.72170585, 0.69796336, 0.67450345, 0.64958209,
     &   0.62229031, 0.60202730, 0.58223885, 0.56328470, 0.54505736,
     &   0.52727163, 0.50990611, 0.49297634, 0.47528338, 0.45593211,
     &   0.43725908, 0.42013198, 0.40361109, 0.38725349, 0.37129331,
     &   0.35583365, 0.34084553, 0.32662845, 0.31328207, 0.30051482,
     &   0.28836867, 0.27716801, 0.26738128, 0.25871325, 0.25169498,
     &   0.24573471, 0.24069007, 0.23662263, 0.23302822, 0.22975561,
     &   0.22687808, 0.22466733, 0.22326955, 0.22239502, 0.22159633,
     &   0.22182521, 0.22313239, 0.22480163, 0.22632575, 0.22770067,
     &   0.22977827, 0.23218539, 0.23447281, 0.23606516, 0.23758155,
     &   0.23897120, 0.24005422, 0.24075750, 0.24140766, 0.24041410,
     &   0.23857823, 0.23654707, 0.23449720, 0.23238145, 0.22981887,
     &   0.22545232, 0.22110908, 0.21650995, 0.21198845, 0.20732638,
     &   0.20268758, 0.19721176, 0.19181943, 0.18613218, 0.18070966,
     &   0.17568652, 0.17069620, 0.16572662, 0.16076547, 0.15587592,
     &   0.15096237, 0.14622231, 0.14167967, 0.13731468, 0.13312168,
     &   0.12914601, 0.12484229, 0.12050779, 0.11640624, 0.11245794,
     &   0.10855427, 0.10464919, 0.10097606, 0.09758507, 0.09428326,
     &   0.09104992, 0.08808544, 0.08513276, 0.08217223, 0.07938257,
     &   0.07679242, 0.07432113, 0.07205009, 0.06984399, 0.06770088,
     &   0.06561888, 0.06359617, 0.06214036, 0.06082397, 0.05953438,
     &   0.05827096, 0.05703313, 0.05586062, 0.05471718, 0.05359612,
     &   0.05249694, 0.05142131, 0.05037135, 0.04934165, 0.04833176,
     &   0.04735898, 0.04651768, 0.04569063, 0.04487758, 0.04408547,
     &   0.04333149, 0.04258981, 0.04186019, 0.04115284, 0.04046454,
     &   0.03978714, 0.03913825, 0.03857718, 0.03802390, 0.03747830,
     &   0.03693016, 0.03638894, 0.03585538, 0.03529256, 0.03472875,
     &   0.03417354, 0.03363883, 0.03311307, 0.03259677, 0.03209499,
     &   0.03160060, 0.03111210, 0.03062983, 0.03015471, 0.02970541,
     &   0.02926306, 0.02880034, 0.02832900, 0.02786013, 0.02738631,
     &   0.02692121, 0.02647947, 0.02604464, 0.02561771, 0.02519749,
     &   0.02479647, 0.02440309, 0.02401588, 0.02363460, 0.02327066,
     &   0.02291242, 0.02255957, 0.02221145, 0.02186741, 0.02152859/
      data (qq(i,2),i=1,mxwv)/
     &   2.13798332, 2.13925219, 2.14121342, 2.14302087, 2.14454389,
     &   2.14633369, 2.14811230, 2.15019870, 2.15216684, 2.15378094,
     &   2.15584373, 2.15777731, 2.15973592, 2.16166997, 2.16378689,
     &   2.16593575, 2.16784549, 2.17012668, 2.17208719, 2.17420554,
     &   2.17655826, 2.17878199, 2.18132281, 2.18330407, 2.18574071,
     &   2.18803787, 2.19039845, 2.19302201, 2.19534039, 2.19793582,
     &   2.20031524, 2.20289874, 2.20567107, 2.20836806, 2.21095133,
     &   2.21373105, 2.21643114, 2.21926165, 2.22209811, 2.22489476,
     &   2.22785592, 2.23078966, 2.23379755, 2.23679447, 2.23991466,
     &   2.24296117, 2.24611163, 2.24927950, 2.25246811, 2.25567436,
     &   2.25902772, 2.26228857, 2.26560974, 2.26899385, 2.27235270,
     &   2.27572775, 2.27916002, 2.28259277, 2.28602648, 2.28948903,
     &   2.29299402, 2.29648638, 2.30001974, 2.30355215, 2.30712700,
     &   2.31072402, 2.31436896, 2.31805515, 2.32180691, 2.32563615,
     &   2.32955813, 2.33359814, 2.33777022, 2.34213328, 2.34669137,
     &   2.35149336, 2.35658002, 2.36199999, 2.36780286, 2.37404275,
     &   2.38078094, 2.38808084, 2.39600778, 2.40463352, 2.41402841,
     &   2.42426777, 2.43542647, 2.44758058, 2.46080685, 2.47520447,
     &   2.49055290, 2.50686765, 2.52442575, 2.54340053, 2.56385159,
     &   2.58563137, 2.60874343, 2.63316607, 2.65884924, 2.68580270,
     &   2.71406054, 2.74336243, 2.77359009, 2.80458641, 2.83639789,
     &   2.86914539, 2.89929056, 2.92447400, 2.95171785, 2.98308206,
     &   3.01519752, 3.04522467, 3.07343721, 3.09903026, 3.12162685,
     &   3.14059973, 3.15504026, 3.16401362, 3.16631794, 3.15989852,
     &   3.14246082, 3.11055136, 3.06122589, 2.98802567, 2.86877966,
     &   2.64914417, 2.24823785, 1.68687320, 1.65082586, 1.80276322,
     &   2.05663753, 2.25913787, 2.40496802, 2.52363515, 2.64168501,
     &   2.77308202, 2.89902425, 3.02475595, 3.13580918, 3.19440866,
     &   3.22508168, 3.23115849, 3.21329999, 3.17614865, 3.12688971,
     &   3.06819391, 3.00411606, 2.93599343, 2.86500192, 2.79205775,
     &   2.71830487, 2.64325356, 2.56808257, 2.49391270, 2.42130661,
     &   2.35278726, 2.28868771, 2.23401618, 2.18567824, 2.13337469,
     &   2.07345867, 2.00666738, 1.93272746, 1.85209572, 1.76304054,
     &   1.66803360, 1.56938481, 1.46480370, 1.35238814, 1.23800766,
     &   1.17062557, 1.25814593, 1.53864408, 1.80666018, 1.74136353,
     &   1.60048425, 1.49206913, 1.39938819, 1.31875503, 1.24944663,
     &   1.18757772, 1.13048494, 1.07846689, 1.02890050, 0.98189175,
     &   0.93626744, 0.89275277, 0.85150170, 0.81159115, 0.77302444,
     &   0.73593479, 0.69997483, 0.66549397, 0.63244063, 0.60123473,
     &   0.57131159, 0.54277855, 0.51594448, 0.48996422, 0.46606418,
     &   0.44387582, 0.42454937, 0.41302297, 0.41110086, 0.42056698,
     &   0.44397256, 0.47131950, 0.50981212, 0.55733794, 0.61353487,
     &   0.66908497, 0.74138194, 0.80845982, 0.87330765, 0.92801374,
     &   0.97697091, 1.02180791, 1.06119645, 1.09199321, 1.11951602,
     &   1.14143550, 1.15647972, 1.17098117, 1.18009853, 1.18668222,
     &   1.19206452, 1.19590342, 1.19766498, 1.19458222, 1.19039881,
     &   1.18394518, 1.17135620, 1.15939510, 1.14124632, 1.12467051,
     &   1.10712790, 1.08210421, 1.05639410, 1.02959049, 0.99969471,
     &   0.96512556, 0.93898529, 0.91308397, 0.88795686, 0.86347985,
     &   0.83926386, 0.81547731, 0.79197890, 0.76698780, 0.73913848,
     &   0.71192718, 0.68603122, 0.66079444, 0.63542688, 0.61038649,
     &   0.58576566, 0.56151837, 0.53822154, 0.51596385, 0.49447951,
     &   0.47369748, 0.45428184, 0.43682915, 0.42111406, 0.40789378,
     &   0.39639977, 0.38641477, 0.37806916, 0.37057266, 0.36368626,
     &   0.35748842, 0.35236308, 0.34856409, 0.34564441, 0.34291050,
     &   0.34179229, 0.34235492, 0.34365222, 0.34482867, 0.34586433,
     &   0.34801191, 0.35076204, 0.35344568, 0.35519850, 0.35690305,
     &   0.35849330, 0.35969409, 0.36039451, 0.36111709, 0.35949779,
     &   0.35661972, 0.35342786, 0.35023332, 0.34698740, 0.34307635,
     &   0.33652633, 0.32998708, 0.32302523, 0.31616625, 0.30910769,
     &   0.30207407, 0.29380319, 0.28564483, 0.27701414, 0.26877081,
     &   0.26110730, 0.25349978, 0.24593145, 0.23838133, 0.23094511,
     &   0.22347386, 0.21626568, 0.20935442, 0.20271324, 0.19634008,
     &   0.19030578, 0.18380356, 0.17726648, 0.17107494, 0.16512010,
     &   0.15924217, 0.15337250, 0.14784652, 0.14274411, 0.13778329,
     &   0.13293341, 0.12848461, 0.12406534, 0.11964417, 0.11547906,
     &   0.11161148, 0.10792426, 0.10453162, 0.10124227, 0.09805273,
     &   0.09495969, 0.09195992, 0.08978231, 0.08781121, 0.08588430,
     &   0.08400035, 0.08215816, 0.08041602, 0.07872003, 0.07706007,
     &   0.07543521, 0.07384758, 0.07230001, 0.07078455, 0.06930044,
     &   0.06787216, 0.06663447, 0.06541952, 0.06422679, 0.06306614,
     &   0.06196215, 0.06087751, 0.05981183, 0.05877952, 0.05777590,
     &   0.05678929, 0.05584471, 0.05502670, 0.05422094, 0.05342721,
     &   0.05263123, 0.05184607, 0.05107273, 0.05025868, 0.04944408,
     &   0.04864248, 0.04787079, 0.04711253, 0.04636840, 0.04564557,
     &   0.04493385, 0.04423108, 0.04353771, 0.04285501, 0.04220954,
     &   0.04157440, 0.04091078, 0.04023536, 0.03956383, 0.03888561,
     &   0.03822013, 0.03758813, 0.03696627, 0.03635591, 0.03575535,
     &   0.03518232, 0.03462039, 0.03406746, 0.03352317, 0.03300372,
     &   0.03249256, 0.03198923, 0.03149279, 0.03100232, 0.03051943/
      data (qq(i,3),i=1,mxwv)/
     &   2.10869312, 2.10994959, 2.11140633, 2.11277103, 2.11422896,
     &   2.11536121, 2.11695004, 2.11851144, 2.11983109, 2.12128448,
     &   2.12290406, 2.12427378, 2.12593651, 2.12752199, 2.12906098,
     &   2.13054109, 2.13223219, 2.13380599, 2.13549471, 2.13716388,
     &   2.13883781, 2.14053965, 2.14241076, 2.14389253, 2.14597774,
     &   2.14763069, 2.14959931, 2.15132356, 2.15322566, 2.15502787,
     &   2.15699601, 2.15899014, 2.16102433, 2.16296744, 2.16500926,
     &   2.16714239, 2.16919565, 2.17152119, 2.17347479, 2.17582440,
     &   2.17812586, 2.18023086, 2.18252206, 2.18482947, 2.18741202,
     &   2.18976593, 2.19206786, 2.19451046, 2.19712734, 2.19927979,
     &   2.20195913, 2.20460868, 2.20721841, 2.21003222, 2.21257138,
     &   2.21535015, 2.21808338, 2.22079682, 2.22379136, 2.22658467,
     &   2.22949386, 2.23260593, 2.23552299, 2.23868084, 2.24167609,
     &   2.24473333, 2.24787998, 2.25107288, 2.25427008, 2.25750327,
     &   2.26077747, 2.26406288, 2.26738620, 2.27073312, 2.27406740,
     &   2.27748823, 2.28088045, 2.28429341, 2.28773665, 2.29119539,
     &   2.29467297, 2.29816723, 2.30168939, 2.30523252, 2.30881763,
     &   2.31244469, 2.31613111, 2.31988573, 2.32373452, 2.32769465,
     &   2.33177805, 2.33604360, 2.34052014, 2.34524989, 2.35028577,
     &   2.35567641, 2.36148620, 2.36778593, 2.37465405, 2.38218617,
     &   2.39050651, 2.39969325, 2.40986061, 2.42113304, 2.43366981,
     &   2.44789767, 2.46370196, 2.47966862, 2.49573636, 2.51447392,
     &   2.53611541, 2.56007242, 2.58641195, 2.61510921, 2.64629149,
     &   2.68005490, 2.71643901, 2.75540113, 2.79707813, 2.84088016,
     &   2.88646746, 2.93235302, 2.97695661, 3.02118087, 3.05438161,
     &   3.03401637, 2.82029462, 2.20736504, 1.96685779, 1.99823010,
     &   2.16106415, 2.28940988, 2.37913322, 2.44473457, 2.50376511,
     &   2.56359792, 2.62238908, 2.68873930, 2.76887298, 2.84917426,
     &   2.92530346, 2.99790573, 3.06162143, 3.11368728, 3.15467215,
     &   3.18417263, 3.20343637, 3.21363235, 3.21554637, 3.20906758,
     &   3.19627380, 3.17594481, 3.14931393, 3.11678267, 3.07932758,
     &   3.03928494, 2.99739718, 2.95979095, 2.93239808, 2.90570259,
     &   2.87324166, 2.83226228, 2.78262997, 2.72195435, 2.64654684,
     &   2.55602670, 2.45047402, 2.32708144, 2.17662430, 1.99636972,
     &   1.82395816, 1.79886127, 2.02898788, 2.35301685, 2.44095850,
     &   2.36579680, 2.27746248, 2.18968177, 2.10384727, 2.02282238,
     &   1.94670093, 1.87331951, 1.80390608, 1.73616374, 1.67005050,
     &   1.60433435, 1.53977203, 1.47672570, 1.41483545, 1.35354042,
     &   1.29323709, 1.23349452, 1.17478526, 1.11712825, 1.06127715,
     &   1.00641990, 0.95282000, 0.90102839, 0.84956110, 0.80019069,
     &   0.75258124, 0.70687234, 0.67304814, 0.64910203, 0.64281952,
     &   0.65700501, 0.67927259, 0.71648842, 0.76698202, 0.82928628,
     &   0.89143950, 0.97439528, 1.05156362, 1.12681627, 1.19099045,
     &   1.24820960, 1.30166233, 1.34947407, 1.38847625, 1.42405236,
     &   1.45427072, 1.47696316, 1.49915755, 1.51610875, 1.53059542,
     &   1.54341340, 1.55549169, 1.56569850, 1.57107580, 1.57504904,
     &   1.57740808, 1.57322407, 1.56969309, 1.55904830, 1.54945028,
     &   1.53938389, 1.52113044, 1.50125325, 1.47851932, 1.45108402,
     &   1.41654611, 1.38996530, 1.36316288, 1.33678174, 1.31068814,
     &   1.28441143, 1.25846064, 1.23237038, 1.20382631, 1.17095113,
     &   1.13821161, 1.10528207, 1.07261980, 1.03890014, 1.00488675,
     &   0.97053176, 0.93571740, 0.90145522, 0.86771047, 0.83448863,
     &   0.80137235, 0.76968843, 0.74003637, 0.71266198, 0.68858790,
     &   0.66707826, 0.64786100, 0.63126296, 0.61609554, 0.60200340,
     &   0.58903724, 0.57771480, 0.56852263, 0.56079382, 0.55345875,
     &   0.54865283, 0.54646742, 0.54585260, 0.54528731, 0.54470569,
     &   0.54592073, 0.54831171, 0.55089056, 0.55235320, 0.55390370,
     &   0.55547148, 0.55662739, 0.55720031, 0.55807185, 0.55571741,
     &   0.55144519, 0.54661435, 0.54182684, 0.53707272, 0.53129399,
     &   0.52163976, 0.51189679, 0.50135422, 0.49089104, 0.48011509,
     &   0.46930960, 0.45660460, 0.44399253, 0.43051162, 0.41755992,
     &   0.40541157, 0.39333448, 0.38130838, 0.36929464, 0.35744789,
     &   0.34551895, 0.33398399, 0.32289293, 0.31221557, 0.30197191,
     &   0.29228595, 0.28192002, 0.27151796, 0.26163468, 0.25213447,
     &   0.24277574, 0.23345183, 0.22464956, 0.21651274, 0.20861855,
     &   0.20092036, 0.19384757, 0.18685448, 0.17988528, 0.17331909,
     &   0.16721793, 0.16140804, 0.15604682, 0.15086702, 0.14586158,
     &   0.14102377, 0.13634714, 0.13289264, 0.12975894, 0.12670794,
     &   0.12373675, 0.12084254, 0.11811424, 0.11546731, 0.11288538,
     &   0.11036633, 0.10791252, 0.10552726, 0.10319855, 0.10092463,
     &   0.09874038, 0.09683983, 0.09497969, 0.09315877, 0.09139106,
     &   0.08971205, 0.08806676, 0.08645427, 0.08489495, 0.08338176,
     &   0.08189763, 0.08047837, 0.07924533, 0.07803354, 0.07684249,
     &   0.07565257, 0.07448127, 0.07332975, 0.07212290, 0.07091793,
     &   0.06973404, 0.06859525, 0.06747789, 0.06638284, 0.06532034,
     &   0.06427555, 0.06324537, 0.06223033, 0.06123209, 0.06028872,
     &   0.05936145, 0.05839503, 0.05741312, 0.05643792, 0.05545427,
     &   0.05448988, 0.05357417, 0.05267388, 0.05179091, 0.05092278,
     &   0.05009467, 0.04928317, 0.04848522, 0.04770028, 0.04695141,
     &   0.04621495, 0.04549025, 0.04477591, 0.04407059, 0.04337656/
      data (qq(i,4),i=1,mxwv)/
     &   2.08608413, 2.08718085, 2.08819842, 2.08928776, 2.09026051,
     &   2.09133458, 2.09247351, 2.09354448, 2.09475374, 2.09592819,
     &   2.09698820, 2.09825253, 2.09944415, 2.10058975, 2.10188723,
     &   2.10301042, 2.10440993, 2.10564876, 2.10713172, 2.10819149,
     &   2.10947013, 2.11083865, 2.11228514, 2.11354899, 2.11504149,
     &   2.11647153, 2.11772227, 2.11944032, 2.12073398, 2.12217474,
     &   2.12372160, 2.12518692, 2.12670422, 2.12827706, 2.12999678,
     &   2.13142085, 2.13316607, 2.13475418, 2.13644004, 2.13801360,
     &   2.13973999, 2.14158034, 2.14336085, 2.14500785, 2.14754248,
     &   2.14867210, 2.15053105, 2.15235233, 2.15427756, 2.15617537,
     &   2.15817475, 2.16008782, 2.16209793, 2.16414022, 2.16627002,
     &   2.16842127, 2.17039990, 2.17264581, 2.17476940, 2.17722082,
     &   2.17922306, 2.18139601, 2.18377066, 2.18595934, 2.18852735,
     &   2.19084144, 2.19325399, 2.19597149, 2.19838595, 2.20085573,
     &   2.20338416, 2.20608950, 2.20859051, 2.21130514, 2.21415663,
     &   2.21691465, 2.21960235, 2.22249460, 2.22532177, 2.22823811,
     &   2.23119760, 2.23420048, 2.23719168, 2.24030447, 2.24336004,
     &   2.24646950, 2.24964976, 2.25286150, 2.25608301, 2.25932121,
     &   2.26253939, 2.26582241, 2.26915288, 2.27252769, 2.27591467,
     &   2.27931309, 2.28272390, 2.28614759, 2.28958941, 2.29304862,
     &   2.29653072, 2.30004168, 2.30359244, 2.30719543, 2.31088018,
     &   2.31466842, 2.31852102, 2.32249141, 2.32671952, 2.33127499,
     &   2.33619928, 2.34156895, 2.34752536, 2.35422683, 2.36188507,
     &   2.37076735, 2.38122392, 2.39368606, 2.40880442, 2.42733002,
     &   2.45043492, 2.47953343, 2.51644492, 2.56633377, 2.63695979,
     &   2.73646903, 2.80438280, 2.48966694, 2.15578151, 2.10177422,
     &   2.18972707, 2.26456690, 2.31655908, 2.35262156, 2.38277960,
     &   2.40946722, 2.43095112, 2.45043802, 2.47089243, 2.49408054,
     &   2.52004957, 2.55064130, 2.58556533, 2.62389326, 2.66443777,
     &   2.70609426, 2.74769092, 2.78883410, 2.82893705, 2.86680984,
     &   2.90274501, 2.93540955, 2.96442008, 2.98866057, 3.00792336,
     &   3.02198839, 3.03077078, 3.03648925, 3.04872918, 3.06593990,
     &   3.08406568, 3.09908509, 3.11101890, 3.11588383, 3.10966635,
     &   3.08762121, 3.04506540, 2.97790289, 2.86905861, 2.69852328,
     &   2.45939159, 2.28735828, 2.37881136, 2.63521552, 2.83085465,
     &   2.87047887, 2.85724258, 2.82630253, 2.78239822, 2.73039556,
     &   2.67559981, 2.61768126, 2.55856657, 2.49776578, 2.43501234,
     &   2.36961031, 2.30192137, 2.23245740, 2.16206431, 2.08944917,
     &   2.01527452, 1.93915737, 1.86155772, 1.78262067, 1.70341992,
     &   1.62305439, 1.54199219, 1.46106601, 1.37812603, 1.29532707,
     &   1.21259439, 1.12785673, 1.05840755, 0.99532914, 0.95742053,
     &   0.94795388, 0.95371586, 0.97912616, 1.02392077, 1.08473706,
     &   1.14701998, 1.23365045, 1.31460798, 1.39419651, 1.46257877,
     &   1.52303052, 1.58031464, 1.63210475, 1.67565691, 1.71578753,
     &   1.75132167, 1.77945089, 1.80704463, 1.83019590, 1.85123301,
     &   1.87025559, 1.88947523, 1.90730131, 1.92113745, 1.93348849,
     &   1.94530141, 1.95149374, 1.95836437, 1.95874786, 1.95952487,
     &   1.96076453, 1.95526111, 1.94776297, 1.93628299, 1.91968000,
     &   1.89515257, 1.87585616, 1.85601318, 1.83620429, 1.81630564,
     &   1.79586506, 1.77574813, 1.75512981, 1.73167431, 1.70336866,
     &   1.67456388, 1.64312804, 1.61127734, 1.57719040, 1.54180443,
     &   1.50471973, 1.46561122, 1.42580879, 1.38490987, 1.34344804,
     &   1.30030930, 1.25759029, 1.21547580, 1.17526007, 1.13806653,
     &   1.10376072, 1.07215190, 1.04398108, 1.01774323, 0.99301237,
     &   0.96975309, 0.94857353, 0.93038440, 0.91433358, 0.89892870,
     &   0.88709855, 0.87900382, 0.87400436, 0.86951578, 0.86537468,
     &   0.86397922, 0.86478364, 0.86644882, 0.86713207, 0.86827159,
     &   0.86983430, 0.87118530, 0.87207621, 0.87397659, 0.87195694,
     &   0.86715698, 0.86135793, 0.85571909, 0.85037875, 0.84350461,
     &   0.83126980, 0.81861663, 0.80435240, 0.78994828, 0.77498752,
     &   0.75973231, 0.74151307, 0.72311902, 0.70289445, 0.68316793,
     &   0.66430342, 0.64537662, 0.62637138, 0.60720849, 0.58815211,
     &   0.56875134, 0.54981065, 0.53141022, 0.51354426, 0.49632597,
     &   0.48001266, 0.46263373, 0.44515380, 0.42837855, 0.41219971,
     &   0.39624578, 0.38034523, 0.36520466, 0.35113859, 0.33749765,
     &   0.32421276, 0.31194299, 0.29987898, 0.28790689, 0.27660036,
     &   0.26606229, 0.25602967, 0.24670805, 0.23774523, 0.22912604,
     &   0.22083592, 0.21286091, 0.20678002, 0.20123938, 0.19587976,
     &   0.19069350, 0.18567327, 0.18096623, 0.17642576, 0.17202219,
     &   0.16775009, 0.16361059, 0.15960635, 0.15571795, 0.15194090,
     &   0.14832486, 0.14515376, 0.14206679, 0.13906078, 0.13615564,
     &   0.13340358, 0.13071999, 0.12810239, 0.12557934, 0.12313955,
     &   0.12075731, 0.11848422, 0.11649707, 0.11455292, 0.11265031,
     &   0.11076353, 0.10891392, 0.10710233, 0.10522015, 0.10334931,
     &   0.10151698, 0.09975738, 0.09803587, 0.09635346, 0.09472474,
     &   0.09312759, 0.09155734, 0.09001437, 0.08850069, 0.08707152,
     &   0.08566996, 0.08421671, 0.08274542, 0.08128752, 0.07982088,
     &   0.07838541, 0.07702288, 0.07568563, 0.07437608, 0.07309065,
     &   0.07186520, 0.07066607, 0.06948867, 0.06833217, 0.06722954,
     &   0.06614668, 0.06508251, 0.06403495, 0.06300194, 0.06198668/
      data (qq(i,5),i=1,mxwv)/
     &   2.06820655, 2.06907678, 2.06985450, 2.07070923, 2.07153821,
     &   2.07240009, 2.07327151, 2.07405877, 2.07502890, 2.07594180,
     &   2.07686520, 2.07781076, 2.07874894, 2.07965040, 2.08064580,
     &   2.08158016, 2.08250928, 2.08357215, 2.08455801, 2.08565044,
     &   2.08637929, 2.08769488, 2.08875489, 2.08978438, 2.09119201,
     &   2.09195542, 2.09307051, 2.09417486, 2.09566021, 2.09656596,
     &   2.09772825, 2.09896684, 2.10003686, 2.10122228, 2.10266829,
     &   2.10368466, 2.10505676, 2.10627294, 2.10756540, 2.10885644,
     &   2.10994577, 2.11153722, 2.11289072, 2.11438799, 2.11563444,
     &   2.11702752, 2.11848927, 2.12003207, 2.12152100, 2.12287998,
     &   2.12438536, 2.12596536, 2.12761021, 2.12916422, 2.13065577,
     &   2.13222671, 2.13384008, 2.13563156, 2.13722587, 2.13888812,
     &   2.14061761, 2.14231634, 2.14399195, 2.14580345, 2.14766812,
     &   2.14946795, 2.15147948, 2.15331149, 2.15531325, 2.15730000,
     &   2.15920115, 2.16103840, 2.16323447, 2.16517234, 2.16729069,
     &   2.16925478, 2.17180204, 2.17365289, 2.17592597, 2.17805815,
     &   2.18034935, 2.18262458, 2.18497086, 2.18732190, 2.18993735,
     &   2.19229698, 2.19462013, 2.19715142, 2.19970536, 2.20224977,
     &   2.20480633, 2.20744205, 2.21013236, 2.21287704, 2.21566105,
     &   2.21848178, 2.22133803, 2.22423530, 2.22717381, 2.23014522,
     &   2.23316383, 2.23621774, 2.23930860, 2.24243641, 2.24560738,
     &   2.24880743, 2.25185871, 2.25477719, 2.25798988, 2.26141644,
     &   2.26488400, 2.26830220, 2.27171969, 2.27512717, 2.27853608,
     &   2.28195477, 2.28539276, 2.28887916, 2.29246712, 2.29624224,
     &   2.30038857, 2.30524015, 2.31143141, 2.32040977, 2.33603311,
     &   2.37137532, 2.45426416, 2.44946384, 2.20604682, 2.13419557,
     &   2.18014622, 2.22571373, 2.25818729, 2.28112984, 2.30040073,
     &   2.31750512, 2.33114266, 2.34305668, 2.35338426, 2.36136699,
     &   2.36845207, 2.37524247, 2.38214278, 2.38966513, 2.39820385,
     &   2.40807438, 2.41940808, 2.43239141, 2.44712353, 2.46356678,
     &   2.48175073, 2.50162148, 2.52295828, 2.54527211, 2.56819105,
     &   2.59079742, 2.61251926, 2.63217568, 2.65303588, 2.67775917,
     &   2.70673823, 2.73902822, 2.77534389, 2.81459999, 2.85631204,
     &   2.89715695, 2.93249226, 2.95910835, 2.96320677, 2.91557240,
     &   2.75280142, 2.52676201, 2.48015690, 2.59384298, 2.74544239,
     &   2.83394647, 2.88375354, 2.91826963, 2.93902302, 2.94725871,
     &   2.94816136, 2.94250584, 2.93109941, 2.91520643, 2.89412355,
     &   2.86783385, 2.83567119, 2.79767823, 2.75550127, 2.70736241,
     &   2.65363932, 2.59395409, 2.52823567, 2.45648146, 2.37949419,
     &   2.29647303, 2.20777822, 2.11415458, 2.01308823, 1.90620482,
     &   1.79376423, 1.67024875, 1.55999899, 1.44535625, 1.36056030,
     &   1.31077492, 1.28606808, 1.28603566, 1.31325352, 1.36183822,
     &   1.41526771, 1.49541938, 1.57121015, 1.64651060, 1.71159971,
     &   1.76844800, 1.82287967, 1.87235153, 1.91487622, 1.95417786,
     &   1.98994458, 2.01923084, 2.04786587, 2.07324982, 2.09701180,
     &   2.11866832, 2.14117742, 2.16281676, 2.18176293, 2.19935513,
     &   2.21735668, 2.23155284, 2.24632907, 2.25650620, 2.26651025,
     &   2.27776742, 2.28519630, 2.29097319, 2.29291487, 2.29087257,
     &   2.28247237, 2.27554417, 2.26810193, 2.26050091, 2.25264978,
     &   2.24424005, 2.23628497, 2.22780561, 2.21717906, 2.20294452,
     &   2.18797851, 2.16855264, 2.14828658, 2.12532520, 2.10047889,
     &   2.07296491, 2.04221296, 2.00941515, 1.97365880, 1.93596601,
     &   1.89444149, 1.85140610, 1.80609035, 1.76090717, 1.71656728,
     &   1.67406821, 1.63346839, 1.59599853, 1.56027949, 1.52597260,
     &   1.49290502, 1.46160078, 1.43346810, 1.40771902, 1.38269222,
     &   1.36168027, 1.34498692, 1.33325493, 1.32279706, 1.31332016,
     &   1.30742574, 1.30502141, 1.30461967, 1.30402958, 1.30455196,
     &   1.30626345, 1.30836785, 1.31056821, 1.31500280, 1.31589973,
     &   1.31346834, 1.30959499, 1.30618668, 1.30368543, 1.29933238,
     &   1.28896904, 1.27771175, 1.26376152, 1.24925303, 1.23394084,
     &   1.21783960, 1.19770801, 1.17674363, 1.15228570, 1.12778544,
     &   1.10357714, 1.07878339, 1.05338395, 1.02719164, 1.00060153,
     &   0.97278589, 0.94501305, 0.91738498, 0.88998175, 0.86317515,
     &   0.83752573, 0.80999488, 0.78189069, 0.75425148, 0.72720861,
     &   0.70021945, 0.67300582, 0.64650851, 0.62151313, 0.59706151,
     &   0.57307357, 0.55058616, 0.52846700, 0.50646114, 0.48545322,
     &   0.46565524, 0.44668886, 0.42877817, 0.41157955, 0.39506987,
     &   0.37922588, 0.36402419, 0.35186824, 0.34069008, 0.32993704,
     &   0.31959191, 0.30963805, 0.30035883, 0.29146108, 0.28288439,
     &   0.27461582, 0.26665196, 0.25899169, 0.25160208, 0.24447189,
     &   0.23767234, 0.23163033, 0.22579171, 0.22014777, 0.21472745,
     &   0.20961125, 0.20465828, 0.19986169, 0.19526058, 0.19083521,
     &   0.18654466, 0.18246461, 0.17885970, 0.17535831, 0.17195611,
     &   0.16862442, 0.16538107, 0.16222458, 0.15899503, 0.15581037,
     &   0.15270871, 0.14973888, 0.14684851, 0.14403814, 0.14132886,
     &   0.13868555, 0.13610087, 0.13357413, 0.13110690, 0.12878151,
     &   0.12651105, 0.12418008, 0.12183660, 0.11952472, 0.11721121,
     &   0.11495440, 0.11281383, 0.11072018, 0.10867617, 0.10667633,
     &   0.10477211, 0.10291418, 0.10109536, 0.09931409, 0.09761816,
     &   0.09595724, 0.09432944, 0.09273133, 0.09115964, 0.08961877/
      data (qq(i,6),i=1,mxwv)/
     &   2.05411696, 2.05476093, 2.05542445, 2.05602145, 2.05673957,
     &   2.05736589, 2.05815721, 2.05881834, 2.05952716, 2.06018996,
     &   2.06094384, 2.06168818, 2.06234145, 2.06310344, 2.06386614,
     &   2.06469393, 2.06542754, 2.06618309, 2.06695819, 2.06776166,
     &   2.06858897, 2.06941533, 2.07024527, 2.07111263, 2.07197547,
     &   2.07282686, 2.07369161, 2.07456946, 2.07545376, 2.07639432,
     &   2.07728839, 2.07823086, 2.07916021, 2.08000159, 2.08109283,
     &   2.08203435, 2.08302379, 2.08403444, 2.08505797, 2.08608317,
     &   2.08717680, 2.08816314, 2.08928466, 2.09025407, 2.09145880,
     &   2.09244537, 2.09361911, 2.09477377, 2.09595418, 2.09715438,
     &   2.09819961, 2.09934044, 2.10062861, 2.10186815, 2.10311317,
     &   2.10433388, 2.10559273, 2.10696149, 2.10810876, 2.10942388,
     &   2.11090374, 2.11212969, 2.11371994, 2.11499071, 2.11631060,
     &   2.11772799, 2.11924076, 2.12064791, 2.12207150, 2.12361336,
     &   2.12504840, 2.12671852, 2.12837768, 2.12970209, 2.13151193,
     &   2.13294792, 2.13467789, 2.13643956, 2.13797808, 2.13971138,
     &   2.14145851, 2.14318466, 2.14502048, 2.14688516, 2.14867878,
     &   2.15072083, 2.15239573, 2.15428305, 2.15624714, 2.15819168,
     &   2.16015983, 2.16214848, 2.16418982, 2.16628361, 2.16840315,
     &   2.17051125, 2.17272449, 2.17495179, 2.17723060, 2.17938137,
     &   2.18178248, 2.18411541, 2.18653321, 2.18895006, 2.19142032,
     &   2.19391632, 2.19634652, 2.19869804, 2.20127082, 2.20402932,
     &   2.20685768, 2.20969129, 2.21256971, 2.21548724, 2.21845365,
     &   2.22147202, 2.22454119, 2.22766376, 2.23084426, 2.23406315,
     &   2.23731828, 2.24058032, 2.24386501, 2.24731398, 2.25083137,
     &   2.25469613, 2.26597571, 2.29258537, 2.18143511, 2.13118839,
     &   2.15849757, 2.18855834, 2.21036124, 2.22608805, 2.23931837,
     &   2.25110126, 2.26072168, 2.26951075, 2.27760100, 2.28415132,
     &   2.29011202, 2.29568028, 2.30075884, 2.30541325, 2.30978560,
     &   2.31392241, 2.31792593, 2.32188511, 2.32587838, 2.32995987,
     &   2.33425617, 2.33882332, 2.34377050, 2.34916782, 2.35509562,
     &   2.36152601, 2.36842346, 2.37546253, 2.38278770, 2.39101505,
     &   2.40057182, 2.41173005, 2.42501092, 2.44101119, 2.46091676,
     &   2.48543286, 2.51500559, 2.55087304, 2.59300518, 2.63166356,
     &   2.61139894, 2.48055840, 2.40311646, 2.43157744, 2.48926401,
     &   2.53388190, 2.57037735, 2.60534668, 2.63847399, 2.66853070,
     &   2.69659019, 2.72295141, 2.74690151, 2.76941872, 2.78971767,
     &   2.80791521, 2.82277799, 2.83356071, 2.84106755, 2.84384060,
     &   2.84131551, 2.83282924, 2.81728816, 2.79389930, 2.76205349,
     &   2.72076297, 2.66931129, 2.60717225, 2.53173470, 2.44213629,
     &   2.33791590, 2.20967865, 2.08132863, 1.92886686, 1.79913509,
     &   1.70239902, 1.63908076, 1.60349882, 1.60286760, 1.62921667,
     &   1.66550636, 1.72914433, 1.79113674, 1.85388613, 1.90855753,
     &   1.95569682, 2.00122190, 2.04268622, 2.07890534, 2.11233115,
     &   2.14328790, 2.16921377, 2.19439626, 2.21748805, 2.23942614,
     &   2.25948095, 2.28053308, 2.30108595, 2.32012796, 2.33808374,
     &   2.35684037, 2.37357402, 2.39066625, 2.40525365, 2.41938996,
     &   2.43496060, 2.44939351, 2.46288323, 2.47371864, 2.48249650,
     &   2.48787975, 2.49208951, 2.49613523, 2.50011563, 2.50398088,
     &   2.50761199, 2.51184559, 2.51587033, 2.51912761, 2.52113247,
     &   2.52276111, 2.52008677, 2.51676702, 2.51152277, 2.50480032,
     &   2.49571466, 2.48363519, 2.46916986, 2.45112824, 2.43061233,
     &   2.40546346, 2.37728953, 2.34430909, 2.30919838, 2.27173185,
     &   2.23384666, 2.19584727, 2.15913296, 2.12301302, 2.08740306,
     &   2.05200076, 2.01700735, 1.98404872, 1.95275223, 1.92183030,
     &   1.89396334, 1.86962521, 1.85129035, 1.83491778, 1.82013488,
     &   1.80899131, 1.80216825, 1.79846919, 1.79589880, 1.79511058,
     &   1.79632819, 1.79877126, 1.80216968, 1.80898035, 1.81408882,
     &   1.81623924, 1.81686163, 1.81838071, 1.82157373, 1.82323492,
     &   1.82032061, 1.81634116, 1.80921245, 1.80132675, 1.79284811,
     &   1.78335631, 1.77006781, 1.75547171, 1.73624921, 1.71625805,
     &   1.69552636, 1.67367744, 1.65065360, 1.62609124, 1.60038745,
     &   1.57227528, 1.54323494, 1.51326966, 1.48253644, 1.45175755,
     &   1.42187381, 1.38910294, 1.35464907, 1.31931233, 1.28376579,
     &   1.24732709, 1.20954096, 1.17119706, 1.13393533, 1.09659779,
     &   1.05911875, 1.02291322, 0.98677623, 0.95014876, 0.91422421,
     &   0.87946033, 0.84543943, 0.81223166, 0.77996767, 0.74865896,
     &   0.71831310, 0.68893379, 0.66402185, 0.64068013, 0.61813128,
     &   0.59636211, 0.57535821, 0.55577022, 0.53695840, 0.51880342,
     &   0.50129026, 0.48441535, 0.46817762, 0.45253184, 0.43746081,
     &   0.42307484, 0.41003764, 0.39748421, 0.38539773, 0.37382850,
     &   0.36291718, 0.35240680, 0.34228238, 0.33259776, 0.32331842,
     &   0.31437802, 0.30589294, 0.29827940, 0.29093742, 0.28385562,
     &   0.27702653, 0.27043021, 0.26405671, 0.25766641, 0.25142890,
     &   0.24539651, 0.23963928, 0.23407461, 0.22870089, 0.22354980,
     &   0.21856016, 0.21371973, 0.20902373, 0.20447034, 0.20018937,
     &   0.19603764, 0.19184370, 0.18767549, 0.18359342, 0.17954448,
     &   0.17561692, 0.17189540, 0.16827704, 0.16476318, 0.16134495,
     &   0.15809684, 0.15494408, 0.15187432, 0.14888415, 0.14604434,
     &   0.14327748, 0.14057951, 0.13794412, 0.13536531, 0.13284895/
      data (qq(i,7),i=1,mxwv)/
     &   2.04292679, 2.04345584, 2.04394841, 2.04449677, 2.04498100,
     &   2.04555130, 2.04607344, 2.04664254, 2.04707193, 2.04773879,
     &   2.04831982, 2.04891729, 2.04947448, 2.05006957, 2.05066252,
     &   2.05136728, 2.05188251, 2.05246615, 2.05314517, 2.05378556,
     &   2.05458999, 2.05503869, 2.05568361, 2.05637288, 2.05705523,
     &   2.05771661, 2.05844760, 2.05907869, 2.05981398, 2.06053400,
     &   2.06122065, 2.06193972, 2.06268358, 2.06346512, 2.06412601,
     &   2.06501889, 2.06571960, 2.06660628, 2.06734228, 2.06816840,
     &   2.06910706, 2.06983709, 2.07069325, 2.07150078, 2.07231116,
     &   2.07326126, 2.07406402, 2.07494712, 2.07588601, 2.07678294,
     &   2.07762551, 2.07860970, 2.07953358, 2.08056903, 2.08146000,
     &   2.08259320, 2.08351636, 2.08446288, 2.08550620, 2.08639598,
     &   2.08754086, 2.08863521, 2.08964777, 2.09077358, 2.09197688,
     &   2.09304690, 2.09418678, 2.09521651, 2.09642196, 2.09757328,
     &   2.09883595, 2.09991646, 2.10112095, 2.10241914, 2.10365915,
     &   2.10490036, 2.10601854, 2.10737395, 2.10869360, 2.11008978,
     &   2.11148000, 2.11276555, 2.11419702, 2.11561060, 2.11698246,
     &   2.11843348, 2.11982489, 2.12152028, 2.12280083, 2.12435079,
     &   2.12588573, 2.12740803, 2.12898946, 2.13058972, 2.13220334,
     &   2.13388658, 2.13556790, 2.13719106, 2.13896751, 2.14072323,
     &   2.14240813, 2.14421701, 2.14604497, 2.14788318, 2.14982080,
     &   2.15166569, 2.15351629, 2.15531826, 2.15728188, 2.15936255,
     &   2.16149378, 2.16363215, 2.16580367, 2.16800785, 2.17025161,
     &   2.17254281, 2.17487764, 2.17727113, 2.17972875, 2.18224859,
     &   2.18484569, 2.18752074, 2.19031525, 2.19340110, 2.19677806,
     &   2.20011997, 2.20142627, 2.19591856, 2.14499521, 2.11690831,
     &   2.13536215, 2.15609050, 2.17118526, 2.18218684, 2.19137931,
     &   2.19948387, 2.20612431, 2.21224546, 2.21794057, 2.22267699,
     &   2.22710562, 2.23136377, 2.23539996, 2.23923302, 2.24294806,
     &   2.24654555, 2.25006080, 2.25352168, 2.25693989, 2.26028681,
     &   2.26362133, 2.26688528, 2.27009749, 2.27323484, 2.27631211,
     &   2.27934670, 2.28233504, 2.28537798, 2.28885293, 2.29267859,
     &   2.29670906, 2.30083036, 2.30511904, 2.30955291, 2.31425476,
     &   2.31943583, 2.32553458, 2.33349633, 2.34518576, 2.36308503,
     &   2.37687349, 2.33985448, 2.30043578, 2.31498837, 2.33906937,
     &   2.35306096, 2.36411214, 2.37522674, 2.38690662, 2.39918685,
     &   2.41229367, 2.42645383, 2.44143510, 2.45760489, 2.47487473,
     &   2.49348688, 2.51311493, 2.53345180, 2.55460024, 2.57626629,
     &   2.59797573, 2.61938453, 2.63957763, 2.65768433, 2.67237854,
     &   2.68265247, 2.68690038, 2.68316579, 2.66907382, 2.64075994,
     &   2.59540057, 2.52163148, 2.42910123, 2.29360533, 2.15646219,
     &   2.03412366, 1.94308162, 1.87650323, 1.84765303, 1.84900343,
     &   1.86534834, 1.90745282, 1.95132077, 1.99729753, 2.03795505,
     &   2.07271862, 2.10661554, 2.13756537, 2.16495347, 2.19017744,
     &   2.21379566, 2.23390079, 2.25330520, 2.27145505, 2.28881383,
     &   2.30472827, 2.32141757, 2.33779430, 2.35337687, 2.36817288,
     &   2.38364816, 2.39816523, 2.41281939, 2.42630529, 2.43934369,
     &   2.45345116, 2.46764350, 2.48141193, 2.49375653, 2.50543261,
     &   2.51599813, 2.52488136, 2.53391647, 2.54307127, 2.55232930,
     &   2.56169391, 2.57165980, 2.58175254, 2.59211874, 2.60297918,
     &   2.61400032, 2.62250805, 2.63096619, 2.63888264, 2.64639163,
     &   2.65289927, 2.65800500, 2.66176081, 2.66321921, 2.66313028,
     &   2.65991879, 2.65403247, 2.64343381, 2.62985611, 2.61218119,
     &   2.59224629, 2.57032895, 2.54737806, 2.52352238, 2.49891210,
     &   2.47318935, 2.44611764, 2.41896105, 2.39189434, 2.36444306,
     &   2.33778596, 2.31246471, 2.29227257, 2.27408814, 2.25759649,
     &   2.24368310, 2.23374009, 2.22726917, 2.22305179, 2.22085643,
     &   2.22100616, 2.22294450, 2.22649193, 2.23376918, 2.24171162,
     &   2.24785662, 2.25278163, 2.25879288, 2.26679277, 2.27401400,
     &   2.27952790, 2.28417325, 2.28622842, 2.28766608, 2.28906918,
     &   2.28970218, 2.28828931, 2.28575087, 2.27923036, 2.27171707,
     &   2.26284385, 2.25299072, 2.24209833, 2.22974467, 2.21617842,
     &   2.20008945, 2.18258739, 2.16349220, 2.14295197, 2.12182307,
     &   2.10113215, 2.07785749, 2.05238986, 2.02459455, 1.99560940,
     &   1.96481466, 1.93160546, 1.89582396, 1.85961556, 1.82205987,
     &   1.78305328, 1.74372792, 1.70350206, 1.66140103, 1.61832249,
     &   1.57487011, 1.53076375, 1.48544359, 1.44022238, 1.39517975,
     &   1.35039401, 1.30594182, 1.26559842, 1.22653472, 1.18808663,
     &   1.15026963, 1.11312151, 1.07805526, 1.04384744, 1.01031101,
     &   0.97747111, 0.94536084, 0.91401523, 0.88342518, 0.85360384,
     &   0.82474309, 0.79783303, 0.77168643, 0.74630278, 0.72180635,
     &   0.69849086, 0.67589438, 0.65400976, 0.63292384, 0.61260086,
     &   0.59295690, 0.57419252, 0.55700046, 0.54040283, 0.52438498,
     &   0.50907892, 0.49430847, 0.48004684, 0.46595258, 0.45226145,
     &   0.43904594, 0.42641178, 0.41423175, 0.40250522, 0.39128989,
     &   0.38047075, 0.37003249, 0.35996124, 0.35024542, 0.34111440,
     &   0.33230707, 0.32356760, 0.31498984, 0.30665052, 0.29845878,
     &   0.29055834, 0.28306866, 0.27583531, 0.26885310, 0.26210791,
     &   0.25570986, 0.24954014, 0.24357499, 0.23780654, 0.23234518,
     &   0.22706228, 0.22194828, 0.21698979, 0.21217412, 0.20750852/
      data (qq(i,8),i=1,mxwv)/
     &   2.03409195, 2.03448105, 2.03489804, 2.03530455, 2.03574800,
     &   2.03614974, 2.03657389, 2.03700686, 2.03747296, 2.03791094,
     &   2.03835917, 2.03877401, 2.03929281, 2.03973341, 2.04024744,
     &   2.04067302, 2.04115009, 2.04165006, 2.04214668, 2.04264116,
     &   2.04315805, 2.04366279, 2.04417443, 2.04472423, 2.04525638,
     &   2.04574156, 2.04634523, 2.04685712, 2.04750466, 2.04800057,
     &   2.04852676, 2.04916525, 2.04978752, 2.05033755, 2.05093026,
     &   2.05153990, 2.05212212, 2.05279970, 2.05340719, 2.05407929,
     &   2.05466604, 2.05531549, 2.05597448, 2.05659604, 2.05730271,
     &   2.05805349, 2.05871820, 2.05937719, 2.06012940, 2.06085539,
     &   2.06144619, 2.06232929, 2.06298208, 2.06386471, 2.06454563,
     &   2.06533241, 2.06608367, 2.06694388, 2.06778264, 2.06849313,
     &   2.06926894, 2.07017803, 2.07093024, 2.07186890, 2.07273865,
     &   2.07356000, 2.07443881, 2.07538271, 2.07627702, 2.07712650,
     &   2.07808161, 2.07902193, 2.07999992, 2.08093429, 2.08202982,
     &   2.08285427, 2.08392286, 2.08491611, 2.08603597, 2.08694625,
     &   2.08802629, 2.08906388, 2.09014988, 2.09125614, 2.09235644,
     &   2.09346247, 2.09453702, 2.09575915, 2.09693122, 2.09810066,
     &   2.09927440, 2.10039735, 2.10170078, 2.10293865, 2.10419941,
     &   2.10545444, 2.10681272, 2.10808325, 2.10936117, 2.11069441,
     &   2.11210871, 2.11342502, 2.11493516, 2.11630774, 2.11775494,
     &   2.11921978, 2.12064910, 2.12204456, 2.12355256, 2.12514472,
     &   2.12676716, 2.12839437, 2.13004255, 2.13171005, 2.13341022,
     &   2.13513088, 2.13689613, 2.13868451, 2.14052343, 2.14239740,
     &   2.14431787, 2.14628696, 2.14833307, 2.15057421, 2.15302491,
     &   2.15553546, 2.15670848, 2.14728212, 2.11633348, 2.10072613,
     &   2.11380601, 2.12835598, 2.13896060, 2.14674926, 2.15321541,
     &   2.15885830, 2.16349316, 2.16778779, 2.17179608, 2.17516732,
     &   2.17834997, 2.18142986, 2.18436670, 2.18718195, 2.18993378,
     &   2.19262481, 2.19528127, 2.19792366, 2.20056152, 2.20317674,
     &   2.20581079, 2.20842195, 2.21102190, 2.21358585, 2.21611333,
     &   2.21860814, 2.22104883, 2.22349954, 2.22630215, 2.22941303,
     &   2.23271680, 2.23610020, 2.23961782, 2.24318075, 2.24674726,
     &   2.25020742, 2.25344920, 2.25650620, 2.25912738, 2.26108980,
     &   2.26009393, 2.24443913, 2.23007250, 2.24403191, 2.26100516,
     &   2.26971340, 2.27555680, 2.28067517, 2.28533864, 2.28971052,
     &   2.29410148, 2.29862785, 2.30333567, 2.30838656, 2.31382656,
     &   2.31980586, 2.32638693, 2.33366442, 2.34181142, 2.35098839,
     &   2.36131430, 2.37298870, 2.38607669, 2.40064406, 2.41649032,
     &   2.43361378, 2.45158386, 2.46953058, 2.48674846, 2.50050831,
     &   2.50798917, 2.50147963, 2.47489643, 2.40910244, 2.31824183,
     &   2.21594810, 2.12855935, 2.05404663, 2.01064205, 1.99500227,
     &   1.99631953, 2.01945996, 2.04691410, 2.07735825, 2.10497737,
     &   2.12864375, 2.15200806, 2.17347598, 2.19270921, 2.21045041,
     &   2.22718883, 2.24163556, 2.25555110, 2.26870418, 2.28133035,
     &   2.29297829, 2.30512023, 2.31704760, 2.32850504, 2.33943748,
     &   2.35078597, 2.36159492, 2.37243485, 2.38264132, 2.39253688,
     &   2.40305591, 2.41374850, 2.42421460, 2.43393707, 2.44344401,
     &   2.45263267, 2.46076131, 2.46909285, 2.47758865, 2.48624396,
     &   2.49508691, 2.50435567, 2.51382422, 2.52374125, 2.53442597,
     &   2.54542685, 2.55545688, 2.56575108, 2.57624912, 2.58698964,
     &   2.59774518, 2.60840964, 2.61889648, 2.62877846, 2.63843656,
     &   2.64721990, 2.65498948, 2.66042519, 2.66411018, 2.66484261,
     &   2.66354728, 2.66024709, 2.65514851, 2.64868021, 2.64100027,
     &   2.63177776, 2.62049747, 2.60764456, 2.59362006, 2.57862043,
     &   2.56234121, 2.54514503, 2.53061891, 2.51734805, 2.50519609,
     &   2.49395919, 2.48521304, 2.47918320, 2.47560477, 2.47366953,
     &   2.47370481, 2.47547722, 2.47892213, 2.48535609, 2.49397779,
     &   2.50208426, 2.50949192, 2.51780367, 2.52775455, 2.53762627,
     &   2.54822516, 2.55829859, 2.56688476, 2.57511592, 2.58370566,
     &   2.59191751, 2.59997296, 2.60741901, 2.61268330, 2.61721063,
     &   2.62046194, 2.62328291, 2.62565899, 2.62730145, 2.62828660,
     &   2.62778473, 2.62622952, 2.62337136, 2.61930823, 2.61469460,
     &   2.61022210, 2.60486174, 2.59815574, 2.58913279, 2.57890749,
     &   2.56721759, 2.55360961, 2.53693962, 2.51882911, 2.49904037,
     &   2.47744846, 2.45416451, 2.42966914, 2.40293121, 2.37384462,
     &   2.34273624, 2.30956244, 2.27293658, 2.23511052, 2.19612813,
     &   2.15603805, 2.11489344, 2.07412601, 2.03304458, 1.99144244,
     &   1.94937754, 1.90690947, 1.86604476, 1.82517612, 1.78406262,
     &   1.74275875, 1.70131993, 1.65980470, 1.61827314, 1.57678246,
     &   1.53555870, 1.49562418, 1.45597863, 1.41658139, 1.37776375,
     &   1.34002185, 1.30269921, 1.26583481, 1.22955716, 1.19388533,
     &   1.15878451, 1.12458396, 1.09232998, 1.06071031, 1.02974343,
     &   1.00002718, 0.97098076, 0.94256669, 0.91443670, 0.88685554,
     &   0.85992628, 0.83381891, 0.80839092, 0.78367817, 0.75982207,
     &   0.73662692, 0.71411031, 0.69225967, 0.67105651, 0.65096444,
     &   0.63149410, 0.61231315, 0.59354216, 0.57526028, 0.55731559,
     &   0.53997177, 0.52339602, 0.50738311, 0.49191684, 0.47698897,
     &   0.46277267, 0.44908273, 0.43587732, 0.42314628, 0.41108274,
     &   0.39945793, 0.38825315, 0.37744167, 0.36699799, 0.35693419/
      data (qq(i,9),i=1,mxwv)/
     &   2.02706456, 2.02738762, 2.02769160, 2.02802467, 2.02832866,
     &   2.02870750, 2.02903938, 2.02938008, 2.02970338, 2.03006792,
     &   2.03045630, 2.03080034, 2.03115582, 2.03152800, 2.03191280,
     &   2.03229547, 2.03267145, 2.03305078, 2.03344083, 2.03384042,
     &   2.03423452, 2.03465509, 2.03505230, 2.03550577, 2.03590846,
     &   2.03633690, 2.03676200, 2.03722191, 2.03764534, 2.03809428,
     &   2.03854060, 2.03899193, 2.03942204, 2.03994226, 2.04045773,
     &   2.04086900, 2.04133153, 2.04185772, 2.04237986, 2.04288411,
     &   2.04338384, 2.04391384, 2.04443431, 2.04493284, 2.04544902,
     &   2.04600072, 2.04654241, 2.04708195, 2.04768729, 2.04823685,
     &   2.04877973, 2.04937148, 2.04998255, 2.05057120, 2.05120730,
     &   2.05176497, 2.05239773, 2.05307794, 2.05365062, 2.05433059,
     &   2.05495453, 2.05556202, 2.05627084, 2.05688620, 2.05757189,
     &   2.05828929, 2.05907989, 2.05966377, 2.06039596, 2.06110954,
     &   2.06183386, 2.06257701, 2.06345081, 2.06412053, 2.06484747,
     &   2.06566453, 2.06648183, 2.06724524, 2.06810451, 2.06882524,
     &   2.06966782, 2.07048106, 2.07134295, 2.07217717, 2.07309365,
     &   2.07394576, 2.07484651, 2.07575560, 2.07664967, 2.07758784,
     &   2.07848716, 2.07943296, 2.08039021, 2.08139348, 2.08239269,
     &   2.08334589, 2.08437324, 2.08535028, 2.08639693, 2.08745694,
     &   2.08852720, 2.08957958, 2.09065318, 2.09178782, 2.09291649,
     &   2.09405613, 2.09515667, 2.09624934, 2.09742165, 2.09865117,
     &   2.09990144, 2.10115409, 2.10242605, 2.10370731, 2.10500646,
     &   2.10634518, 2.10767531, 2.10903764, 2.11040854, 2.11184978,
     &   2.11329412, 2.11476588, 2.11627960, 2.11791515, 2.11966133,
     &   2.12150049, 2.12205315, 2.11473298, 2.09441423, 2.08528495,
     &   2.09466076, 2.10497022, 2.11249757, 2.11806941, 2.12267661,
     &   2.12666559, 2.12995481, 2.13301921, 2.13589001, 2.13833547,
     &   2.14066386, 2.14292574, 2.14509630, 2.14718580, 2.14923573,
     &   2.15124726, 2.15323853, 2.15522313, 2.15720749, 2.15918207,
     &   2.16116810, 2.16314435, 2.16511703, 2.16707134, 2.16900754,
     &   2.17092729, 2.17282081, 2.17473054, 2.17688966, 2.17926335,
     &   2.18177390, 2.18434930, 2.18703222, 2.18976951, 2.19254255,
     &   2.19528675, 2.19792080, 2.20045137, 2.20256662, 2.20351243,
     &   2.20016313, 2.18825698, 2.18170929, 2.19289041, 2.20491266,
     &   2.21122646, 2.21558881, 2.21945190, 2.22294664, 2.22613430,
     &   2.22922921, 2.23227262, 2.23527002, 2.23830414, 2.24134707,
     &   2.24443650, 2.24754119, 2.25066876, 2.25391650, 2.25727272,
     &   2.26078510, 2.26452971, 2.26857209, 2.27302933, 2.27800965,
     &   2.28373218, 2.29038572, 2.29813337, 2.30733371, 2.31782746,
     &   2.32934260, 2.34023809, 2.34478474, 2.33277202, 2.29834437,
     &   2.24271178, 2.18508863, 2.12755895, 2.08808708, 2.06858230,
     &   2.06360292, 2.07496762, 2.09127879, 2.11062574, 2.12875581,
     &   2.14450455, 2.16026139, 2.17488813, 2.18815136, 2.20046353,
     &   2.21215296, 2.22237468, 2.23225284, 2.24164319, 2.25069094,
     &   2.25911760, 2.26784921, 2.27643704, 2.28470492, 2.29263973,
     &   2.30081081, 2.30859947, 2.31640410, 2.32376552, 2.33095145,
     &   2.33849239, 2.34604430, 2.35344172, 2.36038613, 2.36716580,
     &   2.37371111, 2.37976909, 2.38596058, 2.39226937, 2.39869213,
     &   2.40524054, 2.41204119, 2.41897011, 2.42614841, 2.43373775,
     &   2.44152617, 2.44890332, 2.45648527, 2.46426678, 2.47228074,
     &   2.48045182, 2.48877716, 2.49723387, 2.50568938, 2.51434588,
     &   2.52303314, 2.53166080, 2.53969908, 2.54731417, 2.55385303,
     &   2.55962181, 2.56454444, 2.56850100, 2.57175994, 2.57437587,
     &   2.57612371, 2.57655454, 2.57562017, 2.57356119, 2.57069206,
     &   2.56618571, 2.56010008, 2.55461764, 2.54953837, 2.54486489,
     &   2.53999114, 2.53607583, 2.53363347, 2.53294373, 2.53321624,
     &   2.53469968, 2.53739071, 2.54130602, 2.54703045, 2.55498147,
     &   2.56313658, 2.57103634, 2.57950401, 2.58897758, 2.59869242,
     &   2.60991764, 2.62091136, 2.63133740, 2.64163065, 2.65232992,
     &   2.66293454, 2.67423081, 2.68534708, 2.69581771, 2.70595789,
     &   2.71534753, 2.72475457, 2.73420143, 2.74357867, 2.75287127,
     &   2.76181054, 2.77036881, 2.77837110, 2.78587103, 2.79317403,
     &   2.80051494, 2.80838895, 2.81610394, 2.82264066, 2.82867074,
     &   2.83428907, 2.83938265, 2.84270954, 2.84484768, 2.84613419,
     &   2.84651518, 2.84530306, 2.84355998, 2.84073520, 2.83598948,
     &   2.82923722, 2.82062960, 2.80869079, 2.79536510, 2.78061485,
     &   2.76440644, 2.74670935, 2.72556996, 2.70271802, 2.67859459,
     &   2.65320754, 2.62656832, 2.60023999, 2.57302022, 2.54469728,
     &   2.51528883, 2.48479724, 2.45322466, 2.42064023, 2.38707280,
     &   2.35259223, 2.31751513, 2.28167486, 2.24510956, 2.20804667,
     &   2.17101455, 2.13340902, 2.09527230, 2.05666113, 2.01763201,
     &   1.97822452, 1.93874037, 1.90014684, 1.86139333, 1.82252336,
     &   1.78476572, 1.74702537, 1.70924222, 1.67133427, 1.63338828,
     &   1.59547508, 1.55777061, 1.52020788, 1.48288810, 1.44607472,
     &   1.40951252, 1.37331140, 1.33748305, 1.30201316, 1.26772225,
     &   1.23385918, 1.20018911, 1.16681325, 1.13374329, 1.10077119,
     &   1.06835234, 1.03668070, 1.00561583, 0.97514355, 0.94532347,
     &   0.91643113, 0.88825256, 0.86075085, 0.83395207, 0.80823338,
     &   0.78322130, 0.75891024, 0.73527455, 0.71228892, 0.69000363/
      data (qq(i,10),i=1,mxwv)/
     &   2.02147675, 2.02172947, 2.02200747, 2.02225709, 2.02272439,
     &   2.02278566, 2.02306294, 2.02334714, 2.02374697, 2.02388453,
     &   2.02417088, 2.02445960, 2.02470636, 2.02502751, 2.02532864,
     &   2.02562976, 2.02592611, 2.02623630, 2.02637529, 2.02685618,
     &   2.02720785, 2.02751517, 2.02783895, 2.02816486, 2.02848506,
     &   2.02883649, 2.02917671, 2.02951646, 2.02990079, 2.03023982,
     &   2.03061032, 2.03095460, 2.03132558, 2.03167629, 2.03206921,
     &   2.03244925, 2.03284502, 2.03321195, 2.03361368, 2.03397870,
     &   2.03441906, 2.03482938, 2.03522515, 2.03564739, 2.03609538,
     &   2.03650141, 2.03692579, 2.03738785, 2.03781891, 2.03828096,
     &   2.03873539, 2.03917623, 2.03963327, 2.04010248, 2.04059672,
     &   2.04103661, 2.04158521, 2.04216194, 2.04254341, 2.04300737,
     &   2.04357433, 2.04411316, 2.04460979, 2.04514980, 2.04569697,
     &   2.04623652, 2.04676580, 2.04735374, 2.04789972, 2.04850340,
     &   2.04903936, 2.04963541, 2.05018258, 2.05078959, 2.05140066,
     &   2.05199862, 2.05261564, 2.05328894, 2.05387568, 2.05455470,
     &   2.05519652, 2.05587029, 2.05656505, 2.05728483, 2.05786777,
     &   2.05857587, 2.05926442, 2.05999112, 2.06070018, 2.06142879,
     &   2.06215525, 2.06289697, 2.06365967, 2.06441450, 2.06519032,
     &   2.06597161, 2.06676960, 2.06757021, 2.06805110, 2.06920528,
     &   2.07003784, 2.07087803, 2.07172990, 2.07255316, 2.07346630,
     &   2.07436180, 2.07523084, 2.07613683, 2.07700896, 2.07796621,
     &   2.07893777, 2.07991219, 2.08089685, 2.08189154, 2.08290386,
     &   2.08392835, 2.08496237, 2.08601975, 2.08709145, 2.08817816,
     &   2.08928275, 2.09040236, 2.09154844, 2.09276652, 2.09404469,
     &   2.09526467, 2.09566092, 2.09030700, 2.07687855, 2.07135296,
     &   2.07810140, 2.08545446, 2.09084129, 2.09486294, 2.09818316,
     &   2.10104322, 2.10341430, 2.10563540, 2.10772562, 2.10953021,
     &   2.11126280, 2.11295271, 2.11458302, 2.11616135, 2.11771369,
     &   2.11924291, 2.12076020, 2.12227511, 2.12375855, 2.12530088,
     &   2.12682295, 2.12833786, 2.12985158, 2.13135386, 2.13284683,
     &   2.13433123, 2.13579583, 2.13728881, 2.13895106, 2.14075899,
     &   2.14265704, 2.14459348, 2.14660358, 2.14864445, 2.15070367,
     &   2.15273690, 2.15469122, 2.15656996, 2.15815496, 2.15891409,
     &   2.15654969, 2.14837670, 2.14497161, 2.15331197, 2.16165018,
     &   2.16607857, 2.16922259, 2.17204762, 2.17463875, 2.17703485,
     &   2.17937994, 2.18169975, 2.18399501, 2.18632078, 2.18865323,
     &   2.19101191, 2.19336748, 2.19570541, 2.19808841, 2.20047307,
     &   2.20286083, 2.20525622, 2.20764041, 2.21001387, 2.21236968,
     &   2.21473122, 2.21712136, 2.21956944, 2.22216797, 2.22494698,
     &   2.22808456, 2.23142910, 2.23398280, 2.23227978, 2.22178268,
     &   2.19836807, 2.16905904, 2.13491702, 2.10834002, 2.09336996,
     &   2.08833480, 2.09429550, 2.10436440, 2.11691260, 2.12900496,
     &   2.13970351, 2.15051818, 2.16066551, 2.16996241, 2.17866540,
     &   2.18697286, 2.19432569, 2.20147085, 2.20828986, 2.21488857,
     &   2.22109628, 2.22750258, 2.23381543, 2.23990011, 2.24577570,
     &   2.25179362, 2.25752401, 2.26327920, 2.26870084, 2.27403426,
     &   2.27958965, 2.28507376, 2.29045272, 2.29553080, 2.30046177,
     &   2.30516911, 2.30967855, 2.31427073, 2.31894326, 2.32369304,
     &   2.32851934, 2.33350444, 2.33857179, 2.34374261, 2.34908938,
     &   2.35454559, 2.35974050, 2.36504745, 2.37043333, 2.37592864,
     &   2.38148284, 2.38708806, 2.39275408, 2.39840865, 2.40417171,
     &   2.40995002, 2.41574812, 2.42133713, 2.42684722, 2.43201113,
     &   2.43699145, 2.44175911, 2.44625258, 2.45058703, 2.45478177,
     &   2.45872808, 2.46219563, 2.46508646, 2.46748281, 2.46954989,
     &   2.47072363, 2.47093153, 2.47116423, 2.47148681, 2.47192168,
     &   2.47201514, 2.47238588, 2.47341490, 2.47540450, 2.47787404,
     &   2.48097515, 2.48476171, 2.48926878, 2.49473619, 2.50166845,
     &   2.50889707, 2.51607442, 2.52357793, 2.53162980, 2.53991556,
     &   2.54929423, 2.55859137, 2.56771016, 2.57680202, 2.58616376,
     &   2.59553075, 2.60547280, 2.61539578, 2.62523365, 2.63498187,
     &   2.64442420, 2.65400290, 2.66373968, 2.67362142, 2.68362069,
     &   2.69372225, 2.70382214, 2.71385217, 2.72384548, 2.73389602,
     &   2.74401331, 2.75489259, 2.76610613, 2.77710152, 2.78814507,
     &   2.79942012, 2.81100869, 2.82224822, 2.83312082, 2.84403372,
     &   2.85501623, 2.86541033, 2.87603235, 2.88683295, 2.89709496,
     &   2.90660906, 2.91556454, 2.92318439, 2.93033242, 2.93695211,
     &   2.94298434, 2.94836855, 2.95001125, 2.95014143, 2.94935536,
     &   2.94760919, 2.94485998, 2.94154096, 2.93731356, 2.93207979,
     &   2.92580795, 2.91844463, 2.90993023, 2.90029001, 2.88949943,
     &   2.87743449, 2.86355138, 2.84852409, 2.83234382, 2.81507993,
     &   2.79694867, 2.77770758, 2.75735736, 2.73581886, 2.71312761,
     &   2.68935728, 2.66456842, 2.63901377, 2.61253667, 2.58515596,
     &   2.55817580, 2.53045917, 2.50191736, 2.47287703, 2.44308758,
     &   2.41248536, 2.38108945, 2.34894276, 2.31613445, 2.28290224,
     &   2.24901462, 2.21461558, 2.17970276, 2.14422512, 2.10903907,
     &   2.07339764, 2.03740549, 2.00100780, 1.96401107, 1.92618895,
     &   1.88799989, 1.84947896, 1.81071222, 1.77165866, 1.73245895,
     &   1.69337583, 1.65428913, 1.61518610, 1.57615292, 1.53774071,
     &   1.49950778, 1.46149075, 1.42369270, 1.38611686, 1.34889627/
      data (qq(i,11),i=1,mxwv)/
     &   2.01705885, 2.01725364, 2.01745462, 2.01768255, 2.01787853,
     &   2.01808667, 2.01830935, 2.01851845, 2.01874614, 2.01896572,
     &   2.01919580, 2.01940870, 2.01964378, 2.01986694, 2.02000594,
     &   2.02033734, 2.02058840, 2.02082372, 2.02108407, 2.02132821,
     &   2.02157927, 2.02183557, 2.02211046, 2.02240729, 2.02262783,
     &   2.02290177, 2.02315354, 2.02345300, 2.02372336, 2.02400041,
     &   2.02428150, 2.02456665, 2.02484894, 2.02514815, 2.02541971,
     &   2.02577281, 2.02604866, 2.02636933, 2.02667022, 2.02699304,
     &   2.02731180, 2.02764153, 2.02795672, 2.02831459, 2.02862978,
     &   2.02898359, 2.02930617, 2.02966785, 2.03001499, 2.03036737,
     &   2.03072119, 2.03110361, 2.03145957, 2.03184032, 2.03218198,
     &   2.03258729, 2.03298640, 2.03337836, 2.03378534, 2.03416300,
     &   2.03457975, 2.03505421, 2.03540301, 2.03583026, 2.03625107,
     &   2.03667498, 2.03708100, 2.03755593, 2.03800225, 2.03844643,
     &   2.03888988, 2.03937101, 2.03984165, 2.04030776, 2.04077005,
     &   2.04127884, 2.04175138, 2.04224873, 2.04326868, 2.04326773,
     &   2.04378319, 2.04430485, 2.04483294, 2.04535961, 2.04589963,
     &   2.04642892, 2.04700041, 2.04755235, 2.04815197, 2.04868436,
     &   2.04926920, 2.04985523, 2.05044985, 2.05104470, 2.05166435,
     &   2.05228090, 2.05290651, 2.05305171, 2.05418038, 2.05482459,
     &   2.05548382, 2.05614471, 2.05681682, 2.05749893, 2.05818629,
     &   2.05889821, 2.05957127, 2.06025076, 2.06097293, 2.06172323,
     &   2.06248307, 2.06325054, 2.06401348, 2.06479549, 2.06558466,
     &   2.06638288, 2.06718707, 2.06800556, 2.06884456, 2.06968784,
     &   2.07054210, 2.07140565, 2.07228446, 2.07321072, 2.07416940,
     &   2.07505846, 2.07533360, 2.07151031, 2.06256914, 2.05916095,
     &   2.06403255, 2.06930351, 2.07318449, 2.07610846, 2.07852530,
     &   2.08060145, 2.08233595, 2.08396721, 2.08551097, 2.08686185,
     &   2.08816886, 2.08944821, 2.09068918, 2.09189606, 2.09308791,
     &   2.09426498, 2.09543610, 2.09660673, 2.09778023, 2.09895015,
     &   2.10013032, 2.10130668, 2.10248327, 2.10365391, 2.10481906,
     &   2.10598135, 2.10713482, 2.10830593, 2.10960007, 2.11099339,
     &   2.11244678, 2.11392617, 2.11545205, 2.11699677, 2.11854935,
     &   2.12008023, 2.12154722, 2.12295270, 2.12413621, 2.12472034,
     &   2.12314391, 2.11776686, 2.11604309, 2.12218595, 2.12801600,
     &   2.13115859, 2.13344598, 2.13552523, 2.13744807, 2.13924336,
     &   2.14100862, 2.14275813, 2.14449525, 2.14625454, 2.14802027,
     &   2.14980531, 2.15158749, 2.15336061, 2.15516567, 2.15697217,
     &   2.15878057, 2.16059065, 2.16238642, 2.16415977, 2.16589522,
     &   2.16758633, 2.16921687, 2.17075753, 2.17218137, 2.17338824,
     &   2.17434716, 2.17466402, 2.17420268, 2.17114353, 2.16479421,
     &   2.15297270, 2.13799548, 2.11934972, 2.10381079, 2.09455538,
     &   2.09134841, 2.09515166, 2.10185766, 2.11036015, 2.11869526,
     &   2.12618995, 2.13381004, 2.14102221, 2.14768505, 2.15397120,
     &   2.15999866, 2.16539121, 2.17065954, 2.17570567, 2.18060827,
     &   2.18526149, 2.19004965, 2.19477868, 2.19934440, 2.20377803,
     &   2.20830345, 2.21261501, 2.21695566, 2.22104788, 2.22509933,
     &   2.22929955, 2.23341012, 2.23745131, 2.24128985, 2.24500942,
     &   2.24854350, 2.25201011, 2.25553417, 2.25911832, 2.26275969,
     &   2.26645422, 2.27025962, 2.27411914, 2.27803493, 2.28202653,
     &   2.28609037, 2.28997755, 2.29393578, 2.29792714, 2.30197620,
     &   2.30604458, 2.31012011, 2.31421614, 2.31827736, 2.32238030,
     &   2.32644296, 2.33075094, 2.33436775, 2.33818626, 2.34180117,
     &   2.34533000, 2.34876776, 2.35211039, 2.35541320, 2.35869050,
     &   2.36188960, 2.36490631, 2.36770964, 2.37034869, 2.37288904,
     &   2.37506819, 2.37684941, 2.37873340, 2.38070869, 2.38277984,
     &   2.38472009, 2.38684535, 2.38934875, 2.39238524, 2.39571071,
     &   2.39941144, 2.40351701, 2.40804791, 2.41315484, 2.41907406,
     &   2.42516589, 2.43125391, 2.43755817, 2.44420242, 2.45115590,
     &   2.45832825, 2.46563673, 2.47284532, 2.48005819, 2.48742580,
     &   2.49481058, 2.50246072, 2.51010942, 2.51769543, 2.52526116,
     &   2.53270960, 2.54024839, 2.54788971, 2.55562377, 2.56344628,
     &   2.57135105, 2.57930446, 2.58728361, 2.59531140, 2.60342836,
     &   2.61162901, 2.62023425, 2.62907195, 2.63793468, 2.64692616,
     &   2.65614700, 2.66566586, 2.67525935, 2.68487096, 2.69471288,
     &   2.70482683, 2.71492600, 2.72536993, 2.73623919, 2.74724770,
     &   2.75830412, 2.76953673, 2.78077245, 2.79221869, 2.80386710,
     &   2.81570601, 2.82772279, 2.83812761, 2.84811020, 2.85800552,
     &   2.86778307, 2.87740993, 2.88667798, 2.89571810, 2.90452456,
     &   2.91306305, 2.92128515, 2.92913294, 2.93659782, 2.94363856,
     &   2.95009565, 2.95525002, 2.95982337, 2.96377683, 2.96704388,
     &   2.96953249, 2.97130919, 2.97233939, 2.97248626, 2.97173834,
     &   2.97013807, 2.96753669, 2.96352601, 2.95861840, 2.95279360,
     &   2.94672322, 2.93982029, 2.93201256, 2.92378259, 2.91474605,
     &   2.90475702, 2.89367962, 2.88161755, 2.86859179, 2.85467625,
     &   2.83977580, 2.82397890, 2.80725908, 2.78954577, 2.77122831,
     &   2.75196815, 2.73217726, 2.71165991, 2.69008732, 2.66732550,
     &   2.64353466, 2.61848617, 2.59245682, 2.56535125, 2.53728437,
     &   2.50827956, 2.47837162, 2.44753695, 2.41583848, 2.38366485,
     &   2.35071254, 2.31700206, 2.28252101, 2.24725580, 2.21132469/
      data (qq(i,12),i=1,mxwv)/
     &   2.01354194, 2.01370192, 2.01387286, 2.01403880, 2.01418281,
     &   2.01436043, 2.01453876, 2.01471233, 2.01484990, 2.01505995,
     &   2.01522923, 2.01541471, 2.01556444, 2.01577878, 2.01596642,
     &   2.01615214, 2.01634836, 2.01653957, 2.01673484, 2.01694298,
     &   2.01714182, 2.01734066, 2.01754427, 2.01774359, 2.01795840,
     &   2.01817441, 2.01839352, 2.01859856, 2.01882958, 2.01904893,
     &   2.01927423, 2.01949430, 2.01974797, 2.01996326, 2.02019763,
     &   2.02045727, 2.02066445, 2.02092195, 2.02117181, 2.02143431,
     &   2.02168846, 2.02193904, 2.02217889, 2.02245903, 2.02273178,
     &   2.02296925, 2.02329850, 2.02360320, 2.02382398, 2.02410507,
     &   2.02439094, 2.02467680, 2.02496719, 2.02526641, 2.02556205,
     &   2.02587008, 2.02618313, 2.02647424, 2.02680731, 2.02711105,
     &   2.02743530, 2.02776122, 2.02809978, 2.02843332, 2.02876091,
     &   2.02911019, 2.02945375, 2.02979088, 2.03015113, 2.03053951,
     &   2.03086782, 2.03122282, 2.03163075, 2.03199458, 2.03236508,
     &   2.03275037, 2.03313732, 2.03351712, 2.03392220, 2.03432822,
     &   2.03473234, 2.03514481, 2.03556418, 2.03601098, 2.03573537,
     &   2.03665733, 2.03727484, 2.03772402, 2.03816867, 2.03862405,
     &   2.03908014, 2.03954363, 2.04001403, 2.04049110, 2.04097748,
     &   2.04146075, 2.04195523, 2.04245496, 2.04293537, 2.04347205,
     &   2.04398918, 2.04451394, 2.04504442, 2.04558969, 2.04612589,
     &   2.04667521, 2.04721856, 2.04775882, 2.04832792, 2.04891753,
     &   2.04930472, 2.05011559, 2.05071902, 2.05133128, 2.05195117,
     &   2.05257821, 2.05322456, 2.05385518, 2.05450654, 2.05516481,
     &   2.05583143, 2.05650377, 2.05718660, 2.05790234, 2.05862808,
     &   2.05930161, 2.05951357, 2.05682778, 2.05083275, 2.04870749,
     &   2.05223417, 2.05602813, 2.05883932, 2.06097865, 2.06275249,
     &   2.06427646, 2.06555915, 2.06677341, 2.06792712, 2.06895065,
     &   2.06994724, 2.07092595, 2.07188034, 2.07281256, 2.07373571,
     &   2.07465053, 2.07556295, 2.07647538, 2.07738900, 2.07830715,
     &   2.07922912, 2.08015084, 2.08107376, 2.08199382, 2.08291149,
     &   2.08382845, 2.08474112, 2.08566976, 2.08668423, 2.08776784,
     &   2.08888865, 2.09002566, 2.09120727, 2.09239244, 2.09358048,
     &   2.09475064, 2.09587455, 2.09694982, 2.09786034, 2.09833956,
     &   2.09732151, 2.09380412, 2.09299684, 2.09751034, 2.10163355,
     &   2.10390115, 2.10559559, 2.10715199, 2.10860491, 2.10996699,
     &   2.11132312, 2.11266422, 2.11399794, 2.11534739, 2.11670160,
     &   2.11806822, 2.11943245, 2.12078977, 2.12216830, 2.12354541,
     &   2.12492156, 2.12629676, 2.12765837, 2.12900043, 2.13031244,
     &   2.13158584, 2.13281083, 2.13396049, 2.13500476, 2.13586116,
     &   2.13647676, 2.13649225, 2.13581681, 2.13306594, 2.12832260,
     &   2.12079525, 2.11196542, 2.10119700, 2.09220481, 2.08685875,
     &   2.08516884, 2.08795071, 2.09267259, 2.09863901, 2.10454035,
     &   2.10991311, 2.11539102, 2.12061048, 2.12546515, 2.13007569,
     &   2.13451409, 2.13852310, 2.14245749, 2.14623904, 2.14992714,
     &   2.15345311, 2.15707421, 2.16065788, 2.16412592, 2.16750979,
     &   2.17095590, 2.17424536, 2.17756319, 2.18069887, 2.18381834,
     &   2.18704057, 2.19017982, 2.19327331, 2.19622922, 2.19909477,
     &   2.20181513, 2.20452785, 2.20728302, 2.21008420, 2.21292877,
     &   2.21581316, 2.21877766, 2.22178245, 2.22481823, 2.22788811,
     &   2.23101044, 2.23401546, 2.23707080, 2.24014401, 2.24325418,
     &   2.24637413, 2.24949312, 2.25262475, 2.25572896, 2.25885701,
     &   2.26194549, 2.26501775, 2.26797199, 2.27088380, 2.27366400,
     &   2.27639127, 2.27906370, 2.28168869, 2.28429365, 2.28688741,
     &   2.28943634, 2.29188061, 2.29422760, 2.29649925, 2.29872537,
     &   2.30079174, 2.30268431, 2.30472851, 2.30687165, 2.30911231,
     &   2.31134677, 2.31376576, 2.31646466, 2.31950784, 2.32276392,
     &   2.32628655, 2.33008361, 2.33416390, 2.33864856, 2.34359741,
     &   2.34862804, 2.35366130, 2.35885739, 2.36429834, 2.36982322,
     &   2.37560272, 2.38137245, 2.38705826, 2.39275789, 2.39855576,
     &   2.40436959, 2.41028070, 2.41619015, 2.42200470, 2.42781734,
     &   2.43357825, 2.43938923, 2.44525504, 2.45116496, 2.45711613,
     &   2.46308756, 2.46908975, 2.47510314, 2.48114920, 2.48725939,
     &   2.49344921, 2.49979877, 2.50624394, 2.51269889, 2.51922631,
     &   2.52585864, 2.53261685, 2.53941369, 2.54625821, 2.55322480,
     &   2.56033158, 2.56748462, 2.57481694, 2.58235693, 2.59001565,
     &   2.59777403, 2.60568953, 2.61373115, 2.62197328, 2.63042545,
     &   2.63909674, 2.64799666, 2.65652609, 2.66509795, 2.67383909,
     &   2.68275094, 2.69183421, 2.70092940, 2.71015334, 2.71952677,
     &   2.72904682, 2.73870564, 2.74849200, 2.75840831, 2.76844668,
     &   2.77854156, 2.78834128, 2.79818606, 2.80806017, 2.81790709,
     &   2.82760501, 2.83726692, 2.84687138, 2.85634303, 2.86566401,
     &   2.87484646, 2.88375044, 2.89194131, 2.89987636, 2.90752697,
     &   2.91495919, 2.92209101, 2.92888808, 2.93564558, 2.94212103,
     &   2.94820404, 2.95375085, 2.95883107, 2.96341419, 2.96744323,
     &   2.97093081, 2.97388268, 2.97626638, 2.97802806, 2.97907424,
     &   2.97946358, 2.97957826, 2.97924423, 2.97822189, 2.97651219,
     &   2.97399735, 2.97038054, 2.96591520, 2.96051526, 2.95421124,
     &   2.94682193, 2.93847179, 2.92913508, 2.91881537, 2.90753913,
     &   2.89527369, 2.88200498, 2.86770368, 2.85233641, 2.83592820/
      data (qq(i,13),i=1,mxwv)/
     &   2.01075816, 2.01087856, 2.01101637, 2.01113558, 2.01126814,
     &   2.01140094, 2.01153755, 2.01167440, 2.01181149, 2.01195431,
     &   2.01210046, 2.01223683, 2.01238489, 2.01252913, 2.01267481,
     &   2.01283646, 2.01297617, 2.01313329, 2.01329041, 2.01343536,
     &   2.01360726, 2.01375556, 2.01392603, 2.01410127, 2.01426005,
     &   2.01442409, 2.01459432, 2.01479340, 2.01494479, 2.01512313,
     &   2.01530075, 2.01548648, 2.01566434, 2.01585054, 2.01605654,
     &   2.01622248, 2.01641464, 2.01661777, 2.01682472, 2.01701951,
     &   2.01725245, 2.01740909, 2.01762247, 2.01782322, 2.01803946,
     &   2.01825190, 2.01847672, 2.01869965, 2.01890802, 2.01913238,
     &   2.01936293, 2.01958942, 2.01982522, 2.02006221, 2.02029228,
     &   2.02053905, 2.02077842, 2.02102613, 2.02127361, 2.02154088,
     &   2.02178240, 2.02203512, 2.02231574, 2.02254009, 2.02283263,
     &   2.02309966, 2.02337050, 2.02365112, 2.02392817, 2.02422071,
     &   2.02450013, 2.02479649, 2.02508092, 2.02537847, 2.02568221,
     &   2.02598739, 2.02629662, 2.02660894, 2.02685738, 2.02724266,
     &   2.02756715, 2.02788925, 2.02822113, 2.02855563, 2.02889562,
     &   2.02923989, 2.02958441, 2.02993560, 2.03008676, 2.03064799,
     &   2.03100967, 2.03140903, 2.03172946, 2.03212833, 2.03251147,
     &   2.03289580, 2.03328657, 2.03368449, 2.03408241, 2.03448749,
     &   2.03489637, 2.03531194, 2.03573155, 2.03615618, 2.03657889,
     &   2.03704977, 2.03745079, 2.03788042, 2.03833055, 2.03879523,
     &   2.03926849, 2.03973937, 2.04021645, 2.04069924, 2.04118729,
     &   2.04168105, 2.04218030, 2.04268622, 2.04319835, 2.04371572,
     &   2.04423857, 2.04476595, 2.04530001, 2.04585171, 2.04641509,
     &   2.04693317, 2.04711390, 2.04524899, 2.04120779, 2.03987670,
     &   2.04243565, 2.04517651, 2.04722238, 2.04879594, 2.05010700,
     &   2.05123615, 2.05219507, 2.05310750, 2.05397844, 2.05476141,
     &   2.05552769, 2.05628252, 2.05702209, 2.05774736, 2.05847025,
     &   2.05917621, 2.05989480, 2.06061459, 2.06133437, 2.06205368,
     &   2.06277990, 2.06350613, 2.06423378, 2.06496143, 2.06568789,
     &   2.06641531, 2.06714153, 2.06787992, 2.06868005, 2.06952834,
     &   2.07040501, 2.07129335, 2.07220292, 2.07312107, 2.07404017,
     &   2.07494545, 2.07581830, 2.07665396, 2.07736993, 2.07777882,
     &   2.07714748, 2.07485414, 2.07453990, 2.07785678, 2.08080339,
     &   2.08246446, 2.08373833, 2.08492041, 2.08603382, 2.08709073,
     &   2.08813787, 2.08917785, 2.09021831, 2.09126806, 2.09231973,
     &   2.09338188, 2.09444237, 2.09549809, 2.09656811, 2.09763646,
     &   2.09870362, 2.09976792, 2.10082102, 2.10185814, 2.10287237,
     &   2.10385561, 2.10479951, 2.10568666, 2.10648942, 2.10715151,
     &   2.10763121, 2.10766459, 2.10720873, 2.10527134, 2.10201001,
     &   2.09705687, 2.09148693, 2.08488345, 2.07950282, 2.07641959,
     &   2.07563186, 2.07776237, 2.08118582, 2.08546495, 2.08971786,
     &   2.09362769, 2.09761906, 2.10144258, 2.10501957, 2.10843611,
     &   2.11173701, 2.11474466, 2.11770749, 2.12056518, 2.12336159,
     &   2.12605214, 2.12881041, 2.13154626, 2.13420081, 2.13680124,
     &   2.13944578, 2.14197707, 2.14453387, 2.14695859, 2.14937949,
     &   2.15187311, 2.15429735, 2.15669227, 2.15899301, 2.16122675,
     &   2.16335034, 2.16549182, 2.16766453, 2.16987348, 2.17211652,
     &   2.17438984, 2.17672133, 2.17908382, 2.18146420, 2.18385887,
     &   2.18629193, 2.18864775, 2.19104075, 2.19344449, 2.19587374,
     &   2.19830990, 2.20074391, 2.20318723, 2.20561171, 2.20805478,
     &   2.21046710, 2.21286941, 2.21519947, 2.21750164, 2.21972036,
     &   2.22191501, 2.22408152, 2.22622609, 2.22836637, 2.23050308,
     &   2.23261380, 2.23466420, 2.23666525, 2.23862982, 2.24056983,
     &   2.24243021, 2.24420714, 2.24612355, 2.24812412, 2.25020623,
     &   2.25232267, 2.25459886, 2.25708771, 2.25981355, 2.26270223,
     &   2.26578879, 2.26907277, 2.27255750, 2.27633905, 2.28040624,
     &   2.28451324, 2.28862596, 2.29286504, 2.29728794, 2.30176497,
     &   2.30636215, 2.31095672, 2.31548572, 2.32003284, 2.32464814,
     &   2.32927990, 2.33393955, 2.33860064, 2.34316754, 2.34774184,
     &   2.35229516, 2.35688114, 2.36150122, 2.36614299, 2.37081361,
     &   2.37547946, 2.38016772, 2.38486385, 2.38958287, 2.39435244,
     &   2.39919376, 2.40409994, 2.40904832, 2.41399813, 2.41899395,
     &   2.42404270, 2.42914653, 2.43426108, 2.43941689, 2.44463658,
     &   2.44992590, 2.45525169, 2.46066976, 2.46618342, 2.47174382,
     &   2.47736454, 2.48304725, 2.48877788, 2.49460673, 2.50053763,
     &   2.50657392, 2.51271939, 2.51875329, 2.52483582, 2.53101707,
     &   2.53730130, 2.54369259, 2.55015254, 2.55671263, 2.56338263,
     &   2.57016659, 2.57706714, 2.58408689, 2.59123278, 2.59850955,
     &   2.60590315, 2.61330748, 2.62084007, 2.62850451, 2.63628745,
     &   2.64414763, 2.65213752, 2.66025829, 2.66849399, 2.67684627,
     &   2.68532658, 2.69388723, 2.70235443, 2.71091461, 2.71956277,
     &   2.72824407, 2.73699689, 2.74581790, 2.75480890, 2.76389456,
     &   2.77304029, 2.78218746, 2.79136539, 2.80055547, 2.80970764,
     &   2.81885386, 2.82798076, 2.83707476, 2.84611869, 2.85493803,
     &   2.86366272, 2.87246227, 2.88126445, 2.88998961, 2.89868832,
     &   2.90722013, 2.91542506, 2.92341113, 2.93114090, 2.93859768,
     &   2.94561195, 2.95227122, 2.95855927, 2.96444917, 2.96978474,
     &   2.97466135, 2.97905111, 2.98292804, 2.98626351, 2.98901725/

c        ---------------- single scatter albedo ---------------

      data (ww(i,1),i=1,mxwv)/
     &   0.99999952, 0.99999958, 0.99999964, 0.99999964, 0.99999970,
     &   0.99999970, 0.99999976, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999982, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999994, 0.99999994, 0.99999994, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999994, 0.99999994, 0.99999994,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999982,
     &   0.99999976, 0.99999964, 0.99999946, 0.99999940, 0.99999940,
     &   0.99999940, 0.99999934, 0.99999923, 0.99999917, 0.99999905,
     &   0.99999875, 0.99999797, 0.99999660, 0.99999517, 0.99999487,
     &   0.99999499, 0.99999559, 0.99999613, 0.99999607, 0.99999499,
     &   0.99999338, 0.99999148, 0.99999005, 0.99998879, 0.99998742,
     &   0.99998277, 0.99996358, 0.99993312, 0.99991757, 0.99992037,
     &   0.99993020, 0.99994451, 0.99995929, 0.99997067, 0.99997354,
     &   0.99996907, 0.99995905, 0.99992585, 0.99983281, 0.99979675,
     &   0.99979216, 0.99979687, 0.99980885, 0.99982476, 0.99982220,
     &   0.99978274, 0.99970102, 0.99943924, 0.99916118, 0.99816114,
     &   0.99572027, 0.99538642, 0.99598861, 0.99725771, 0.99804133,
     &   0.99851036, 0.99880695, 0.99903840, 0.99916226, 0.99924058,
     &   0.99922901, 0.99915224, 0.99901688, 0.99877852, 0.99873787,
     &   0.99808365, 0.99248528, 0.98448712, 0.98503619, 0.98936582,
     &   0.99313819, 0.99495143, 0.99624306, 0.99692607, 0.99727446,
     &   0.99735862, 0.99709493, 0.99653637, 0.99570632, 0.99422938,
     &   0.99204558, 0.98870075, 0.98481202, 0.98276567, 0.97890717,
     &   0.95922488, 0.87456477, 0.60092169, 0.39636961, 0.35611215,
     &   0.38785917, 0.42182928, 0.45341501, 0.48493326, 0.52896410,
     &   0.59844899, 0.67909014, 0.76488978, 0.84285098, 0.88823092,
     &   0.92191178, 0.94717973, 0.96350604, 0.97250253, 0.97750479,
     &   0.97891426, 0.97817403, 0.97593778, 0.97256535, 0.96757478,
     &   0.96229076, 0.95535046, 0.94727427, 0.93761307, 0.92654824,
     &   0.91459662, 0.90150052, 0.88984787, 0.88724649, 0.89048690,
     &   0.89601666, 0.90116936, 0.90720439, 0.91192007, 0.91372395,
     &   0.91012162, 0.89826524, 0.87755698, 0.83687615, 0.75934839,
     &   0.60984266, 0.44399109, 0.40469515, 0.49654633, 0.62870169,
     &   0.68542868, 0.70718223, 0.71863103, 0.72157675, 0.71745110,
     &   0.71096838, 0.70258021, 0.69270551, 0.68257952, 0.67147034,
     &   0.65969020, 0.64646697, 0.63167769, 0.61690199, 0.60082132,
     &   0.58359200, 0.56517518, 0.54522538, 0.52371144, 0.50063592,
     &   0.47589877, 0.44957092, 0.42163384, 0.39161080, 0.35907152,
     &   0.32457307, 0.28465000, 0.24550866, 0.19957438, 0.16065010,
     &   0.12966332, 0.10869911, 0.09359177, 0.08540367, 0.08223289,
     &   0.08193011, 0.08527038, 0.08943519, 0.09422144, 0.09848119,
     &   0.10209934, 0.10566373, 0.10885277, 0.11145543, 0.11376107,
     &   0.11578444, 0.11712978, 0.11838754, 0.11935266, 0.12015285,
     &   0.12065148, 0.12131826, 0.12191113, 0.12227121, 0.12243631,
     &   0.12274905, 0.12273871, 0.12276960, 0.12238671, 0.12188148,
     &   0.12162974, 0.12124781, 0.12069120, 0.11966123, 0.11832535,
     &   0.11652559, 0.11441365, 0.11232819, 0.11027414, 0.10824618,
     &   0.10623882, 0.10437321, 0.10253117, 0.10071009, 0.09893207,
     &   0.09719741, 0.09498110, 0.09280986, 0.09058782, 0.08836666,
     &   0.08606426, 0.08364838, 0.08116627, 0.07851537, 0.07584449,
     &   0.07299417, 0.07006160, 0.06685416, 0.06360495, 0.06020356,
     &   0.05685322, 0.05357836, 0.05042693, 0.04742674, 0.04458369,
     &   0.04186176, 0.03922241, 0.03670470, 0.03432534, 0.03208752,
     &   0.02995682, 0.02793958, 0.02618165, 0.02459365, 0.02315000,
     &   0.02181572, 0.02063180, 0.01959152, 0.01867770, 0.01784808,
     &   0.01710322, 0.01643152, 0.01582579, 0.01530599, 0.01486420,
     &   0.01443098, 0.01399682, 0.01359221, 0.01322973, 0.01287905,
     &   0.01256918, 0.01225852, 0.01193481, 0.01161513, 0.01131371,
     &   0.01101675, 0.01073934, 0.01046383, 0.01017510, 0.00988740,
     &   0.00959218, 0.00930665, 0.00903108, 0.00876284, 0.00850220,
     &   0.00824333, 0.00798780, 0.00773332, 0.00748180, 0.00723900,
     &   0.00700847, 0.00679591, 0.00659144, 0.00638216, 0.00617820,
     &   0.00598140, 0.00579133, 0.00559511, 0.00539979, 0.00521059,
     &   0.00502781, 0.00484428, 0.00467142, 0.00450606, 0.00434081,
     &   0.00417539, 0.00401277, 0.00384502, 0.00368480, 0.00353176,
     &   0.00338558, 0.00324592, 0.00309221, 0.00294171, 0.00279859,
     &   0.00266246, 0.00253301, 0.00241182, 0.00229673, 0.00218714,
     &   0.00208281, 0.00198339, 0.00188853, 0.00179823, 0.00171227,
     &   0.00162999, 0.00154891, 0.00147186, 0.00139864, 0.00132906,
     &   0.00126291, 0.00120005, 0.00114032, 0.00108331, 0.00102897,
     &   0.00097735, 0.00092809, 0.00088032, 0.00083500, 0.00079201,
     &   0.00075214, 0.00071434, 0.00067843, 0.00064498, 0.00061332,
     &   0.00058321, 0.00055441, 0.00052701, 0.00050096, 0.00047619,
     &   0.00045264, 0.00043030, 0.00040908, 0.00038891, 0.00036973,
     &   0.00035149, 0.00033445, 0.00031841, 0.00030314, 0.00028863,
     &   0.00027480, 0.00026146, 0.00024876, 0.00023665, 0.00022511,
     &   0.00021406, 0.00020353, 0.00019351, 0.00018398, 0.00017489,
     &   0.00016624, 0.00015802, 0.00015020, 0.00014275, 0.00013568/
      data (ww(i,2),i=1,mxwv)/
     &   0.99999940, 0.99999946, 0.99999952, 0.99999952, 0.99999958,
     &   0.99999964, 0.99999964, 0.99999964, 0.99999970, 0.99999970,
     &   0.99999970, 0.99999976, 0.99999970, 0.99999976, 0.99999982,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999994, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999982, 0.99999982, 0.99999976, 0.99999976,
     &   0.99999964, 0.99999946, 0.99999928, 0.99999917, 0.99999911,
     &   0.99999911, 0.99999899, 0.99999887, 0.99999881, 0.99999863,
     &   0.99999815, 0.99999702, 0.99999499, 0.99999291, 0.99999249,
     &   0.99999267, 0.99999350, 0.99999428, 0.99999422, 0.99999255,
     &   0.99999017, 0.99998742, 0.99998528, 0.99998343, 0.99998128,
     &   0.99997431, 0.99994552, 0.99989980, 0.99987602, 0.99987978,
     &   0.99989426, 0.99991554, 0.99993777, 0.99995500, 0.99995917,
     &   0.99995196, 0.99993604, 0.99988347, 0.99973547, 0.99967641,
     &   0.99966687, 0.99967229, 0.99968952, 0.99971348, 0.99970734,
     &   0.99964023, 0.99950194, 0.99906087, 0.99858838, 0.99689466,
     &   0.99276316, 0.99217606, 0.99317580, 0.99532056, 0.99665207,
     &   0.99745274, 0.99796146, 0.99835998, 0.99857551, 0.99871409,
     &   0.99870151, 0.99858171, 0.99836808, 0.99799073, 0.99794477,
     &   0.99691707, 0.98812354, 0.97594953, 0.97705644, 0.98385495,
     &   0.98971093, 0.99254555, 0.99454623, 0.99561977, 0.99619257,
     &   0.99638742, 0.99611533, 0.99547791, 0.99453545, 0.99285358,
     &   0.99043518, 0.98684013, 0.98290801, 0.98133349, 0.97818160,
     &   0.96029204, 0.88629788, 0.65051126, 0.45335221, 0.40789646,
     &   0.42965531, 0.45185778, 0.47039634, 0.48706186, 0.51111752,
     &   0.55724645, 0.62581104, 0.71359581, 0.80470860, 0.86261636,
     &   0.90569735, 0.93765694, 0.95804328, 0.96924108, 0.97549391,
     &   0.97760212, 0.97735327, 0.97557360, 0.97271365, 0.96835762,
     &   0.96384770, 0.95789891, 0.95104575, 0.94289029, 0.93362486,
     &   0.92370141, 0.91287541, 0.90336311, 0.90186918, 0.90555465,
     &   0.91124725, 0.91657686, 0.92260861, 0.92745936, 0.92986423,
     &   0.92785108, 0.91924220, 0.90373415, 0.87244433, 0.81095517,
     &   0.68393046, 0.52605706, 0.47914237, 0.56073743, 0.68641496,
     &   0.74120396, 0.76305342, 0.77524769, 0.77978271, 0.77807325,
     &   0.77423626, 0.76871663, 0.76185495, 0.75474048, 0.74674398,
     &   0.73814428, 0.72825056, 0.71692353, 0.70551914, 0.69287282,
     &   0.67907649, 0.66405845, 0.64744067, 0.62911725, 0.60897517,
     &   0.58685577, 0.56266069, 0.53622192, 0.50689054, 0.47391659,
     &   0.43757614, 0.39347333, 0.34805253, 0.29157364, 0.24104093,
     &   0.19887660, 0.16943091, 0.14767697, 0.13574652, 0.13116366,
     &   0.13088927, 0.13592756, 0.14218697, 0.14930777, 0.15567274,
     &   0.16113099, 0.16650350, 0.17136121, 0.17544205, 0.17913465,
     &   0.18247540, 0.18494239, 0.18730941, 0.18932550, 0.19115071,
     &   0.19258994, 0.19428684, 0.19591540, 0.19727910, 0.19840063,
     &   0.19976383, 0.20075156, 0.20181023, 0.20236614, 0.20274742,
     &   0.20351337, 0.20419501, 0.20465152, 0.20444603, 0.20383513,
     &   0.20259498, 0.20075272, 0.19891614, 0.19708408, 0.19524902,
     &   0.19340676, 0.19174626, 0.19008461, 0.18844388, 0.18688156,
     &   0.18534511, 0.18294172, 0.18054605, 0.17800565, 0.17539743,
     &   0.17257227, 0.16946495, 0.16614150, 0.16240478, 0.15852132,
     &   0.15418683, 0.14956179, 0.14425512, 0.13870710, 0.13268358,
     &   0.12659618, 0.12050370, 0.11451343, 0.10870444, 0.10310488,
     &   0.09764888, 0.09225909, 0.08702935, 0.08201165, 0.07722890,
     &   0.07260516, 0.06816494, 0.06426598, 0.06071645, 0.05746546,
     &   0.05443297, 0.05172713, 0.04934153, 0.04724295, 0.04533013,
     &   0.04360836, 0.04205271, 0.04064818, 0.03944657, 0.03843470,
     &   0.03743645, 0.03642629, 0.03548203, 0.03463707, 0.03381536,
     &   0.03309447, 0.03236434, 0.03159376, 0.03082663, 0.03010079,
     &   0.02938045, 0.02870655, 0.02803204, 0.02731759, 0.02660035,
     &   0.02585787, 0.02513627, 0.02443675, 0.02375264, 0.02308489,
     &   0.02241828, 0.02175712, 0.02109545, 0.02043863, 0.01980242,
     &   0.01919672, 0.01863739, 0.01809792, 0.01754330, 0.01700119,
     &   0.01647685, 0.01596922, 0.01544324, 0.01491820, 0.01440852,
     &   0.01391513, 0.01341846, 0.01295000, 0.01250115, 0.01205164,
     &   0.01160076, 0.01115674, 0.01069771, 0.01025871, 0.00983886,
     &   0.00943731, 0.00905325, 0.00862958, 0.00821424, 0.00781882,
     &   0.00744237, 0.00708401, 0.00674826, 0.00642911, 0.00612499,
     &   0.00583521, 0.00555884, 0.00529499, 0.00504363, 0.00480419,
     &   0.00457485, 0.00434868, 0.00413362, 0.00392915, 0.00373473,
     &   0.00354981, 0.00337400, 0.00320684, 0.00304724, 0.00289504,
     &   0.00275041, 0.00261232, 0.00247835, 0.00235122, 0.00223057,
     &   0.00211866, 0.00201252, 0.00191166, 0.00181767, 0.00172872,
     &   0.00164408, 0.00156309, 0.00148603, 0.00141273, 0.00134305,
     &   0.00127677, 0.00121388, 0.00115416, 0.00109735, 0.00104333,
     &   0.00099195, 0.00094395, 0.00089874, 0.00085571, 0.00081482,
     &   0.00077582, 0.00073821, 0.00070241, 0.00066825, 0.00063573,
     &   0.00060454, 0.00057485, 0.00054658, 0.00051968, 0.00049403,
     &   0.00046963, 0.00044642, 0.00042434, 0.00040333, 0.00038334/
      data (ww(i,3),i=1,mxwv)/
     &   0.99999917, 0.99999928, 0.99999934, 0.99999928, 0.99999940,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999958, 0.99999958,
     &   0.99999964, 0.99999958, 0.99999964, 0.99999964, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999976, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999988, 0.99999988, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999988, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999976, 0.99999976, 0.99999970, 0.99999958,
     &   0.99999946, 0.99999923, 0.99999899, 0.99999881, 0.99999875,
     &   0.99999827, 0.99999857, 0.99999839, 0.99999827, 0.99999803,
     &   0.99999732, 0.99999565, 0.99999279, 0.99998915, 0.99998927,
     &   0.99998927, 0.99999070, 0.99999183, 0.99999154, 0.99998909,
     &   0.99998540, 0.99998146, 0.99997848, 0.99997544, 0.99997264,
     &   0.99996269, 0.99992073, 0.99985343, 0.99981958, 0.99982470,
     &   0.99984527, 0.99987638, 0.99990886, 0.99993390, 0.99994010,
     &   0.99992919, 0.99990588, 0.99982846, 0.99961066, 0.99952310,
     &   0.99950802, 0.99951512, 0.99953949, 0.99957418, 0.99956387,
     &   0.99946231, 0.99925345, 0.99858880, 0.99787223, 0.99531174,
     &   0.98908353, 0.98814297, 0.98958594, 0.99279594, 0.99480408,
     &   0.99601656, 0.99678808, 0.99739659, 0.99772197, 0.99792838,
     &   0.99789298, 0.99768221, 0.99731505, 0.99667311, 0.99657536,
     &   0.99483740, 0.98018080, 0.96016127, 0.96183759, 0.97290260,
     &   0.98260385, 0.98735601, 0.99074084, 0.99257505, 0.99357086,
     &   0.99393755, 0.99353850, 0.99256557, 0.99114746, 0.98863602,
     &   0.98513579, 0.98011905, 0.97502923, 0.97378892, 0.97095239,
     &   0.95134300, 0.87732273, 0.67051172, 0.49031708, 0.44467977,
     &   0.45737207, 0.47020876, 0.47862408, 0.48441643, 0.49194479,
     &   0.51126546, 0.55337983, 0.62692142, 0.72493762, 0.79927546,
     &   0.85966301, 0.90686882, 0.93791169, 0.95533168, 0.96522480,
     &   0.96902776, 0.96950114, 0.96796286, 0.96513134, 0.96058547,
     &   0.95605785, 0.95004308, 0.94324231, 0.93524051, 0.92630577,
     &   0.91691488, 0.90679765, 0.89813751, 0.89775699, 0.90271699,
     &   0.90972978, 0.91633248, 0.92356056, 0.92953640, 0.93311667,
     &   0.93253571, 0.92599750, 0.91360217, 0.88790166, 0.83683336,
     &   0.72771877, 0.58183151, 0.52751464, 0.59064770, 0.70826775,
     &   0.76383793, 0.78741699, 0.80138403, 0.80796903, 0.80866116,
     &   0.80728340, 0.80433774, 0.80012804, 0.79567492, 0.79043794,
     &   0.78469211, 0.77782631, 0.76971328, 0.76152706, 0.75226837,
     &   0.74199599, 0.73064053, 0.71783692, 0.70345432, 0.68732345,
     &   0.66925496, 0.64906478, 0.62647063, 0.60075158, 0.57091689,
     &   0.53691775, 0.49382934, 0.44729593, 0.38603896, 0.32800245,
     &   0.27696961, 0.23993467, 0.21168204, 0.19576751, 0.18940198,
     &   0.18878667, 0.19497463, 0.20273750, 0.21151233, 0.21931687,
     &   0.22597772, 0.23249297, 0.23837221, 0.24334975, 0.24787061,
     &   0.25200266, 0.25517175, 0.25823256, 0.26093772, 0.26345515,
     &   0.26555657, 0.26797572, 0.27034950, 0.27248591, 0.27437881,
     &   0.27658796, 0.27846465, 0.28044030, 0.28195259, 0.28327638,
     &   0.28507885, 0.28692988, 0.28857699, 0.28952676, 0.29010555,
     &   0.29008725, 0.28924155, 0.28841889, 0.28760645, 0.28679666,
     &   0.28599212, 0.28541559, 0.28485230, 0.28437382, 0.28409219,
     &   0.28385824, 0.28253910, 0.28122342, 0.27973992, 0.27816752,
     &   0.27630550, 0.27405846, 0.27147841, 0.26828146, 0.26481298,
     &   0.26064011, 0.25592932, 0.25007156, 0.24365802, 0.23628671,
     &   0.22856160, 0.22056787, 0.21246286, 0.20440146, 0.19644561,
     &   0.18849455, 0.18041290, 0.17235830, 0.16444524, 0.15674609,
     &   0.14910845, 0.14159213, 0.13490824, 0.12874855, 0.12303955,
     &   0.11762637, 0.11274956, 0.10842826, 0.10462550, 0.10113855,
     &   0.09799003, 0.09514111, 0.09256967, 0.09038886, 0.08860181,
     &   0.08682457, 0.08499306, 0.08327505, 0.08174816, 0.08025210,
     &   0.07897643, 0.07765616, 0.07622296, 0.07477175, 0.07339215,
     &   0.07200155, 0.07070382, 0.06938177, 0.06794605, 0.06647844,
     &   0.06492463, 0.06339803, 0.06190287, 0.06042439, 0.05896550,
     &   0.05749057, 0.05600950, 0.05450822, 0.05300087, 0.05152777,
     &   0.05011496, 0.04880668, 0.04753597, 0.04621365, 0.04491090,
     &   0.04364245, 0.04240669, 0.04111309, 0.03981156, 0.03854068,
     &   0.03730373, 0.03604970, 0.03486226, 0.03371971, 0.03256883,
     &   0.03140800, 0.03025947, 0.02906530, 0.02791914, 0.02681920,
     &   0.02576379, 0.02475123, 0.02362765, 0.02252247, 0.02146737,
     &   0.02046029, 0.01949918, 0.01859670, 0.01773696, 0.01691595,
     &   0.01613203, 0.01538294, 0.01466641, 0.01398261, 0.01333013,
     &   0.01270413, 0.01208560, 0.01149662, 0.01093583, 0.01040188,
     &   0.00989335, 0.00940927, 0.00894850, 0.00850803, 0.00808756,
     &   0.00768757, 0.00730528, 0.00693402, 0.00658139, 0.00624646,
     &   0.00593555, 0.00564046, 0.00535982, 0.00509814, 0.00485033,
     &   0.00461439, 0.00438849, 0.00417339, 0.00396871, 0.00377401,
     &   0.00358872, 0.00341282, 0.00324571, 0.00308667, 0.00293539,
     &   0.00279142, 0.00265689, 0.00253014, 0.00240945, 0.00229471,
     &   0.00218526, 0.00207967, 0.00197912, 0.00188315, 0.00179176,
     &   0.00170412, 0.00162064, 0.00154113, 0.00146547, 0.00139330,
     &   0.00132463, 0.00125930, 0.00119713, 0.00113798, 0.00108170/
      data (ww(i,4),i=1,mxwv)/
     &   0.99999887, 0.99999899, 0.99999905, 0.99999911, 0.99999923,
     &   0.99999911, 0.99999928, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999958, 0.99999958,
     &   0.99999964, 0.99999958, 0.99999970, 0.99999970, 0.99999976,
     &   0.99999976, 0.99999976, 0.99999976, 0.99999982, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999982, 0.99999976, 0.99999976,
     &   0.99999976, 0.99999970, 0.99999964, 0.99999958, 0.99999946,
     &   0.99999928, 0.99999893, 0.99999857, 0.99999833, 0.99999821,
     &   0.99999815, 0.99999803, 0.99999756, 0.99999762, 0.99999678,
     &   0.99999624, 0.99999398, 0.99999011, 0.99998540, 0.99998403,
     &   0.99998415, 0.99998647, 0.99998820, 0.99998796, 0.99998474,
     &   0.99997979, 0.99997467, 0.99996984, 0.99996549, 0.99996096,
     &   0.99994606, 0.99988651, 0.99979222, 0.99973822, 0.99974656,
     &   0.99977833, 0.99982131, 0.99986899, 0.99990559, 0.99991339,
     &   0.99989599, 0.99986440, 0.99975067, 0.99943900, 0.99931085,
     &   0.99929076, 0.99929857, 0.99933290, 0.99938214, 0.99936688,
     &   0.99922097, 0.99891806, 0.99795890, 0.99692863, 0.99327600,
     &   0.98449999, 0.98315758, 0.98514807, 0.98965138, 0.99249220,
     &   0.99421740, 0.99531949, 0.99619287, 0.99665838, 0.99695224,
     &   0.99689221, 0.99657279, 0.99601942, 0.99505502, 0.99488884,
     &   0.99227744, 0.97066647, 0.94169754, 0.94376308, 0.95944023,
     &   0.97354060, 0.98052859, 0.98557317, 0.98830819, 0.98977381,
     &   0.99026293, 0.98952681, 0.98784614, 0.98541331, 0.98115838,
     &   0.97525418, 0.96688467, 0.95853454, 0.95671052, 0.95287877,
     &   0.92544538, 0.83741432, 0.65511763, 0.50525880, 0.46748200,
     &   0.47553518, 0.48387331, 0.48764753, 0.48938096, 0.49024406,
     &   0.49463868, 0.51274043, 0.55958045, 0.64279693, 0.71924150,
     &   0.79127181, 0.85442847, 0.89963853, 0.92649579, 0.94236356,
     &   0.94876999, 0.94995117, 0.94805747, 0.94431269, 0.93817466,
     &   0.93237096, 0.92472178, 0.91635615, 0.90675557, 0.89637601,
     &   0.88583744, 0.87477809, 0.86573219, 0.86687821, 0.87461221,
     &   0.88493383, 0.89477652, 0.90533727, 0.91432559, 0.92047620,
     &   0.92187285, 0.91684639, 0.90615088, 0.88299054, 0.83725315,
     &   0.73965657, 0.60575593, 0.54493225, 0.58336729, 0.69001663,
     &   0.75009263, 0.77825075, 0.79611355, 0.80611801, 0.80990529,
     &   0.81139052, 0.81122148, 0.80970329, 0.80790627, 0.80534387,
     &   0.80232823, 0.79829264, 0.79313236, 0.78792620, 0.78180289,
     &   0.77480936, 0.76690537, 0.75775534, 0.74724162, 0.73517931,
     &   0.72140920, 0.70572525, 0.68780404, 0.66697538, 0.64214760,
     &   0.61307013, 0.57478946, 0.53163868, 0.47177818, 0.41189998,
     &   0.35638931, 0.31445366, 0.28130773, 0.26201409, 0.25386393,
     &   0.25262198, 0.25915343, 0.26748633, 0.27683929, 0.28506193,
     &   0.29198697, 0.29868269, 0.30466282, 0.30970261, 0.31425050,
     &   0.31840214, 0.32161641, 0.32471052, 0.32748109, 0.33008018,
     &   0.33227944, 0.33479723, 0.33728719, 0.33959681, 0.34168798,
     &   0.34411466, 0.34630778, 0.34860191, 0.35055777, 0.35233122,
     &   0.35459948, 0.35707170, 0.35939974, 0.36112005, 0.36259350,
     &   0.36365095, 0.36374980, 0.36389840, 0.36407065, 0.36426190,
     &   0.36448589, 0.36495835, 0.36547357, 0.36616361, 0.36719865,
     &   0.36831990, 0.36836678, 0.36844605, 0.36841890, 0.36834788,
     &   0.36802483, 0.36735117, 0.36634630, 0.36470050, 0.36277446,
     &   0.36010337, 0.35680372, 0.35215285, 0.34676129, 0.34010065,
     &   0.33284795, 0.32508805, 0.31697530, 0.30872646, 0.30042335,
     &   0.29193345, 0.28305614, 0.27396300, 0.26482034, 0.25575772,
     &   0.24650718, 0.23714961, 0.22872876, 0.22088939, 0.21355106,
     &   0.20646422, 0.20002249, 0.19430555, 0.18930928, 0.18471621,
     &   0.18057714, 0.17685220, 0.17352052, 0.17076226, 0.16865648,
     &   0.16655932, 0.16434015, 0.16226850, 0.16048065, 0.15872598,
     &   0.15737888, 0.15592508, 0.15425314, 0.15251103, 0.15086417,
     &   0.14916088, 0.14762111, 0.14599922, 0.14414302, 0.14217509,
     &   0.13998438, 0.13779536, 0.13561723, 0.13342257, 0.13121460,
     &   0.12892659, 0.12656879, 0.12411072, 0.12158048, 0.11906129,
     &   0.11660824, 0.11433789, 0.11210244, 0.10970278, 0.10729378,
     &   0.10491285, 0.10256008, 0.10002839, 0.09742652, 0.09484830,
     &   0.09230429, 0.08967441, 0.08715999, 0.08471477, 0.08221217,
     &   0.07964855, 0.07707956, 0.07436566, 0.07173564, 0.06918841,
     &   0.06672274, 0.06433730, 0.06164570, 0.05897395, 0.05640470,
     &   0.05393530, 0.05156305, 0.04932224, 0.04717493, 0.04511272,
     &   0.04313302, 0.04123148, 0.03940355, 0.03765094, 0.03597111,
     &   0.03435239, 0.03274507, 0.03120859, 0.02974024, 0.02833720,
     &   0.02699645, 0.02571607, 0.02449361, 0.02332157, 0.02219959,
     &   0.02112947, 0.02010402, 0.01910548, 0.01815485, 0.01724998,
     &   0.01640845, 0.01560817, 0.01484566, 0.01413354, 0.01345804,
     &   0.01281384, 0.01219609, 0.01160701, 0.01104566, 0.01051094,
     &   0.01000144, 0.00951716, 0.00905655, 0.00861770, 0.00819981,
     &   0.00780170, 0.00742938, 0.00707828, 0.00674368, 0.00642529,
     &   0.00612135, 0.00582790, 0.00554824, 0.00528112, 0.00502660,
     &   0.00478233, 0.00454951, 0.00432765, 0.00411640, 0.00391478,
     &   0.00372285, 0.00354015, 0.00336623, 0.00320065, 0.00304307/
      data (ww(i,5),i=1,mxwv)/
     &   0.99999827, 0.99999851, 0.99999869, 0.99999881, 0.99999881,
     &   0.99999905, 0.99999899, 0.99999911, 0.99999917, 0.99999923,
     &   0.99999893, 0.99999923, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999952, 0.99999946, 0.99999958, 0.99999958, 0.99999964,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999976, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999976, 0.99999970, 0.99999970,
     &   0.99999964, 0.99999958, 0.99999952, 0.99999940, 0.99999928,
     &   0.99999905, 0.99999845, 0.99999791, 0.99999768, 0.99999768,
     &   0.99999762, 0.99999738, 0.99999690, 0.99999642, 0.99999624,
     &   0.99999493, 0.99999201, 0.99998295, 0.99997985, 0.99997914,
     &   0.99997908, 0.99998230, 0.99998391, 0.99998373, 0.99997807,
     &   0.99997193, 0.99996245, 0.99995834, 0.99995255, 0.99994677,
     &   0.99992687, 0.99984056, 0.99970990, 0.99963808, 0.99965358,
     &   0.99968612, 0.99975443, 0.99981660, 0.99986738, 0.99987733,
     &   0.99985737, 0.99979556, 0.99965656, 0.99920702, 0.99904925,
     &   0.99900645, 0.99902117, 0.99906808, 0.99913728, 0.99910432,
     &   0.99889934, 0.99847943, 0.99714142, 0.99570125, 0.99066454,
     &   0.97868037, 0.97683543, 0.97951096, 0.98563069, 0.98951364,
     &   0.99188161, 0.99339950, 0.99460733, 0.99525052, 0.99565774,
     &   0.99556446, 0.99510586, 0.99431616, 0.99294412, 0.99269658,
     &   0.98900241, 0.95917624, 0.92044318, 0.92303604, 0.94375324,
     &   0.96278375, 0.97235131, 0.97934133, 0.98315072, 0.98518610,
     &   0.98583293, 0.98470944, 0.98219794, 0.97855645, 0.97220153,
     &   0.96333444, 0.95069396, 0.93778253, 0.93398577, 0.92673880,
     &   0.88355070, 0.76343107, 0.60543478, 0.50374311, 0.48122457,
     &   0.48914397, 0.49653402, 0.49924690, 0.50052184, 0.50034410,
     &   0.50020581, 0.50602216, 0.53118163, 0.59188259, 0.65918767,
     &   0.73197234, 0.80295813, 0.85798895, 0.89238095, 0.91332805,
     &   0.92149520, 0.92226446, 0.91853690, 0.91206974, 0.90206772,
     &   0.89267725, 0.88071656, 0.86798245, 0.85384184, 0.83908880,
     &   0.82468057, 0.81015295, 0.79879010, 0.80100536, 0.81207871,
     &   0.82696491, 0.84160978, 0.85761040, 0.87177008, 0.88230020,
     &   0.88655806, 0.88261306, 0.87232518, 0.84883171, 0.80405754,
     &   0.71336055, 0.59438878, 0.53352088, 0.54631108, 0.62969011,
     &   0.69243264, 0.72668505, 0.75047392, 0.76572019, 0.77394283,
     &   0.77938986, 0.78290725, 0.78480917, 0.78634250, 0.78703117,
     &   0.78727025, 0.78648353, 0.78458613, 0.78265750, 0.77990109,
     &   0.77636588, 0.77205396, 0.76665354, 0.76007897, 0.75214350,
     &   0.74276060, 0.73174030, 0.71876371, 0.70330232, 0.68425786,
     &   0.66133195, 0.62993312, 0.59300065, 0.53919178, 0.48265457,
     &   0.42755735, 0.38433865, 0.34895119, 0.32762694, 0.31809533,
     &   0.31612349, 0.32215178, 0.33005863, 0.33886451, 0.34648415,
     &   0.35280025, 0.35880163, 0.36407465, 0.36845443, 0.37235758,
     &   0.37587947, 0.37859002, 0.38117957, 0.38349196, 0.38565677,
     &   0.38748786, 0.38957593, 0.39164153, 0.39358085, 0.39534932,
     &   0.39740473, 0.39932731, 0.40133637, 0.40315419, 0.40482590,
     &   0.40692589, 0.40932977, 0.41167146, 0.41358945, 0.41542530,
     &   0.41710067, 0.41784167, 0.41865551, 0.41950476, 0.42038724,
     &   0.42132595, 0.42249408, 0.42372724, 0.42520896, 0.42715588,
     &   0.42921808, 0.43038887, 0.43161848, 0.43283346, 0.43406591,
     &   0.43513489, 0.43595809, 0.43651140, 0.43651205, 0.43628186,
     &   0.43541083, 0.43393400, 0.43113503, 0.42754358, 0.42258534,
     &   0.41690972, 0.41059387, 0.40375650, 0.39665961, 0.38939673,
     &   0.38182691, 0.37371078, 0.36518854, 0.35645732, 0.34769824,
     &   0.33852866, 0.32902873, 0.32039779, 0.31233096, 0.30475488,
     &   0.29733425, 0.29055685, 0.28456804, 0.27942029, 0.27471524,
     &   0.27052361, 0.26681754, 0.26358268, 0.26102126, 0.25932878,
     &   0.25768870, 0.25590375, 0.25428963, 0.25301823, 0.25180683,
     &   0.25120455, 0.25046924, 0.24947682, 0.24838723, 0.24742316,
     &   0.24638113, 0.24560864, 0.24472474, 0.24355973, 0.24222068,
     &   0.24054006, 0.23884106, 0.23713596, 0.23538755, 0.23359150,
     &   0.23166233, 0.22958685, 0.22731146, 0.22487009, 0.22237784,
     &   0.21990776, 0.21770556, 0.21551722, 0.21299908, 0.21039037,
     &   0.20776133, 0.20511732, 0.20208521, 0.19882190, 0.19550437,
     &   0.19215217, 0.18852834, 0.18501490, 0.18153778, 0.17784585,
     &   0.17392287, 0.16987768, 0.16543354, 0.16103637, 0.15669055,
     &   0.15240027, 0.14816944, 0.14316705, 0.13808939, 0.13312280,
     &   0.12827019, 0.12353409, 0.11899475, 0.11458165, 0.11028384,
     &   0.10610224, 0.10203318, 0.09807210, 0.09422879, 0.09050281,
     &   0.08687121, 0.08321727, 0.07968922, 0.07628519, 0.07300249,
     &   0.06983741, 0.06678957, 0.06385642, 0.06102223, 0.05828896,
     &   0.05566394, 0.05313128, 0.05064725, 0.04826819, 0.04599070,
     &   0.04386223, 0.04182776, 0.03987980, 0.03805298, 0.03631263,
     &   0.03464588, 0.03304086, 0.03150441, 0.03003492, 0.02863020,
     &   0.02728725, 0.02600678, 0.02478521, 0.02361796, 0.02250339,
     &   0.02143877, 0.02044085, 0.01949774, 0.01859690, 0.01773790,
     &   0.01691617, 0.01612116, 0.01536204, 0.01463568, 0.01394237,
     &   0.01327586, 0.01263957, 0.01203232, 0.01145329, 0.01089989,
     &   0.01037237, 0.00986963, 0.00939046, 0.00893374, 0.00849863/
      data (ww(i,6),i=1,mxwv)/
     &   0.99999779, 0.99999803, 0.99999815, 0.99999833, 0.99999851,
     &   0.99999863, 0.99999869, 0.99999881, 0.99999887, 0.99999887,
     &   0.99999899, 0.99999905, 0.99999911, 0.99999917, 0.99999928,
     &   0.99999928, 0.99999934, 0.99999940, 0.99999940, 0.99999952,
     &   0.99999958, 0.99999958, 0.99999964, 0.99999970, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999964, 0.99999958, 0.99999958,
     &   0.99999946, 0.99999940, 0.99999928, 0.99999917, 0.99999899,
     &   0.99999851, 0.99999797, 0.99999714, 0.99999696, 0.99999672,
     &   0.99999666, 0.99999636, 0.99999577, 0.99999517, 0.99999475,
     &   0.99999285, 0.99998879, 0.99998069, 0.99997270, 0.99997175,
     &   0.99997175, 0.99997509, 0.99997723, 0.99997783, 0.99997085,
     &   0.99996144, 0.99995142, 0.99994242, 0.99993420, 0.99992651,
     &   0.99989372, 0.99978262, 0.99960804, 0.99951243, 0.99952787,
     &   0.99957955, 0.99966121, 0.99974710, 0.99981964, 0.99982995,
     &   0.99980646, 0.99973977, 0.99951392, 0.99893206, 0.99868143,
     &   0.99864382, 0.99865776, 0.99871540, 0.99881184, 0.99877453,
     &   0.99848175, 0.99791205, 0.99608070, 0.99410695, 0.98723787,
     &   0.97100145, 0.96852177, 0.97212708, 0.98039186, 0.98565477,
     &   0.98888034, 0.99093372, 0.99257398, 0.99343705, 0.99400586,
     &   0.99386013, 0.99322599, 0.99212170, 0.99023223, 0.98987991,
     &   0.98480844, 0.94469929, 0.89431983, 0.89759159, 0.92429322,
     &   0.94930178, 0.96203774, 0.97142589, 0.97656375, 0.97930455,
     &   0.98014748, 0.97854900, 0.97503030, 0.96994895, 0.96115196,
     &   0.94895494, 0.93172091, 0.91418493, 0.90859026, 0.89802313,
     &   0.83923179, 0.68789148, 0.54769576, 0.49790719, 0.49115402,
     &   0.50062710, 0.50812578, 0.51078123, 0.51220894, 0.51208192,
     &   0.51082146, 0.51121312, 0.52162242, 0.55934936, 0.61264759,
     &   0.68033111, 0.75452387, 0.81690586, 0.85782444, 0.88340223,
     &   0.89320207, 0.89352912, 0.88795704, 0.87872696, 0.86482048,
     &   0.85161948, 0.83504152, 0.81745207, 0.79813778, 0.77819657,
     &   0.75894421, 0.73990202, 0.72509640, 0.72633702, 0.73800731,
     &   0.75453442, 0.77127439, 0.79036361, 0.80777729, 0.82094085,
     &   0.82601845, 0.82028323, 0.80680519, 0.77821529, 0.72972155,
     &   0.64642614, 0.55490118, 0.50947350, 0.50876105, 0.55671793,
     &   0.60750794, 0.64055783, 0.66612941, 0.68434411, 0.69578612,
     &   0.70453066, 0.71143627, 0.71671116, 0.72173482, 0.72598380,
     &   0.72992301, 0.73290980, 0.73484367, 0.73683637, 0.73812288,
     &   0.73873734, 0.73872375, 0.73776776, 0.73581129, 0.73265004,
     &   0.72829282, 0.72257310, 0.71518564, 0.70580566, 0.69335550,
     &   0.67759717, 0.65455604, 0.62577599, 0.58137441, 0.53237879,
     &   0.48222941, 0.44152692, 0.40703008, 0.38548797, 0.37531108,
     &   0.37271580, 0.37758833, 0.38429201, 0.39171141, 0.39801484,
     &   0.40317157, 0.40795425, 0.41206545, 0.41538501, 0.41829151,
     &   0.42083937, 0.42274860, 0.42455667, 0.42612034, 0.42755932,
     &   0.42876539, 0.43012545, 0.43145454, 0.43267551, 0.43378356,
     &   0.43507403, 0.43627325, 0.43754238, 0.43870670, 0.43979588,
     &   0.44118357, 0.44283551, 0.44449821, 0.44594786, 0.44744259,
     &   0.44898903, 0.44985032, 0.45080546, 0.45181695, 0.45288339,
     &   0.45402959, 0.45536569, 0.45678768, 0.45849144, 0.46070597,
     &   0.46306923, 0.46486565, 0.46676037, 0.46875286, 0.47084397,
     &   0.47290668, 0.47488853, 0.47672343, 0.47819361, 0.47953582,
     &   0.48045415, 0.48088700, 0.48021448, 0.47880489, 0.47610408,
     &   0.47264701, 0.46849397, 0.46370420, 0.45856440, 0.45316905,
     &   0.44739127, 0.44098380, 0.43402719, 0.42673072, 0.41931728,
     &   0.41131791, 0.40280104, 0.39493766, 0.38754988, 0.38058457,
     &   0.37365144, 0.36725846, 0.36160558, 0.35680848, 0.35244104,
     &   0.34858087, 0.34522140, 0.34235722, 0.34015980, 0.33893844,
     &   0.33784586, 0.33664334, 0.33562192, 0.33495161, 0.33438855,
     &   0.33457017, 0.33464342, 0.33451194, 0.33430308, 0.33425161,
     &   0.33414680, 0.33441204, 0.33459413, 0.33457085, 0.33438531,
     &   0.33385202, 0.33332759, 0.33282652, 0.33231512, 0.33178043,
     &   0.33115318, 0.33038655, 0.32941738, 0.32827669, 0.32707641,
     &   0.32587686, 0.32505047, 0.32428128, 0.32314694, 0.32190636,
     &   0.32065806, 0.31942075, 0.31771848, 0.31568480, 0.31356093,
     &   0.31136954, 0.30876270, 0.30625364, 0.30376503, 0.30092660,
     &   0.29768655, 0.29418859, 0.29003516, 0.28581765, 0.28153861,
     &   0.27720094, 0.27280796, 0.26699835, 0.26086479, 0.25470918,
     &   0.24853899, 0.24236177, 0.23629387, 0.23024882, 0.22421750,
     &   0.21820731, 0.21221797, 0.20624751, 0.20032099, 0.19444548,
     &   0.18858343, 0.18250582, 0.17651449, 0.17061552, 0.16481219,
     &   0.15910415, 0.15350497, 0.14801894, 0.14262158, 0.13732573,
     &   0.13215627, 0.12708600, 0.12202205, 0.11710132, 0.11232451,
     &   0.10780685, 0.10343372, 0.09919482, 0.09517878, 0.09131093,
     &   0.08756629, 0.08392116, 0.08039680, 0.07699367, 0.07371057,
     &   0.07054438, 0.06750056, 0.06457378, 0.06175584, 0.05904507,
     &   0.05643783, 0.05397955, 0.05164262, 0.04939717, 0.04724408,
     &   0.04517306, 0.04315819, 0.04122469, 0.03936569, 0.03758327,
     &   0.03586195, 0.03421184, 0.03263084, 0.03111764, 0.02966609,
     &   0.02827772, 0.02695023, 0.02568108, 0.02446782, 0.02330868/
      data (ww(i,7),i=1,mxwv)/
     &   0.99999696, 0.99999726, 0.99999750, 0.99999762, 0.99999797,
     &   0.99999809, 0.99999815, 0.99999827, 0.99999839, 0.99999851,
     &   0.99999857, 0.99999869, 0.99999851, 0.99999887, 0.99999893,
     &   0.99999905, 0.99999911, 0.99999917, 0.99999923, 0.99999934,
     &   0.99999940, 0.99999946, 0.99999952, 0.99999958, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999946, 0.99999946, 0.99999940,
     &   0.99999928, 0.99999917, 0.99999905, 0.99999887, 0.99999857,
     &   0.99999809, 0.99999708, 0.99999601, 0.99999577, 0.99999547,
     &   0.99999529, 0.99999505, 0.99999428, 0.99999380, 0.99999285,
     &   0.99999022, 0.99998415, 0.99997383, 0.99996281, 0.99996006,
     &   0.99995869, 0.99996525, 0.99996990, 0.99996901, 0.99996042,
     &   0.99994701, 0.99993157, 0.99992120, 0.99991137, 0.99989760,
     &   0.99985856, 0.99970168, 0.99946046, 0.99932796, 0.99934280,
     &   0.99941963, 0.99953932, 0.99965447, 0.99974871, 0.99977320,
     &   0.99972832, 0.99964190, 0.99935168, 0.99853688, 0.99819899,
     &   0.99813479, 0.99815875, 0.99823213, 0.99836111, 0.99831593,
     &   0.99793899, 0.99714297, 0.99460208, 0.99195045, 0.98254484,
     &   0.96055329, 0.95722944, 0.96207827, 0.97324985, 0.98040140,
     &   0.98476905, 0.98758531, 0.98984998, 0.99102640, 0.99176180,
     &   0.99160558, 0.99072623, 0.98922503, 0.98664337, 0.98614794,
     &   0.97925824, 0.92574805, 0.86137813, 0.86548412, 0.89930022,
     &   0.93179733, 0.94863033, 0.96115303, 0.96803856, 0.97171205,
     &   0.97282249, 0.97062778, 0.96583647, 0.95894688, 0.94711155,
     &   0.93085378, 0.90819287, 0.88545144, 0.87800407, 0.86414713,
     &   0.79163742, 0.62828273, 0.51430357, 0.49666989, 0.49972558,
     &   0.51026201, 0.51780307, 0.52047658, 0.52198386, 0.52193785,
     &   0.52046502, 0.51892745, 0.52125508, 0.53972298, 0.57556754,
     &   0.63175446, 0.70367491, 0.77106827, 0.81826645, 0.84883434,
     &   0.86063880, 0.86081958, 0.85372406, 0.84218729, 0.82511300,
     &   0.80907834, 0.78929752, 0.76866978, 0.74647868, 0.72404939,
     &   0.70285225, 0.68236494, 0.66668802, 0.66723871, 0.67824376,
     &   0.69439167, 0.71106184, 0.73064148, 0.74874330, 0.76213342,
     &   0.76590765, 0.75642806, 0.73726624, 0.70047885, 0.64488518,
     &   0.57066619, 0.51562226, 0.49563545, 0.49591392, 0.51720703,
     &   0.54778099, 0.57052344, 0.58946574, 0.60341918, 0.61214048,
     &   0.61896259, 0.62454045, 0.62902325, 0.63367558, 0.63800019,
     &   0.64245516, 0.64642900, 0.64982152, 0.65369338, 0.65734094,
     &   0.66078472, 0.66408938, 0.66694117, 0.66928709, 0.67088288,
     &   0.67179817, 0.67186391, 0.67076021, 0.66836745, 0.66362113,
     &   0.65635419, 0.64339846, 0.62475526, 0.59286040, 0.55517751,
     &   0.51424795, 0.47986630, 0.44967666, 0.43016621, 0.42056099,
     &   0.41787335, 0.42152852, 0.42683262, 0.43264690, 0.43748897,
     &   0.44144249, 0.44499472, 0.44797298, 0.45026886, 0.45224181,
     &   0.45387471, 0.45502672, 0.45611566, 0.45696723, 0.45771068,
     &   0.45832485, 0.45898971, 0.45960811, 0.46009460, 0.46051937,
     &   0.46100765, 0.46136740, 0.46178961, 0.46209058, 0.46239656,
     &   0.46285695, 0.46339101, 0.46396005, 0.46445397, 0.46500123,
     &   0.46563107, 0.46599215, 0.46644634, 0.46697384, 0.46756977,
     &   0.46825263, 0.46907449, 0.46998698, 0.47113091, 0.47268438,
     &   0.47440302, 0.47591543, 0.47756547, 0.47939780, 0.48140597,
     &   0.48352951, 0.48575589, 0.48800933, 0.49016392, 0.49235114,
     &   0.49442068, 0.49624112, 0.49736533, 0.49795496, 0.49754000,
     &   0.49646416, 0.49475563, 0.49240166, 0.48966914, 0.48664290,
     &   0.48322368, 0.47918665, 0.47453797, 0.46947220, 0.46422803,
     &   0.45830297, 0.45174348, 0.44552276, 0.43962339, 0.43401754,
     &   0.42830691, 0.42294353, 0.41815439, 0.41410413, 0.41039804,
     &   0.40710753, 0.40424913, 0.40182683, 0.39994857, 0.39899695,
     &   0.39822730, 0.39739525, 0.39671689, 0.39633107, 0.39607856,
     &   0.39658380, 0.39701837, 0.39733472, 0.39760372, 0.39803261,
     &   0.39844099, 0.39926174, 0.40004364, 0.40074477, 0.40132892,
     &   0.40162846, 0.40197185, 0.40237418, 0.40281451, 0.40327391,
     &   0.40372035, 0.40407923, 0.40429342, 0.40438780, 0.40444684,
     &   0.40450126, 0.40499541, 0.40561378, 0.40594786, 0.40621892,
     &   0.40654138, 0.40695241, 0.40698671, 0.40671328, 0.40639758,
     &   0.40606648, 0.40533641, 0.40474132, 0.40423581, 0.40341619,
     &   0.40220651, 0.40075895, 0.39867204, 0.39651304, 0.39428037,
     &   0.39197248, 0.38958779, 0.38550645, 0.38096195, 0.37631887,
     &   0.37157178, 0.36672136, 0.36184514, 0.35688305, 0.35182512,
     &   0.34667295, 0.34141862, 0.33605176, 0.33059773, 0.32505965,
     &   0.31937593, 0.31319681, 0.30695316, 0.30065057, 0.29428700,
     &   0.28785008, 0.28137523, 0.27486938, 0.26829374, 0.26166910,
     &   0.25503841, 0.24835582, 0.24145187, 0.23457880, 0.22774509,
     &   0.22115138, 0.21462186, 0.20814767, 0.20190477, 0.19576646,
     &   0.18969476, 0.18365113, 0.17768589, 0.17180875, 0.16602533,
     &   0.16034096, 0.15477599, 0.14932926, 0.14399321, 0.13876948,
     &   0.13366316, 0.12878339, 0.12407988, 0.11949507, 0.11503880,
     &   0.11069335, 0.10640509, 0.10223781, 0.09818173, 0.09424719,
     &   0.09040257, 0.08667712, 0.08307081, 0.07958516, 0.07620896,
     &   0.07295066, 0.06980845, 0.06677964, 0.06386153, 0.06105271/
      data (ww(i,8),i=1,mxwv)/
     &   0.99999571, 0.99999619, 0.99999648, 0.99999678, 0.99999714,
     &   0.99999738, 0.99999750, 0.99999762, 0.99999774, 0.99999791,
     &   0.99999803, 0.99999821, 0.99999833, 0.99999839, 0.99999827,
     &   0.99999863, 0.99999881, 0.99999887, 0.99999899, 0.99999905,
     &   0.99999911, 0.99999923, 0.99999934, 0.99999940, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999928, 0.99999923, 0.99999911,
     &   0.99999899, 0.99999887, 0.99999869, 0.99999845, 0.99999815,
     &   0.99999738, 0.99999613, 0.99999458, 0.99999380, 0.99999374,
     &   0.99999344, 0.99999273, 0.99999201, 0.99999106, 0.99998963,
     &   0.99998653, 0.99997860, 0.99996340, 0.99994862, 0.99994433,
     &   0.99994594, 0.99995279, 0.99995714, 0.99995506, 0.99994540,
     &   0.99992621, 0.99990547, 0.99988961, 0.99987245, 0.99985975,
     &   0.99980903, 0.99959201, 0.99924839, 0.99907559, 0.99910349,
     &   0.99920541, 0.99935955, 0.99952328, 0.99965537, 0.99968237,
     &   0.99963135, 0.99950504, 0.99909872, 0.99796885, 0.99752194,
     &   0.99742806, 0.99746478, 0.99758458, 0.99774879, 0.99769086,
     &   0.99715376, 0.99606317, 0.99260443, 0.98891127, 0.97606587,
     &   0.94637978, 0.94197839, 0.94842881, 0.96346414, 0.97316384,
     &   0.97913885, 0.98297501, 0.98605442, 0.98769617, 0.98873019,
     &   0.98846889, 0.98728544, 0.98521662, 0.98170143, 0.98104608,
     &   0.97165918, 0.90081954, 0.82066298, 0.82560819, 0.86722666,
     &   0.90869105, 0.93071711, 0.94733310, 0.95655078, 0.96148878,
     &   0.96298546, 0.96002597, 0.95359266, 0.94438517, 0.92870623,
     &   0.90745121, 0.87838811, 0.84988409, 0.84060633, 0.82353926,
     &   0.73942572, 0.58155888, 0.50410366, 0.50006592, 0.50712168,
     &   0.51790988, 0.52533835, 0.52800852, 0.52954912, 0.52956444,
     &   0.52812338, 0.52614981, 0.52527016, 0.53153592, 0.55081689,
     &   0.59057903, 0.65284228, 0.72026330, 0.77196527, 0.80724251,
     &   0.82120597, 0.82138956, 0.81291533, 0.79930365, 0.77951872,
     &   0.76131135, 0.73937720, 0.71712875, 0.69393373, 0.67128944,
     &   0.65065366, 0.63144523, 0.61724895, 0.61762714, 0.62734431,
     &   0.64203137, 0.65763968, 0.67656338, 0.69452238, 0.70791548,
     &   0.71118766, 0.70025867, 0.67912430, 0.64063907, 0.58693731,
     &   0.52763253, 0.49975356, 0.49687436, 0.50038028, 0.50803709,
     &   0.52313036, 0.53619015, 0.54788685, 0.55658621, 0.56159216,
     &   0.56522149, 0.56787872, 0.56969255, 0.57163125, 0.57330906,
     &   0.57509106, 0.57652462, 0.57757354, 0.57916224, 0.58074737,
     &   0.58242452, 0.58431560, 0.58625752, 0.58829898, 0.59032673,
     &   0.59249544, 0.59475237, 0.59689248, 0.59898192, 0.60017991,
     &   0.60037875, 0.59744680, 0.58990473, 0.57293612, 0.54981077,
     &   0.52197582, 0.49737710, 0.47477964, 0.45970696, 0.45224896,
     &   0.45038220, 0.45341295, 0.45773000, 0.46232629, 0.46607372,
     &   0.46918657, 0.47188887, 0.47411263, 0.47573653, 0.47712332,
     &   0.47818312, 0.47886923, 0.47953627, 0.47995767, 0.48028380,
     &   0.48055974, 0.48081815, 0.48102009, 0.48105374, 0.48106185,
     &   0.48104978, 0.48084226, 0.48070434, 0.48037475, 0.48011309,
     &   0.47990721, 0.47955656, 0.47922388, 0.47885057, 0.47844091,
     &   0.47797751, 0.47761700, 0.47731656, 0.47707853, 0.47689968,
     &   0.47677901, 0.47674122, 0.47676498, 0.47688511, 0.47717208,
     &   0.47758165, 0.47798744, 0.47851592, 0.47919846, 0.48004359,
     &   0.48104617, 0.48222280, 0.48354158, 0.48496580, 0.48655117,
     &   0.48827028, 0.49001774, 0.49156687, 0.49292228, 0.49377176,
     &   0.49424085, 0.49431369, 0.49391729, 0.49323365, 0.49231362,
     &   0.49109134, 0.48939493, 0.48716658, 0.48454672, 0.48173919,
     &   0.47830203, 0.47425243, 0.47023952, 0.46637306, 0.46264949,
     &   0.45873535, 0.45496231, 0.45153886, 0.44863561, 0.44595274,
     &   0.44354466, 0.44144091, 0.43965071, 0.43822026, 0.43750161,
     &   0.43695912, 0.43639296, 0.43592978, 0.43566197, 0.43551177,
     &   0.43596619, 0.43638298, 0.43675491, 0.43710467, 0.43758050,
     &   0.43805769, 0.43889073, 0.43972120, 0.44057271, 0.44136351,
     &   0.44196999, 0.44263929, 0.44338575, 0.44420338, 0.44507301,
     &   0.44600129, 0.44690421, 0.44774032, 0.44852549, 0.44930539,
     &   0.45007688, 0.45128375, 0.45266217, 0.45388472, 0.45509908,
     &   0.45641956, 0.45789796, 0.45915303, 0.46017474, 0.46121436,
     &   0.46229985, 0.46306923, 0.46399617, 0.46507964, 0.46594489,
     &   0.46650687, 0.46690693, 0.46681890, 0.46668798, 0.46651277,
     &   0.46629190, 0.46602389, 0.46411601, 0.46177402, 0.45933539,
     &   0.45679948, 0.45416573, 0.45144036, 0.44862196, 0.44570914,
     &   0.44270119, 0.43958747, 0.43635434, 0.43302307, 0.42959279,
     &   0.42599064, 0.42181838, 0.41755345, 0.41316116, 0.40866005,
     &   0.40400085, 0.39923838, 0.39437270, 0.38935030, 0.38418511,
     &   0.37891829, 0.37347856, 0.36762190, 0.36167359, 0.35563606,
     &   0.34972176, 0.34374201, 0.33768234, 0.33177167, 0.32584321,
     &   0.31984174, 0.31370595, 0.30750200, 0.30123872, 0.29491660,
     &   0.28854883, 0.28216323, 0.27575970, 0.26932919, 0.26286006,
     &   0.25637922, 0.25007147, 0.24386452, 0.23767109, 0.23151650,
     &   0.22536871, 0.21913663, 0.21293789, 0.20676257, 0.20063575,
     &   0.19450325, 0.18842950, 0.18242370, 0.17649744, 0.17063391,
     &   0.16486311, 0.15919085, 0.15362130, 0.14815830, 0.14280744/
      data (ww(i,9),i=1,mxwv)/
     &   0.99999404, 0.99999464, 0.99999475, 0.99999559, 0.99999607,
     &   0.99999624, 0.99999654, 0.99999666, 0.99999684, 0.99999708,
     &   0.99999726, 0.99999744, 0.99999768, 0.99999785, 0.99999774,
     &   0.99999809, 0.99999827, 0.99999845, 0.99999857, 0.99999869,
     &   0.99999881, 0.99999893, 0.99999905, 0.99999917, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999911, 0.99999887, 0.99999875,
     &   0.99999863, 0.99999845, 0.99999815, 0.99999785, 0.99999744,
     &   0.99999636, 0.99999446, 0.99999261, 0.99999166, 0.99999130,
     &   0.99999106, 0.99999034, 0.99998897, 0.99998784, 0.99998546,
     &   0.99998188, 0.99997008, 0.99994898, 0.99992692, 0.99992388,
     &   0.99992508, 0.99993426, 0.99994141, 0.99993998, 0.99992335,
     &   0.99989915, 0.99987030, 0.99984717, 0.99982709, 0.99980617,
     &   0.99973416, 0.99942654, 0.99896801, 0.99871832, 0.99875355,
     &   0.99889970, 0.99911547, 0.99934524, 0.99951988, 0.99956429,
     &   0.99948490, 0.99931473, 0.99875510, 0.99719375, 0.99656361,
     &   0.99644774, 0.99648577, 0.99665350, 0.99689418, 0.99681121,
     &   0.99606395, 0.99454731, 0.98977774, 0.98470414, 0.96713376,
     &   0.92733806, 0.92149788, 0.93006569, 0.95011801, 0.96320558,
     &   0.97134346, 0.97659338, 0.98080754, 0.98305422, 0.98446804,
     &   0.98412359, 0.98248798, 0.97970134, 0.97487390, 0.97397560,
     &   0.96123880, 0.86875015, 0.77265191, 0.77831656, 0.82735258,
     &   0.87873805, 0.90701669, 0.92878002, 0.94100970, 0.94760847,
     &   0.94960958, 0.94565701, 0.93708622, 0.92492121, 0.90442699,
     &   0.87717646, 0.84091264, 0.80649567, 0.79549813, 0.77560693,
     &   0.68449700, 0.54871678, 0.50434560, 0.50493133, 0.51308179,
     &   0.52368975, 0.53095746, 0.53362697, 0.53519613, 0.53526932,
     &   0.53390634, 0.53193331, 0.53018475, 0.53075910, 0.53834158,
     &   0.56128734, 0.60773057, 0.66828561, 0.72060710, 0.75891036,
     &   0.77460575, 0.77478981, 0.76517802, 0.74997932, 0.72840691,
     &   0.70910048, 0.68656749, 0.66454720, 0.64252847, 0.62200046,
     &   0.60417295, 0.58837819, 0.57722980, 0.57747865, 0.58499628,
     &   0.59679538, 0.60985690, 0.62638408, 0.64270645, 0.65521634,
     &   0.65817130, 0.64749140, 0.62755817, 0.59349418, 0.55094743,
     &   0.51251608, 0.50073999, 0.50425029, 0.50891298, 0.51045990,
     &   0.51587820, 0.52185225, 0.52787751, 0.53257489, 0.53516585,
     &   0.53693587, 0.53807092, 0.53863579, 0.53923851, 0.53960818,
     &   0.53996527, 0.53999466, 0.53966260, 0.53960264, 0.53941596,
     &   0.53917396, 0.53895575, 0.53866750, 0.53837222, 0.53804654,
     &   0.53789169, 0.53794116, 0.53822345, 0.53894281, 0.53982806,
     &   0.54108602, 0.54187447, 0.54088992, 0.53577495, 0.52640712,
     &   0.51261115, 0.49915206, 0.48585257, 0.47674894, 0.47262692,
     &   0.47235355, 0.47552928, 0.47947150, 0.48344275, 0.48661661,
     &   0.48932725, 0.49162400, 0.49350846, 0.49484307, 0.49600008,
     &   0.49684086, 0.49736652, 0.49790561, 0.49819306, 0.49839520,
     &   0.49859226, 0.49874216, 0.49883372, 0.49873531, 0.49863249,
     &   0.49847269, 0.49807492, 0.49775848, 0.49720061, 0.49674633,
     &   0.49630898, 0.49560115, 0.49489620, 0.49414292, 0.49327517,
     &   0.49222109, 0.49145856, 0.49072209, 0.49002972, 0.48937556,
     &   0.48874521, 0.48816031, 0.48760155, 0.48701799, 0.48638541,
     &   0.48580870, 0.48523387, 0.48471850, 0.48424196, 0.48383030,
     &   0.48348591, 0.48322001, 0.48305288, 0.48298237, 0.48303759,
     &   0.48322356, 0.48353454, 0.48391443, 0.48435351, 0.48473415,
     &   0.48506218, 0.48530582, 0.48540068, 0.48540348, 0.48532408,
     &   0.48512626, 0.48471555, 0.48402339, 0.48310295, 0.48206130,
     &   0.48063084, 0.47879905, 0.47686487, 0.47494754, 0.47305602,
     &   0.47099811, 0.46895677, 0.46706593, 0.46544430, 0.46393085,
     &   0.46256009, 0.46135595, 0.46032673, 0.45949793, 0.45906067,
     &   0.45872283, 0.45837650, 0.45809096, 0.45791757, 0.45780969,
     &   0.45801291, 0.45819938, 0.45836237, 0.45851943, 0.45874882,
     &   0.45898536, 0.45941672, 0.45985907, 0.46033314, 0.46079540,
     &   0.46117935, 0.46161947, 0.46212626, 0.46270213, 0.46333757,
     &   0.46405247, 0.46479201, 0.46553540, 0.46629158, 0.46707642,
     &   0.46787009, 0.46899712, 0.47029966, 0.47158030, 0.47291282,
     &   0.47438663, 0.47605723, 0.47768718, 0.47921264, 0.48082471,
     &   0.48254940, 0.48410562, 0.48583555, 0.48777831, 0.48964730,
     &   0.49135771, 0.49301913, 0.49442053, 0.49582791, 0.49723774,
     &   0.49864665, 0.50005138, 0.50007892, 0.49974161, 0.49932608,
     &   0.49883074, 0.49825415, 0.49752435, 0.49670503, 0.49580607,
     &   0.49482647, 0.49375695, 0.49258560, 0.49133003, 0.48998946,
     &   0.48849440, 0.48646173, 0.48433390, 0.48211044, 0.47976923,
     &   0.47725648, 0.47464770, 0.47194314, 0.46909025, 0.46610066,
     &   0.46301225, 0.45974633, 0.45603374, 0.45222092, 0.44830763,
     &   0.44445652, 0.44052249, 0.43649137, 0.43259045, 0.42865169,
     &   0.42461532, 0.42040977, 0.41609955, 0.41168573, 0.40715492,
     &   0.40252465, 0.39781508, 0.39302084, 0.38812730, 0.38308606,
     &   0.37794676, 0.37291804, 0.36791551, 0.36283702, 0.35771403,
     &   0.35248733, 0.34703451, 0.34149152, 0.33584082, 0.33010495,
     &   0.32419968, 0.31820750, 0.31213710, 0.30599910, 0.29975379,
     &   0.29345250, 0.28710207, 0.28070864, 0.27427855, 0.26781899/
      data (ww(i,10),i=1,mxwv)/
     &   0.99999154, 0.99999249, 0.99999315, 0.99999374, 0.99999410,
     &   0.99999481, 0.99999487, 0.99999535, 0.99999547, 0.99999595,
     &   0.99999595, 0.99999642, 0.99999660, 0.99999696, 0.99999720,
     &   0.99999738, 0.99999756, 0.99999774, 0.99999803, 0.99999815,
     &   0.99999833, 0.99999851, 0.99999869, 0.99999887, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999875, 0.99999851, 0.99999833,
     &   0.99999809, 0.99999779, 0.99999714, 0.99999690, 0.99999636,
     &   0.99999493, 0.99999249, 0.99998987, 0.99998844, 0.99998772,
     &   0.99998736, 0.99998623, 0.99998432, 0.99998337, 0.99998015,
     &   0.99997413, 0.99995822, 0.99993020, 0.99989992, 0.99989367,
     &   0.99989611, 0.99990797, 0.99991739, 0.99991751, 0.99989474,
     &   0.99985975, 0.99981785, 0.99978757, 0.99975973, 0.99973071,
     &   0.99962676, 0.99921668, 0.99855906, 0.99821609, 0.99826503,
     &   0.99846816, 0.99876815, 0.99908876, 0.99933749, 0.99939471,
     &   0.99929035, 0.99905360, 0.99827147, 0.99610591, 0.99522638,
     &   0.99506927, 0.99512196, 0.99534708, 0.99567366, 0.99557096,
     &   0.99454027, 0.99244213, 0.98585135, 0.97886097, 0.95490962,
     &   0.90218180, 0.89463973, 0.90574646, 0.93213814, 0.94965965,
     &   0.96064270, 0.96779346, 0.97354442, 0.97671461, 0.97856516,
     &   0.97809803, 0.97585803, 0.97203457, 0.96548039, 0.96426469,
     &   0.94706756, 0.82881528, 0.71952885, 0.72557294, 0.77997142,
     &   0.84104294, 0.87632841, 0.90425920, 0.92024285, 0.92895001,
     &   0.93161029, 0.92637753, 0.91510540, 0.89926368, 0.87306631,
     &   0.83912361, 0.79562330, 0.75616127, 0.74388063, 0.72217172,
     &   0.63179404, 0.52959031, 0.50758821, 0.50938952, 0.51760924,
     &   0.52790254, 0.53501958, 0.53769839, 0.53929943, 0.53942776,
     &   0.53814071, 0.53624463, 0.53438050, 0.53303868, 0.53451931,
     &   0.54466611, 0.57336801, 0.62064832, 0.66825229, 0.70647293,
     &   0.72286558, 0.72301728, 0.71283954, 0.69709510, 0.67542171,
     &   0.65673369, 0.63581055, 0.61632711, 0.59786159, 0.58162230,
     &   0.56834280, 0.55725783, 0.54988527, 0.54997671, 0.55480009,
     &   0.56275725, 0.57203341, 0.58446050, 0.59738922, 0.60768461,
     &   0.61006331, 0.60089445, 0.58450639, 0.55864626, 0.53052944,
     &   0.51033640, 0.50665969, 0.51177692, 0.51659578, 0.51598895,
     &   0.51661372, 0.51827407, 0.52041686, 0.52222663, 0.52314138,
     &   0.52370220, 0.52395427, 0.52392316, 0.52390730, 0.52375799,
     &   0.52357739, 0.52319080, 0.52257162, 0.52206910, 0.52144396,
     &   0.52073109, 0.51995981, 0.51906615, 0.51807833, 0.51698244,
     &   0.51584721, 0.51471019, 0.51359105, 0.51258242, 0.51161897,
     &   0.51092780, 0.51029193, 0.50959748, 0.50818676, 0.50570929,
     &   0.50118840, 0.49608642, 0.49053922, 0.48687029, 0.48602432,
     &   0.48741135, 0.49109644, 0.49503064, 0.49877843, 0.50173390,
     &   0.50431311, 0.50648063, 0.50827461, 0.50955003, 0.51068044,
     &   0.51150548, 0.51204520, 0.51261473, 0.51293415, 0.51317507,
     &   0.51343042, 0.51362926, 0.51377177, 0.51371872, 0.51367104,
     &   0.51355565, 0.51318949, 0.51291168, 0.51237661, 0.51195949,
     &   0.51155007, 0.51083016, 0.51011050, 0.50933880, 0.50842458,
     &   0.50727171, 0.50646877, 0.50567830, 0.50492328, 0.50419760,
     &   0.50348127, 0.50279832, 0.50212294, 0.50138062, 0.50049812,
     &   0.49963945, 0.49874625, 0.49787250, 0.49696398, 0.49605134,
     &   0.49512431, 0.49417606, 0.49324512, 0.49231893, 0.49141866,
     &   0.49051768, 0.48966935, 0.48886353, 0.48813403, 0.48747391,
     &   0.48688352, 0.48634976, 0.48586416, 0.48540962, 0.48497814,
     &   0.48455444, 0.48412001, 0.48365369, 0.48315015, 0.48261616,
     &   0.48199400, 0.48126784, 0.48052007, 0.47976378, 0.47900230,
     &   0.47819948, 0.47742113, 0.47670799, 0.47608343, 0.47551212,
     &   0.47501308, 0.47458851, 0.47424036, 0.47401908, 0.47389984,
     &   0.47377008, 0.47362068, 0.47351608, 0.47348842, 0.47346413,
     &   0.47345230, 0.47342825, 0.47334927, 0.47326836, 0.47321311,
     &   0.47315633, 0.47310504, 0.47305265, 0.47294831, 0.47285077,
     &   0.47273803, 0.47264880, 0.47258931, 0.47255245, 0.47254822,
     &   0.47256935, 0.47261894, 0.47269148, 0.47279370, 0.47293845,
     &   0.47312281, 0.47345018, 0.47387952, 0.47434682, 0.47488499,
     &   0.47553566, 0.47633493, 0.47719988, 0.47809085, 0.47909930,
     &   0.48024616, 0.48139265, 0.48270515, 0.48423266, 0.48583215,
     &   0.48744982, 0.48914790, 0.49084195, 0.49262810, 0.49450317,
     &   0.49646378, 0.49850613, 0.49969435, 0.50067568, 0.50163859,
     &   0.50257909, 0.50349319, 0.50425279, 0.50495934, 0.50562865,
     &   0.50626075, 0.50683790, 0.50735843, 0.50782931, 0.50824797,
     &   0.50855482, 0.50842553, 0.50822258, 0.50794423, 0.50756317,
     &   0.50701421, 0.50638485, 0.50567383, 0.50483686, 0.50388187,
     &   0.50283968, 0.50163400, 0.50000316, 0.49827811, 0.49645847,
     &   0.49463409, 0.49272847, 0.49073365, 0.48884690, 0.48692361,
     &   0.48491400, 0.48275119, 0.48049387, 0.47814044, 0.47567013,
     &   0.47310922, 0.47047171, 0.46775335, 0.46494353, 0.46196574,
     &   0.45889407, 0.45592174, 0.45297226, 0.44995612, 0.44691068,
     &   0.44376299, 0.44039291, 0.43692589, 0.43334460, 0.42966378,
     &   0.42578912, 0.42180189, 0.41770780, 0.41351065, 0.40914690,
     &   0.40467870, 0.40010694, 0.39543277, 0.39065722, 0.38577753/
      data (ww(i,11),i=1,mxwv)/
     &   0.99998826, 0.99998951, 0.99999034, 0.99999130, 0.99999213,
     &   0.99999267, 0.99999315, 0.99999350, 0.99999368, 0.99999428,
     &   0.99999464, 0.99999493, 0.99999541, 0.99999571, 0.99999601,
     &   0.99999630, 0.99999660, 0.99999690, 0.99999714, 0.99999744,
     &   0.99999756, 0.99999791, 0.99999809, 0.99999833, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999827, 0.99999791, 0.99999768,
     &   0.99999726, 0.99999696, 0.99999648, 0.99999589, 0.99999493,
     &   0.99999285, 0.99998951, 0.99998581, 0.99998379, 0.99998283,
     &   0.99998248, 0.99998111, 0.99997783, 0.99997628, 0.99997324,
     &   0.99996394, 0.99994171, 0.99990135, 0.99986047, 0.99985313,
     &   0.99985576, 0.99987149, 0.99988687, 0.99988478, 0.99985260,
     &   0.99980354, 0.99974012, 0.99970430, 0.99966669, 0.99962330,
     &   0.99948180, 0.99891329, 0.99799120, 0.99751437, 0.99758202,
     &   0.99786550, 0.99828392, 0.99872899, 0.99907577, 0.99915826,
     &   0.99900669, 0.99867791, 0.99759334, 0.99457920, 0.99335223,
     &   0.99313205, 0.99321020, 0.99353194, 0.99399441, 0.99383706,
     &   0.99240619, 0.98948658, 0.98038191, 0.97075796, 0.93834168,
     &   0.86975056, 0.86027735, 0.87428302, 0.90826225, 0.93136489,
     &   0.94606501, 0.95572597, 0.96363080, 0.96774822, 0.97040725,
     &   0.96976870, 0.96671301, 0.96150815, 0.95262158, 0.95098943,
     &   0.92799157, 0.78130013, 0.66559464, 0.67143440, 0.72704762,
     &   0.79553103, 0.83779490, 0.87257665, 0.89298767, 0.90426397,
     &   0.90773314, 0.90092915, 0.88640666, 0.86628938, 0.83381474,
     &   0.79324943, 0.74386543, 0.70172065, 0.68907005, 0.66737163,
     &   0.58727348, 0.52125454, 0.51085520, 0.51290441, 0.52090764,
     &   0.53088248, 0.53787529, 0.54057050, 0.54220438, 0.54238307,
     &   0.54116094, 0.53933865, 0.53751868, 0.53574479, 0.53491622,
     &   0.53783709, 0.55182058, 0.58272618, 0.62041819, 0.65449154,
     &   0.66999578, 0.67007560, 0.66025966, 0.64548475, 0.62599725,
     &   0.60996044, 0.59293437, 0.57801348, 0.56478149, 0.55394202,
     &   0.54569107, 0.53928715, 0.53527886, 0.53524321, 0.53767914,
     &   0.54197127, 0.54731357, 0.55499411, 0.56352991, 0.57063991,
     &   0.57217115, 0.56534159, 0.55380243, 0.53726029, 0.52184290,
     &   0.51299405, 0.51249599, 0.51784253, 0.52258897, 0.52146447,
     &   0.52037179, 0.52008021, 0.52019638, 0.52035844, 0.52028197,
     &   0.52012175, 0.51986271, 0.51951057, 0.51917118, 0.51877660,
     &   0.51836056, 0.51784754, 0.51722413, 0.51664025, 0.51598096,
     &   0.51526064, 0.51448327, 0.51361877, 0.51266956, 0.51163274,
     &   0.51051986, 0.50934023, 0.50809717, 0.50679797, 0.50541514,
     &   0.50404149, 0.50255233, 0.50123072, 0.49987581, 0.49881411,
     &   0.49759549, 0.49626884, 0.49493143, 0.49461335, 0.49593323,
     &   0.49843776, 0.50252366, 0.50650394, 0.51016790, 0.51304650,
     &   0.51559108, 0.51773584, 0.51953286, 0.52083665, 0.52201247,
     &   0.52289498, 0.52350938, 0.52415973, 0.52456617, 0.52489865,
     &   0.52525240, 0.52554828, 0.52578992, 0.52583832, 0.52589554,
     &   0.52588326, 0.52562135, 0.52544898, 0.52502024, 0.52471268,
     &   0.52441055, 0.52379322, 0.52317727, 0.52251107, 0.52169913,
     &   0.52064121, 0.51994342, 0.51925528, 0.51860154, 0.51797473,
     &   0.51735491, 0.51676661, 0.51618659, 0.51552397, 0.51470715,
     &   0.51390791, 0.51306248, 0.51222670, 0.51133692, 0.51042408,
     &   0.50947207, 0.50846666, 0.50744760, 0.50639313, 0.50532103,
     &   0.50418299, 0.50301391, 0.50188947, 0.50076807, 0.49969733,
     &   0.49868780, 0.49773532, 0.49686292, 0.49603325, 0.49523497,
     &   0.49446154, 0.49372375, 0.49304709, 0.49241284, 0.49177986,
     &   0.49121657, 0.49072316, 0.49033463, 0.48996609, 0.48961225,
     &   0.48933128, 0.48914230, 0.48902327, 0.48892611, 0.48887861,
     &   0.48889205, 0.48894811, 0.48904189, 0.48925638, 0.48941678,
     &   0.48948872, 0.48952365, 0.48960450, 0.48976126, 0.48986524,
     &   0.48984149, 0.48978671, 0.48963889, 0.48948768, 0.48933986,
     &   0.48918116, 0.48891234, 0.48862675, 0.48820338, 0.48778403,
     &   0.48736411, 0.48693687, 0.48650199, 0.48604777, 0.48558843,
     &   0.48508239, 0.48457673, 0.48406851, 0.48356435, 0.48308903,
     &   0.48267010, 0.48222694, 0.48178303, 0.48133978, 0.48092932,
     &   0.48054633, 0.48019201, 0.47986147, 0.47958946, 0.47937834,
     &   0.47923756, 0.47915673, 0.47916785, 0.47928381, 0.47948849,
     &   0.47977671, 0.48016873, 0.48065081, 0.48125663, 0.48199022,
     &   0.48285523, 0.48385507, 0.48462045, 0.48537570, 0.48619249,
     &   0.48706883, 0.48800245, 0.48889479, 0.48981449, 0.49077284,
     &   0.49176708, 0.49279159, 0.49383947, 0.49491304, 0.49600902,
     &   0.49708924, 0.49794662, 0.49879551, 0.49963215, 0.50043100,
     &   0.50113046, 0.50180322, 0.50244606, 0.50302738, 0.50354880,
     &   0.50402778, 0.50440162, 0.50445557, 0.50444722, 0.50437433,
     &   0.50424838, 0.50406104, 0.50380951, 0.50363374, 0.50343323,
     &   0.50317001, 0.50279057, 0.50233907, 0.50181055, 0.50117886,
     &   0.50047457, 0.49970368, 0.49886355, 0.49794662, 0.49686217,
     &   0.49569660, 0.49460500, 0.49352908, 0.49240464, 0.49127194,
     &   0.49005523, 0.48865545, 0.48717344, 0.48559779, 0.48393694,
     &   0.48210391, 0.48017195, 0.47814772, 0.47603261, 0.47375360,
     &   0.47138065, 0.46891433, 0.46635649, 0.46370876, 0.46096510/
      data (ww(i,12),i=1,mxwv)/
     &   0.99998367, 0.99998516, 0.99998641, 0.99998671, 0.99998891,
     &   0.99998975, 0.99999005, 0.99999088, 0.99999136, 0.99999195,
     &   0.99999237, 0.99999303, 0.99999356, 0.99999398, 0.99999440,
     &   0.99999475, 0.99999523, 0.99999565, 0.99999595, 0.99999642,
     &   0.99999654, 0.99999708, 0.99999738, 0.99999774, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999750, 0.99999702, 0.99999672,
     &   0.99999613, 0.99999571, 0.99999505, 0.99999434, 0.99999261,
     &   0.99999011, 0.99998516, 0.99998015, 0.99997747, 0.99997598,
     &   0.99997556, 0.99997342, 0.99996805, 0.99996746, 0.99996263,
     &   0.99994940, 0.99991870, 0.99986297, 0.99980479, 0.99979115,
     &   0.99979824, 0.99982071, 0.99984294, 0.99983919, 0.99979413,
     &   0.99972707, 0.99964803, 0.99958742, 0.99953276, 0.99947369,
     &   0.99927741, 0.99847138, 0.99719435, 0.99652433, 0.99661607,
     &   0.99701500, 0.99760425, 0.99822330, 0.99870783, 0.99882370,
     &   0.99861240, 0.99815071, 0.99664146, 0.99242145, 0.99072492,
     &   0.99041867, 0.99053085, 0.99097902, 0.99161476, 0.99137789,
     &   0.98942226, 0.98537213, 0.97278976, 0.95964092, 0.91620308,
     &   0.82934898, 0.81791693, 0.83488947, 0.87728554, 0.90709072,
     &   0.92645705, 0.93934751, 0.94987309, 0.95558423, 0.95918828,
     &   0.95832604, 0.95418096, 0.94714957, 0.93522829, 0.93305647,
     &   0.90277272, 0.72818893, 0.61663979, 0.62161571, 0.67265534,
     &   0.74365085, 0.79146600, 0.83295405, 0.85816348, 0.87235892,
     &   0.87676525, 0.86814731, 0.84997398, 0.82531673, 0.78680372,
     &   0.74102783, 0.68900931, 0.64806253, 0.63633054, 0.61697817,
     &   0.55529356, 0.51913083, 0.51342607, 0.51549911, 0.52323282,
     &   0.53292847, 0.53982377, 0.54253823, 0.54420334, 0.54442590,
     &   0.54325706, 0.54149371, 0.53973138, 0.53792590, 0.53648901,
     &   0.53642637, 0.54132771, 0.55749613, 0.58249271, 0.60871667,
     &   0.62156630, 0.62153476, 0.61314666, 0.60097593, 0.58575076,
     &   0.57393336, 0.56218159, 0.55260664, 0.54475439, 0.53882056,
     &   0.53464520, 0.53164095, 0.52988291, 0.52978468, 0.53067207,
     &   0.53239948, 0.53473121, 0.53842795, 0.54288268, 0.54677999,
     &   0.54744238, 0.54310179, 0.53626949, 0.52745527, 0.52028865,
     &   0.51664722, 0.51711321, 0.52239591, 0.52704054, 0.52587491,
     &   0.52438992, 0.52349859, 0.52287918, 0.52237952, 0.52189505,
     &   0.52143091, 0.52096063, 0.52048671, 0.52002162, 0.51954073,
     &   0.51904315, 0.51850718, 0.51792824, 0.51735377, 0.51673776,
     &   0.51608467, 0.51538956, 0.51464206, 0.51383883, 0.51298213,
     &   0.51206148, 0.51107579, 0.51002443, 0.50887591, 0.50762594,
     &   0.50628752, 0.50476021, 0.50336117, 0.50190330, 0.50089693,
     &   0.50034648, 0.50011188, 0.50035065, 0.50150532, 0.50384003,
     &   0.50685602, 0.51108670, 0.51504248, 0.51863509, 0.52146900,
     &   0.52399635, 0.52614397, 0.52796543, 0.52931750, 0.53055203,
     &   0.53150576, 0.53220356, 0.53293908, 0.53343862, 0.53386754,
     &   0.53431964, 0.53471518, 0.53505844, 0.53521305, 0.53537756,
     &   0.53547335, 0.53532451, 0.53526402, 0.53495216, 0.53476071,
     &   0.53457385, 0.53407598, 0.53358060, 0.53303754, 0.53235155,
     &   0.53142375, 0.53085399, 0.53029364, 0.52976733, 0.52926815,
     &   0.52877611, 0.52831501, 0.52786243, 0.52732784, 0.52663970,
     &   0.52596855, 0.52525204, 0.52454430, 0.52378142, 0.52299386,
     &   0.52216494, 0.52127844, 0.52037382, 0.51942778, 0.51845640,
     &   0.51740730, 0.51634568, 0.51525152, 0.51417619, 0.51313752,
     &   0.51214772, 0.51120228, 0.51033401, 0.50949842, 0.50868434,
     &   0.50788593, 0.50711751, 0.50642282, 0.50577909, 0.50513494,
     &   0.50459474, 0.50417155, 0.50390625, 0.50367826, 0.50347930,
     &   0.50340003, 0.50345218, 0.50359178, 0.50374365, 0.50395268,
     &   0.50422800, 0.50454181, 0.50488526, 0.50535673, 0.50572324,
     &   0.50597090, 0.50617272, 0.50642306, 0.50675279, 0.50703073,
     &   0.50709087, 0.50713426, 0.50706822, 0.50699782, 0.50692517,
     &   0.50683814, 0.50660378, 0.50634700, 0.50591826, 0.50549072,
     &   0.50506252, 0.50461549, 0.50414753, 0.50364155, 0.50311673,
     &   0.50251251, 0.50188839, 0.50125057, 0.50059307, 0.49995330,
     &   0.49937397, 0.49870476, 0.49798447, 0.49722430, 0.49646413,
     &   0.49567708, 0.49484241, 0.49396318, 0.49311718, 0.49226695,
     &   0.49141058, 0.49058509, 0.48977375, 0.48894894, 0.48813671,
     &   0.48735335, 0.48659441, 0.48583680, 0.48513409, 0.48449144,
     &   0.48391429, 0.48340845, 0.48296252, 0.48256728, 0.48222676,
     &   0.48194385, 0.48172176, 0.48156258, 0.48145896, 0.48141161,
     &   0.48142233, 0.48149225, 0.48162180, 0.48181364, 0.48206899,
     &   0.48237854, 0.48267868, 0.48302883, 0.48342839, 0.48386738,
     &   0.48431823, 0.48480886, 0.48533773, 0.48589200, 0.48647061,
     &   0.48707908, 0.48768571, 0.48817801, 0.48867664, 0.48917863,
     &   0.48965901, 0.49013594, 0.49060857, 0.49114311, 0.49169242,
     &   0.49223676, 0.49274421, 0.49323672, 0.49370763, 0.49413335,
     &   0.49453697, 0.49491659, 0.49526981, 0.49559346, 0.49580130,
     &   0.49597004, 0.49619228, 0.49643481, 0.49666432, 0.49691519,
     &   0.49712375, 0.49722409, 0.49727795, 0.49727952, 0.49722940,
     &   0.49705952, 0.49682295, 0.49652508, 0.49616399, 0.49566761,
     &   0.49510157, 0.49446541, 0.49376118, 0.49299094, 0.49214512/
      data (ww(i,13),i=1,mxwv)/
     &   0.99997699, 0.99997914, 0.99998087, 0.99998271, 0.99998432,
     &   0.99998552, 0.99998641, 0.99998713, 0.99998784, 0.99998844,
     &   0.99998939, 0.99999022, 0.99999082, 0.99999154, 0.99999213,
     &   0.99999267, 0.99999332, 0.99999380, 0.99999422, 0.99999493,
     &   0.99999529, 0.99999589, 0.99999619, 0.99999678, 1.00000000,
     &   1.00000000, 1.00000000, 1.00000000, 1.00000000, 1.00000000,
     &   1.00000000, 1.00000000, 0.99999654, 0.99999577, 0.99999541,
     &   0.99999475, 0.99999398, 0.99999309, 0.99999183, 0.99998987,
     &   0.99998599, 0.99997926, 0.99997205, 0.99996847, 0.99996632,
     &   0.99996573, 0.99996263, 0.99995756, 0.99995434, 0.99994737,
     &   0.99992925, 0.99988627, 0.99980742, 0.99972659, 0.99971008,
     &   0.99971664, 0.99974847, 0.99977845, 0.99977440, 0.99971014,
     &   0.99961638, 0.99950755, 0.99942231, 0.99934632, 0.99926257,
     &   0.99898750, 0.99785775, 0.99606615, 0.99513042, 0.99526417,
     &   0.99581885, 0.99664354, 0.99751544, 0.99819231, 0.99835342,
     &   0.99805808, 0.99741244, 0.99529612, 0.98941237, 0.98704678,
     &   0.98662060, 0.98677868, 0.98740399, 0.98830020, 0.98799449,
     &   0.98522592, 0.97961640, 0.96232492, 0.94450611, 0.88726640,
     &   0.78134179, 0.76828617, 0.78776211, 0.83839124, 0.87564391,
     &   0.90055680, 0.91744858, 0.93141639, 0.93904877, 0.94391155,
     &   0.94275421, 0.93719256, 0.92781103, 0.91206563, 0.90922564,
     &   0.87025243, 0.67358041, 0.57815468, 0.58176100, 0.62253773,
     &   0.68885893, 0.73896134, 0.78560424, 0.81530929, 0.83247226,
     &   0.83786315, 0.82734984, 0.80553913, 0.77677506, 0.73380107,
     &   0.68600887, 0.63639218, 0.60120153, 0.59166449, 0.57667440,
     &   0.53647351, 0.51946235, 0.51529539, 0.51734155, 0.52482080,
     &   0.53428578, 0.54110628, 0.54383987, 0.54553270, 0.54579216,
     &   0.54466587, 0.54294807, 0.54123217, 0.53946346, 0.53793836,
     &   0.53699470, 0.53776848, 0.54409206, 0.55735433, 0.57408190,
     &   0.58304834, 0.58288193, 0.57670957, 0.56817120, 0.55816764,
     &   0.55091363, 0.54423040, 0.53920668, 0.53541619, 0.53276902,
     &   0.53103119, 0.52985555, 0.52920467, 0.52909732, 0.52926201,
     &   0.52968055, 0.53033358, 0.53156704, 0.53323275, 0.53474528,
     &   0.53474915, 0.53235066, 0.52888763, 0.52482665, 0.52165365,
     &   0.51982540, 0.52052975, 0.52569902, 0.53024846, 0.52913237,
     &   0.52763742, 0.52665609, 0.52588081, 0.52522361, 0.52464688,
     &   0.52412492, 0.52362269, 0.52314711, 0.52267784, 0.52220684,
     &   0.52172065, 0.52121770, 0.52069759, 0.52016979, 0.51961285,
     &   0.51902980, 0.51841223, 0.51775807, 0.51706308, 0.51633191,
     &   0.51554906, 0.51471114, 0.51381814, 0.51283193, 0.51175648,
     &   0.51057911, 0.50920898, 0.50793040, 0.50652736, 0.50550497,
     &   0.50504297, 0.50505602, 0.50576413, 0.50742036, 0.51009744,
     &   0.51325291, 0.51745033, 0.52132201, 0.52483356, 0.52762604,
     &   0.53013498, 0.53228742, 0.53413320, 0.53553152, 0.53681993,
     &   0.53783953, 0.53861254, 0.53942364, 0.54000670, 0.54052180,
     &   0.54106021, 0.54154438, 0.54197824, 0.54222816, 0.54248822,
     &   0.54268128, 0.54263502, 0.54267538, 0.54246980, 0.54238254,
     &   0.54229957, 0.54191178, 0.54152656, 0.54109567, 0.54052532,
     &   0.53971875, 0.53926480, 0.53882104, 0.53841120, 0.53802848,
     &   0.53765333, 0.53730905, 0.53697371, 0.53655839, 0.53599262,
     &   0.53544432, 0.53485334, 0.53427172, 0.53363669, 0.53297794,
     &   0.53227925, 0.53152478, 0.53075349, 0.52994215, 0.52910578,
     &   0.52819300, 0.52726769, 0.52630955, 0.52536845, 0.52446365,
     &   0.52360415, 0.52278620, 0.52204347, 0.52132982, 0.52063495,
     &   0.51995260, 0.51929742, 0.51871455, 0.51818156, 0.51764596,
     &   0.51721650, 0.51690984, 0.51677066, 0.51667279, 0.51660782,
     &   0.51667255, 0.51687872, 0.51717865, 0.51749021, 0.51786232,
     &   0.51830304, 0.51878226, 0.51928973, 0.51992661, 0.52044719,
     &   0.52084255, 0.52119052, 0.52158695, 0.52206236, 0.52248263,
     &   0.52267432, 0.52284706, 0.52290916, 0.52296698, 0.52302158,
     &   0.52306092, 0.52294701, 0.52281052, 0.52249783, 0.52218646,
     &   0.52187479, 0.52154326, 0.52118880, 0.52079451, 0.52037787,
     &   0.51987779, 0.51935685, 0.51881534, 0.51825202, 0.51770395,
     &   0.51721668, 0.51662940, 0.51598221, 0.51528519, 0.51458043,
     &   0.51383698, 0.51302892, 0.51215541, 0.51130283, 0.51042604,
     &   0.50951910, 0.50862426, 0.50771803, 0.50675750, 0.50577450,
     &   0.50477928, 0.50377035, 0.50269592, 0.50162750, 0.50056690,
     &   0.49951559, 0.49847540, 0.49752092, 0.49658957, 0.49566665,
     &   0.49475387, 0.49385306, 0.49302822, 0.49222529, 0.49143639,
     &   0.49066344, 0.48990822, 0.48917246, 0.48845887, 0.48776966,
     &   0.48710936, 0.48648968, 0.48589623, 0.48533344, 0.48480418,
     &   0.48431745, 0.48386234, 0.48344052, 0.48305261, 0.48269993,
     &   0.48238465, 0.48210701, 0.48185655, 0.48163879, 0.48145369,
     &   0.48130804, 0.48119280, 0.48110735, 0.48106733, 0.48106393,
     &   0.48109344, 0.48114845, 0.48123422, 0.48134902, 0.48148555,
     &   0.48164955, 0.48183998, 0.48205581, 0.48229593, 0.48252445,
     &   0.48276937, 0.48306236, 0.48339355, 0.48375392, 0.48416081,
     &   0.48458549, 0.48500076, 0.48542881, 0.48586783, 0.48631513,
     &   0.48673546, 0.48715225, 0.48756766, 0.48797828, 0.48833558,
     &   0.48867881, 0.48900640, 0.48931867, 0.48961639, 0.48989040/

c         ---------------- asymmetry factor ---------------

      data (gg(i,1),i=1,mxwv)/
     &   0.82582545, 0.82592523, 0.82585710, 0.82588977, 0.82582527,
     &   0.82564992, 0.82560641, 0.82534742, 0.82520568, 0.82506657,
     &   0.82481307, 0.82456845, 0.82425445, 0.82396966, 0.82363415,
     &   0.82320780, 0.82287830, 0.82244104, 0.82199508, 0.82151937,
     &   0.82103729, 0.82049131, 0.81991357, 0.81933504, 0.81872535,
     &   0.81806701, 0.81734812, 0.81660944, 0.81584901, 0.81504452,
     &   0.81418365, 0.81330740, 0.81237507, 0.81141019, 0.81038910,
     &   0.80933768, 0.80824000, 0.80711931, 0.80594260, 0.80473220,
     &   0.80347955, 0.80219209, 0.80086428, 0.79951847, 0.79814166,
     &   0.79673654, 0.79531223, 0.79386520, 0.79240590, 0.79093921,
     &   0.78946817, 0.78800279, 0.78654683, 0.78510952, 0.78369778,
     &   0.78231937, 0.78098428, 0.77969915, 0.77847517, 0.77732205,
     &   0.77624935, 0.77526689, 0.77438474, 0.77361232, 0.77296138,
     &   0.77244097, 0.77206379, 0.77183712, 0.77176172, 0.77184343,
     &   0.77209342, 0.77251720, 0.77311635, 0.77389461, 0.77485329,
     &   0.77599204, 0.77730536, 0.77879637, 0.78048605, 0.78226358,
     &   0.78420770, 0.78628397, 0.78848392, 0.79079866, 0.79321790,
     &   0.79573429, 0.79833770, 0.80104482, 0.80381238, 0.80679411,
     &   0.80994910, 0.81254768, 0.81496698, 0.81730813, 0.81980264,
     &   0.82232887, 0.82485163, 0.82733917, 0.82979083, 0.83220702,
     &   0.83462101, 0.83697563, 0.83926922, 0.84150988, 0.84367168,
     &   0.84602505, 0.84901446, 0.85165107, 0.85245973, 0.85309666,
     &   0.85411465, 0.85543901, 0.85682338, 0.85824859, 0.85968673,
     &   0.86113864, 0.86262619, 0.86414945, 0.86575198, 0.86746395,
     &   0.86934125, 0.87142861, 0.87368357, 0.87624252, 0.87963700,
     &   0.88497370, 0.89308161, 0.90238464, 0.90146482, 0.89375055,
     &   0.88199556, 0.87077630, 0.86228466, 0.85282397, 0.84088981,
     &   0.82518446, 0.81116998, 0.79965270, 0.79298598, 0.79247075,
     &   0.79226148, 0.79255801, 0.79326576, 0.79408979, 0.79461628,
     &   0.79487628, 0.79475111, 0.79429221, 0.79352623, 0.79250586,
     &   0.79118663, 0.78969026, 0.78798294, 0.78607655, 0.78397077,
     &   0.78159428, 0.77897614, 0.77590501, 0.77216172, 0.76814109,
     &   0.76407272, 0.76004201, 0.75599086, 0.75198478, 0.74809271,
     &   0.74431711, 0.74065375, 0.73707110, 0.73376703, 0.73080313,
     &   0.72807783, 0.72367293, 0.71595693, 0.70734733, 0.69888288,
     &   0.69204396, 0.68571329, 0.67933685, 0.67286748, 0.66625392,
     &   0.65941244, 0.65234286, 0.64503986, 0.63748878, 0.62969744,
     &   0.62165493, 0.61336565, 0.60482770, 0.59603113, 0.58697444,
     &   0.57765901, 0.56807870, 0.55823809, 0.54813796, 0.53778493,
     &   0.52719527, 0.51635122, 0.50527877, 0.49393955, 0.48236141,
     &   0.47054112, 0.45834744, 0.44618133, 0.43352759, 0.42102569,
     &   0.40871832, 0.39671010, 0.38475269, 0.37320104, 0.36203402,
     &   0.35123599, 0.34106970, 0.33149248, 0.32256624, 0.31413832,
     &   0.30581799, 0.29793465, 0.29031041, 0.28297898, 0.27577883,
     &   0.26885036, 0.26200894, 0.25526813, 0.24875644, 0.24239631,
     &   0.23606311, 0.23000668, 0.22412345, 0.21843770, 0.21282479,
     &   0.20746984, 0.20228088, 0.19719486, 0.19221528, 0.18725319,
     &   0.18251030, 0.17796002, 0.17343809, 0.16885823, 0.16430296,
     &   0.15972769, 0.15507665, 0.15053827, 0.14611103, 0.14179477,
     &   0.13758878, 0.13352397, 0.12956649, 0.12571009, 0.12194680,
     &   0.11827938, 0.11458509, 0.11099261, 0.10747156, 0.10403536,
     &   0.10066457, 0.09735073, 0.09411050, 0.09092317, 0.08781720,
     &   0.08475711, 0.08177216, 0.07883084, 0.07597638, 0.07319383,
     &   0.07050925, 0.06791995, 0.06543289, 0.06303990, 0.06073668,
     &   0.05851362, 0.05636254, 0.05429049, 0.05229382, 0.05036614,
     &   0.04850581, 0.04670938, 0.04500944, 0.04338047, 0.04181745,
     &   0.04031225, 0.03887641, 0.03750962, 0.03621034, 0.03496517,
     &   0.03377532, 0.03263836, 0.03155303, 0.03052563, 0.02956484,
     &   0.02863942, 0.02774037, 0.02687770, 0.02605632, 0.02526627,
     &   0.02452936, 0.02381342, 0.02311428, 0.02243586, 0.02178509,
     &   0.02115416, 0.02055400, 0.01997020, 0.01939676, 0.01883660,
     &   0.01828493, 0.01775069, 0.01723347, 0.01673143, 0.01624404,
     &   0.01576786, 0.01530308, 0.01484825, 0.01440405, 0.01397328,
     &   0.01355793, 0.01315873, 0.01277072, 0.01238773, 0.01201483,
     &   0.01165209, 0.01129850, 0.01094837, 0.01060642, 0.01027373,
     &   0.00995009, 0.00963359, 0.00932789, 0.00903054, 0.00873936,
     &   0.00845465, 0.00817702, 0.00790264, 0.00763738, 0.00738094,
     &   0.00713302, 0.00689335, 0.00665735, 0.00642852, 0.00620756,
     &   0.00599419, 0.00578815, 0.00559134, 0.00540154, 0.00521817,
     &   0.00504102, 0.00486984, 0.00470440, 0.00454457, 0.00439016,
     &   0.00424092, 0.00409623, 0.00395646, 0.00382146, 0.00369117,
     &   0.00356567, 0.00344443, 0.00332730, 0.00321406, 0.00310461,
     &   0.00299887, 0.00289673, 0.00279810, 0.00270280, 0.00261074,
     &   0.00252270, 0.00243769, 0.00235553, 0.00227646, 0.00220013,
     &   0.00212634, 0.00205495, 0.00198594, 0.00191926, 0.00185490,
     &   0.00179268, 0.00173259, 0.00167456, 0.00161845, 0.00156442,
     &   0.00151218, 0.00146191, 0.00141343, 0.00136652, 0.00132110,
     &   0.00127717, 0.00123457, 0.00119338, 0.00115352, 0.00111497,
     &   0.00107767, 0.00104160, 0.00100672, 0.00097300, 0.00094044,
     &   0.00090897, 0.00087854, 0.00084912, 0.00082066, 0.00079314/
      data (gg(i,2),i=1,mxwv)/
     &   0.83627009, 0.83659846, 0.83663005, 0.83670270, 0.83686882,
     &   0.83690906, 0.83694035, 0.83684474, 0.83677542, 0.83682454,
     &   0.83668655, 0.83658659, 0.83646691, 0.83634633, 0.83614570,
     &   0.83592373, 0.83578694, 0.83549982, 0.83532858, 0.83509028,
     &   0.83475798, 0.83446884, 0.83405447, 0.83384895, 0.83346409,
     &   0.83313030, 0.83276808, 0.83230066, 0.83194649, 0.83148319,
     &   0.83109748, 0.83063084, 0.83008826, 0.82956862, 0.82908487,
     &   0.82852161, 0.82798141, 0.82738441, 0.82677656, 0.82617378,
     &   0.82549828, 0.82482111, 0.82410330, 0.82337433, 0.82258445,
     &   0.82180250, 0.82095975, 0.82008660, 0.81918013, 0.81824225,
     &   0.81722403, 0.81621188, 0.81514990, 0.81403714, 0.81290424,
     &   0.81173629, 0.81051993, 0.80927426, 0.80800170, 0.80669385,
     &   0.80534750, 0.80398440, 0.80258876, 0.80117899, 0.79974592,
     &   0.79830289, 0.79685289, 0.79540467, 0.79394943, 0.79249275,
     &   0.79104722, 0.78961992, 0.78822207, 0.78685427, 0.78553617,
     &   0.78427351, 0.78307498, 0.78195798, 0.78094471, 0.78000134,
     &   0.77915174, 0.77841294, 0.77779859, 0.77732247, 0.77700287,
     &   0.77685970, 0.77690613, 0.77720606, 0.77771062, 0.77871221,
     &   0.78036481, 0.78132290, 0.78213137, 0.78293908, 0.78420019,
     &   0.78578764, 0.78764629, 0.78972620, 0.79203528, 0.79455960,
     &   0.79734278, 0.80031121, 0.80345273, 0.80677307, 0.81018698,
     &   0.81414741, 0.81973636, 0.82522154, 0.82739776, 0.82911015,
     &   0.83158195, 0.83482122, 0.83832872, 0.84206462, 0.84595090,
     &   0.84996974, 0.85414451, 0.85845757, 0.86296248, 0.86770225,
     &   0.87274581, 0.87814724, 0.88379961, 0.88986969, 0.89718425,
     &   0.90732378, 0.92101502, 0.93518555, 0.93590158, 0.92911202,
     &   0.91754192, 0.90637332, 0.89815760, 0.88919264, 0.87704295,
     &   0.85732710, 0.83534956, 0.81497556, 0.80224621, 0.80126292,
     &   0.80209434, 0.80450195, 0.80812585, 0.81229156, 0.81617731,
     &   0.81976593, 0.82282186, 0.82540864, 0.82756245, 0.82936561,
     &   0.83074903, 0.83190638, 0.83278763, 0.83341277, 0.83378547,
     &   0.83379912, 0.83350146, 0.83257240, 0.83071029, 0.82852954,
     &   0.82638258, 0.82439715, 0.82249790, 0.82078964, 0.81939876,
     &   0.81833935, 0.81762302, 0.81725740, 0.81763804, 0.81899768,
     &   0.82127768, 0.82124031, 0.81415951, 0.80162561, 0.79155678,
     &   0.78684068, 0.78330159, 0.77986991, 0.77648473, 0.77305311,
     &   0.76942849, 0.76564348, 0.76167184, 0.75751472, 0.75319499,
     &   0.74871570, 0.74408275, 0.73928791, 0.73429054, 0.72911859,
     &   0.72375411, 0.71818864, 0.71241111, 0.70640588, 0.70015448,
     &   0.69364077, 0.68684214, 0.67973685, 0.67229271, 0.66447908,
     &   0.65624845, 0.64749771, 0.63831258, 0.62834412, 0.61784518,
     &   0.60682100, 0.59547061, 0.58352464, 0.57138306, 0.55910558,
     &   0.54688400, 0.53486782, 0.52341312, 0.51261789, 0.50243384,
     &   0.49230000, 0.48262390, 0.47319272, 0.46409330, 0.45504105,
     &   0.44625315, 0.43751350, 0.42875323, 0.42021552, 0.41176525,
     &   0.40322748, 0.39493841, 0.38678047, 0.37883997, 0.37090263,
     &   0.36324027, 0.35580987, 0.34842101, 0.34120595, 0.33391252,
     &   0.32686862, 0.32016787, 0.31346345, 0.30663735, 0.29984090,
     &   0.29303080, 0.28588077, 0.27882427, 0.27185482, 0.26497954,
     &   0.25820878, 0.25159669, 0.24509515, 0.23872399, 0.23249003,
     &   0.22634941, 0.22007392, 0.21390516, 0.20780051, 0.20178203,
     &   0.19581491, 0.18988562, 0.18402441, 0.17819431, 0.17245887,
     &   0.16675198, 0.16113392, 0.15554406, 0.15007971, 0.14471497,
     &   0.13951389, 0.13447754, 0.12962505, 0.12494500, 0.12043165,
     &   0.11606720, 0.11183643, 0.10775577, 0.10381959, 0.10001610,
     &   0.09634242, 0.09279249, 0.08943407, 0.08621509, 0.08312551,
     &   0.08014889, 0.07730906, 0.07460557, 0.07203546, 0.06957155,
     &   0.06721646, 0.06496548, 0.06281620, 0.06078145, 0.05887942,
     &   0.05704686, 0.05526551, 0.05355568, 0.05192748, 0.05036100,
     &   0.04890113, 0.04748214, 0.04609576, 0.04474987, 0.04345868,
     &   0.04220640, 0.04101567, 0.03985688, 0.03871810, 0.03760508,
     &   0.03650808, 0.03544545, 0.03441646, 0.03341743, 0.03244727,
     &   0.03149910, 0.03057328, 0.02966688, 0.02878136, 0.02792246,
     &   0.02709422, 0.02629833, 0.02552464, 0.02476056, 0.02401645,
     &   0.02329252, 0.02258671, 0.02188747, 0.02120440, 0.02053976,
     &   0.01989313, 0.01926063, 0.01864974, 0.01805548, 0.01747345,
     &   0.01690425, 0.01634916, 0.01580043, 0.01526996, 0.01475714,
     &   0.01426139, 0.01378215, 0.01331013, 0.01285247, 0.01241057,
     &   0.01198387, 0.01157186, 0.01117836, 0.01079888, 0.01043228,
     &   0.01007812, 0.00973592, 0.00940517, 0.00908566, 0.00877699,
     &   0.00847863, 0.00818938, 0.00790999, 0.00764011, 0.00737965,
     &   0.00712877, 0.00688641, 0.00665228, 0.00642591, 0.00620711,
     &   0.00599573, 0.00579156, 0.00559438, 0.00540389, 0.00521985,
     &   0.00504386, 0.00487392, 0.00470968, 0.00455163, 0.00439903,
     &   0.00425152, 0.00410881, 0.00397086, 0.00383756, 0.00370889,
     &   0.00358451, 0.00346439, 0.00334836, 0.00323620, 0.00312818,
     &   0.00302375, 0.00292324, 0.00282631, 0.00273253, 0.00264173,
     &   0.00255391, 0.00246873, 0.00238637, 0.00230667, 0.00222961,
     &   0.00215503, 0.00208291, 0.00201317, 0.00194574, 0.00188065,
     &   0.00181772, 0.00175687, 0.00169804, 0.00164112, 0.00158610/
      data (gg(i,3),i=1,mxwv)/
     &   0.84455985, 0.84485096, 0.84503460, 0.84523511, 0.84538072,
     &   0.84563208, 0.84568828, 0.84574807, 0.84588081, 0.84594083,
     &   0.84592026, 0.84598678, 0.84592396, 0.84588182, 0.84584671,
     &   0.84582376, 0.84571087, 0.84563249, 0.84550291, 0.84537148,
     &   0.84523302, 0.84507620, 0.84484732, 0.84476697, 0.84444332,
     &   0.84428525, 0.84400076, 0.84380674, 0.84354097, 0.84331173,
     &   0.84301376, 0.84270489, 0.84237897, 0.84208763, 0.84175521,
     &   0.84138793, 0.84104955, 0.84060657, 0.84030765, 0.83985507,
     &   0.83942181, 0.83906525, 0.83863693, 0.83820271, 0.83766359,
     &   0.83721220, 0.83677375, 0.83627933, 0.83571291, 0.83531868,
     &   0.83471936, 0.83412755, 0.83354622, 0.83288312, 0.83231533,
     &   0.83165139, 0.83099717, 0.83034146, 0.82957327, 0.82887071,
     &   0.82811481, 0.82727188, 0.82648861, 0.82560199, 0.82476044,
     &   0.82388192, 0.82296246, 0.82201469, 0.82103533, 0.82001179,
     &   0.81894821, 0.81785631, 0.81672752, 0.81556922, 0.81439739,
     &   0.81317675, 0.81194711, 0.81070381, 0.80946487, 0.80816841,
     &   0.80683345, 0.80547619, 0.80409986, 0.80271494, 0.80133480,
     &   0.79997998, 0.79865843, 0.79746318, 0.79630262, 0.79561704,
     &   0.79577571, 0.79469544, 0.79312056, 0.79122537, 0.78971910,
     &   0.78849256, 0.78747916, 0.78662771, 0.78598046, 0.78552574,
     &   0.78532881, 0.78536803, 0.78565985, 0.78625280, 0.78700769,
     &   0.78856128, 0.79322964, 0.79878032, 0.79930472, 0.79850447,
     &   0.79875422, 0.80056041, 0.80311215, 0.80641943, 0.81035179,
     &   0.81488729, 0.82007921, 0.82590097, 0.83241725, 0.83973670,
     &   0.84794080, 0.85712224, 0.86701638, 0.87769902, 0.89064717,
     &   0.90846258, 0.93150747, 0.95311749, 0.95590609, 0.94998789,
     &   0.93872488, 0.92796981, 0.92080271, 0.91387290, 0.90524715,
     &   0.88846231, 0.86165524, 0.82762671, 0.79805332, 0.78733510,
     &   0.78259206, 0.78254759, 0.78648686, 0.79288423, 0.79984164,
     &   0.80701917, 0.81376028, 0.82000190, 0.82570952, 0.83098125,
     &   0.83563316, 0.83999014, 0.84394395, 0.84752542, 0.85073042,
     &   0.85338926, 0.85558379, 0.85681653, 0.85660261, 0.85592538,
     &   0.85535437, 0.85510170, 0.85505164, 0.85536933, 0.85624504,
     &   0.85768920, 0.85970110, 0.86227852, 0.86597824, 0.87107486,
     &   0.87732667, 0.88030589, 0.87369573, 0.85839933, 0.84612209,
     &   0.84258795, 0.84101349, 0.83970332, 0.83854777, 0.83739966,
     &   0.83604926, 0.83455569, 0.83287942, 0.83103675, 0.82906634,
     &   0.82698691, 0.82481235, 0.82253838, 0.82011151, 0.81759226,
     &   0.81496322, 0.81222945, 0.80938870, 0.80643511, 0.80335027,
     &   0.80014056, 0.79679084, 0.79328346, 0.78963310, 0.78581131,
     &   0.78178263, 0.77758008, 0.77298433, 0.76805568, 0.76247734,
     &   0.75616741, 0.74919164, 0.74137366, 0.73283529, 0.72361571,
     &   0.71401465, 0.70386934, 0.69391835, 0.68428326, 0.67516202,
     &   0.66605043, 0.65725845, 0.64864951, 0.64036721, 0.63205945,
     &   0.62395519, 0.61590636, 0.60770988, 0.59967655, 0.59162897,
     &   0.58339453, 0.57524943, 0.56711388, 0.55913305, 0.55104345,
     &   0.54309249, 0.53536582, 0.52752781, 0.51989329, 0.51204062,
     &   0.50430495, 0.49698111, 0.48959526, 0.48207411, 0.47460863,
     &   0.46722409, 0.45919818, 0.45116219, 0.44309568, 0.43501252,
     &   0.42693928, 0.41892570, 0.41093770, 0.40306294, 0.39536369,
     &   0.38767549, 0.37972692, 0.37179247, 0.36385259, 0.35591438,
     &   0.34793031, 0.33987877, 0.33177257, 0.32355452, 0.31531486,
     &   0.30695871, 0.29854858, 0.28997570, 0.28140029, 0.27277303,
     &   0.26423383, 0.25580850, 0.24754657, 0.23946342, 0.23156868,
     &   0.22384234, 0.21626292, 0.20887040, 0.20167324, 0.19466785,
     &   0.18784128, 0.18119201, 0.17486876, 0.16878493, 0.16292517,
     &   0.15725191, 0.15181945, 0.14663446, 0.14169870, 0.13695525,
     &   0.13241124, 0.12806056, 0.12390025, 0.11995343, 0.11626942,
     &   0.11272001, 0.10926517, 0.10594362, 0.10277622, 0.09972786,
     &   0.09689724, 0.09414237, 0.09144932, 0.08883125, 0.08631821,
     &   0.08387809, 0.08156190, 0.07930480, 0.07708517, 0.07491116,
     &   0.07276264, 0.07067894, 0.06865892, 0.06669539, 0.06478614,
     &   0.06291765, 0.06109011, 0.05929774, 0.05754391, 0.05584076,
     &   0.05419688, 0.05261735, 0.05108058, 0.04956020, 0.04807804,
     &   0.04663493, 0.04522685, 0.04382974, 0.04246368, 0.04113378,
     &   0.03983932, 0.03857229, 0.03734849, 0.03615772, 0.03499085,
     &   0.03384928, 0.03273576, 0.03163438, 0.03056980, 0.02954081,
     &   0.02854623, 0.02758493, 0.02663783, 0.02571967, 0.02483334,
     &   0.02397770, 0.02315169, 0.02236311, 0.02160278, 0.02086838,
     &   0.02015901, 0.01947367, 0.01881138, 0.01817165, 0.01755372,
     &   0.01695649, 0.01637751, 0.01581832, 0.01527823, 0.01475703,
     &   0.01425508, 0.01377020, 0.01330180, 0.01284898, 0.01241130,
     &   0.01198851, 0.01158013, 0.01118577, 0.01080479, 0.01043675,
     &   0.01008483, 0.00974502, 0.00941661, 0.00910059, 0.00879549,
     &   0.00850058, 0.00821526, 0.00793944, 0.00767294, 0.00741568,
     &   0.00716701, 0.00692687, 0.00669490, 0.00647065, 0.00625469,
     &   0.00604592, 0.00584498, 0.00565121, 0.00546373, 0.00528220,
     &   0.00510663, 0.00493634, 0.00477169, 0.00461234, 0.00445828,
     &   0.00430916, 0.00416498, 0.00402554, 0.00389072, 0.00376058,
     &   0.00363476, 0.00351311, 0.00339547, 0.00328168, 0.00317167/
      data (gg(i,4),i=1,mxwv)/
     &   0.85081232, 0.85113579, 0.85146528, 0.85174489, 0.85205144,
     &   0.85229868, 0.85249537, 0.85271430, 0.85285896, 0.85299277,
     &   0.85316002, 0.85322952, 0.85331571, 0.85341138, 0.85343051,
     &   0.85350990, 0.85346949, 0.85348326, 0.85338873, 0.85345846,
     &   0.85343117, 0.85336089, 0.85325092, 0.85320866, 0.85306942,
     &   0.85294843, 0.85289627, 0.85264999, 0.85257018, 0.85242772,
     &   0.85223860, 0.85207891, 0.85189581, 0.85168809, 0.85141820,
     &   0.85126549, 0.85098267, 0.85076016, 0.85049802, 0.85028118,
     &   0.85000056, 0.84967661, 0.84937501, 0.84912705, 0.84937263,
     &   0.84848493, 0.84814936, 0.84782654, 0.84745914, 0.84710097,
     &   0.84670240, 0.84633636, 0.84593064, 0.84551305, 0.84506303,
     &   0.84460324, 0.84421033, 0.84371644, 0.84327072, 0.84270066,
     &   0.84230822, 0.84185076, 0.84131831, 0.84085870, 0.84025544,
     &   0.83975512, 0.83923119, 0.83860266, 0.83808148, 0.83751965,
     &   0.83692896, 0.83627075, 0.83568621, 0.83502293, 0.83431572,
     &   0.83365059, 0.83300757, 0.83231050, 0.83168280, 0.83096534,
     &   0.83019263, 0.82938969, 0.82857323, 0.82769787, 0.82684958,
     &   0.82600260, 0.82515913, 0.82445931, 0.82375586, 0.82368267,
     &   0.82475203, 0.82395566, 0.82234728, 0.82012844, 0.81829959,
     &   0.81671995, 0.81527698, 0.81387836, 0.81257057, 0.81130725,
     &   0.81014717, 0.80905986, 0.80804741, 0.80716157, 0.80614150,
     &   0.80580586, 0.80975509, 0.81504214, 0.81339782, 0.80906433,
     &   0.80539101, 0.80350810, 0.80230856, 0.80189788, 0.80210501,
     &   0.80293375, 0.80451918, 0.80689418, 0.81018490, 0.81473875,
     &   0.82077187, 0.82867533, 0.83823228, 0.84930396, 0.86486763,
     &   0.89046401, 0.92773694, 0.96182382, 0.96755475, 0.96242714,
     &   0.95152372, 0.94128400, 0.93526137, 0.93043298, 0.92634338,
     &   0.91900939, 0.90185332, 0.86948657, 0.82818484, 0.80124068,
     &   0.78145677, 0.76838702, 0.76273990, 0.76313430, 0.76671398,
     &   0.77272439, 0.77976054, 0.78729051, 0.79496920, 0.80280739,
     &   0.81020558, 0.81765705, 0.82484084, 0.83175385, 0.83829325,
     &   0.84412849, 0.84935659, 0.85311532, 0.85436392, 0.85473776,
     &   0.85526478, 0.85638726, 0.85790098, 0.86012465, 0.86336493,
     &   0.86764228, 0.87291807, 0.87909293, 0.88695550, 0.89673805,
     &   0.90785444, 0.91431409, 0.90862209, 0.89094377, 0.87365401,
     &   0.86945015, 0.86895913, 0.86916959, 0.86979979, 0.87056822,
     &   0.87112111, 0.87152964, 0.87174284, 0.87177110, 0.87167621,
     &   0.87148762, 0.87122667, 0.87088853, 0.87040097, 0.86985612,
     &   0.86922836, 0.86853516, 0.86777627, 0.86695004, 0.86603528,
     &   0.86505330, 0.86399275, 0.86283880, 0.86163843, 0.86036617,
     &   0.85900027, 0.85766286, 0.85596895, 0.85424614, 0.85188890,
     &   0.84881270, 0.84503669, 0.84049070, 0.83512670, 0.82896310,
     &   0.82231402, 0.81478590, 0.80723333, 0.79974216, 0.79265410,
     &   0.78559810, 0.77874410, 0.77204543, 0.76567292, 0.75928175,
     &   0.75306910, 0.74698240, 0.74073350, 0.73463285, 0.72849774,
     &   0.72219795, 0.71588129, 0.70952016, 0.70327723, 0.69690311,
     &   0.69054306, 0.68438345, 0.67802691, 0.67187929, 0.66547132,
     &   0.65901881, 0.65291739, 0.64672405, 0.64045709, 0.63427418,
     &   0.62827706, 0.62162584, 0.61488318, 0.60801888, 0.60104525,
     &   0.59399271, 0.58686709, 0.57967335, 0.57254034, 0.56557262,
     &   0.55853474, 0.55130911, 0.54400909, 0.53667504, 0.52928299,
     &   0.52180803, 0.51423794, 0.50653684, 0.49866205, 0.49065956,
     &   0.48246896, 0.47406906, 0.46533754, 0.45636645, 0.44706321,
     &   0.43756029, 0.42788863, 0.41808772, 0.40822211, 0.39832425,
     &   0.38838124, 0.37836081, 0.36830625, 0.35826531, 0.34828216,
     &   0.33829290, 0.32831287, 0.31860197, 0.30910546, 0.29982051,
     &   0.29066294, 0.28174645, 0.27312136, 0.26483384, 0.25677997,
     &   0.24898212, 0.24144872, 0.23418625, 0.22721982, 0.22070839,
     &   0.21442641, 0.20828547, 0.20233916, 0.19662584, 0.19111435,
     &   0.18602991, 0.18106507, 0.17621298, 0.17147864, 0.16692272,
     &   0.16248700, 0.15829873, 0.15420739, 0.15019397, 0.14624439,
     &   0.14231730, 0.13849746, 0.13478395, 0.13116483, 0.12763371,
     &   0.12416931, 0.12076423, 0.11740698, 0.11410501, 0.11088280,
     &   0.10775743, 0.10475539, 0.10182589, 0.09890985, 0.09605380,
     &   0.09326269, 0.09052978, 0.08780178, 0.08512080, 0.08250186,
     &   0.07994477, 0.07743157, 0.07499962, 0.07262825, 0.07029751,
     &   0.06801128, 0.06577688, 0.06356100, 0.06141751, 0.05934443,
     &   0.05733975, 0.05540150, 0.05348876, 0.05163427, 0.04984452,
     &   0.04811725, 0.04645030, 0.04486012, 0.04332745, 0.04184749,
     &   0.04041842, 0.03903818, 0.03770475, 0.03641716, 0.03517382,
     &   0.03397249, 0.03280810, 0.03168384, 0.03059833, 0.02955112,
     &   0.02854295, 0.02756932, 0.02662903, 0.02572020, 0.02484196,
     &   0.02399380, 0.02317473, 0.02238391, 0.02162009, 0.02088232,
     &   0.02017713, 0.01949631, 0.01883842, 0.01820550, 0.01759452,
     &   0.01700399, 0.01643272, 0.01588055, 0.01534705, 0.01483212,
     &   0.01433441, 0.01385382, 0.01338962, 0.01294090, 0.01250880,
     &   0.01209109, 0.01168911, 0.01130149, 0.01092647, 0.01056337,
     &   0.01021221, 0.00987158, 0.00954226, 0.00922354, 0.00891541,
     &   0.00861717, 0.00832881, 0.00804993, 0.00778031, 0.00752004,
     &   0.00726841, 0.00702514, 0.00678989, 0.00656233, 0.00634234/
      data (gg(i,5),i=1,mxwv)/
     &   0.85563320, 0.85602611, 0.85642934, 0.85678023, 0.85712159,
     &   0.85742575, 0.85771060, 0.85802025, 0.85823357, 0.85844839,
     &   0.85864413, 0.85881537, 0.85897744, 0.85914212, 0.85925424,
     &   0.85938138, 0.85950154, 0.85955465, 0.85963023, 0.85965389,
     &   0.85981476, 0.85973406, 0.85974711, 0.85976636, 0.85962546,
     &   0.85973829, 0.85970360, 0.85966837, 0.85947204, 0.85950708,
     &   0.85943341, 0.85932559, 0.85928148, 0.85918808, 0.85898405,
     &   0.85895330, 0.85877466, 0.85865641, 0.85850585, 0.85835487,
     &   0.85828495, 0.85800630, 0.85782504, 0.85758477, 0.85744548,
     &   0.85724580, 0.85701180, 0.85674244, 0.85649019, 0.85628843,
     &   0.85602474, 0.85572892, 0.85540909, 0.85511899, 0.85485429,
     &   0.85455692, 0.85424262, 0.85385680, 0.85355061, 0.85322076,
     &   0.85286623, 0.85252750, 0.85219628, 0.85181367, 0.85141426,
     &   0.85104787, 0.85062158, 0.85028034, 0.84986031, 0.84942102,
     &   0.84902501, 0.84864128, 0.84812886, 0.84772360, 0.84726638,
     &   0.84688348, 0.84629887, 0.84599727, 0.84563702, 0.84523875,
     &   0.84477299, 0.84429729, 0.84379554, 0.84329247, 0.84273010,
     &   0.84231198, 0.84197360, 0.84180075, 0.84165174, 0.84237945,
     &   0.84462035, 0.84444076, 0.84322512, 0.84121060, 0.83970523,
     &   0.83853668, 0.83755708, 0.83663619, 0.83583832, 0.83509105,
     &   0.83446813, 0.83392000, 0.83343679, 0.83307809, 0.83244199,
     &   0.83266592, 0.83866966, 0.84606677, 0.84443051, 0.83904845,
     &   0.83410203, 0.83122104, 0.82893401, 0.82738811, 0.82629806,
     &   0.82560354, 0.82541573, 0.82568347, 0.82642299, 0.82802439,
     &   0.83057570, 0.83449143, 0.83935499, 0.84425050, 0.85305637,
     &   0.87465060, 0.91838354, 0.96582550, 0.97472960, 0.97002351,
     &   0.95938611, 0.94948053, 0.94408786, 0.94023293, 0.93829083,
     &   0.93697971, 0.93150645, 0.91420490, 0.88200998, 0.85356599,
     &   0.82767940, 0.80577749, 0.79053813, 0.78191030, 0.77732217,
     &   0.77637595, 0.77755451, 0.78014785, 0.78372443, 0.78844458,
     &   0.79331106, 0.79912007, 0.80537939, 0.81213993, 0.81915784,
     &   0.82593966, 0.83253032, 0.83757389, 0.83869624, 0.83809745,
     &   0.83748662, 0.83783835, 0.83885479, 0.84125024, 0.84561515,
     &   0.85218364, 0.86096013, 0.87158108, 0.88527256, 0.90213203,
     &   0.92095721, 0.93291497, 0.92945647, 0.91155660, 0.88634503,
     &   0.87739742, 0.87581998, 0.87618655, 0.87779272, 0.88003355,
     &   0.88217384, 0.88424939, 0.88613790, 0.88782692, 0.88940191,
     &   0.89089185, 0.89233148, 0.89370960, 0.89490968, 0.89606810,
     &   0.89715308, 0.89818305, 0.89916134, 0.90008616, 0.90092939,
     &   0.90172273, 0.90245342, 0.90310466, 0.90374744, 0.90435123,
     &   0.90489548, 0.90556073, 0.90581805, 0.90616655, 0.90579468,
     &   0.90462792, 0.90270078, 0.90000910, 0.89644581, 0.89204508,
     &   0.88715261, 0.88126206, 0.87529397, 0.86930245, 0.86368835,
     &   0.85814977, 0.85277587, 0.84756732, 0.84270507, 0.83785760,
     &   0.83319962, 0.82873565, 0.82413954, 0.81971198, 0.81527448,
     &   0.81073093, 0.80614471, 0.80152237, 0.79702759, 0.79243410,
     &   0.78781486, 0.78340095, 0.77878529, 0.77439260, 0.76977295,
     &   0.76503599, 0.76058877, 0.75605810, 0.75152385, 0.74709457,
     &   0.74290597, 0.73820293, 0.73339558, 0.72845364, 0.72338611,
     &   0.71821892, 0.71291840, 0.70751894, 0.70213932, 0.69687551,
     &   0.69150925, 0.68611801, 0.68062937, 0.67512399, 0.66956145,
     &   0.66395724, 0.65832376, 0.65260470, 0.64681178, 0.64092660,
     &   0.63498062, 0.62888819, 0.62261558, 0.61611891, 0.60932821,
     &   0.60226041, 0.59491754, 0.58727568, 0.57939565, 0.57129735,
     &   0.56296992, 0.55436850, 0.54547197, 0.53632367, 0.52699274,
     &   0.51733893, 0.50735623, 0.49724823, 0.48707953, 0.47686851,
     &   0.46647856, 0.45603278, 0.44563818, 0.43541983, 0.42525369,
     &   0.41517654, 0.40523326, 0.39545381, 0.38582683, 0.37670127,
     &   0.36782694, 0.35906076, 0.35042527, 0.34196472, 0.33372378,
     &   0.32610542, 0.31860468, 0.31126782, 0.30404913, 0.29704034,
     &   0.29017192, 0.28371057, 0.27737492, 0.27121717, 0.26512331,
     &   0.25902608, 0.25307116, 0.24726143, 0.24158759, 0.23603161,
     &   0.23058526, 0.22520241, 0.21986203, 0.21457312, 0.20936629,
     &   0.20425813, 0.19937396, 0.19459642, 0.18979658, 0.18505114,
     &   0.18038221, 0.17578372, 0.17113484, 0.16649671, 0.16191971,
     &   0.15740684, 0.15290625, 0.14851534, 0.14419943, 0.13990602,
     &   0.13564324, 0.13143528, 0.12721278, 0.12309697, 0.11908830,
     &   0.11518696, 0.11139281, 0.10760438, 0.10391353, 0.10034099,
     &   0.09688427, 0.09354073, 0.09034693, 0.08726393, 0.08428287,
     &   0.08140097, 0.07861479, 0.07592086, 0.07331795, 0.07080332,
     &   0.06837264, 0.06601496, 0.06373856, 0.06154075, 0.05942081,
     &   0.05738054, 0.05541059, 0.05350857, 0.05167064, 0.04989509,
     &   0.04818093, 0.04652613, 0.04492895, 0.04338687, 0.04189798,
     &   0.04047595, 0.03910358, 0.03777786, 0.03650309, 0.03527293,
     &   0.03408432, 0.03293475, 0.03182394, 0.03075101, 0.02971576,
     &   0.02871538, 0.02774968, 0.02681715, 0.02591596, 0.02504839,
     &   0.02420989, 0.02340322, 0.02262557, 0.02187332, 0.02114508,
     &   0.02044090, 0.01975791, 0.01909768, 0.01845881, 0.01784124,
     &   0.01724357, 0.01666576, 0.01610701, 0.01556690, 0.01504556,
     &   0.01454159, 0.01405440, 0.01358331, 0.01312769, 0.01268724/
      data (gg(i,6),i=1,mxwv)/
     &   0.85929167, 0.85976028, 0.86019212, 0.86063051, 0.86099911,
     &   0.86138177, 0.86167926, 0.86202049, 0.86232346, 0.86261982,
     &   0.86286354, 0.86309612, 0.86335456, 0.86355489, 0.86373919,
     &   0.86388850, 0.86406577, 0.86422247, 0.86436141, 0.86447895,
     &   0.86457914, 0.86467016, 0.86475188, 0.86481208, 0.86486536,
     &   0.86491764, 0.86495835, 0.86498827, 0.86501014, 0.86500251,
     &   0.86500931, 0.86499393, 0.86497670, 0.86499381, 0.86490285,
     &   0.86487079, 0.86481661, 0.86474985, 0.86467570, 0.86459881,
     &   0.86449087, 0.86442584, 0.86430335, 0.86424226, 0.86408174,
     &   0.86400896, 0.86385453, 0.86370313, 0.86353713, 0.86335838,
     &   0.86324632, 0.86308134, 0.86286002, 0.86265850, 0.86244744,
     &   0.86224681, 0.86202961, 0.86176515, 0.86159194, 0.86135387,
     &   0.86104822, 0.86084682, 0.86049867, 0.86028200, 0.86004639,
     &   0.85978341, 0.85949945, 0.85927629, 0.85903096, 0.85871029,
     &   0.85843241, 0.85805285, 0.85768116, 0.85744888, 0.85704774,
     &   0.85680598, 0.85646445, 0.85616624, 0.85603315, 0.85575974,
     &   0.85542423, 0.85509664, 0.85472423, 0.85433608, 0.85402268,
     &   0.85368657, 0.85356581, 0.85368198, 0.85381329, 0.85506266,
     &   0.85823816, 0.85838354, 0.85723448, 0.85508394, 0.85358322,
     &   0.85253185, 0.85172462, 0.85101396, 0.85048389, 0.85007709,
     &   0.84980023, 0.84967667, 0.84966195, 0.84984535, 0.84969229,
     &   0.85069638, 0.85910267, 0.86902767, 0.86779785, 0.86194444,
     &   0.85653585, 0.85367090, 0.85151732, 0.85028118, 0.84958971,
     &   0.84934813, 0.84967870, 0.85046208, 0.85163581, 0.85362011,
     &   0.85635072, 0.86013454, 0.86411071, 0.86640042, 0.87073863,
     &   0.88604736, 0.92385626, 0.97045618, 0.97967595, 0.97477221,
     &   0.96424443, 0.95446974, 0.94927704, 0.94566309, 0.94422799,
     &   0.94475108, 0.94465792, 0.93854845, 0.91979569, 0.89799803,
     &   0.87444109, 0.85206801, 0.83507347, 0.82450604, 0.81793797,
     &   0.81521076, 0.81464916, 0.81540346, 0.81696570, 0.81961769,
     &   0.82212168, 0.82553607, 0.82933891, 0.83374876, 0.83854997,
     &   0.84336340, 0.84831989, 0.85206133, 0.85110927, 0.84744126,
     &   0.84298253, 0.83916247, 0.83556783, 0.83341736, 0.83366227,
     &   0.83725107, 0.84480339, 0.85592026, 0.87289488, 0.89640242,
     &   0.92505765, 0.94439995, 0.94432628, 0.93090129, 0.90374762,
     &   0.88687646, 0.88011390, 0.87701094, 0.87668604, 0.87825525,
     &   0.88032520, 0.88276213, 0.88532305, 0.88783354, 0.89040345,
     &   0.89301199, 0.89571756, 0.89848292, 0.90108061, 0.90371174,
     &   0.90631616, 0.90890384, 0.91147548, 0.91401726, 0.91648364,
     &   0.91890728, 0.92126507, 0.92353141, 0.92579752, 0.92802411,
     &   0.93017966, 0.93250895, 0.93431711, 0.93627900, 0.93735570,
     &   0.93750960, 0.93679178, 0.93526882, 0.93280530, 0.92945629,
     &   0.92559671, 0.92066520, 0.91563153, 0.91053802, 0.90582031,
     &   0.90120852, 0.89675355, 0.89248037, 0.88857770, 0.88471848,
     &   0.88106930, 0.87766242, 0.87415129, 0.87083548, 0.86753780,
     &   0.86417490, 0.86077863, 0.85736954, 0.85411203, 0.85078931,
     &   0.84744477, 0.84432554, 0.84102315, 0.83796579, 0.83470976,
     &   0.83132714, 0.82821244, 0.82502866, 0.82187921, 0.81884527,
     &   0.81606352, 0.81287253, 0.80958760, 0.80618149, 0.80266201,
     &   0.79905111, 0.79529691, 0.79144692, 0.78759903, 0.78382635,
     &   0.77994519, 0.77615237, 0.77226698, 0.76838386, 0.76445860,
     &   0.76053238, 0.75663388, 0.75270784, 0.74880815, 0.74487662,
     &   0.74101228, 0.73711175, 0.73323888, 0.72926736, 0.72519070,
     &   0.72093689, 0.71649629, 0.71181434, 0.70692742, 0.70184612,
     &   0.69657362, 0.69108450, 0.68531299, 0.67927724, 0.67303401,
     &   0.66643900, 0.65945828, 0.65209872, 0.64450473, 0.63669407,
     &   0.62852752, 0.62004244, 0.61133218, 0.60253489, 0.59355718,
     &   0.58441556, 0.57516497, 0.56583744, 0.55635160, 0.54712659,
     &   0.53803444, 0.52893031, 0.51975524, 0.51051927, 0.50136918,
     &   0.49274170, 0.48412910, 0.47564912, 0.46718740, 0.45882064,
     &   0.45051351, 0.44260776, 0.43478283, 0.42721927, 0.41967836,
     &   0.41210115, 0.40463924, 0.39730519, 0.39011225, 0.38303441,
     &   0.37612051, 0.36927024, 0.36246416, 0.35570660, 0.34900266,
     &   0.34233505, 0.33600390, 0.32983753, 0.32365528, 0.31751642,
     &   0.31147602, 0.30554634, 0.29953808, 0.29345411, 0.28740442,
     &   0.28139395, 0.27529019, 0.26927748, 0.26332793, 0.25731283,
     &   0.25122163, 0.24510422, 0.23884074, 0.23262228, 0.22645408,
     &   0.22034155, 0.21429011, 0.20801441, 0.20177998, 0.19565679,
     &   0.18964890, 0.18375994, 0.17805597, 0.17248595, 0.16704223,
     &   0.16172631, 0.15653774, 0.15147556, 0.14654395, 0.14174297,
     &   0.13706727, 0.13249052, 0.12804706, 0.12373540, 0.11955727,
     &   0.11551898, 0.11160593, 0.10781575, 0.10414216, 0.10058396,
     &   0.09714140, 0.09381126, 0.09058962, 0.08747548, 0.08446581,
     &   0.08159129, 0.07881538, 0.07613228, 0.07355253, 0.07106236,
     &   0.06865555, 0.06632702, 0.06407673, 0.06190313, 0.05980603,
     &   0.05777977, 0.05582414, 0.05393611, 0.05211189, 0.05035639,
     &   0.04866016, 0.04702923, 0.04545765, 0.04393779, 0.04246686,
     &   0.04104489, 0.03966596, 0.03833342, 0.03704438, 0.03579873,
     &   0.03459356, 0.03342883, 0.03230290, 0.03121489, 0.03016505,
     &   0.02915052, 0.02817006, 0.02722230, 0.02630590, 0.02542028/
      data (gg(i,7),i=1,mxwv)/
     &   0.86207455, 0.86257666, 0.86306560, 0.86350971, 0.86396039,
     &   0.86435080, 0.86474526, 0.86510903, 0.86551118, 0.86579049,
     &   0.86608958, 0.86636627, 0.86664778, 0.86689991, 0.86713850,
     &   0.86731941, 0.86756980, 0.86777949, 0.86793941, 0.86810613,
     &   0.86819422, 0.86842436, 0.86856353, 0.86867672, 0.86878502,
     &   0.86889547, 0.86897022, 0.86908108, 0.86914277, 0.86920488,
     &   0.86927605, 0.86932909, 0.86936700, 0.86938471, 0.86944938,
     &   0.86941278, 0.86945337, 0.86941302, 0.86943287, 0.86941338,
     &   0.86933976, 0.86935657, 0.86931652, 0.86929446, 0.86927021,
     &   0.86918890, 0.86915433, 0.86908621, 0.86898899, 0.86890602,
     &   0.86884302, 0.86871678, 0.86861378, 0.86846226, 0.86836773,
     &   0.86817008, 0.86805546, 0.86793029, 0.86776626, 0.86766773,
     &   0.86746418, 0.86728209, 0.86713338, 0.86693603, 0.86671311,
     &   0.86655134, 0.86638767, 0.86629379, 0.86610472, 0.86590576,
     &   0.86564541, 0.86545140, 0.86521626, 0.86494672, 0.86472255,
     &   0.86452222, 0.86438060, 0.86420000, 0.86415708, 0.86398172,
     &   0.86374831, 0.86354035, 0.86327761, 0.86300689, 0.86280411,
     &   0.86264759, 0.86262721, 0.86290652, 0.86333585, 0.86512309,
     &   0.86935747, 0.86975902, 0.86852658, 0.86602479, 0.86433774,
     &   0.86320662, 0.86239874, 0.86172986, 0.86127347, 0.86094379,
     &   0.86084938, 0.86091030, 0.86112398, 0.86158848, 0.86161458,
     &   0.86311716, 0.87385577, 0.88621402, 0.88506049, 0.87830758,
     &   0.87197226, 0.86874187, 0.86639869, 0.86522096, 0.86475271,
     &   0.86488169, 0.86577117, 0.86727154, 0.86930096, 0.87238264,
     &   0.87640309, 0.88168210, 0.88704604, 0.89003420, 0.89466888,
     &   0.90973759, 0.94324309, 0.97778761, 0.98319238, 0.97774953,
     &   0.96725190, 0.95752180, 0.95237738, 0.94881213, 0.94746643,
     &   0.94839060, 0.95002520, 0.94938946, 0.94093490, 0.92652410,
     &   0.90686762, 0.88531911, 0.86768013, 0.85658240, 0.84993112,
     &   0.84778047, 0.84818071, 0.85008180, 0.85282248, 0.85667843,
     &   0.86014611, 0.86441463, 0.86884248, 0.87365776, 0.87860757,
     &   0.88333714, 0.88802540, 0.89144772, 0.89013404, 0.88581043,
     &   0.88018429, 0.87464923, 0.86862922, 0.86337954, 0.85981280,
     &   0.85916781, 0.86242914, 0.86913294, 0.88225913, 0.90376258,
     &   0.93442363, 0.95580459, 0.95670068, 0.94861346, 0.93219966,
     &   0.91596824, 0.90607238, 0.89903319, 0.89481127, 0.89303768,
     &   0.89222622, 0.89216334, 0.89266610, 0.89337748, 0.89446342,
     &   0.89584047, 0.89765006, 0.89986634, 0.90212381, 0.90468973,
     &   0.90748316, 0.91048831, 0.91370845, 0.91710025, 0.92058939,
     &   0.92417336, 0.92779744, 0.93140864, 0.93508708, 0.93878102,
     &   0.94240910, 0.94629371, 0.94951808, 0.95296246, 0.95527506,
     &   0.95647794, 0.95663744, 0.95592147, 0.95418257, 0.95150518,
     &   0.94828665, 0.94394696, 0.93948275, 0.93493557, 0.93076235,
     &   0.92671180, 0.92281651, 0.91911542, 0.91580313, 0.91255444,
     &   0.90953428, 0.90678710, 0.90395832, 0.90134603, 0.89877629,
     &   0.89616847, 0.89354527, 0.89093292, 0.88849837, 0.88602483,
     &   0.88355142, 0.88133001, 0.87894893, 0.87683713, 0.87454921,
     &   0.87215495, 0.87004328, 0.86787903, 0.86576110, 0.86376363,
     &   0.86201274, 0.85988253, 0.85766900, 0.85534459, 0.85292184,
     &   0.85041523, 0.84777260, 0.84504122, 0.84230274, 0.83961022,
     &   0.83680511, 0.83411443, 0.83133417, 0.82855523, 0.82573348,
     &   0.82292247, 0.82015675, 0.81739283, 0.81470668, 0.81202734,
     &   0.80948502, 0.80698735, 0.80465901, 0.80234158, 0.80007792,
     &   0.79774535, 0.79532844, 0.79276079, 0.79006183, 0.78723145,
     &   0.78428817, 0.78123093, 0.77798194, 0.77454501, 0.77096391,
     &   0.76713169, 0.76301253, 0.75847739, 0.75370234, 0.74870586,
     &   0.74338508, 0.73770672, 0.73172987, 0.72556484, 0.71916372,
     &   0.71251845, 0.70567226, 0.69864374, 0.69130498, 0.68400335,
     &   0.67674839, 0.66943181, 0.66193014, 0.65420926, 0.64645523,
     &   0.63897115, 0.63141298, 0.62392348, 0.61635256, 0.60872155,
     &   0.60103858, 0.59356993, 0.58608156, 0.57881510, 0.57149494,
     &   0.56410319, 0.55671930, 0.54935801, 0.54205120, 0.53477561,
     &   0.52762944, 0.52050149, 0.51339209, 0.50630122, 0.49919593,
     &   0.49201944, 0.48512930, 0.47839844, 0.47171226, 0.46506217,
     &   0.45852342, 0.45214090, 0.44578296, 0.43934736, 0.43296462,
     &   0.42664650, 0.42022955, 0.41389546, 0.40767094, 0.40139902,
     &   0.39503446, 0.38862991, 0.38208720, 0.37552065, 0.36892974,
     &   0.36231411, 0.35567373, 0.34850025, 0.34121543, 0.33392268,
     &   0.32663003, 0.31934234, 0.31208676, 0.30485946, 0.29766282,
     &   0.29050201, 0.28337955, 0.27629736, 0.26926753, 0.26229560,
     &   0.25537178, 0.24842130, 0.24155413, 0.23477581, 0.22809340,
     &   0.22151759, 0.21504791, 0.20868857, 0.20243350, 0.19628976,
     &   0.19026835, 0.18436581, 0.17856580, 0.17290026, 0.16737048,
     &   0.16204309, 0.15685491, 0.15180029, 0.14691252, 0.14216325,
     &   0.13754274, 0.13304298, 0.12866995, 0.12442411, 0.12030791,
     &   0.11631385, 0.11244454, 0.10869617, 0.10506305, 0.10155616,
     &   0.09815933, 0.09488908, 0.09173303, 0.08867542, 0.08571182,
     &   0.08284263, 0.08005548, 0.07735942, 0.07474899, 0.07222472,
     &   0.06978065, 0.06741765, 0.06513274, 0.06292444, 0.06079339,
     &   0.05873407, 0.05674412, 0.05482080, 0.05296148, 0.05116508/
      data (gg(i,8),i=1,mxwv)/
     &   0.86415082, 0.86470020, 0.86520845, 0.86570001, 0.86615551,
     &   0.86660403, 0.86702698, 0.86743528, 0.86780620, 0.86817056,
     &   0.86851180, 0.86885196, 0.86913478, 0.86943799, 0.86969519,
     &   0.86997896, 0.87023062, 0.87046015, 0.87068135, 0.87089401,
     &   0.87108797, 0.87127811, 0.87145698, 0.87161267, 0.87176722,
     &   0.87193519, 0.87204623, 0.87219024, 0.87225848, 0.87240922,
     &   0.87252927, 0.87259823, 0.87266862, 0.87276524, 0.87283915,
     &   0.87290239, 0.87297350, 0.87300140, 0.87305409, 0.87307775,
     &   0.87313449, 0.87316227, 0.87318403, 0.87321877, 0.87321573,
     &   0.87319076, 0.87319630, 0.87319952, 0.87315768, 0.87312305,
     &   0.87314165, 0.87303305, 0.87302113, 0.87290806, 0.87287748,
     &   0.87279671, 0.87272930, 0.87261528, 0.87251043, 0.87245822,
     &   0.87238336, 0.87225145, 0.87218541, 0.87204075, 0.87192309,
     &   0.87183547, 0.87176627, 0.87169826, 0.87160873, 0.87149715,
     &   0.87133116, 0.87116259, 0.87097770, 0.87081748, 0.87061965,
     &   0.87055415, 0.87041134, 0.87036735, 0.87042910, 0.87042522,
     &   0.87029284, 0.87014300, 0.86996472, 0.86978400, 0.86965275,
     &   0.86963290, 0.86974692, 0.87031490, 0.87094545, 0.87338996,
     &   0.87894279, 0.87968528, 0.87821656, 0.87520415, 0.87320584,
     &   0.87192732, 0.87104112, 0.87033731, 0.86993456, 0.86965406,
     &   0.86965293, 0.86988896, 0.87027019, 0.87099916, 0.87116754,
     &   0.87315321, 0.88664716, 0.90159076, 0.90042353, 0.89252526,
     &   0.88491464, 0.88100398, 0.87815440, 0.87674141, 0.87620276,
     &   0.87640339, 0.87753987, 0.87942922, 0.88195461, 0.88576460,
     &   0.89069593, 0.89711684, 0.90359700, 0.90726560, 0.91293132,
     &   0.93018579, 0.96187919, 0.98405242, 0.98544466, 0.97960538,
     &   0.96913284, 0.95942426, 0.95429432, 0.95074743, 0.94943005,
     &   0.95044482, 0.95253944, 0.95404065, 0.95199859, 0.94462174,
     &   0.93044513, 0.91118264, 0.89316422, 0.88099957, 0.87352097,
     &   0.87124997, 0.87206668, 0.87473559, 0.87844884, 0.88347143,
     &   0.88804162, 0.89346004, 0.89895397, 0.90471143, 0.91041011,
     &   0.91565698, 0.92062378, 0.92419791, 0.92344326, 0.91996497,
     &   0.91519511, 0.91035396, 0.90483624, 0.89981353, 0.89617991,
     &   0.89517766, 0.89772904, 0.90304500, 0.91356725, 0.93036991,
     &   0.95338988, 0.96704364, 0.96518725, 0.95902288, 0.95322114,
     &   0.94489378, 0.93828285, 0.93264616, 0.92850924, 0.92599154,
     &   0.92407984, 0.92263377, 0.92159110, 0.92061025, 0.91986811,
     &   0.91927475, 0.91902769, 0.91916168, 0.91933924, 0.91984361,
     &   0.92065424, 0.92178738, 0.92331278, 0.92523396, 0.92754012,
     &   0.93022317, 0.93325114, 0.93658423, 0.94027334, 0.94429559,
     &   0.94850278, 0.95326775, 0.95749336, 0.96211427, 0.96542346,
     &   0.96744597, 0.96822953, 0.96805865, 0.96679002, 0.96453911,
     &   0.96171957, 0.95775354, 0.95364600, 0.94944060, 0.94560605,
     &   0.94190180, 0.93835312, 0.93500626, 0.93205947, 0.92918909,
     &   0.92656058, 0.92422372, 0.92181820, 0.91964871, 0.91753769,
     &   0.91540474, 0.91327417, 0.91117370, 0.90927404, 0.90735394,
     &   0.90545923, 0.90384471, 0.90209353, 0.90064007, 0.89902645,
     &   0.89733958, 0.89598036, 0.89459163, 0.89326131, 0.89207298,
     &   0.89114714, 0.88981378, 0.88841277, 0.88691640, 0.88533175,
     &   0.88367867, 0.88191020, 0.88006604, 0.87822866, 0.87644827,
     &   0.87455797, 0.87274027, 0.87083000, 0.86890316, 0.86692011,
     &   0.86492109, 0.86293536, 0.86092854, 0.85897261, 0.85700941,
     &   0.85516244, 0.85336882, 0.85177433, 0.85024595, 0.84885985,
     &   0.84749174, 0.84612721, 0.84470499, 0.84322476, 0.84167957,
     &   0.84009296, 0.83847815, 0.83675838, 0.83492464, 0.83300519,
     &   0.83092761, 0.82865560, 0.82599431, 0.82312179, 0.82005829,
     &   0.81673300, 0.81307983, 0.80913806, 0.80499786, 0.80064309,
     &   0.79605663, 0.79127485, 0.78631002, 0.78102297, 0.77569026,
     &   0.77039492, 0.76506007, 0.75953996, 0.75377899, 0.74796700,
     &   0.74228400, 0.73652250, 0.73081684, 0.72501558, 0.71909958,
     &   0.71310055, 0.70718139, 0.70119941, 0.69538325, 0.68948323,
     &   0.68351406, 0.67748243, 0.67139417, 0.66528070, 0.65912098,
     &   0.65301645, 0.64687574, 0.64071226, 0.63452029, 0.62824333,
     &   0.62180895, 0.61547744, 0.60920388, 0.60297668, 0.59673238,
     &   0.59052736, 0.58441383, 0.57838833, 0.57230306, 0.56624472,
     &   0.56022912, 0.55416441, 0.54812407, 0.54218316, 0.53626698,
     &   0.53033215, 0.52441192, 0.51852584, 0.51261747, 0.50668722,
     &   0.50073504, 0.49476111, 0.48838347, 0.48187745, 0.47532281,
     &   0.46872017, 0.46207014, 0.45529485, 0.44847125, 0.44161275,
     &   0.43472052, 0.42779359, 0.42083013, 0.41383624, 0.40681353,
     &   0.39974120, 0.39249662, 0.38520989, 0.37793794, 0.37062272,
     &   0.36326882, 0.35590875, 0.34854671, 0.34117097, 0.33379012,
     &   0.32642135, 0.31904757, 0.31160146, 0.30418992, 0.29681903,
     &   0.28956163, 0.28236568, 0.27523109, 0.26822975, 0.26131544,
     &   0.25447536, 0.24769610, 0.24099973, 0.23439258, 0.22788079,
     &   0.22146688, 0.21516205, 0.20896754, 0.20288108, 0.19691488,
     &   0.19106369, 0.18537574, 0.17983045, 0.17440125, 0.16908896,
     &   0.16389233, 0.15878820, 0.15380514, 0.14893770, 0.14419208,
     &   0.13955739, 0.13504395, 0.13065058, 0.12637830, 0.12222937,
     &   0.11819944, 0.11428691, 0.11048917, 0.10680356, 0.10322997/
      data (gg(i,9),i=1,mxwv)/
     &   0.86570579, 0.86627316, 0.86681956, 0.86733145, 0.86779755,
     &   0.86828339, 0.86873442, 0.86917073, 0.86959422, 0.86997551,
     &   0.87033057, 0.87068945, 0.87102950, 0.87134981, 0.87164986,
     &   0.87193936, 0.87222141, 0.87248987, 0.87274337, 0.87298322,
     &   0.87321627, 0.87342876, 0.87364298, 0.87382525, 0.87402141,
     &   0.87419945, 0.87437236, 0.87452382, 0.87468487, 0.87482899,
     &   0.87496865, 0.87510151, 0.87523818, 0.87533218, 0.87542295,
     &   0.87555480, 0.87566102, 0.87573594, 0.87580919, 0.87588662,
     &   0.87596458, 0.87602592, 0.87608939, 0.87615889, 0.87621909,
     &   0.87626052, 0.87630016, 0.87633586, 0.87633747, 0.87636369,
     &   0.87635893, 0.87637556, 0.87636262, 0.87635684, 0.87632316,
     &   0.87631983, 0.87628138, 0.87622112, 0.87620938, 0.87614995,
     &   0.87611711, 0.87609148, 0.87602305, 0.87599266, 0.87593222,
     &   0.87587076, 0.87583029, 0.87590492, 0.87587899, 0.87580025,
     &   0.87569672, 0.87557459, 0.87539381, 0.87531763, 0.87524295,
     &   0.87516278, 0.87510377, 0.87515438, 0.87537348, 0.87545365,
     &   0.87538868, 0.87531090, 0.87518978, 0.87507123, 0.87499201,
     &   0.87508547, 0.87531817, 0.87619644, 0.87712485, 0.88037568,
     &   0.88756454, 0.88851970, 0.88688254, 0.88316619, 0.88072366,
     &   0.87918729, 0.87817532, 0.87737513, 0.87693816, 0.87667322,
     &   0.87677860, 0.87715858, 0.87776077, 0.87876087, 0.87905347,
     &   0.88162667, 0.89844346, 0.91603237, 0.91482747, 0.90577930,
     &   0.89671189, 0.89194459, 0.88841474, 0.88663977, 0.88593906,
     &   0.88613766, 0.88749206, 0.88974845, 0.89274335, 0.89725423,
     &   0.90302020, 0.91041005, 0.91766083, 0.92147869, 0.92741328,
     &   0.94565761, 0.97404289, 0.98733282, 0.98675025, 0.98076940,
     &   0.97032666, 0.96062624, 0.95549929, 0.95196092, 0.95065993,
     &   0.95170647, 0.95391148, 0.95605910, 0.95706600, 0.95487237,
     &   0.94684064, 0.93207133, 0.91532111, 0.90266180, 0.89436305,
     &   0.89169329, 0.89253563, 0.89550281, 0.89964533, 0.90519583,
     &   0.91019273, 0.91599697, 0.92174923, 0.92760092, 0.93319523,
     &   0.93815994, 0.94265550, 0.94579136, 0.94539768, 0.94282341,
     &   0.93912029, 0.93525386, 0.93067020, 0.92643458, 0.92348337,
     &   0.92311096, 0.92619044, 0.93172222, 0.94134641, 0.95450735,
     &   0.96872693, 0.97408485, 0.96965992, 0.96367317, 0.96263880,
     &   0.96023405, 0.95755613, 0.95488662, 0.95277768, 0.95147562,
     &   0.95044219, 0.94961959, 0.94897753, 0.94829738, 0.94769430,
     &   0.94709843, 0.94664198, 0.94635075, 0.94598436, 0.94573158,
     &   0.94558370, 0.94554502, 0.94568688, 0.94602907, 0.94659823,
     &   0.94744015, 0.94854909, 0.94997758, 0.95179760, 0.95411015,
     &   0.95686358, 0.96048772, 0.96415257, 0.96859968, 0.97204489,
     &   0.97432017, 0.97534108, 0.97540921, 0.97437757, 0.97236228,
     &   0.96977526, 0.96604151, 0.96215922, 0.95817345, 0.95455498,
     &   0.95106894, 0.94773823, 0.94461238, 0.94189209, 0.93925470,
     &   0.93686682, 0.93478042, 0.93263322, 0.93073279, 0.92890084,
     &   0.92705578, 0.92522466, 0.92343563, 0.92186278, 0.92028093,
     &   0.91874194, 0.91750497, 0.91614407, 0.91510576, 0.91391778,
     &   0.91268128, 0.91181505, 0.91094255, 0.91014570, 0.90952241,
     &   0.90920025, 0.90843731, 0.90762824, 0.90674227, 0.90578622,
     &   0.90478575, 0.90369719, 0.90255398, 0.90146142, 0.90048724,
     &   0.89943242, 0.89842397, 0.89734226, 0.89626122, 0.89513749,
     &   0.89399552, 0.89285290, 0.89166564, 0.89048034, 0.88925511,
     &   0.88807750, 0.88688982, 0.88579792, 0.88471991, 0.88372427,
     &   0.88274103, 0.88177294, 0.88078195, 0.87977344, 0.87874413,
     &   0.87771624, 0.87670648, 0.87565374, 0.87454766, 0.87340820,
     &   0.87218004, 0.87082809, 0.86915582, 0.86731839, 0.86533248,
     &   0.86313832, 0.86066127, 0.85793573, 0.85503560, 0.85195470,
     &   0.84867424, 0.84522712, 0.84162420, 0.83772933, 0.83379042,
     &   0.82990581, 0.82601041, 0.82195979, 0.81769669, 0.81340295,
     &   0.80921412, 0.80497015, 0.80079460, 0.79654425, 0.79218918,
     &   0.78776568, 0.78338808, 0.77895194, 0.77465773, 0.77028584,
     &   0.76586223, 0.76136482, 0.75679362, 0.75217336, 0.74748415,
     &   0.74281126, 0.73808306, 0.73331618, 0.72850215, 0.72358161,
     &   0.71848762, 0.71337032, 0.70822829, 0.70311564, 0.69794226,
     &   0.69273740, 0.68753779, 0.68242276, 0.67725766, 0.67207170,
     &   0.66687483, 0.66165996, 0.65639079, 0.65114713, 0.64594370,
     &   0.64076197, 0.63560587, 0.63059086, 0.62553954, 0.62045336,
     &   0.61533380, 0.61018234, 0.60496104, 0.59967589, 0.59433323,
     &   0.58893341, 0.58347678, 0.57783246, 0.57211447, 0.56634384,
     &   0.56052184, 0.55464941, 0.54872745, 0.54275668, 0.53673822,
     &   0.53065979, 0.52444249, 0.51817000, 0.51184350, 0.50545025,
     &   0.49895629, 0.49241287, 0.48582074, 0.47917122, 0.47246668,
     &   0.46571642, 0.45889390, 0.45190099, 0.44486344, 0.43778354,
     &   0.43065661, 0.42350060, 0.41631943, 0.40917939, 0.40204087,
     &   0.39489061, 0.38770762, 0.38051581, 0.37331700, 0.36610496,
     &   0.35889679, 0.35170040, 0.34451881, 0.33735266, 0.33017612,
     &   0.32302499, 0.31597349, 0.30899784, 0.30206871, 0.29520088,
     &   0.28837287, 0.28154647, 0.27477068, 0.26804280, 0.26137316,
     &   0.25473723, 0.24816571, 0.24166402, 0.23523824, 0.22888158,
     &   0.22261043, 0.21642868, 0.21033935, 0.20434508, 0.19845000/
      data (gg(i,10),i=1,mxwv)/
     &   0.86686021, 0.86744881, 0.86799252, 0.86853611, 0.86895543,
     &   0.86953557, 0.87000090, 0.87045169, 0.87083346, 0.87130159,
     &   0.87169057, 0.87206268, 0.87244010, 0.87277144, 0.87309670,
     &   0.87341088, 0.87371850, 0.87400341, 0.87433618, 0.87454772,
     &   0.87478763, 0.87503749, 0.87527174, 0.87549752, 0.87571692,
     &   0.87591606, 0.87611336, 0.87630445, 0.87646973, 0.87664890,
     &   0.87680864, 0.87697446, 0.87712395, 0.87727755, 0.87740767,
     &   0.87753981, 0.87766117, 0.87779158, 0.87790227, 0.87802625,
     &   0.87811464, 0.87821335, 0.87831587, 0.87840378, 0.87847877,
     &   0.87856746, 0.87864316, 0.87869686, 0.87875730, 0.87880206,
     &   0.87884343, 0.87888986, 0.87892693, 0.87895787, 0.87896967,
     &   0.87899965, 0.87898052, 0.87894756, 0.87899673, 0.87901521,
     &   0.87899166, 0.87898177, 0.87898558, 0.87897056, 0.87895143,
     &   0.87895536, 0.87902099, 0.87911314, 0.87916017, 0.87910670,
     &   0.87905478, 0.87896180, 0.87888515, 0.87879819, 0.87875277,
     &   0.87874502, 0.87876129, 0.87886900, 0.87927407, 0.87939793,
     &   0.87940049, 0.87935853, 0.87928092, 0.87917894, 0.87921232,
     &   0.87938583, 0.87977445, 0.88099146, 0.88227248, 0.88656443,
     &   0.89577186, 0.89704549, 0.89507163, 0.89045733, 0.88740164,
     &   0.88548803, 0.88425052, 0.88326609, 0.88282037, 0.88249201,
     &   0.88268197, 0.88320935, 0.88403571, 0.88537598, 0.88576812,
     &   0.88906181, 0.90978378, 0.92973238, 0.92852092, 0.91851294,
     &   0.90793192, 0.90216047, 0.89778507, 0.89552546, 0.89457971,
     &   0.89474243, 0.89632291, 0.89899188, 0.90251720, 0.90781522,
     &   0.91447639, 0.92282015, 0.93070608, 0.93453318, 0.94044930,
     &   0.95837849, 0.98088449, 0.98866546, 0.98750383, 0.98151189,
     &   0.97109073, 0.96138501, 0.95625317, 0.95271593, 0.95142490,
     &   0.95249093, 0.95473087, 0.95703691, 0.95912594, 0.95972371,
     &   0.95663917, 0.94748896, 0.93396294, 0.92200840, 0.91344202,
     &   0.91050118, 0.91131347, 0.91441357, 0.91874492, 0.92439854,
     &   0.92937386, 0.93496805, 0.94029838, 0.94547522, 0.95017070,
     &   0.95411140, 0.95747131, 0.95968848, 0.95947570, 0.95779967,
     &   0.95526469, 0.95248771, 0.94899625, 0.94561923, 0.94323462,
     &   0.94311613, 0.94609022, 0.95112145, 0.95909560, 0.96844149,
     &   0.97612560, 0.97728866, 0.97189605, 0.96588910, 0.96636462,
     &   0.96668011, 0.96625668, 0.96554589, 0.96492714, 0.96460730,
     &   0.96437061, 0.96422642, 0.96413827, 0.96402669, 0.96393830,
     &   0.96385384, 0.96383405, 0.96388811, 0.96388161, 0.96391714,
     &   0.96398133, 0.96407139, 0.96422249, 0.96442050, 0.96470135,
     &   0.96505171, 0.96548605, 0.96602982, 0.96672130, 0.96765304,
     &   0.96882635, 0.97057033, 0.97255701, 0.97531992, 0.97769487,
     &   0.97937530, 0.98013508, 0.98010373, 0.97909510, 0.97716653,
     &   0.97469872, 0.97110492, 0.96736628, 0.96352291, 0.96004313,
     &   0.95669538, 0.95350122, 0.95051217, 0.94792974, 0.94543296,
     &   0.94318873, 0.94125086, 0.93925613, 0.93751335, 0.93584430,
     &   0.93416697, 0.93250906, 0.93089908, 0.92951322, 0.92812437,
     &   0.92678648, 0.92576188, 0.92462009, 0.92381424, 0.92286450,
     &   0.92187721, 0.92128247, 0.92069417, 0.92019409, 0.91988826,
     &   0.91991156, 0.91948193, 0.91902107, 0.91849655, 0.91791737,
     &   0.91730845, 0.91663092, 0.91592097, 0.91529202, 0.91483450,
     &   0.91432649, 0.91387600, 0.91338164, 0.91292650, 0.91246617,
     &   0.91202241, 0.91161251, 0.91118222, 0.91076481, 0.91032660,
     &   0.90993643, 0.90951794, 0.90912014, 0.90867430, 0.90820146,
     &   0.90767443, 0.90709656, 0.90644801, 0.90576094, 0.90504396,
     &   0.90431547, 0.90357941, 0.90278995, 0.90195560, 0.90111178,
     &   0.90018398, 0.89914638, 0.89785725, 0.89645648, 0.89495516,
     &   0.89327663, 0.89136374, 0.88924867, 0.88701135, 0.88462818,
     &   0.88208151, 0.87940365, 0.87660491, 0.87355155, 0.87050384,
     &   0.86753935, 0.86458558, 0.86150503, 0.85824525, 0.85498452,
     &   0.85187787, 0.84874016, 0.84569210, 0.84259349, 0.83942139,
     &   0.83620602, 0.83307183, 0.82990098, 0.82688332, 0.82380325,
     &   0.82068020, 0.81750131, 0.81426382, 0.81099129, 0.80765885,
     &   0.80434060, 0.80096751, 0.79755175, 0.79408419, 0.79051226,
     &   0.78677911, 0.78299886, 0.77916563, 0.77532864, 0.77140749,
     &   0.76741654, 0.76337540, 0.75936723, 0.75529140, 0.75115168,
     &   0.74695116, 0.74272257, 0.73838937, 0.73400855, 0.72963804,
     &   0.72528166, 0.72092873, 0.71672416, 0.71247023, 0.70816988,
     &   0.70382553, 0.69943988, 0.69525683, 0.69108391, 0.68686813,
     &   0.68260872, 0.67830497, 0.67383534, 0.66929805, 0.66471082,
     &   0.66006964, 0.65538663, 0.65065324, 0.64586812, 0.64103103,
     &   0.63614351, 0.63120854, 0.62621135, 0.62115180, 0.61601764,
     &   0.61077696, 0.60547191, 0.60010302, 0.59467161, 0.58917648,
     &   0.58361703, 0.57797801, 0.57219887, 0.56634778, 0.56042576,
     &   0.55435091, 0.54820609, 0.54200029, 0.53574818, 0.52945143,
     &   0.52310598, 0.51670229, 0.51025212, 0.50375324, 0.49719042,
     &   0.49058831, 0.48394558, 0.47726658, 0.47055566, 0.46375072,
     &   0.45691743, 0.45010355, 0.44330046, 0.43650109, 0.42973113,
     &   0.42295146, 0.41613054, 0.40930209, 0.40246725, 0.39562780,
     &   0.38874793, 0.38186124, 0.37497419, 0.36808857, 0.36116758,
     &   0.35425279, 0.34734738, 0.34045577, 0.33358216, 0.32672521/
      data (gg(i,11),i=1,mxwv)/
     &   0.86769879, 0.86830479, 0.86887866, 0.86941910, 0.86995268,
     &   0.87045628, 0.87093663, 0.87141192, 0.87185776, 0.87228376,
     &   0.87268829, 0.87308472, 0.87345773, 0.87382299, 0.87425536,
     &   0.87450135, 0.87481707, 0.87512726, 0.87541640, 0.87570256,
     &   0.87597644, 0.87623835, 0.87648392, 0.87671274, 0.87696451,
     &   0.87718719, 0.87741238, 0.87760931, 0.87781465, 0.87800986,
     &   0.87819755, 0.87837833, 0.87855524, 0.87872010, 0.87889177,
     &   0.87902391, 0.87918544, 0.87932426, 0.87946743, 0.87959832,
     &   0.87972724, 0.87984902, 0.87997448, 0.88007826, 0.88019824,
     &   0.88029712, 0.88040406, 0.88048887, 0.88057297, 0.88065052,
     &   0.88072395, 0.88078338, 0.88085365, 0.88091004, 0.88097358,
     &   0.88100523, 0.88103610, 0.88106620, 0.88109136, 0.88113493,
     &   0.88116443, 0.88117802, 0.88122392, 0.88124335, 0.88126785,
     &   0.88131064, 0.88143629, 0.88160843, 0.88170332, 0.88170034,
     &   0.88165879, 0.88158000, 0.88150090, 0.88144118, 0.88143736,
     &   0.88146198, 0.88153738, 0.88174969, 0.88216257, 0.88254052,
     &   0.88259017, 0.88258922, 0.88254511, 0.88248152, 0.88253087,
     &   0.88282365, 0.88337606, 0.88502938, 0.88673556, 0.89237070,
     &   0.90404981, 0.90566897, 0.90328348, 0.89754307, 0.89368612,
     &   0.89125937, 0.88969344, 0.88857144, 0.88782907, 0.88749391,
     &   0.88775873, 0.88845557, 0.88953441, 0.89125228, 0.89176673,
     &   0.89594769, 0.92105860, 0.94258136, 0.94139826, 0.93092847,
     &   0.91901135, 0.91216093, 0.90679568, 0.90392965, 0.90265220,
     &   0.90273219, 0.90455574, 0.90768677, 0.91179812, 0.91794336,
     &   0.92549652, 0.93465507, 0.94290626, 0.94661087, 0.95227814,
     &   0.96858722, 0.98432010, 0.98923337, 0.98796481, 0.98199207,
     &   0.97157800, 0.96185619, 0.95671403, 0.95317286, 0.95188731,
     &   0.95296842, 0.95522892, 0.95757639, 0.95996046, 0.96166778,
     &   0.96148670, 0.95733601, 0.94833308, 0.93849123, 0.93051738,
     &   0.92757487, 0.92833453, 0.93136650, 0.93552524, 0.94083095,
     &   0.94531018, 0.95011455, 0.95444673, 0.95839572, 0.96173847,
     &   0.96434933, 0.96641308, 0.96766859, 0.96757686, 0.96670902,
     &   0.96532935, 0.96372485, 0.96155018, 0.95930046, 0.95767146,
     &   0.95772099, 0.96012956, 0.96397567, 0.96953607, 0.97522020,
     &   0.97902775, 0.97873878, 0.97315747, 0.96715951, 0.96802419,
     &   0.96931630, 0.96988648, 0.97014832, 0.97031194, 0.97050172,
     &   0.97067177, 0.97085249, 0.97103441, 0.97119737, 0.97136986,
     &   0.97155011, 0.97176301, 0.97201234, 0.97224039, 0.97249955,
     &   0.97278112, 0.97308803, 0.97342920, 0.97380525, 0.97421300,
     &   0.97465765, 0.97513765, 0.97565758, 0.97623014, 0.97689450,
     &   0.97763813, 0.97862279, 0.97965258, 0.98107171, 0.98230588,
     &   0.98315412, 0.98346281, 0.98317885, 0.98208773, 0.98016989,
     &   0.97776127, 0.97425348, 0.97060800, 0.96685737, 0.96346617,
     &   0.96020532, 0.95709550, 0.95418948, 0.95168948, 0.94927609,
     &   0.94711614, 0.94526505, 0.94335914, 0.94170755, 0.94013232,
     &   0.93855172, 0.93699270, 0.93548453, 0.93420422, 0.93292421,
     &   0.93169850, 0.93079114, 0.92976981, 0.92909050, 0.92827064,
     &   0.92741728, 0.92696416, 0.92652309, 0.92617661, 0.92603266,
     &   0.92622972, 0.92597336, 0.92569226, 0.92535400, 0.92496747,
     &   0.92455876, 0.92408890, 0.92359430, 0.92319554, 0.92298812,
     &   0.92274362, 0.92256969, 0.92236710, 0.92222512, 0.92209977,
     &   0.92201525, 0.92199337, 0.92197663, 0.92199928, 0.92203152,
     &   0.92214644, 0.92228144, 0.92238599, 0.92247021, 0.92249405,
     &   0.92243588, 0.92230439, 0.92206103, 0.92175967, 0.92141414,
     &   0.92103702, 0.92061985, 0.92010897, 0.91953057, 0.91894144,
     &   0.91822892, 0.91737491, 0.91628599, 0.91511405, 0.91386843,
     &   0.91244704, 0.91081208, 0.90900815, 0.90712595, 0.90512133,
     &   0.90297800, 0.90073067, 0.89838773, 0.89581561, 0.89330280,
     &   0.89090312, 0.88853091, 0.88604951, 0.88340968, 0.88084328,
     &   0.87839675, 0.87598664, 0.87369108, 0.87136465, 0.86899370,
     &   0.86660266, 0.86434829, 0.86208344, 0.86000967, 0.85789454,
     &   0.85574782, 0.85357600, 0.85137928, 0.84917855, 0.84695011,
     &   0.84477043, 0.84255904, 0.84032351, 0.83805454, 0.83570200,
     &   0.83320785, 0.83073115, 0.82823753, 0.82573926, 0.82316840,
     &   0.82054639, 0.81789088, 0.81523770, 0.81248903, 0.80966538,
     &   0.80676520, 0.80378997, 0.80069959, 0.79753119, 0.79430979,
     &   0.79103601, 0.78770423, 0.78441918, 0.78103977, 0.77756798,
     &   0.77400637, 0.77035755, 0.76690704, 0.76346880, 0.75998688,
     &   0.75646311, 0.75289893, 0.74922132, 0.74549109, 0.74172008,
     &   0.73790908, 0.73406154, 0.73018110, 0.72626269, 0.72230655,
     &   0.71832496, 0.71438324, 0.71040094, 0.70637739, 0.70230538,
     &   0.69816750, 0.69398296, 0.68975061, 0.68547952, 0.68116546,
     &   0.67679948, 0.67238116, 0.66790652, 0.66336769, 0.65876353,
     &   0.65398246, 0.64912528, 0.64420074, 0.63917285, 0.63407224,
     &   0.62890851, 0.62368530, 0.61839801, 0.61304212, 0.60760248,
     &   0.60209876, 0.59652245, 0.59087747, 0.58517408, 0.57934386,
     &   0.57345390, 0.56749666, 0.56148392, 0.55543697, 0.54937977,
     &   0.54328197, 0.53714406, 0.53096622, 0.52475727, 0.51851255,
     &   0.51221269, 0.50587606, 0.49950999, 0.49311405, 0.48664403,
     &   0.48014596, 0.47362301, 0.46708125, 0.46052697, 0.45395344/
      data (gg(i,12),i=1,mxwv)/
     &   0.86831170, 0.86892611, 0.86950588, 0.87006736, 0.87061650,
     &   0.87112576, 0.87161839, 0.87210166, 0.87257916, 0.87300217,
     &   0.87342572, 0.87382627, 0.87422889, 0.87459022, 0.87494785,
     &   0.87529498, 0.87562650, 0.87594783, 0.87625712, 0.87655067,
     &   0.87683904, 0.87711751, 0.87738538, 0.87764758, 0.87789375,
     &   0.87813270, 0.87836349, 0.87859350, 0.87880582, 0.87901694,
     &   0.87921977, 0.87941945, 0.87959969, 0.87979174, 0.87996984,
     &   0.88013351, 0.88031197, 0.88047129, 0.88062680, 0.88077426,
     &   0.88092113, 0.88106740, 0.88121545, 0.88134259, 0.88147074,
     &   0.88160163, 0.88170147, 0.88179803, 0.88192976, 0.88202780,
     &   0.88212055, 0.88221151, 0.88230157, 0.88238519, 0.88245958,
     &   0.88251948, 0.88257474, 0.88263494, 0.88268197, 0.88274676,
     &   0.88280630, 0.88286680, 0.88291723, 0.88296872, 0.88302237,
     &   0.88309300, 0.88327658, 0.88354516, 0.88369679, 0.88370341,
     &   0.88367778, 0.88361090, 0.88352108, 0.88346839, 0.88348472,
     &   0.88355798, 0.88367873, 0.88399142, 0.88477212, 0.88510680,
     &   0.88520294, 0.88522756, 0.88519591, 0.88513058, 0.88582873,
     &   0.88569438, 0.88639212, 0.88861156, 0.89090234, 0.89821237,
     &   0.91279000, 0.91476500, 0.91191447, 0.90486270, 0.90000987,
     &   0.89689684, 0.89489949, 0.89328557, 0.89248276, 0.89204133,
     &   0.89237589, 0.89326817, 0.89465004, 0.89682239, 0.89747024,
     &   0.90274251, 0.93240190, 0.95398837, 0.95294517, 0.94287133,
     &   0.93035108, 0.92231518, 0.91589338, 0.91232193, 0.91063148,
     &   0.91058540, 0.91265982, 0.91629642, 0.92102641, 0.92801011,
     &   0.93632239, 0.94594473, 0.95408165, 0.95748490, 0.96259135,
     &   0.97595477, 0.98579675, 0.98953760, 0.98826092, 0.98230350,
     &   0.97188365, 0.96213871, 0.95698321, 0.95343494, 0.95215195,
     &   0.95324516, 0.95552206, 0.95788860, 0.96034259, 0.96238190,
     &   0.96345633, 0.96254766, 0.95800221, 0.95132494, 0.94493496,
     &   0.94237447, 0.94307411, 0.94578785, 0.94940567, 0.95381033,
     &   0.95734525, 0.96090704, 0.96390760, 0.96644437, 0.96843421,
     &   0.96987444, 0.97092569, 0.97149986, 0.97147447, 0.97115546,
     &   0.97062767, 0.96997440, 0.96898466, 0.96787870, 0.96708417,
     &   0.96733034, 0.96907258, 0.97164625, 0.97502488, 0.97819084,
     &   0.98021048, 0.97953385, 0.97394341, 0.96795470, 0.96890521,
     &   0.97047478, 0.97139144, 0.97203684, 0.97253901, 0.97295654,
     &   0.97330928, 0.97363591, 0.97393155, 0.97421092, 0.97448736,
     &   0.97477245, 0.97507119, 0.97538418, 0.97569376, 0.97602737,
     &   0.97637981, 0.97675818, 0.97716480, 0.97760242, 0.97806376,
     &   0.97856259, 0.97909743, 0.97966540, 0.98029608, 0.98098314,
     &   0.98172832, 0.98261595, 0.98342681, 0.98439896, 0.98514861,
     &   0.98555487, 0.98559064, 0.98513323, 0.98397642, 0.98206002,
     &   0.97968984, 0.97624022, 0.97265702, 0.96896666, 0.96563119,
     &   0.96242404, 0.95936501, 0.95650822, 0.95405662, 0.95169181,
     &   0.94958073, 0.94778019, 0.94592583, 0.94432724, 0.94280648,
     &   0.94128221, 0.93978047, 0.93833113, 0.93711191, 0.93589497,
     &   0.93473387, 0.93389410, 0.93294197, 0.93233526, 0.93158978,
     &   0.93081254, 0.93043905, 0.93008006, 0.92981893, 0.92976379,
     &   0.93005407, 0.92989242, 0.92970878, 0.92947048, 0.92918658,
     &   0.92888361, 0.92852235, 0.92813957, 0.92785692, 0.92777091,
     &   0.92765224, 0.92760998, 0.92754418, 0.92754591, 0.92757118,
     &   0.92764586, 0.92779255, 0.92795444, 0.92816734, 0.92840248,
     &   0.92873639, 0.92907625, 0.92945611, 0.92979825, 0.93008447,
     &   0.93029219, 0.93042833, 0.93044806, 0.93041074, 0.93033087,
     &   0.93021911, 0.93006051, 0.92979568, 0.92945373, 0.92910099,
     &   0.92860514, 0.92794603, 0.92704523, 0.92606854, 0.92502654,
     &   0.92380172, 0.92236531, 0.92077053, 0.91911727, 0.91735113,
     &   0.91545671, 0.91347104, 0.91140532, 0.90911734, 0.90691692,
     &   0.90484828, 0.90281838, 0.90068787, 0.89840829, 0.89616877,
     &   0.89418095, 0.89219296, 0.89033717, 0.88846296, 0.88656044,
     &   0.88465244, 0.88291430, 0.88118339, 0.87967390, 0.87813967,
     &   0.87658715, 0.87503219, 0.87347674, 0.87194699, 0.87041092,
     &   0.86896044, 0.86750668, 0.86605018, 0.86458904, 0.86306953,
     &   0.86142761, 0.85986912, 0.85834521, 0.85684979, 0.85531831,
     &   0.85378611, 0.85228217, 0.85081601, 0.84927106, 0.84769505,
     &   0.84609044, 0.84441513, 0.84267652, 0.84092027, 0.83912432,
     &   0.83726996, 0.83536339, 0.83346951, 0.83148152, 0.82939583,
     &   0.82720900, 0.82491755, 0.82259738, 0.82022142, 0.81777608,
     &   0.81526119, 0.81267691, 0.80996311, 0.80717444, 0.80432075,
     &   0.80140257, 0.79842228, 0.79538316, 0.79228258, 0.78912175,
     &   0.78591174, 0.78271145, 0.77946240, 0.77616566, 0.77282053,
     &   0.76942307, 0.76598138, 0.76249629, 0.75897950, 0.75542951,
     &   0.75183886, 0.74821746, 0.74459738, 0.74093604, 0.73723286,
     &   0.73339868, 0.72951096, 0.72557640, 0.72153991, 0.71744168,
     &   0.71329522, 0.70911330, 0.70488280, 0.70059961, 0.69625455,
     &   0.69185656, 0.68739510, 0.68287206, 0.67829376, 0.67362326,
     &   0.66889244, 0.66405725, 0.65913820, 0.65416813, 0.64915788,
     &   0.64409477, 0.63900304, 0.63385499, 0.62866044, 0.62341017,
     &   0.61810613, 0.61274499, 0.60733068, 0.60186023, 0.59630424,
     &   0.59069055, 0.58502078, 0.57929963, 0.57353246, 0.56771332/
      data (gg(i,13),i=1,mxwv)/
     &   0.86874950, 0.86937523, 0.86996394, 0.87053841, 0.87108672,
     &   0.87160975, 0.87211424, 0.87260741, 0.87307900, 0.87352484,
     &   0.87395179, 0.87436712, 0.87476403, 0.87514901, 0.87551814,
     &   0.87583959, 0.87621832, 0.87654752, 0.87686646, 0.87718022,
     &   0.87747312, 0.87776649, 0.87804168, 0.87830687, 0.87856972,
     &   0.87882382, 0.87906879, 0.87929326, 0.87953258, 0.87975377,
     &   0.87996948, 0.88017637, 0.88038152, 0.88057828, 0.88076061,
     &   0.88095659, 0.88113701, 0.88130826, 0.88147444, 0.88164258,
     &   0.88178986, 0.88196921, 0.88212043, 0.88227338, 0.88241756,
     &   0.88255817, 0.88268816, 0.88281339, 0.88293791, 0.88305181,
     &   0.88315976, 0.88326931, 0.88337559, 0.88348174, 0.88357466,
     &   0.88365263, 0.88372821, 0.88379669, 0.88387007, 0.88394415,
     &   0.88403374, 0.88411933, 0.88418859, 0.88428080, 0.88434279,
     &   0.88445109, 0.88470691, 0.88507169, 0.88528919, 0.88531703,
     &   0.88527870, 0.88518900, 0.88509685, 0.88503420, 0.88506156,
     &   0.88517106, 0.88534236, 0.88576943, 0.88687420, 0.88729548,
     &   0.88743240, 0.88747567, 0.88744146, 0.88736647, 0.88749766,
     &   0.88804781, 0.88907462, 0.89202386, 0.89513886, 0.90447342,
     &   0.92223132, 0.92451614, 0.92123073, 0.91278315, 0.90676606,
     &   0.90284044, 0.90024561, 0.89813787, 0.89707673, 0.89647794,
     &   0.89688146, 0.89800590, 0.89975536, 0.90253252, 0.90327531,
     &   0.90980607, 0.94359696, 0.96316302, 0.96236646, 0.95378017,
     &   0.94128442, 0.93276876, 0.92537850, 0.92106515, 0.91890836,
     &   0.91870040, 0.92103314, 0.92517585, 0.93048990, 0.93817258,
     &   0.94690663, 0.95637083, 0.96372402, 0.96663129, 0.97086459,
     &   0.98050284, 0.98635459, 0.98972684, 0.98845464, 0.98250371,
     &   0.97206926, 0.96229768, 0.95712745, 0.95357060, 0.95228815,
     &   0.95339119, 0.95568192, 0.95806253, 0.96053821, 0.96265823,
     &   0.96413678, 0.96477824, 0.96341497, 0.96000099, 0.95585865,
     &   0.95408416, 0.95473015, 0.95690602, 0.95966935, 0.96282035,
     &   0.96521020, 0.96744937, 0.96921140, 0.97059846, 0.97162348,
     &   0.97232646, 0.97280991, 0.97304076, 0.97304714, 0.97300011,
     &   0.97293079, 0.97284108, 0.97263026, 0.97235894, 0.97224039,
     &   0.97266018, 0.97385889, 0.97547901, 0.97746164, 0.97937697,
     &   0.98078090, 0.98002952, 0.97444737, 0.96845937, 0.96943128,
     &   0.97106755, 0.97208017, 0.97284484, 0.97346163, 0.97396207,
     &   0.97438413, 0.97476941, 0.97511011, 0.97543699, 0.97575569,
     &   0.97608328, 0.97641665, 0.97675514, 0.97709650, 0.97745770,
     &   0.97783530, 0.97823769, 0.97866422, 0.97911817, 0.97959244,
     &   0.98010278, 0.98064935, 0.98122925, 0.98187536, 0.98257619,
     &   0.98334038, 0.98423564, 0.98503250, 0.98593891, 0.98659509,
     &   0.98688847, 0.98684496, 0.98633164, 0.98515832, 0.98325598,
     &   0.98091769, 0.97750992, 0.97396809, 0.97031564, 0.96701366,
     &   0.96383786, 0.96080774, 0.95797849, 0.95555395, 0.95321625,
     &   0.95113254, 0.94936079, 0.94753569, 0.94596756, 0.94447821,
     &   0.94298655, 0.94151789, 0.94010264, 0.93891925, 0.93773943,
     &   0.93661636, 0.93581706, 0.93490613, 0.93434328, 0.93364257,
     &   0.93291098, 0.93258572, 0.93227649, 0.93206710, 0.93206584,
     &   0.93241268, 0.93230790, 0.93218237, 0.93200350, 0.93178052,
     &   0.93153977, 0.93124205, 0.93092436, 0.93070871, 0.93069190,
     &   0.93064398, 0.93067515, 0.93068463, 0.93076396, 0.93086922,
     &   0.93102646, 0.93125862, 0.93150884, 0.93181348, 0.93214464,
     &   0.93257880, 0.93302339, 0.93351418, 0.93397123, 0.93437696,
     &   0.93470854, 0.93497258, 0.93512315, 0.93522054, 0.93527919,
     &   0.93530971, 0.93529636, 0.93517762, 0.93498313, 0.93478125,
     &   0.93443388, 0.93392032, 0.93316239, 0.93233097, 0.93143696,
     &   0.93035889, 0.92906940, 0.92762434, 0.92612636, 0.92451924,
     &   0.92278761, 0.92096949, 0.91907638, 0.91696388, 0.91494834,
     &   0.91307294, 0.91124249, 0.90931600, 0.90724456, 0.90522027,
     &   0.90346152, 0.90170640, 0.90009391, 0.89846987, 0.89682484,
     &   0.89518207, 0.89372188, 0.89227760, 0.89106870, 0.88984412,
     &   0.88861006, 0.88738424, 0.88616949, 0.88499063, 0.88382292,
     &   0.88275450, 0.88169515, 0.88065046, 0.87961489, 0.87853497,
     &   0.87734401, 0.87626350, 0.87524211, 0.87427205, 0.87328798,
     &   0.87233150, 0.87143826, 0.87061542, 0.86973715, 0.86886311,
     &   0.86800015, 0.86709261, 0.86616564, 0.86528027, 0.86439383,
     &   0.86349124, 0.86257839, 0.86172563, 0.86082596, 0.85987681,
     &   0.85887539, 0.85781848, 0.85663241, 0.85538512, 0.85409141,
     &   0.85274929, 0.85135686, 0.84982216, 0.84822315, 0.84657174,
     &   0.84486634, 0.84310573, 0.84128928, 0.83941376, 0.83747768,
     &   0.83547878, 0.83341616, 0.83129555, 0.82911557, 0.82687014,
     &   0.82454526, 0.82216167, 0.81971902, 0.81722116, 0.81466764,
     &   0.81205553, 0.80938566, 0.80666482, 0.80389136, 0.80106574,
     &   0.79812801, 0.79513252, 0.79208440, 0.78894991, 0.78575200,
     &   0.78249961, 0.77920234, 0.77585262, 0.77244878, 0.76898670,
     &   0.76547277, 0.76190025, 0.75827098, 0.75459015, 0.75084156,
     &   0.74704069, 0.74314433, 0.73916888, 0.73514116, 0.73106354,
     &   0.72693551, 0.72278655, 0.71858615, 0.71434247, 0.71004838,
     &   0.70571768, 0.70133740, 0.69690901, 0.69242942, 0.68788868,
     &   0.68329453, 0.67864645, 0.67394644, 0.66919690, 0.66439301/

c     ice data

      data (qqi(i,1),i=1,mxwv)/
     &   2.17622876, 2.17840743, 2.18082023, 2.18302011, 2.18548632,
     &   2.18786287, 2.19010639, 2.19279814, 2.19516540, 2.19764400,
     &   2.20041800, 2.20275927, 2.20552278, 2.20830655, 2.21089935,
     &   2.21359396, 2.21642351, 2.21922898, 2.22207069, 2.22500086,
     &   2.22788191, 2.23084331, 2.23384261, 2.23687792, 2.23992443,
     &   2.24301553, 2.24619794, 2.24927902, 2.25247335, 2.25576949,
     &   2.25899458, 2.26225591, 2.26553059, 2.26884747, 2.27218223,
     &   2.27553463, 2.27889204, 2.28227782, 2.28567743, 2.28909802,
     &   2.29251885, 2.29597735, 2.29945612, 2.30296636, 2.30651546,
     &   2.31011558, 2.31378007, 2.31752086, 2.32136011, 2.32531667,
     &   2.32941508, 2.33368397, 2.33815837, 2.34287238, 2.34786654,
     &   2.35318375, 2.35891867, 2.36498785, 2.37157607, 2.37870026,
     &   2.38642001, 2.39478970, 2.40387583, 2.41373992, 2.42443871,
     &   2.43602967, 2.44856882, 2.46211982, 2.47670054, 2.49238348,
     &   2.50916147, 2.52708030, 2.54614997, 2.56637406, 2.58772302,
     &   2.61017585, 2.63372922, 2.65826130, 2.68377376, 2.71013165,
     &   2.73726797, 2.76503778, 2.79333735, 2.82198977, 2.85090923,
     &   2.87991333, 2.90880156, 2.93736267, 2.96550608, 2.99295211,
     &   3.01942420, 3.04412246, 3.06594467, 3.08665228, 3.10705876,
     &   3.12583137, 3.14284706, 3.15725636, 3.16872931, 3.17751122,
     &   3.18325520, 3.18570590, 3.18458247, 3.17972922, 3.17122531,
     &   3.15818191, 3.13956642, 3.11378121, 3.08274722, 3.05257297,
     &   3.02228522, 2.98808670, 2.94926620, 2.90207314, 2.84588504,
     &   2.78239703, 2.71123385, 2.63061213, 2.54344487, 2.44561505,
     &   2.33848596, 2.21706152, 2.07900643, 1.91886818, 1.72695100,
     &   1.48603010, 1.21580958, 0.91254288, 0.65315205, 0.60195059,
     &   1.00365186, 1.58387733, 1.88239682, 2.20290422, 2.48098278,
     &   2.62484097, 2.80629730, 2.98096180, 3.11367393, 3.18251824,
     &   3.15866184, 3.06346154, 2.93697429, 2.79611111, 2.66531825,
     &   2.53107524, 2.40736461, 2.29285240, 2.18345046, 2.08598804,
     &   1.99160314, 1.90376842, 1.81958508, 1.74686456, 1.69509971,
     &   1.66831744, 1.65986133, 1.61267698, 1.54809415, 1.46908534,
     &   1.39147377, 1.31403470, 1.24162161, 1.17426014, 1.10881054,
     &   1.04882538, 0.99551463, 0.95016962, 0.91752762, 0.89655596,
     &   0.91106629, 0.95512557, 0.98618406, 1.00193572, 0.99133372,
     &   0.96269393, 0.91042566, 0.87280458, 0.85396403, 0.83193302,
     &   0.80359888, 0.77466649, 0.74798989, 0.71396124, 0.68591940,
     &   0.66213685, 0.63786584, 0.60576016, 0.56587964, 0.51968181,
     &   0.47602180, 0.45039597, 0.42647868, 0.40444461, 0.38474351,
     &   0.36861840, 0.35014328, 0.32921728, 0.30804271, 0.28535312,
     &   0.26718712, 0.26637501, 0.28706035, 0.33275387, 0.43039846,
     &   0.57374614, 0.72907251, 0.85454166, 0.95909679, 1.03006446,
     &   1.08678842, 1.12298036, 1.14409387, 1.15106499, 1.13177979,
     &   1.10395014, 1.06566918, 1.00922060, 0.95248640, 0.89178675,
     &   0.82101321, 0.74446529, 0.66645521, 0.59811872, 0.53704667,
     &   0.48804408, 0.44788849, 0.41426891, 0.38678095, 0.36748898,
     &   0.34543756, 0.31804702, 0.28799936, 0.25950566, 0.23379852,
     &   0.21510667, 0.19971426, 0.19236670, 0.18734805, 0.17555737,
     &   0.15685396, 0.13550365, 0.11661259, 0.10335232, 0.09162591,
     &   0.08123308, 0.07280353, 0.06721842, 0.06320556, 0.06017902,
     &   0.05795182, 0.05598997, 0.05456356, 0.05396695, 0.05378709,
     &   0.05430345, 0.05572163, 0.05745927, 0.05951934, 0.06190668,
     &   0.06458828, 0.06724051, 0.07020132, 0.07347767, 0.07707818,
     &   0.08124653, 0.08608044, 0.09136911, 0.09713465, 0.10508864,
     &   0.11696254, 0.13124792, 0.14863731, 0.16540682, 0.18176426,
     &   0.19979778, 0.21965837, 0.23817964, 0.25584555, 0.27318257,
     &   0.29061607, 0.30880630, 0.32765067, 0.35039961, 0.37857530,
     &   0.39115745, 0.40933737, 0.42001513, 0.38951984, 0.33109453,
     &   0.28626463, 0.26747465, 0.25387740, 0.24094407, 0.22599810,
     &   0.21020570, 0.19229940, 0.16412415, 0.14013043, 0.11989243,
     &   0.12261753, 0.13981597, 0.16021460, 0.17959534, 0.17506924,
     &   0.17012131, 0.16349205, 0.15453941, 0.14214496, 0.12838894,
     &   0.11683186, 0.10673799, 0.09783781, 0.08968746, 0.08241411,
     &   0.07573187, 0.06907845, 0.06251855, 0.05657453, 0.05141868,
     &   0.04678006, 0.04380222, 0.04141493, 0.03917862, 0.03728823,
     &   0.03549027, 0.03383607, 0.03229109, 0.03081769, 0.02958884,
     &   0.02841510, 0.02732820, 0.02647163, 0.02564286, 0.02484093,
     &   0.02406491, 0.02331390, 0.02258705, 0.02188353, 0.02120255,
     &   0.02054335, 0.01990519, 0.01921475, 0.01853583, 0.01788126,
     &   0.01725013, 0.01664159, 0.01605481, 0.01548898, 0.01494335,
     &   0.01441717, 0.01390973, 0.01342036, 0.01294840, 0.01247364,
     &   0.01194608, 0.01144096, 0.01095730, 0.01049418, 0.01005072,
     &   0.00959127, 0.00893763, 0.00832855, 0.00776099, 0.00721005,
     &   0.00668365, 0.00619571, 0.00572624, 0.00525644, 0.00482520,
     &   0.00443282, 0.00411935, 0.00382806, 0.00355738, 0.00330431,
     &   0.00306892, 0.00285031, 0.00264949, 0.00246466, 0.00229273,
     &   0.00213532, 0.00199453, 0.00186304, 0.00174044, 0.00163781,
     &   0.00154123, 0.00145036, 0.00137313, 0.00130258, 0.00123566,
     &   0.00118145, 0.00113826, 0.00109665, 0.00105834, 0.00102656,
     &   0.00099573, 0.00096584, 0.00093824, 0.00091145, 0.00088544/
      data (qqi(i,2),i=1,mxwv)/
     &   2.13844347, 2.14020967, 2.14191651, 2.14379859, 2.14552712,
     &   2.14745545, 2.14914727, 2.15127063, 2.15299201, 2.15473437,
     &   2.15690827, 2.15877390, 2.16086197, 2.16280413, 2.16478133,
     &   2.16730762, 2.16917205, 2.17120576, 2.17324471, 2.17546558,
     &   2.17773080, 2.18014693, 2.18241286, 2.18474793, 2.18714905,
     &   2.18934178, 2.19198346, 2.19438457, 2.19686413, 2.19929576,
     &   2.20192432, 2.20472288, 2.20711827, 2.20986152, 2.21259117,
     &   2.21536827, 2.21810532, 2.22090411, 2.22387075, 2.22673106,
     &   2.22968006, 2.23255682, 2.23556828, 2.23860598, 2.24170113,
     &   2.24479318, 2.24790931, 2.25106692, 2.25423717, 2.25744915,
     &   2.26070762, 2.26395273, 2.26723504, 2.27052283, 2.27385092,
     &   2.27716708, 2.28051353, 2.28388762, 2.28726268, 2.29066586,
     &   2.29408693, 2.29754019, 2.30102706, 2.30455637, 2.30814004,
     &   2.31178784, 2.31551933, 2.31935453, 2.32331324, 2.32742500,
     &   2.33171844, 2.33622861, 2.34099531, 2.34606194, 2.35147524,
     &   2.35728669, 2.36355591, 2.37033868, 2.37770367, 2.38571095,
     &   2.39443612, 2.40393996, 2.41430068, 2.42557383, 2.43785000,
     &   2.45119357, 2.46566033, 2.48129749, 2.49821568, 2.51643324,
     &   2.53610277, 2.55700517, 2.57854009, 2.60104275, 2.62498665,
     &   2.65020180, 2.67701030, 2.70498109, 2.73373914, 2.76343942,
     &   2.79398084, 2.82515192, 2.85669684, 2.88839674, 2.91969538,
     &   2.95110106, 2.98144031, 3.00848579, 3.03000975, 3.05139709,
     &   3.07364488, 3.09542751, 3.11624742, 3.13357115, 3.14477921,
     &   3.14942765, 3.14683795, 3.13508582, 3.11346745, 3.08003068,
     &   3.03372025, 2.96922565, 2.88140988, 2.76193190, 2.59412169,
     &   2.34119678, 2.00220299, 1.54631793, 1.05867374, 0.82390863,
     &   1.22847044, 1.77170849, 2.01606393, 2.25810289, 2.45560288,
     &   2.54650140, 2.64603257, 2.75744605, 2.89142537, 3.03634715,
     &   3.16649795, 3.25008917, 3.29144359, 3.29079938, 3.26440048,
     &   3.22071433, 3.16458178, 3.09872317, 3.02458405, 2.95260167,
     &   2.87637329, 2.79716229, 2.71426439, 2.63025618, 2.55903172,
     &   2.51861954, 2.50597167, 2.49389577, 2.45457292, 2.37898636,
     &   2.29970217, 2.21191835, 2.12133861, 2.03167748, 1.94079149,
     &   1.84890532, 1.76070380, 1.67803979, 1.60632670, 1.54543018,
     &   1.52200961, 1.55972993, 1.59004354, 1.61774421, 1.62015855,
     &   1.59965158, 1.54210770, 1.48926651, 1.45810449, 1.42946732,
     &   1.39335346, 1.35423195, 1.31725109, 1.27142775, 1.22802567,
     &   1.19125080, 1.15622306, 1.11251223, 1.05680764, 0.98802310,
     &   0.91524905, 0.86232048, 0.81475025, 0.76977462, 0.72773045,
     &   0.69077808, 0.65147018, 0.60738575, 0.56211621, 0.51264364,
     &   0.46627894, 0.44185257, 0.44811985, 0.48908699, 0.60091811,
     &   0.77474719, 0.96311653, 1.11815608, 1.24877810, 1.34379911,
     &   1.42493224, 1.48635495, 1.53401840, 1.56883049, 1.57682109,
     &   1.57609081, 1.56286883, 1.52647507, 1.48473012, 1.43583250,
     &   1.37027097, 1.28872204, 1.19588041, 1.10519755, 1.01775694,
     &   0.94071412, 0.87438101, 0.81669694, 0.76722527, 0.73137200,
     &   0.69463980, 0.65020460, 0.59880012, 0.54828155, 0.50159025,
     &   0.46389210, 0.43090943, 0.41124824, 0.39788011, 0.37629962,
     &   0.34386116, 0.30570680, 0.26976365, 0.24215673, 0.21733588,
     &   0.19500025, 0.17597774, 0.16157694, 0.15015827, 0.14074734,
     &   0.13299713, 0.12598202, 0.12005414, 0.11564625, 0.11213247,
     &   0.10991973, 0.10929409, 0.10934937, 0.11007236, 0.11145350,
     &   0.11341944, 0.11541645, 0.11801048, 0.12119722, 0.12497555,
     &   0.12957683, 0.13514075, 0.14147094, 0.14858384, 0.15887335,
     &   0.17477009, 0.19426578, 0.21832667, 0.24159069, 0.26432636,
     &   0.28951985, 0.31739646, 0.34351948, 0.36854199, 0.39320028,
     &   0.41807386, 0.44410795, 0.47118396, 0.50413126, 0.54586518,
     &   0.56629974, 0.59570009, 0.61682498, 0.58111680, 0.49998391,
     &   0.43374577, 0.40518713, 0.38458696, 0.36533761, 0.34326765,
     &   0.31987113, 0.29343894, 0.25123379, 0.21432634, 0.18262807,
     &   0.18443300, 0.20829123, 0.23743929, 0.26605448, 0.26007557,
     &   0.25316370, 0.24369745, 0.23076627, 0.21266459, 0.19230510,
     &   0.17505014, 0.15990594, 0.14649780, 0.13422830, 0.12325247,
     &   0.11318128, 0.10316494, 0.09330244, 0.08437853, 0.07660814,
     &   0.06962012, 0.06507754, 0.06142005, 0.05800861, 0.05512039,
     &   0.05238067, 0.04986439, 0.04751896, 0.04528759, 0.04342024,
     &   0.04164085, 0.03999520, 0.03869200, 0.03743440, 0.03622060,
     &   0.03504890, 0.03391764, 0.03282528, 0.03177032, 0.03075133,
     &   0.02976696, 0.02881590, 0.02779409, 0.02679164, 0.02582652,
     &   0.02489725, 0.02400242, 0.02314070, 0.02231080, 0.02151148,
     &   0.02074157, 0.01999992, 0.01928545, 0.01859713, 0.01790612,
     &   0.01714122, 0.01640937, 0.01570912, 0.01503908, 0.01439792,
     &   0.01373478, 0.01279589, 0.01192130, 0.01110659, 0.01031613,
     &   0.00956121, 0.00886162, 0.00818886, 0.00751612, 0.00689872,
     &   0.00633703, 0.00588786, 0.00547059, 0.00508293, 0.00472060,
     &   0.00438367, 0.00407082, 0.00378348, 0.00351903, 0.00327310,
     &   0.00304795, 0.00284657, 0.00265851, 0.00248320, 0.00233635,
     &   0.00219820, 0.00206824, 0.00195772, 0.00185676, 0.00176102,
     &   0.00168337, 0.00162140, 0.00156174, 0.00150681, 0.00146117,
     &   0.00141694, 0.00137406, 0.00133448, 0.00129609, 0.00125882/
      data (qqi(i,3),i=1,mxwv)/
     &   2.10912704, 2.11047482, 2.11185670, 2.11333418, 2.11451697,
     &   2.11596370, 2.11755061, 2.11892009, 2.12046695, 2.12174582,
     &   2.12333250, 2.12470937, 2.12653637, 2.12807012, 2.12947583,
     &   2.13122869, 2.13274050, 2.13453484, 2.13634610, 2.13780355,
     &   2.13976932, 2.14120197, 2.14308929, 2.14486170, 2.14653039,
     &   2.14841819, 2.15024185, 2.15211272, 2.15407586, 2.15600681,
     &   2.15803742, 2.15996170, 2.16192961, 2.16433668, 2.16607332,
     &   2.16826630, 2.17059374, 2.17236471, 2.17468643, 2.17677665,
     &   2.17910814, 2.18134856, 2.18357873, 2.18591547, 2.18841672,
     &   2.19085431, 2.19314718, 2.19590282, 2.19823122, 2.20077944,
     &   2.20335174, 2.20599461, 2.20862412, 2.21133494, 2.21392989,
     &   2.21685743, 2.21964192, 2.22243190, 2.22529125, 2.22820115,
     &   2.23110461, 2.23414183, 2.23713183, 2.24016404, 2.24328041,
     &   2.24637675, 2.24951816, 2.25266671, 2.25586319, 2.25907397,
     &   2.26230240, 2.26554990, 2.26884675, 2.27213287, 2.27544141,
     &   2.27876210, 2.28209996, 2.28545332, 2.28883600, 2.29224110,
     &   2.29567170, 2.29914474, 2.30266285, 2.30624413, 2.30990386,
     &   2.31365919, 2.31753778, 2.32156610, 2.32578325, 2.33022785,
     &   2.33495879, 2.34002662, 2.34543991, 2.35125256, 2.35753846,
     &   2.36438370, 2.37193894, 2.38027859, 2.38945651, 2.39960217,
     &   2.41085887, 2.42334628, 2.43719149, 2.45252323, 2.46930289,
     &   2.48811483, 2.50900865, 2.53171468, 2.55479455, 2.57894468,
     &   2.60502243, 2.63386321, 2.66580629, 2.70135212, 2.73960590,
     &   2.77976322, 2.82174850, 2.86532068, 2.90805674, 2.95089626,
     &   2.99170876, 3.02866173, 3.05728602, 3.07104445, 3.05439281,
     &   2.96271729, 2.74242926, 2.29272008, 1.60469317, 1.09595001,
     &   1.45002508, 1.91166735, 2.09488654, 2.26609492, 2.39857769,
     &   2.45013642, 2.49635768, 2.53859949, 2.58617806, 2.64452219,
     &   2.72089672, 2.80699158, 2.89493060, 2.97395849, 3.03641391,
     &   3.09296346, 3.13498378, 3.16245985, 3.17771864, 3.18501472,
     &   3.18402195, 3.17172670, 3.14919043, 3.11012578, 3.06345654,
     &   3.03310537, 3.02376533, 3.06278372, 3.08195829, 3.06136203,
     &   3.03242683, 2.98750710, 2.92848277, 2.86144638, 2.78682327,
     &   2.69871545, 2.60421920, 2.50503850, 2.40495729, 2.30628633,
     &   2.22859573, 2.23733568, 2.25205088, 2.28566027, 2.30720615,
     &   2.30918622, 2.27060366, 2.21733618, 2.17954206, 2.15280604,
     &   2.12011790, 2.08144569, 2.04344058, 1.99806333, 1.94681633,
     &   1.90303886, 1.86425710, 1.81940436, 1.76005924, 1.67999816,
     &   1.58302963, 1.49788773, 1.42233217, 1.34870613, 1.27698743,
     &   1.21047115, 1.14201140, 1.06471801, 0.98346233, 0.89214396,
     &   0.79739177, 0.72496331, 0.69346803, 0.70915234, 0.82013714,
     &   1.01435554, 1.22508085, 1.40000474, 1.54653978, 1.65771794,
     &   1.75563025, 1.83641243, 1.90621173, 1.96637034, 2.00497699,
     &   2.03762293, 2.06005812, 2.06227136, 2.05674267, 2.04502392,
     &   2.01686335, 1.96677935, 1.89656687, 1.81441295, 1.72551215,
     &   1.63601422, 1.55328846, 1.47726953, 1.40784883, 1.35561085,
     &   1.30780327, 1.25126922, 1.18015587, 1.10612345, 1.03474343,
     &   0.96968007, 0.90869385, 0.86627674, 0.83786100, 0.80334443,
     &   0.75340998, 0.69105023, 0.62666255, 0.57183754, 0.52115196,
     &   0.47438848, 0.43252265, 0.39751506, 0.36805269, 0.34262568,
     &   0.32063994, 0.30045673, 0.28247720, 0.26732805, 0.25411877,
     &   0.24343042, 0.23564772, 0.22937135, 0.22454426, 0.22111660,
     &   0.21892329, 0.21695390, 0.21622698, 0.21670479, 0.21835579,
     &   0.22116962, 0.22535719, 0.23096238, 0.23797186, 0.24959111,
     &   0.26914024, 0.29423958, 0.32616884, 0.35726491, 0.38780433,
     &   0.42196566, 0.46006322, 0.49605936, 0.53077066, 0.56518644,
     &   0.60004812, 0.63667256, 0.67494959, 0.72198164, 0.78339040,
     &   0.81756455, 0.86632842, 0.90959901, 0.88131291, 0.77864099,
     &   0.68342340, 0.64017057, 0.60912377, 0.58099228, 0.54904801,
     &   0.51478410, 0.47616655, 0.41193762, 0.35218906, 0.29862487,
     &   0.29401717, 0.32526189, 0.36646384, 0.40989909, 0.40327445,
     &   0.39427873, 0.38118854, 0.36272904, 0.33611083, 0.30517074,
     &   0.27838334, 0.25456762, 0.23324765, 0.21372707, 0.19613935,
     &   0.18001536, 0.16398746, 0.14822228, 0.13398001, 0.12145641,
     &   0.11018909, 0.10266139, 0.09654187, 0.09088080, 0.08606996,
     &   0.08152871, 0.07737080, 0.07350947, 0.06985270, 0.06676956,
     &   0.06384503, 0.06114662, 0.05898772, 0.05691514, 0.05492483,
     &   0.05301298, 0.05117594, 0.04941029, 0.04771279, 0.04608036,
     &   0.04451010, 0.04299925, 0.04140033, 0.03983944, 0.03834114,
     &   0.03690268, 0.03552146, 0.03419498, 0.03292089, 0.03169692,
     &   0.03052095, 0.02939092, 0.02830490, 0.02726102, 0.02621774,
     &   0.02507306, 0.02397957, 0.02293489, 0.02193677, 0.02098306,
     &   0.02000059, 0.01862537, 0.01734515, 0.01615330, 0.01499815,
     &   0.01389590, 0.01287497, 0.01189426, 0.01091518, 0.01001695,
     &   0.00919992, 0.00854486, 0.00793663, 0.00737185, 0.00684429,
     &   0.00635394, 0.00589884, 0.00548092, 0.00509638, 0.00473890,
     &   0.00441166, 0.00411884, 0.00384552, 0.00359084, 0.00337710,
     &   0.00317614, 0.00298719, 0.00282622, 0.00267916, 0.00253981,
     &   0.00242643, 0.00233556, 0.00224819, 0.00216773, 0.00210067,
     &   0.00203577, 0.00197295, 0.00191496, 0.00185878, 0.00180432/
      data (qqi(i,4),i=1,mxwv)/
     &   2.08637619, 2.08733582, 2.08842611, 2.08950615, 2.09065223,
     &   2.09175014, 2.09284091, 2.09383965, 2.09489131, 2.09612989,
     &   2.09734106, 2.09851289, 2.09978771, 2.10105252, 2.10210276,
     &   2.10354662, 2.10474396, 2.10591626, 2.10727596, 2.10860682,
     &   2.10981297, 2.11134982, 2.11269855, 2.11378336, 2.11535501,
     &   2.11676455, 2.11821985, 2.11963844, 2.12145305, 2.12273550,
     &   2.12408733, 2.12577462, 2.12733221, 2.12870502, 2.13030434,
     &   2.13195348, 2.13385749, 2.13525105, 2.13705873, 2.13860583,
     &   2.14035511, 2.14207268, 2.14374876, 2.14577222, 2.14760113,
     &   2.14936996, 2.15115857, 2.15301728, 2.15507650, 2.15708542,
     &   2.15883398, 2.16083527, 2.16298389, 2.16497898, 2.16705346,
     &   2.16919971, 2.17127991, 2.17358375, 2.17574239, 2.17792177,
     &   2.18029904, 2.18236995, 2.18482637, 2.18719554, 2.18978119,
     &   2.19213462, 2.19447112, 2.19699121, 2.19958258, 2.20226550,
     &   2.20458674, 2.20733809, 2.21005058, 2.21274400, 2.21545863,
     &   2.21828365, 2.22108722, 2.22395611, 2.22686195, 2.22977543,
     &   2.23276234, 2.23575830, 2.23881364, 2.24188256, 2.24497175,
     &   2.24810505, 2.25126338, 2.25438356, 2.25763535, 2.26086402,
     &   2.26410127, 2.26732469, 2.27051520, 2.27375221, 2.27706599,
     &   2.28040600, 2.28377676, 2.28716207, 2.29056692, 2.29401731,
     &   2.29752469, 2.30110812, 2.30479288, 2.30861521, 2.31261134,
     &   2.31685972, 2.32143807, 2.32644844, 2.33190513, 2.33783674,
     &   2.34434175, 2.35168314, 2.36008286, 2.37006068, 2.38202214,
     &   2.39624858, 2.41332245, 2.43416071, 2.45882320, 2.48909879,
     &   2.52564621, 2.57095504, 2.62703514, 2.69707489, 2.78407216,
     &   2.88072062, 2.93037391, 2.78529954, 2.17223597, 1.39844012,
     &   1.64626133, 2.00189137, 2.13090229, 2.24829459, 2.33679891,
     &   2.36693525, 2.39179063, 2.41252327, 2.43121004, 2.44788837,
     &   2.46446753, 2.48372364, 2.50807405, 2.53809047, 2.57077360,
     &   2.60942554, 2.64961314, 2.68995166, 2.73056865, 2.76914001,
     &   2.80732846, 2.84171224, 2.87246895, 2.89160395, 2.89631581,
     &   2.89615154, 2.89789367, 2.94628763, 2.99557209, 3.03389764,
     &   3.06906438, 3.09596634, 3.11041546, 3.11476994, 3.11056185,
     &   3.08813906, 3.05047083, 2.99647570, 2.92428279, 2.83698368,
     &   2.73502922, 2.70526671, 2.69538260, 2.71974421, 2.75239015,
     &   2.77927518, 2.78344941, 2.75785112, 2.73153639, 2.72107124,
     &   2.70971274, 2.69159055, 2.67173743, 2.65033150, 2.61419344,
     &   2.58244848, 2.55766749, 2.53388023, 2.49945474, 2.44297576,
     &   2.35605192, 2.25865841, 2.17129350, 2.08190227, 1.98980606,
     &   1.89886510, 1.80608106, 1.69882047, 1.58160162, 1.44403017,
     &   1.28715348, 1.14065719, 1.04025495, 0.99976498, 1.08353901,
     &   1.27777624, 1.49107862, 1.66840136, 1.81396818, 1.92674875,
     &   2.02665734, 2.11248851, 2.18986750, 2.26054764, 2.31750584,
     &   2.37166333, 2.42002869, 2.45725393, 2.48834777, 2.51810169,
     &   2.54033756, 2.54654336, 2.53508186, 2.50283146, 2.45590711,
     &   2.39375639, 2.32884336, 2.26361489, 2.19812560, 2.14589882,
     &   2.10522819, 2.05952501, 1.99371696, 1.91901660, 1.84232521,
     &   1.76079226, 1.67773986, 1.61141121, 1.56728971, 1.52656972,
     &   1.47021914, 1.39292753, 1.30238318, 1.21530604, 1.13114715,
     &   1.05025196, 0.97335202, 0.90302396, 0.84061813, 0.78453475,
     &   0.73412764, 0.68700016, 0.64344925, 0.60443068, 0.56903261,
     &   0.53809428, 0.51216537, 0.48933408, 0.46949074, 0.45252860,
     &   0.43812108, 0.42442867, 0.41331890, 0.40468410, 0.39842474,
     &   0.39371136, 0.39088637, 0.39081490, 0.39339948, 0.40252429,
     &   0.42246720, 0.45115024, 0.49007583, 0.52863681, 0.56691670,
     &   0.61042285, 0.65950638, 0.70641452, 0.75201118, 0.79750943,
     &   0.84374189, 0.89240605, 0.94342530, 1.00657511, 1.09151220,
     &   1.14550054, 1.22135746, 1.30054808, 1.30675840, 1.20567024,
     &   1.08690560, 1.02847099, 0.98698264, 0.95140296, 0.91148263,
     &   0.86745435, 0.81782097, 0.72648573, 0.62974322, 0.53463721,
     &   0.50587660, 0.53943092, 0.59424764, 0.66091073, 0.65857595,
     &   0.65013647, 0.63497168, 0.61141551, 0.57453686, 0.52807832,
     &   0.48581052, 0.44698703, 0.41121691, 0.37820187, 0.34784165,
     &   0.31988499, 0.29192510, 0.26429752, 0.23929028, 0.21676758,
     &   0.19640298, 0.18207903, 0.17021935, 0.15937994, 0.15009585,
     &   0.14138769, 0.13344489, 0.12610441, 0.11919758, 0.11329256,
     &   0.10772855, 0.10261066, 0.09844240, 0.09447247, 0.09069010,
     &   0.08708513, 0.08364797, 0.08036962, 0.07724157, 0.07425582,
     &   0.07140484, 0.06868153, 0.06588019, 0.06317053, 0.06058364,
     &   0.05811334, 0.05575379, 0.05349945, 0.05134509, 0.04928578,
     &   0.04731685, 0.04543388, 0.04363267, 0.04190929, 0.04020274,
     &   0.03836540, 0.03661575, 0.03494937, 0.03336208, 0.03184990,
     &   0.03030562, 0.02819982, 0.02624166, 0.02442067, 0.02265960,
     &   0.02098207, 0.01942971, 0.01794195, 0.01646203, 0.01510505,
     &   0.01387083, 0.01287470, 0.01195065, 0.01109340, 0.01029366,
     &   0.00955100, 0.00886227, 0.00822992, 0.00764824, 0.00710791,
     &   0.00661329, 0.00617015, 0.00575689, 0.00537212, 0.00504764,
     &   0.00474294, 0.00445680, 0.00421196, 0.00398815, 0.00377641,
     &   0.00360276, 0.00346217, 0.00332737, 0.00320320, 0.00309886,
     &   0.00299824, 0.00290118, 0.00281158, 0.00272507, 0.00264147/
      data (qqi(i,5),i=1,mxwv)/
     &   2.06840134, 2.06917286, 2.07003403, 2.07085156, 2.07173562,
     &   2.07264829, 2.07337642, 2.07427597, 2.07516885, 2.07611537,
     &   2.07706308, 2.07797146, 2.07889605, 2.07990432, 2.08089304,
     &   2.08172369, 2.08285546, 2.08371210, 2.08479357, 2.08581638,
     &   2.08680725, 2.08784652, 2.08899927, 2.09001684, 2.09109592,
     &   2.09214067, 2.09338617, 2.09438825, 2.09560418, 2.09680820,
     &   2.09795880, 2.09917188, 2.10032535, 2.10159039, 2.10278964,
     &   2.10406780, 2.10529828, 2.10652757, 2.10790539, 2.10918736,
     &   2.11052871, 2.11198092, 2.11340809, 2.11461234, 2.11601210,
     &   2.11746883, 2.11886144, 2.12043524, 2.12188363, 2.12328315,
     &   2.12492537, 2.12634993, 2.12803745, 2.12961531, 2.13112164,
     &   2.13270140, 2.13440347, 2.13597798, 2.13784099, 2.13953805,
     &   2.14124727, 2.14293361, 2.14478254, 2.14634585, 2.14842224,
     &   2.15030313, 2.15213847, 2.15405345, 2.15583014, 2.15791059,
     &   2.15971184, 2.16179180, 2.16406846, 2.16599011, 2.16802454,
     &   2.17008257, 2.17249346, 2.17464900, 2.17666841, 2.17906451,
     &   2.18136048, 2.18370390, 2.18589664, 2.18853164, 2.19086266,
     &   2.19336724, 2.19577360, 2.19855404, 2.20098042, 2.20357537,
     &   2.20622873, 2.20890141, 2.21156335, 2.21428347, 2.21709037,
     &   2.21994400, 2.22285032, 2.22578907, 2.22875261, 2.23175669,
     &   2.23480058, 2.23787618, 2.24097943, 2.24411416, 2.24726963,
     &   2.25045729, 2.25363922, 2.25673628, 2.25971365, 2.26282024,
     &   2.26607919, 2.26941776, 2.27283669, 2.27627301, 2.27970600,
     &   2.28319144, 2.28681946, 2.29069901, 2.29503441, 2.30015445,
     &   2.30653524, 2.31516051, 2.32771373, 2.34753132, 2.38195634,
     &   2.44882083, 2.56232905, 2.69884896, 2.50515223, 1.68738198,
     &   1.79994905, 2.05135584, 2.13943720, 2.21960449, 2.27977991,
     &   2.29837823, 2.31312561, 2.32606387, 2.33858490, 2.35022449,
     &   2.36025834, 2.36832190, 2.37528086, 2.38158941, 2.38789487,
     &   2.39516330, 2.40344524, 2.41303754, 2.42436981, 2.43697691,
     &   2.45152235, 2.46766996, 2.48564649, 2.50365376, 2.51835155,
     &   2.52800345, 2.53410935, 2.55215406, 2.57512665, 2.60421014,
     &   2.63620186, 2.67144632, 2.70738435, 2.74305677, 2.77916050,
     &   2.81103683, 2.83606434, 2.85055041, 2.84783530, 2.82645798,
     &   2.77004099, 2.73213482, 2.71314311, 2.72039723, 2.74413133,
     &   2.77386403, 2.80627298, 2.81688905, 2.81575346, 2.82521415,
     &   2.83858252, 2.84891987, 2.85699677, 2.87000346, 2.86927629,
     &   2.86782122, 2.87155628, 2.88328433, 2.89546585, 2.89833713,
     &   2.87387133, 2.81677794, 2.76260519, 2.70036721, 2.62831831,
     &   2.54857731, 2.46552420, 2.36362791, 2.24333549, 2.09014130,
     &   1.89154458, 1.66858363, 1.47730279, 1.34942794, 1.37293005,
     &   1.53940380, 1.73051584, 1.88946915, 2.01595616, 2.11444187,
     &   2.20059848, 2.27541757, 2.34328485, 2.40592360, 2.46095872,
     &   2.51436043, 2.56490779, 2.61224627, 2.65633011, 2.70277405,
     &   2.75118423, 2.79617882, 2.83636880, 2.86208224, 2.87693715,
     &   2.87312150, 2.86004806, 2.84029889, 2.81313801, 2.78764653,
     &   2.77672601, 2.76910233, 2.74638605, 2.71261549, 2.67201996,
     &   2.61234307, 2.54235220, 2.47472620, 2.42910361, 2.40133452,
     &   2.36786723, 2.31201291, 2.23024774, 2.13636899, 2.03938270,
     &   1.94033885, 1.83861887, 1.73605573, 1.63942754, 1.54840922,
     &   1.46301568, 1.38083661, 1.30187643, 1.22742963, 1.15740180,
     &   1.09296513, 1.03500283, 0.98174757, 0.93319917, 0.88933307,
     &   0.84973013, 0.81145489, 0.77772868, 0.74843395, 0.72344291,
     &   0.70012480, 0.67909384, 0.66318756, 0.65219313, 0.64996612,
     &   0.66155028, 0.68689269, 0.72738498, 0.76924253, 0.81185335,
     &   0.86172092, 0.91903877, 0.97464699, 1.02914095, 1.08377242,
     &   1.13927460, 1.19756460, 1.25858164, 1.33423209, 1.43811381,
     &   1.51145744, 1.61186886, 1.72698736, 1.78812373, 1.72982061,
     &   1.62023771, 1.56037664, 1.51929426, 1.48819876, 1.45474899,
     &   1.41615498, 1.37358952, 1.27565444, 1.14465642, 0.99289840,
     &   0.90358597, 0.91856778, 0.97861540, 1.07337689, 1.08848250,
     &   1.09051549, 1.08290029, 1.06410956, 1.02602053, 0.96799725,
     &   0.90967745, 0.85233516, 0.79610342, 0.74278831, 0.69131768,
     &   0.64289355, 0.59306967, 0.54254669, 0.49600682, 0.45179304,
     &   0.41097355, 0.37994561, 0.35348606, 0.32952106, 0.30869791,
     &   0.28919494, 0.27140298, 0.25497919, 0.23957260, 0.22612610,
     &   0.21350606, 0.20190273, 0.19223826, 0.18308853, 0.17442583,
     &   0.16622368, 0.15845691, 0.15110153, 0.14413479, 0.13753508,
     &   0.13128191, 0.12535587, 0.11949492, 0.11389305, 0.10857955,
     &   0.10353884, 0.09875617, 0.09421755, 0.08990970, 0.08582009,
     &   0.08193684, 0.07824872, 0.07474511, 0.07141599, 0.06816937,
     &   0.06478877, 0.06158564, 0.05855018, 0.05567316, 0.05294586,
     &   0.05020552, 0.04666094, 0.04337005, 0.04031442, 0.03737089,
     &   0.03457541, 0.03199183, 0.02952653, 0.02709229, 0.02486145,
     &   0.02283146, 0.02116776, 0.01962683, 0.01819946, 0.01687092,
     &   0.01563914, 0.01449831, 0.01345086, 0.01248751, 0.01159388,
     &   0.01077550, 0.01004003, 0.00935532, 0.00871874, 0.00817597,
     &   0.00766754, 0.00719123, 0.00677955, 0.00640272, 0.00604742,
     &   0.00575093, 0.00550560, 0.00527177, 0.00505622, 0.00487205,
     &   0.00469570, 0.00452676, 0.00437085, 0.00422133, 0.00407780/
      data (qqi(i,6),i=1,mxwv)/
     &   2.05414009, 2.05484343, 2.05547309, 2.05614829, 2.05680466,
     &   2.05752754, 2.05816746, 2.05879092, 2.05958652, 2.06034422,
     &   2.06102395, 2.06176543, 2.06248307, 2.06322861, 2.06400514,
     &   2.06483269, 2.06559658, 2.06636500, 2.06709790, 2.06792831,
     &   2.06867027, 2.06954741, 2.07042003, 2.07122993, 2.07217002,
     &   2.07289910, 2.07382560, 2.07470727, 2.07562089, 2.07652378,
     &   2.07741165, 2.07842040, 2.07929254, 2.08026028, 2.08134055,
     &   2.08218384, 2.08319497, 2.08423281, 2.08527708, 2.08613849,
     &   2.08737206, 2.08835506, 2.08945274, 2.09051514, 2.09164906,
     &   2.09271026, 2.09364557, 2.09501958, 2.09612608, 2.09721446,
     &   2.09839177, 2.09967399, 2.10096455, 2.10209394, 2.10340047,
     &   2.10464168, 2.10616946, 2.10711622, 2.10842109, 2.10982442,
     &   2.11104131, 2.11248112, 2.11385560, 2.11526966, 2.11661434,
     &   2.11809349, 2.11970735, 2.12088847, 2.12259912, 2.12393022,
     &   2.12580299, 2.12709117, 2.12871528, 2.13031507, 2.13178086,
     &   2.13353825, 2.13521600, 2.13675404, 2.13873053, 2.14031839,
     &   2.14192390, 2.14389205, 2.14562011, 2.14739490, 2.14918590,
     &   2.15118051, 2.15298128, 2.15498114, 2.15746307, 2.15893912,
     &   2.16097784, 2.16296124, 2.16503263, 2.16711068, 2.16925669,
     &   2.17143273, 2.17365670, 2.17590451, 2.17818952, 2.18049002,
     &   2.18286753, 2.18524480, 2.18771982, 2.19018960, 2.19270897,
     &   2.19526958, 2.19788504, 2.20045519, 2.20295644, 2.20558023,
     &   2.20834827, 2.21123075, 2.21423030, 2.21729875, 2.22038460,
     &   2.22349477, 2.22664237, 2.22981882, 2.23298454, 2.23618817,
     &   2.23941660, 2.24265838, 2.24591470, 2.24931288, 2.25335217,
     &   2.26087332, 2.28386521, 2.36612654, 2.46265244, 1.90700400,
     &   1.90563703, 2.07321739, 2.13306093, 2.18826938, 2.22995281,
     &   2.24190640, 2.25087118, 2.25901461, 2.26738071, 2.27561998,
     &   2.28325796, 2.28993726, 2.29602838, 2.30138588, 2.30612373,
     &   2.31078744, 2.31511474, 2.31916142, 2.32307148, 2.32703543,
     &   2.33110428, 2.33522725, 2.33957815, 2.34396338, 2.34821892,
     &   2.35224175, 2.35603690, 2.36227036, 2.36877966, 2.37612462,
     &   2.38424897, 2.39373732, 2.40482211, 2.41764379, 2.43274093,
     &   2.45044065, 2.47002125, 2.49042296, 2.50862169, 2.52191496,
     &   2.51750088, 2.50290298, 2.49504781, 2.49596810, 2.50562406,
     &   2.52026725, 2.54245090, 2.55951381, 2.56948066, 2.58215451,
     &   2.59831476, 2.61516571, 2.63154483, 2.65310597, 2.67076063,
     &   2.68663597, 2.70434499, 2.72943068, 2.76185799, 2.80038166,
     &   2.83418846, 2.84326959, 2.84923148, 2.84785700, 2.83653808,
     &   2.81335235, 2.78631282, 2.74491382, 2.68342066, 2.58691883,
     &   2.42543864, 2.18780613, 1.93148971, 1.71310532, 1.65343451,
     &   1.76727152, 1.91590166, 2.04095793, 2.13720536, 2.21190524,
     &   2.27585602, 2.33101392, 2.38033199, 2.42514348, 2.46530604,
     &   2.50349736, 2.53948307, 2.57471919, 2.60794210, 2.64299011,
     &   2.68164372, 2.72267532, 2.76681066, 2.80798674, 2.84814239,
     &   2.88049245, 2.90746284, 2.92966247, 2.94547606, 2.95486498,
     &   2.97390175, 3.00209141, 3.02980423, 3.05343676, 3.07220387,
     &   3.07102966, 3.05742431, 3.02874255, 3.00806546, 3.01162267,
     &   3.02593946, 3.03191638, 3.01379442, 2.97040629, 2.91680050,
     &   2.85357428, 2.77745295, 2.68662834, 2.59199452, 2.49568963,
     &   2.39904571, 2.30133367, 2.20203638, 2.10233641, 2.00400066,
     &   1.90835845, 1.81671584, 1.72897267, 1.64561510, 1.56706178,
     &   1.49316049, 1.42029095, 1.35311937, 1.29180408, 1.23644173,
     &   1.18170178, 1.12861598, 1.08345068, 1.04606843, 1.01887786,
     &   1.00716889, 1.01605296, 1.04642928, 1.08225620, 1.12135482,
     &   1.17016292, 1.22840774, 1.28635800, 1.34380162, 1.40166783,
     &   1.46031797, 1.52158463, 1.58537996, 1.66426528, 1.77390254,
     &   1.85718465, 1.96708477, 2.09822130, 2.20214462, 2.20512557,
     &   2.14100218, 2.10021138, 2.07498336, 2.06263328, 2.05302954,
     &   2.04062366, 2.03098059, 1.97579741, 1.85931444, 1.68616581,
     &   1.51365185, 1.47587705, 1.51545286, 1.62261951, 1.66748977,
     &   1.69199848, 1.70672929, 1.71199167, 1.69870281, 1.65721583,
     &   1.60610628, 1.54940557, 1.48744845, 1.42591751, 1.36117327,
     &   1.29756963, 1.22793591, 1.15287805, 1.08051765, 1.00458109,
     &   0.93060005, 0.86807078, 0.81206536, 0.76087564, 0.71510607,
     &   0.67153752, 0.63120979, 0.59350115, 0.55772626, 0.52559066,
     &   0.49517971, 0.46695781, 0.44292811, 0.42007884, 0.39836875,
     &   0.37775555, 0.35819653, 0.33964851, 0.32206863, 0.30541417,
     &   0.28964308, 0.27471411, 0.26044431, 0.24689436, 0.23405482,
     &   0.22189255, 0.21037532, 0.19947186, 0.18915188, 0.17938614,
     &   0.17014648, 0.16140586, 0.15313829, 0.14531891, 0.13780901,
     &   0.13029560, 0.12320205, 0.11650559, 0.11018451, 0.10421816,
     &   0.09834570, 0.09133003, 0.08481809, 0.07877392, 0.07297626,
     &   0.06748662, 0.06241503, 0.05760361, 0.05290620, 0.04859865,
     &   0.04466986, 0.04135692, 0.03829319, 0.03545967, 0.03283034,
     &   0.03039692, 0.02814641, 0.02607840, 0.02417555, 0.02241348,
     &   0.02079758, 0.01933659, 0.01797976, 0.01672091, 0.01562639,
     &   0.01460505, 0.01365187, 0.01281357, 0.01204427, 0.01132288,
     &   0.01070326, 0.01017276, 0.00967169, 0.00920929, 0.00880417,
     &   0.00842033, 0.00805647, 0.00772082, 0.00740228, 0.00709971/
      data (qqi(i,7),i=1,mxwv)/
     &   2.04299641, 2.04347539, 2.04397154, 2.04451704, 2.04506326,
     &   2.04560447, 2.04616117, 2.04668808, 2.04722118, 2.04781461,
     &   2.04820299, 2.04895973, 2.04951787, 2.05010843, 2.05071950,
     &   2.05137110, 2.05185175, 2.05260062, 2.05318785, 2.05395913,
     &   2.05446959, 2.05514812, 2.05582356, 2.05649710, 2.05714536,
     &   2.05781865, 2.05847478, 2.05918527, 2.05989599, 2.06057692,
     &   2.06134200, 2.06205177, 2.06284833, 2.06352210, 2.06432819,
     &   2.06509495, 2.06589985, 2.06660342, 2.06717896, 2.06852555,
     &   2.06905770, 2.06990790, 2.07080817, 2.07158422, 2.07226682,
     &   2.07328510, 2.07423162, 2.07507205, 2.07600760, 2.07685685,
     &   2.07781148, 2.07885551, 2.07969737, 2.08070087, 2.08161998,
     &   2.08258843, 2.08365560, 2.08473372, 2.08580041, 2.08676910,
     &   2.08782864, 2.08877134, 2.08989692, 2.09095526, 2.09213519,
     &   2.09319067, 2.09428477, 2.09547615, 2.09658313, 2.09778738,
     &   2.09897661, 2.10011005, 2.10149837, 2.10259986, 2.10396862,
     &   2.10507703, 2.10646129, 2.10773826, 2.10896945, 2.11041689,
     &   2.11168885, 2.11318231, 2.11448097, 2.11591959, 2.11749816,
     &   2.11876893, 2.12024951, 2.12175417, 2.12322569, 2.12485027,
     &   2.12632990, 2.12788272, 2.12945342, 2.13104248, 2.13268542,
     &   2.13434601, 2.13610554, 2.13775468, 2.13945580, 2.14126468,
     &   2.14306617, 2.14487219, 2.14671803, 2.14857292, 2.15050936,
     &   2.15239835, 2.15441132, 2.15635920, 2.15826321, 2.16025925,
     &   2.16236067, 2.16449237, 2.16681528, 2.16914415, 2.17150044,
     &   2.17390323, 2.17636061, 2.17887998, 2.18144965, 2.18412423,
     &   2.18691468, 2.18985009, 2.19295144, 2.19628358, 2.19986343,
     &   2.20338345, 2.20646787, 2.21510601, 2.27427673, 2.02623439,
     &   1.96984124, 2.07889199, 2.11985946, 2.15826654, 2.18755126,
     &   2.19550347, 2.20106816, 2.20624042, 2.21179318, 2.21740007,
     &   2.22268200, 2.22741389, 2.23185277, 2.23590779, 2.23963594,
     &   2.24341941, 2.24704337, 2.25049043, 2.25383401, 2.25716782,
     &   2.26049304, 2.26365471, 2.26675820, 2.26923466, 2.27128077,
     &   2.27347589, 2.27607322, 2.28145528, 2.28660083, 2.29098439,
     &   2.29533172, 2.29947734, 2.30337548, 2.30723596, 2.31125140,
     &   2.31531882, 2.31971240, 2.32457376, 2.32973146, 2.33490658,
     &   2.33673358, 2.33503556, 2.33488560, 2.33742166, 2.34267974,
     &   2.34927034, 2.35776162, 2.36512947, 2.37085199, 2.37724781,
     &   2.38478327, 2.39288282, 2.40114164, 2.41123986, 2.42139482,
     &   2.43124700, 2.44155860, 2.45442700, 2.47106051, 2.49316716,
     &   2.52038717, 2.54436231, 2.56804776, 2.59184241, 2.61446857,
     &   2.63315845, 2.65226960, 2.67049909, 2.68216395, 2.67964125,
     &   2.63170600, 2.48914385, 2.26327038, 2.01131678, 1.88095009,
     &   1.93488169, 2.03488255, 2.12247181, 2.18848968, 2.23972082,
     &   2.28287029, 2.31974196, 2.35217810, 2.38108444, 2.40664816,
     &   2.43023944, 2.45186663, 2.47254658, 2.49160314, 2.51095176,
     &   2.53155923, 2.55320191, 2.57663488, 2.60009956, 2.62493539,
     &   2.64925528, 2.67325044, 2.69691443, 2.71946049, 2.73778772,
     &   2.75904560, 2.78669357, 2.82083154, 2.85803795, 2.89640260,
     &   2.92941165, 2.95919967, 2.97491741, 2.98542285, 3.00959873,
     &   3.04990935, 3.09956717, 3.14617872, 3.17746663, 3.20218372,
     &   3.21921182, 3.22368574, 3.20923162, 3.18187642, 3.14400673,
     &   3.09685349, 3.04179811, 2.97742224, 2.90339017, 2.82267404,
     &   2.73580098, 2.64382505, 2.54945207, 2.45386791, 2.35823965,
     &   2.26322579, 2.16609740, 2.07185364, 1.98143923, 1.89568675,
     &   1.80696118, 1.71660841, 1.63489473, 1.56239152, 1.49843514,
     &   1.44777226, 1.42441881, 1.42897141, 1.44595683, 1.47069740,
     &   1.50788200, 1.55651689, 1.60757554, 1.65943933, 1.71228468,
     &   1.76586890, 1.82161891, 1.87941158, 1.95068491, 2.05061698,
     &   2.13081217, 2.23207784, 2.35396695, 2.46795940, 2.51293445,
     &   2.50024652, 2.48674631, 2.48250747, 2.49033809, 2.50520372,
     &   2.52120900, 2.54655075, 2.55799341, 2.51145482, 2.39639163,
     &   2.18772554, 2.08666420, 2.07993460, 2.16377711, 2.22968030,
     &   2.27263379, 2.30913854, 2.34236836, 2.36736059, 2.36949205,
     &   2.35712862, 2.33537364, 2.30394864, 2.27060223, 2.22909832,
     &   2.18577743, 2.13318396, 2.07055616, 2.00583410, 1.92614543,
     &   1.84140837, 1.75880599, 1.67922258, 1.60417891, 1.53377450,
     &   1.46408594, 1.39725757, 1.33260572, 1.26917601, 1.20958769,
     &   1.15151608, 1.09609103, 1.04751754, 1.00027108, 0.95439738,
     &   0.90993512, 0.86691445, 0.82535809, 0.78528094, 0.74669093,
     &   0.70958906, 0.67396981, 0.64045894, 0.60838681, 0.57761472,
     &   0.54812777, 0.51990694, 0.49292925, 0.46716866, 0.44259605,
     &   0.41917989, 0.39688650, 0.37568066, 0.35552576, 0.33625799,
     &   0.31750858, 0.29972365, 0.28286576, 0.26689741, 0.25178108,
     &   0.23708622, 0.22078836, 0.20556864, 0.19136283, 0.17770717,
     &   0.16474128, 0.15270717, 0.14130796, 0.13026813, 0.12009300,
     &   0.11074781, 0.10256451, 0.09498315, 0.08796031, 0.08144604,
     &   0.07541162, 0.06982425, 0.06467105, 0.05991494, 0.05550934,
     &   0.05145399, 0.04775423, 0.04432165, 0.04113924, 0.03830560,
     &   0.03566916, 0.03321621, 0.03101414, 0.02898577, 0.02709341,
     &   0.02541591, 0.02392892, 0.02253618, 0.02124907, 0.02009377,
     &   0.01900979, 0.01799243, 0.01705430, 0.01617302, 0.01534467/
      data (qqi(i,8),i=1,mxwv)/
     &   2.03409576, 2.03452325, 2.03492522, 2.03533363, 2.03576612,
     &   2.03618002, 2.03602099, 2.03706980, 2.03749776, 2.03793144,
     &   2.03836179, 2.03883648, 2.03929710, 2.03974628, 2.04023528,
     &   2.04072165, 2.04119349, 2.04169106, 2.04217863, 2.04268503,
     &   2.04319239, 2.04372787, 2.04419446, 2.04474831, 2.04529643,
     &   2.04582500, 2.04635286, 2.04694247, 2.04749298, 2.04805136,
     &   2.04865432, 2.04920316, 2.04977274, 2.05036426, 2.05098200,
     &   2.05157590, 2.05220795, 2.05279803, 2.05344844, 2.05409122,
     &   2.05475664, 2.05538273, 2.05606985, 2.05670309, 2.05746841,
     &   2.05806589, 2.05879378, 2.05949116, 2.06019926, 2.06088829,
     &   2.06164265, 2.06236458, 2.06320357, 2.06388879, 2.06466007,
     &   2.06542182, 2.06621408, 2.06701851, 2.06777787, 2.06852531,
     &   2.06941676, 2.07022500, 2.07107663, 2.07194424, 2.07270718,
     &   2.07371283, 2.07452774, 2.07549644, 2.07638311, 2.07728982,
     &   2.07824063, 2.07917428, 2.08017802, 2.08108354, 2.08204079,
     &   2.08302355, 2.08406758, 2.08509111, 2.08628845, 2.08717632,
     &   2.08822608, 2.08934236, 2.09046292, 2.09148121, 2.09254622,
     &   2.09352589, 2.09484124, 2.09601593, 2.09714150, 2.09842062,
     &   2.09953213, 2.10074544, 2.10195732, 2.10319924, 2.10446596,
     &   2.10575366, 2.10706306, 2.10838246, 2.10970902, 2.11108470,
     &   2.11246920, 2.11391878, 2.11527610, 2.11675024, 2.11815429,
     &   2.11966944, 2.12115359, 2.12264967, 2.12411046, 2.12563515,
     &   2.12723160, 2.12888455, 2.13059640, 2.13234568, 2.13411236,
     &   2.13590407, 2.13773060, 2.13960052, 2.14149857, 2.14347100,
     &   2.14552164, 2.14767504, 2.14995766, 2.15244198, 2.15521789,
     &   2.15828133, 2.16139102, 2.16376305, 2.16923070, 2.06401730,
     &   2.00478601, 2.07603669, 2.10436010, 2.13129568, 2.15207386,
     &   2.15753531, 2.16108918, 2.16444802, 2.16820621, 2.17208457,
     &   2.17577624, 2.17912459, 2.18228912, 2.18521261, 2.18793845,
     &   2.19070864, 2.19339180, 2.19597411, 2.19850659, 2.20105529,
     &   2.20361853, 2.20609331, 2.20850515, 2.21057034, 2.21227384,
     &   2.21407175, 2.21616364, 2.22031283, 2.22433543, 2.22785568,
     &   2.23140621, 2.23484540, 2.23807716, 2.24120665, 2.24432945,
     &   2.24712133, 2.24963903, 2.25178242, 2.25328422, 2.25410509,
     &   2.25293183, 2.25247288, 2.25327873, 2.25620484, 2.26057124,
     &   2.26540160, 2.27046394, 2.27413583, 2.27705479, 2.28072739,
     &   2.28483629, 2.28894925, 2.29304957, 2.29777384, 2.30204391,
     &   2.30632806, 2.31095695, 2.31641555, 2.32269001, 2.32992125,
     &   2.33808160, 2.34620571, 2.35527515, 2.36569357, 2.37761307,
     &   2.39057827, 2.40594482, 2.42553210, 2.44893670, 2.47750568,
     &   2.50145054, 2.47889209, 2.36759710, 2.17385006, 2.02502894,
     &   2.03491163, 2.09516048, 2.15226531, 2.19547153, 2.22935343,
     &   2.25786400, 2.28220797, 2.30352354, 2.32238960, 2.33879614,
     &   2.35374308, 2.36724424, 2.37974477, 2.39114523, 2.40245819,
     &   2.41398025, 2.42545176, 2.43713188, 2.44826555, 2.45950651,
     &   2.47034001, 2.48114467, 2.49207139, 2.50303459, 2.51292706,
     &   2.52351856, 2.53604889, 2.55134916, 2.56850100, 2.58701062,
     &   2.60664177, 2.62798095, 2.64624667, 2.66058612, 2.67719364,
     &   2.70059943, 2.73225760, 2.77090812, 2.81122851, 2.85437131,
     &   2.89985585, 2.94605279, 2.98891139, 3.02643609, 3.05822873,
     &   3.08337402, 3.10278249, 3.11435699, 3.11547399, 3.10687995,
     &   3.08677387, 3.05397797, 3.01141524, 2.95964193, 2.89947486,
     &   2.83180189, 2.75620198, 2.67520881, 2.59036708, 2.50327682,
     &   2.40665174, 2.30126381, 2.19892120, 2.10176229, 2.00557756,
     &   1.91366434, 1.85052168, 1.81812787, 1.80636275, 1.80797637,
     &   1.82439697, 1.85397887, 1.88970113, 1.92824256, 1.96874487,
     &   2.01029825, 2.05371809, 2.09886003, 2.15468812, 2.23385000,
     &   2.30045843, 2.38088107, 2.47699070, 2.57277966, 2.62602282,
     &   2.64224529, 2.64797521, 2.65752268, 2.67524099, 2.70015621,
     &   2.72803283, 2.76564240, 2.81457543, 2.83656812, 2.81958008,
     &   2.66851163, 2.54450011, 2.49552941, 2.53217506, 2.59421158,
     &   2.63722587, 2.67753696, 2.71947098, 2.76339388, 2.79834199,
     &   2.82267714, 2.84056687, 2.85162520, 2.86158109, 2.86505151,
     &   2.86674786, 2.86349106, 2.85380769, 2.84016728, 2.80927992,
     &   2.76905870, 2.71633220, 2.65887356, 2.60186243, 2.54422474,
     &   2.48373604, 2.42256975, 2.36033630, 2.29617548, 2.23205948,
     &   2.16675973, 2.10174918, 2.04252291, 1.98285389, 1.92288041,
     &   1.86273944, 1.80256641, 1.74249399, 1.68265235, 1.62316740,
     &   1.56416142, 1.50575149, 1.45071518, 1.39675212, 1.34345138,
     &   1.29090428, 1.23919773, 1.18841279, 1.13862455, 1.08990276,
     &   1.04231012, 0.99590307, 0.95073152, 0.90683818, 0.86427677,
     &   0.82307559, 0.78316641, 0.74457407, 0.70731634, 0.67140520,
     &   0.63625270, 0.59878933, 0.56307817, 0.52909553, 0.49580854,
     &   0.46364224, 0.43329951, 0.40422350, 0.37587196, 0.34936279,
     &   0.32464477, 0.30214044, 0.28110039, 0.26144493, 0.24309598,
     &   0.22597744, 0.21001559, 0.19516663, 0.18135357, 0.16848685,
     &   0.15654950, 0.14553407, 0.13527206, 0.12571844, 0.11703059,
     &   0.10892849, 0.10137452, 0.09446930, 0.08806790, 0.08209395,
     &   0.07666761, 0.07173526, 0.06712465, 0.06284733, 0.05894117,
     &   0.05528787, 0.05187134, 0.04871543, 0.04576318, 0.04300106/
      data (qqi(i,9),i=1,mxwv)/
     &   2.02707410, 2.02737999, 2.02772069, 2.02804041, 2.02843022,
     &   2.02871370, 2.02905655, 2.02940607, 2.02972746, 2.03009272,
     &   2.03036857, 2.03082561, 2.03122354, 2.03155708, 2.03193307,
     &   2.03231239, 2.03269315, 2.03306532, 2.03346181, 2.03389406,
     &   2.03423953, 2.03466845, 2.03510165, 2.03550243, 2.03591371,
     &   2.03638196, 2.03678393, 2.03723025, 2.03766203, 2.03811598,
     &   2.03856874, 2.03904772, 2.03949666, 2.03997064, 2.04042530,
     &   2.04090977, 2.04142857, 2.04186583, 2.04240727, 2.04294920,
     &   2.04340386, 2.04393697, 2.04445362, 2.04495430, 2.04566073,
     &   2.04605174, 2.04662204, 2.04719663, 2.04769778, 2.04837918,
     &   2.04886174, 2.04944634, 2.05003834, 2.05064225, 2.05123043,
     &   2.05183506, 2.05244279, 2.05306768, 2.05374503, 2.05437350,
     &   2.05498981, 2.05563784, 2.05635905, 2.05698562, 2.05764580,
     &   2.05836177, 2.05905294, 2.05976009, 2.06048846, 2.06114149,
     &   2.06195307, 2.06267357, 2.06338334, 2.06417823, 2.06490850,
     &   2.06574678, 2.06653714, 2.06731153, 2.06816816, 2.06896615,
     &   2.06978512, 2.07060146, 2.07146978, 2.07229972, 2.07317662,
     &   2.07409191, 2.07501602, 2.07588553, 2.07690024, 2.07770896,
     &   2.07865286, 2.07959580, 2.08053756, 2.08150101, 2.08249116,
     &   2.08349586, 2.08451486, 2.08554268, 2.08659983, 2.08764648,
     &   2.08871651, 2.08974123, 2.09090543, 2.09201431, 2.09313393,
     &   2.09429336, 2.09545326, 2.09660363, 2.09773827, 2.09891534,
     &   2.10014796, 2.10141635, 2.10272527, 2.10405803, 2.10540271,
     &   2.10676026, 2.10814953, 2.10955572, 2.11098123, 2.11245155,
     &   2.11397099, 2.11555099, 2.11720824, 2.11898279, 2.12141538,
     &   2.12301135, 2.12512732, 2.12685990, 2.12458944, 2.06434035,
     &   2.02183080, 2.06915641, 2.08890295, 2.10790396, 2.12274575,
     &   2.12660432, 2.12895322, 2.13119483, 2.13379788, 2.13653541,
     &   2.13917017, 2.14158797, 2.14389396, 2.14604306, 2.14806914,
     &   2.15012455, 2.15212679, 2.15406632, 2.15597320, 2.15789723,
     &   2.15983295, 2.16171527, 2.16354895, 2.16515851, 2.16653872,
     &   2.16800833, 2.16970205, 2.17277288, 2.17574453, 2.17836118,
     &   2.18100595, 2.18358254, 2.18603110, 2.18842626, 2.19083548,
     &   2.19303036, 2.19504786, 2.19678664, 2.19800496, 2.19861317,
     &   2.19764185, 2.19751525, 2.19838023, 2.20083022, 2.20424962,
     &   2.20794821, 2.21171999, 2.21442485, 2.21659684, 2.21937227,
     &   2.22244692, 2.22548008, 2.22848582, 2.23191833, 2.23485970,
     &   2.23780751, 2.24105501, 2.24491572, 2.24922371, 2.25380516,
     &   2.25805140, 2.26111364, 2.26447964, 2.26797557, 2.27161717,
     &   2.27539468, 2.28003144, 2.28595066, 2.29364634, 2.30492163,
     &   2.32102036, 2.33209300, 2.30511475, 2.20496607, 2.08951259,
     &   2.08093476, 2.11615133, 2.15291715, 2.18140626, 2.20413160,
     &   2.22344923, 2.24004221, 2.25462985, 2.26757193, 2.27872109,
     &   2.28888392, 2.29805374, 2.30640316, 2.31407428, 2.32168245,
     &   2.32932878, 2.33683252, 2.34434652, 2.35139036, 2.35831976,
     &   2.36470461, 2.37086940, 2.37690592, 2.38274074, 2.38814330,
     &   2.39416528, 2.40099859, 2.40842676, 2.41609740, 2.42387128,
     &   2.43142557, 2.43927169, 2.44649172, 2.45309758, 2.46036863,
     &   2.46908236, 2.47940421, 2.49138832, 2.50475240, 2.51991153,
     &   2.53728533, 2.55755806, 2.58116102, 2.60712361, 2.63523483,
     &   2.66503620, 2.69683242, 2.73007107, 2.76327252, 2.79550529,
     &   2.82466936, 2.84837675, 2.86654949, 2.87791848, 2.88138700,
     &   2.87626743, 2.86300635, 2.83978772, 2.80674314, 2.76442099,
     &   2.70858979, 2.63778138, 2.55912018, 2.47545862, 2.38050604,
     &   2.27472758, 2.18771744, 2.12555480, 2.08813000, 2.06770420,
     &   2.06258059, 2.07109737, 2.08897114, 2.11164546, 2.13736582,
     &   2.16475296, 2.19402218, 2.22501159, 2.26384425, 2.31985593,
     &   2.36908722, 2.42648315, 2.49430251, 2.56200433, 2.60211396,
     &   2.62142015, 2.63237882, 2.64467692, 2.66121602, 2.68223071,
     &   2.70534325, 2.73397994, 2.77637577, 2.81576753, 2.85181427,
     &   2.80064750, 2.71374965, 2.65823150, 2.65854597, 2.69962692,
     &   2.72929931, 2.75829387, 2.78962350, 2.82545853, 2.86123538,
     &   2.89270043, 2.92237520, 2.95053148, 2.97930956, 3.00681567,
     &   3.03456473, 3.06404281, 3.09491920, 3.12496114, 3.14906502,
     &   3.16901398, 3.17440009, 3.17271638, 3.16850615, 3.15983963,
     &   3.14739943, 3.13164306, 3.11253595, 3.08973193, 3.06270981,
     &   3.03226185, 2.99904156, 2.96599960, 2.93051815, 2.89265752,
     &   2.85248685, 2.81008601, 2.76554394, 2.71895862, 2.67043543,
     &   2.62008691, 2.56803274, 2.51873684, 2.46876335, 2.41742134,
     &   2.36481929, 2.31106973, 2.25628805, 2.20059180, 2.14410138,
     &   2.08693767, 2.02922320, 1.97108030, 1.91263187, 1.85443163,
     &   1.79767799, 1.74080372, 1.68392742, 1.62716603, 1.57063401,
     &   1.51392055, 1.45420742, 1.39505649, 1.33661664, 1.27697194,
     &   1.21700883, 1.15836775, 1.10024226, 1.04171085, 0.98506343,
     &   0.93040204, 0.87804401, 0.82772702, 0.77947956, 0.73337197,
     &   0.68935537, 0.64740503, 0.60751837, 0.56965786, 0.53376418,
     &   0.49986601, 0.46800008, 0.43790659, 0.40953076, 0.38319454,
     &   0.35838455, 0.33503577, 0.31333360, 0.29300734, 0.27390742,
     &   0.25624394, 0.23990451, 0.22456138, 0.21021703, 0.19695532,
     &   0.18450384, 0.17281733, 0.16196734, 0.15178496, 0.14223060/
      data (qqi(i,10),i=1,mxwv)/
     &   2.02149177, 2.02173114, 2.02199864, 2.02226591, 2.02252579,
     &   2.02279162, 2.02305508, 2.02328157, 2.02361155, 2.02388763,
     &   2.02418017, 2.02446055, 2.02475286, 2.02503181, 2.02533603,
     &   2.02563858, 2.02595234, 2.02624726, 2.02657533, 2.02688265,
     &   2.02719307, 2.02751732, 2.02784109, 2.02818274, 2.02852440,
     &   2.02884173, 2.02918386, 2.02953410, 2.02988100, 2.03024626,
     &   2.03060198, 2.03096318, 2.03133535, 2.03171754, 2.03208518,
     &   2.03246689, 2.03283381, 2.03323269, 2.03362060, 2.03397799,
     &   2.03443241, 2.03487420, 2.03526545, 2.03568625, 2.03609300,
     &   2.03655076, 2.03696489, 2.03741097, 2.03783250, 2.03827882,
     &   2.03873897, 2.03917813, 2.03939867, 2.04015231, 2.04063606,
     &   2.04108191, 2.04159760, 2.04208136, 2.04262400, 2.04309273,
     &   2.04361653, 2.04412079, 2.04467916, 2.04520702, 2.04573941,
     &   2.04627728, 2.04683971, 2.04735541, 2.04792976, 2.04846501,
     &   2.04906106, 2.04966784, 2.05022240, 2.05085897, 2.05147338,
     &   2.05206203, 2.05269003, 2.05326152, 2.05397034, 2.05464482,
     &   2.05526328, 2.05593777, 2.05658579, 2.05726290, 2.05794668,
     &   2.05866694, 2.05934763, 2.06007910, 2.06075501, 2.06148744,
     &   2.06224084, 2.06298232, 2.06373477, 2.06449151, 2.06526804,
     &   2.06605482, 2.06684494, 2.06764984, 2.06847501, 2.06930542,
     &   2.07013822, 2.07099342, 2.07184458, 2.07273173, 2.07360506,
     &   2.07450414, 2.07540178, 2.07630467, 2.07718992, 2.07810974,
     &   2.07906675, 2.08005047, 2.08106184, 2.08208990, 2.08312440,
     &   2.08416986, 2.08526492, 2.08630919, 2.08739972, 2.08851695,
     &   2.08966470, 2.09085083, 2.09208107, 2.09338093, 2.09477472,
     &   2.09622240, 2.09761548, 2.09856796, 2.09601760, 2.05609179,
     &   2.02878428, 2.06075239, 2.07460999, 2.08807158, 2.09872007,
     &   2.10151005, 2.10312176, 2.10466361, 2.10650706, 2.10847783,
     &   2.11039352, 2.11217332, 2.11388397, 2.11549377, 2.11702681,
     &   2.11858010, 2.12010145, 2.12158084, 2.12304354, 2.12452006,
     &   2.12600589, 2.12745237, 2.12887239, 2.13014007, 2.13126087,
     &   2.13246059, 2.13382936, 2.13612485, 2.13834405, 2.14030385,
     &   2.14228487, 2.14421582, 2.14605761, 2.14786339, 2.14967632,
     &   2.15134215, 2.15288401, 2.15423775, 2.15524006, 2.15582585,
     &   2.15535331, 2.15555525, 2.15644407, 2.15842628, 2.16104054,
     &   2.16381955, 2.16660547, 2.16866302, 2.17038178, 2.17252636,
     &   2.17487049, 2.17718410, 2.17948508, 2.18207622, 2.18432379,
     &   2.18658805, 2.18907237, 2.19199085, 2.19522309, 2.19865322,
     &   2.20183134, 2.20407772, 2.20649409, 2.20888257, 2.21116972,
     &   2.21326542, 2.21565223, 2.21821022, 2.22089100, 2.22389030,
     &   2.22694993, 2.22825456, 2.21912813, 2.17481732, 2.10457802,
     &   2.09460211, 2.11623287, 2.14050460, 2.15983486, 2.17549872,
     &   2.18897438, 2.20063162, 2.21094060, 2.22012544, 2.22800112,
     &   2.23519993, 2.24169946, 2.24756718, 2.25299788, 2.25838375,
     &   2.26375484, 2.26900411, 2.27425432, 2.27922177, 2.28413653,
     &   2.28870106, 2.29313684, 2.29749346, 2.30169845, 2.30564785,
     &   2.31010985, 2.31515694, 2.32052612, 2.32600689, 2.33151746,
     &   2.33655524, 2.34145832, 2.34562922, 2.34964800, 2.35461140,
     &   2.36058903, 2.36714554, 2.37369561, 2.37977552, 2.38589215,
     &   2.39209008, 2.39837909, 2.40483999, 2.41172242, 2.41927958,
     &   2.42772007, 2.43738675, 2.44858122, 2.46151996, 2.47635221,
     &   2.49299550, 2.51108861, 2.53040504, 2.55034041, 2.57007122,
     &   2.58872366, 2.60642076, 2.62063622, 2.63003540, 2.63339853,
     &   2.62913108, 2.61413860, 2.58785367, 2.55071831, 2.49621272,
     &   2.42047501, 2.34565783, 2.28129816, 2.23676753, 2.20739198,
     &   2.19062948, 2.18570995, 2.19090056, 2.20165825, 2.21594429,
     &   2.23231244, 2.25060654, 2.27065301, 2.29629397, 2.33393788,
     &   2.36822820, 2.40759182, 2.45376110, 2.49839211, 2.52312922,
     &   2.53548241, 2.54416776, 2.55371857, 2.56540227, 2.57918763,
     &   2.59379292, 2.61055279, 2.63357854, 2.65762472, 2.68698835,
     &   2.68745780, 2.65238166, 2.61937046, 2.61196280, 2.63462090,
     &   2.65195966, 2.66909528, 2.68730140, 2.70776439, 2.72908425,
     &   2.74897718, 2.76866698, 2.78864861, 2.80924058, 2.83054137,
     &   2.85254645, 2.87729216, 2.90529013, 2.93445706, 2.96618819,
     &   2.99974370, 3.02883339, 3.05599213, 3.08234763, 3.10656118,
     &   3.13027906, 3.15261483, 3.17363286, 3.19345808, 3.21009398,
     &   3.22500515, 3.23765993, 3.24670959, 3.25395632, 3.25927043,
     &   3.26252866, 3.26361561, 3.26242471, 3.25885820, 3.25283003,
     &   3.24426484, 3.23309946, 3.22235560, 3.20980120, 3.19486237,
     &   3.17751098, 3.15772891, 3.13550901, 3.11085391, 3.08377624,
     &   3.05429912, 3.02245498, 2.98828578, 2.95184326, 2.91393995,
     &   2.87668443, 2.83730769, 2.79587889, 2.75247526, 2.70718002,
     &   2.66020584, 2.61194873, 2.56145382, 2.50888181, 2.45207357,
     &   2.39173770, 2.32971191, 2.26522756, 2.19714355, 2.12793708,
     &   2.05781102, 1.98613548, 1.91425502, 1.84240031, 1.77096450,
     &   1.70001626, 1.62971270, 1.56020129, 1.49166703, 1.42430556,
     &   1.35835803, 1.29409003, 1.23137569, 1.17032743, 1.11165118,
     &   1.05475295, 0.99969161, 0.94692320, 0.89617562, 0.84733474,
     &   0.80088335, 0.75676143, 0.71449327, 0.67422539, 0.63622546,
     &   0.59996223, 0.56540412, 0.53288305, 0.50195295, 0.47256675/
      data (qqi(i,11),i=1,mxwv)/
     &   2.01706696, 2.01726270, 2.01746988, 2.01766729, 2.01788950,
     &   2.01809287, 2.01830769, 2.01852012, 2.01873446, 2.01896501,
     &   2.01918650, 2.01929808, 2.01970196, 2.01987743, 2.02011609,
     &   2.02034593, 2.02059007, 2.02083564, 2.02108550, 2.02133036,
     &   2.02158332, 2.02183604, 2.02210855, 2.02235866, 2.02263665,
     &   2.02289891, 2.02317238, 2.02343822, 2.02373123, 2.02400422,
     &   2.02429342, 2.02458191, 2.02486324, 2.02515817, 2.02545094,
     &   2.02576733, 2.02607250, 2.02638268, 2.02669740, 2.02699971,
     &   2.02732825, 2.02769685, 2.02796841, 2.02831316, 2.02854180,
     &   2.02898431, 2.02932954, 2.02967262, 2.03001928, 2.03039908,
     &   2.03074312, 2.03109789, 2.03148437, 2.03184772, 2.03221822,
     &   2.03257179, 2.03300762, 2.03338075, 2.03379369, 2.03418589,
     &   2.03455877, 2.03499556, 2.03542495, 2.03585482, 2.03626299,
     &   2.03668714, 2.03713036, 2.03755832, 2.03801107, 2.03847528,
     &   2.03892899, 2.03939056, 2.04008389, 2.04039931, 2.04079795,
     &   2.04128981, 2.04178619, 2.04228425, 2.04274917, 2.04328823,
     &   2.04381895, 2.04433417, 2.04486060, 2.04539251, 2.04593229,
     &   2.04646802, 2.04702759, 2.04760027, 2.04824996, 2.04873991,
     &   2.04932070, 2.04990363, 2.05049539, 2.05109072, 2.05170655,
     &   2.05232453, 2.05295467, 2.05359197, 2.05421853, 2.05488300,
     &   2.05554366, 2.05620694, 2.05688167, 2.05756927, 2.05825472,
     &   2.05896354, 2.05967188, 2.06037164, 2.06107688, 2.06179929,
     &   2.06254935, 2.06331706, 2.06410456, 2.06490350, 2.06570768,
     &   2.06651974, 2.06732750, 2.06817722, 2.06901860, 2.06987834,
     &   2.07079554, 2.07166362, 2.07259440, 2.07356715, 2.07459378,
     &   2.07563424, 2.07659340, 2.07713652, 2.07490039, 2.04777932,
     &   2.03028774, 2.05216765, 2.06194377, 2.07151222, 2.07917452,
     &   2.08122802, 2.08237505, 2.08346868, 2.08480287, 2.08624649,
     &   2.08766246, 2.08899307, 2.09028172, 2.09150457, 2.09268069,
     &   2.09387088, 2.09504199, 2.09618855, 2.09732270, 2.09846997,
     &   2.09962273, 2.10075283, 2.10186410, 2.10287356, 2.10378861,
     &   2.10476828, 2.10587382, 2.10760736, 2.10928273, 2.11077476,
     &   2.11191344, 2.11375046, 2.11515880, 2.11654282, 2.11792994,
     &   2.11921382, 2.12041187, 2.12147903, 2.12230706, 2.12285161,
     &   2.12269449, 2.12306380, 2.12390518, 2.12549710, 2.12750244,
     &   2.12960458, 2.13168263, 2.13326502, 2.13463283, 2.13630509,
     &   2.13809872, 2.13987494, 2.14164448, 2.14360666, 2.14533281,
     &   2.14707875, 2.14898157, 2.15117836, 2.15357757, 2.15609312,
     &   2.15842032, 2.16011548, 2.16193652, 2.16373634, 2.16545987,
     &   2.16703939, 2.16879988, 2.17061520, 2.17239690, 2.17407537,
     &   2.17481470, 2.17244172, 2.16384459, 2.13931155, 2.09924865,
     &   2.09275317, 2.10697818, 2.12348819, 2.13692570, 2.14794898,
     &   2.15753174, 2.16587710, 2.17330003, 2.17994428, 2.18563294,
     &   2.19084764, 2.19556403, 2.19979525, 2.20373535, 2.20763731,
     &   2.21149659, 2.21525407, 2.21899986, 2.22258162, 2.22612619,
     &   2.22946548, 2.23274231, 2.23598504, 2.23914266, 2.24216509,
     &   2.24555087, 2.24932742, 2.25330639, 2.25737214, 2.26148176,
     &   2.26528025, 2.26899385, 2.27220583, 2.27533627, 2.27918315,
     &   2.28378844, 2.28885293, 2.29396176, 2.29873085, 2.30354619,
     &   2.30840564, 2.31319904, 2.31774235, 2.32213354, 2.32643795,
     &   2.33067703, 2.33496332, 2.33927846, 2.34361148, 2.34811020,
     &   2.35281634, 2.35780859, 2.36332965, 2.36949348, 2.37637305,
     &   2.38400459, 2.39268255, 2.40190506, 2.41126108, 2.42014599,
     &   2.42827296, 2.43412519, 2.43543696, 2.43073678, 2.41506433,
     &   2.38240242, 2.34099483, 2.29805398, 2.26492643, 2.24099469,
     &   2.22497177, 2.21720910, 2.21760798, 2.22277308, 2.23103666,
     &   2.24124408, 2.25315166, 2.26660824, 2.28402424, 2.30971360,
     &   2.33358598, 2.36103511, 2.39311838, 2.42293668, 2.43774271,
     &   2.44440603, 2.45025349, 2.45697641, 2.46502900, 2.47410274,
     &   2.48344016, 2.49369812, 2.50578952, 2.51748133, 2.53102827,
     &   2.53520417, 2.52462983, 2.51201940, 2.51070261, 2.52476096,
     &   2.53615713, 2.54729772, 2.55865455, 2.57050633, 2.58217382,
     &   2.59306359, 2.60376120, 2.61452055, 2.62542009, 2.63658714,
     &   2.64794612, 2.66025233, 2.67372489, 2.68761754, 2.70315957,
     &   2.72002101, 2.73692775, 2.75437331, 2.77231193, 2.79047251,
     &   2.80969834, 2.82952118, 2.85007071, 2.87161684, 2.89321160,
     &   2.91559625, 2.93829322, 2.95931840, 2.98073411, 3.00246549,
     &   3.02443242, 3.04653788, 3.06867981, 3.09074306, 3.11261010,
     &   3.13414693, 3.15521646, 3.17581224, 3.19588876, 3.21531510,
     &   3.23396492, 3.25170875, 3.26841497, 3.28395224, 3.29818821,
     &   3.31099296, 3.32223821, 3.33179927, 3.33955550, 3.34588337,
     &   3.35213113, 3.35642481, 3.35866499, 3.35875940, 3.35662317,
     &   3.35294557, 3.35161543, 3.34729028, 3.33989477, 3.32864690,
     &   3.31333447, 3.29435301, 3.27149296, 3.24415851, 3.21272373,
     &   3.17713547, 3.13551426, 3.09031391, 3.04169297, 2.99004555,
     &   2.93540049, 2.87790966, 2.81766725, 2.75492120, 2.69000816,
     &   2.62319493, 2.55479741, 2.48497844, 2.41397309, 2.34249783,
     &   2.27032781, 2.19766784, 2.12507129, 2.05252457, 1.98009789,
     &   1.90844297, 1.83773613, 1.76766837, 1.69863617, 1.63127947,
     &   1.56492341, 1.49967611, 1.43647873, 1.37454689, 1.31395245/
      data (qqi(i,12),i=1,mxwv)/
     &   2.01354265, 2.01370215, 2.01386571, 2.01402664, 2.01419997,
     &   2.01436305, 2.01452351, 2.01466107, 2.01488471, 2.01505852,
     &   2.01523352, 2.01541877, 2.01559353, 2.01578236, 2.01598287,
     &   2.01616073, 2.01635575, 2.01654506, 2.01673841, 2.01693749,
     &   2.01715159, 2.01733899, 2.01754665, 2.01777101, 2.01796818,
     &   2.01817060, 2.01838708, 2.01860404, 2.01883149, 2.01905394,
     &   2.01927590, 2.01951218, 2.01974273, 2.01996303, 2.02021337,
     &   2.02044463, 2.02068686, 2.02093172, 2.02121234, 2.02143908,
     &   2.02169085, 2.02194428, 2.02220511, 2.02253580, 2.02273083,
     &   2.02300692, 2.02327681, 2.02355218, 2.02382827, 2.02411842,
     &   2.02439046, 2.02471852, 2.02497554, 2.02527165, 2.02557445,
     &   2.02589250, 2.02618480, 2.02649236, 2.02680540, 2.02713037,
     &   2.02745199, 2.02777958, 2.02812147, 2.02845168, 2.02877641,
     &   2.02911448, 2.02946544, 2.02981448, 2.03016543, 2.03051519,
     &   2.03090143, 2.03125286, 2.03161335, 2.03200078, 2.03237844,
     &   2.03275514, 2.03314090, 2.03351521, 2.03394175, 2.03434253,
     &   2.03475380, 2.03514838, 2.03556466, 2.03600407, 2.03643084,
     &   2.03685760, 2.03730631, 2.03774691, 2.03819656, 2.03864932,
     &   2.03911567, 2.03957152, 2.04003882, 2.04051280, 2.04099607,
     &   2.04148507, 2.04198170, 2.04248333, 2.04299092, 2.04350376,
     &   2.04402423, 2.04454994, 2.04507971, 2.04562211, 2.04616642,
     &   2.04672050, 2.04727840, 2.04784274, 2.04838610, 2.04895782,
     &   2.04954672, 2.05014896, 2.05076623, 2.05139160, 2.05204558,
     &   2.05265498, 2.05329585, 2.05394435, 2.05460453, 2.05527282,
     &   2.05595541, 2.05665278, 2.05736804, 2.05810332, 2.05887914,
     &   2.05964565, 2.06033635, 2.06067848, 2.05891538, 2.04039288,
     &   2.02896667, 2.04408884, 2.05101514, 2.05783510, 2.06335974,
     &   2.06489062, 2.06573367, 2.06653237, 2.06751704, 2.06859088,
     &   2.06965160, 2.07065964, 2.07164192, 2.07258201, 2.07349348,
     &   2.07441378, 2.07532477, 2.07622004, 2.07710767, 2.07800651,
     &   2.07891107, 2.07979393, 2.08067751, 2.08148646, 2.08223438,
     &   2.08303332, 2.08392501, 2.08524537, 2.08652329, 2.08767080,
     &   2.08882737, 2.08996916, 2.09103656, 2.09220099, 2.09319997,
     &   2.09420681, 2.09515166, 2.09600639, 2.09669805, 2.09719586,
     &   2.09722519, 2.09765339, 2.09840751, 2.09967947, 2.10122538,
     &   2.10282874, 2.10439992, 2.10563207, 2.10672879, 2.10803413,
     &   2.10942864, 2.11080623, 2.11218190, 2.11368656, 2.11502886,
     &   2.11638999, 2.11786318, 2.11953425, 2.12133765, 2.12320828,
     &   2.12493610, 2.12623167, 2.12761807, 2.12898660, 2.13029790,
     &   2.13150835, 2.13283324, 2.13417649, 2.13546944, 2.13663483,
     &   2.13702440, 2.13491654, 2.12815237, 2.11205435, 2.08809233,
     &   2.08451200, 2.09429336, 2.10576701, 2.11525750, 2.12312102,
     &   2.13001871, 2.13606477, 2.14147496, 2.14634204, 2.15051436,
     &   2.15435266, 2.15783310, 2.16094613, 2.16386151, 2.16674519,
     &   2.16957903, 2.17232561, 2.17505646, 2.17768145, 2.18029857,
     &   2.18278551, 2.18524504, 2.18769336, 2.19009519, 2.19243073,
     &   2.19501996, 2.19786334, 2.20082474, 2.20384264, 2.20689487,
     &   2.20974374, 2.21253586, 2.21501613, 2.21747112, 2.22041631,
     &   2.22385979, 2.22759628, 2.23136020, 2.23491144, 2.23851514,
     &   2.24217534, 2.24582434, 2.24933958, 2.25278544, 2.25619888,
     &   2.25958014, 2.26299572, 2.26639795, 2.26971555, 2.27299047,
     &   2.27614784, 2.27911639, 2.28200150, 2.28480220, 2.28752518,
     &   2.29019737, 2.29295230, 2.29571652, 2.29850030, 2.30127692,
     &   2.30397940, 2.30637717, 2.30798435, 2.30812168, 2.30435944,
     &   2.29281068, 2.27486944, 2.25311351, 2.23471665, 2.22058487,
     &   2.21055555, 2.20542932, 2.20568085, 2.20924306, 2.21503401,
     &   2.22228980, 2.23083329, 2.24056101, 2.25307488, 2.27131081,
     &   2.28834534, 2.30807185, 2.33106565, 2.35176992, 2.36107898,
     &   2.36476207, 2.36884785, 2.37376523, 2.37963653, 2.38608289,
     &   2.39261842, 2.39965606, 2.40701270, 2.41364074, 2.42056966,
     &   2.42156529, 2.41712046, 2.41311407, 2.41624999, 2.42666292,
     &   2.43530631, 2.44362879, 2.45185971, 2.45997286, 2.46751642,
     &   2.47456789, 2.48147655, 2.48835754, 2.49532533, 2.50235558,
     &   2.50947165, 2.51689649, 2.52468300, 2.53256750, 2.54081035,
     &   2.54932785, 2.55766344, 2.56609058, 2.57465672, 2.58330703,
     &   2.59229994, 2.60154343, 2.61110830, 2.62111163, 2.63138723,
     &   2.64216828, 2.65337467, 2.66442966, 2.67598772, 2.68808246,
     &   2.70074749, 2.71401477, 2.72791982, 2.74247742, 2.75772667,
     &   2.77368450, 2.79036760, 2.80715132, 2.82449460, 2.84251380,
     &   2.86120868, 2.88057351, 2.90059495, 2.92125297, 2.94252014,
     &   2.96436048, 2.98672938, 3.00957441, 3.03283310, 3.05644703,
     &   3.08042097, 3.10466290, 3.12908936, 3.15360975, 3.17812467,
     &   3.20308948, 3.23159504, 3.25987005, 3.28775239, 3.31603265,
     &   3.34414434, 3.37115145, 3.39739990, 3.42324567, 3.44702911,
     &   3.46837997, 3.48565984, 3.50008798, 3.51144004, 3.51959229,
     &   3.52431679, 3.52543759, 3.52271032, 3.51604009, 3.50544357,
     &   3.49079013, 3.47198200, 3.44923449, 3.42256856, 3.39187837,
     &   3.35751891, 3.31957793, 3.27808738, 3.23331404, 3.18543100,
     &   3.13461208, 3.08113933, 3.02518940, 2.96711850, 2.90753794,
     &   2.84607840, 2.78291082, 2.71919131, 2.65415311, 2.58794975/
      data (qqi(i,13),i=1,mxwv)/
     &   2.01075125, 2.01087999, 2.01100898, 2.01114488, 2.01127291,
     &   2.01139784, 2.01153874, 2.01167774, 2.01183462, 2.01194906,
     &   2.01209188, 2.01223850, 2.01239705, 2.01253176, 2.01268744,
     &   2.01282692, 2.01298237, 2.01314306, 2.01328754, 2.01344466,
     &   2.01360583, 2.01375937, 2.01392555, 2.01410031, 2.01426530,
     &   2.01442957, 2.01460767, 2.01477909, 2.01494956, 2.01512027,
     &   2.01530647, 2.01549172, 2.01566911, 2.01585436, 2.01604438,
     &   2.01623201, 2.01642251, 2.01661921, 2.01681757, 2.01701117,
     &   2.01721787, 2.01741147, 2.01763415, 2.01783013, 2.01788425,
     &   2.01826239, 2.01847291, 2.01869464, 2.01891685, 2.01912975,
     &   2.01937270, 2.01959133, 2.01983070, 2.02006292, 2.02029133,
     &   2.02056670, 2.02079177, 2.02102566, 2.02127767, 2.02152681,
     &   2.02179527, 2.02204680, 2.02230954, 2.02256536, 2.02283812,
     &   2.02312875, 2.02338505, 2.02366638, 2.02394271, 2.02422166,
     &   2.02449536, 2.02480316, 2.02508783, 2.02543569, 2.02570152,
     &   2.02601409, 2.02630568, 2.02661729, 2.02693033, 2.02725077,
     &   2.02758288, 2.02790356, 2.02823424, 2.02856827, 2.02891183,
     &   2.02943587, 2.02959561, 2.02994728, 2.03034019, 2.03066397,
     &   2.03102708, 2.03135395, 2.03176355, 2.03213906, 2.03252125,
     &   2.03290844, 2.03330135, 2.03369784, 2.03409934, 2.03450727,
     &   2.03491592, 2.03533173, 2.03575134, 2.03616214, 2.03654456,
     &   2.03704667, 2.03748655, 2.03792739, 2.03836536, 2.03881645,
     &   2.03928113, 2.03975582, 2.04024100, 2.04073191, 2.04122615,
     &   2.04172468, 2.04222369, 2.04273748, 2.04325390, 2.04377675,
     &   2.04430985, 2.04485178, 2.04540610, 2.04597664, 2.04658818,
     &   2.04714394, 2.04765749, 2.04789758, 2.04658651, 2.03388643,
     &   2.02630115, 2.03682971, 2.04175520, 2.04662776, 2.05061722,
     &   2.05176854, 2.05240488, 2.05300307, 2.05374169, 2.05455065,
     &   2.05535650, 2.05612636, 2.05688238, 2.05761170, 2.05832148,
     &   2.05904245, 2.05974722, 2.06045699, 2.06115627, 2.06186461,
     &   2.06257749, 2.06328154, 2.06397796, 2.06462812, 2.06523895,
     &   2.06588912, 2.06660724, 2.06761980, 2.06860161, 2.06949186,
     &   2.07038951, 2.07126999, 2.07212210, 2.07296348, 2.07380652,
     &   2.07459950, 2.07535195, 2.07604241, 2.07662129, 2.07706785,
     &   2.07719898, 2.07762671, 2.07828069, 2.07929349, 2.08049011,
     &   2.08172178, 2.08292270, 2.08389068, 2.08477235, 2.08580089,
     &   2.08688998, 2.08796787, 2.08904576, 2.09021139, 2.09126425,
     &   2.09233475, 2.09348512, 2.09477043, 2.09614301, 2.09755564,
     &   2.09886265, 2.09987307, 2.10094881, 2.10201073, 2.10303092,
     &   2.10398030, 2.10500455, 2.10602593, 2.10699725, 2.10784912,
     &   2.10811186, 2.10658646, 2.10182309, 2.09103370, 2.07597995,
     &   2.07410622, 2.08099151, 2.08907819, 2.09585190, 2.10151386,
     &   2.10652089, 2.11093855, 2.11491585, 2.11851358, 2.12160921,
     &   2.12446904, 2.12707210, 2.12939978, 2.13159156, 2.13375759,
     &   2.13587570, 2.13792372, 2.13995504, 2.14192057, 2.14388537,
     &   2.14577127, 2.14764810, 2.14952493, 2.15137720, 2.15320110,
     &   2.15520287, 2.15737128, 2.15960836, 2.16188169, 2.16418195,
     &   2.16634607, 2.16847444, 2.17040563, 2.17234063, 2.17461205,
     &   2.17720938, 2.17998958, 2.18277717, 2.18542194, 2.18810296,
     &   2.19082403, 2.19353771, 2.19616151, 2.19874477, 2.20131350,
     &   2.20386934, 2.20645356, 2.20903921, 2.21157908, 2.21410036,
     &   2.21655393, 2.21887827, 2.22113442, 2.22330236, 2.22535944,
     &   2.22728848, 2.22913408, 2.23080277, 2.23226547, 2.23349214,
     &   2.23432374, 2.23461366, 2.23437333, 2.23345065, 2.23069978,
     &   2.22437525, 2.21531892, 2.20428061, 2.19477391, 2.18739986,
     &   2.18224406, 2.17995644, 2.18097234, 2.18409872, 2.18869400,
     &   2.19428968, 2.20078874, 2.20813084, 2.21743202, 2.23072553,
     &   2.24313188, 2.25758004, 2.27437901, 2.28919911, 2.29540229,
     &   2.29759645, 2.30062127, 2.30436325, 2.30880737, 2.31359601,
     &   2.31840014, 2.32348728, 2.32831478, 2.33251214, 2.33674049,
     &   2.33708692, 2.33489728, 2.33399820, 2.33840609, 2.34647369,
     &   2.35324144, 2.35969234, 2.36595130, 2.37189198, 2.37722802,
     &   2.38225365, 2.38720584, 2.39215255, 2.39719582, 2.40229273,
     &   2.40748334, 2.41285396, 2.41844583, 2.42414856, 2.43003750,
     &   2.43608999, 2.44197631, 2.44785523, 2.45378089, 2.45969343,
     &   2.46570778, 2.47177482, 2.47791243, 2.48415375, 2.49040031,
     &   2.49675417, 2.50318122, 2.50950909, 2.51595163, 2.52251744,
     &   2.52921581, 2.53605890, 2.54305911, 2.55023146, 2.55759335,
     &   2.56516123, 2.57295871, 2.58084679, 2.58893991, 2.59728456,
     &   2.60590410, 2.61482334, 2.62406874, 2.63366842, 2.64365244,
     &   2.65405202, 2.66489935, 2.67622805, 2.68807220, 2.70040751,
     &   2.71308088, 2.72632527, 2.74017310, 2.75465560, 2.76980257,
     &   2.78573871, 2.80308914, 2.82129693, 2.84039593, 2.86108255,
     &   2.88330436, 2.90663099, 2.93144345, 2.95831037, 2.98638630,
     &   3.01561761, 3.04593062, 3.07719517, 3.10928273, 3.14198756,
     &   3.17517710, 3.20867181, 3.24224591, 3.27566600, 3.30869842,
     &   3.34099936, 3.37217402, 3.40217161, 3.43071437, 3.45698953,
     &   3.48132038, 3.50346351, 3.52277875, 3.53934979, 3.55310965,
     &   3.56341362, 3.57017779, 3.57373309, 3.57390571, 3.57055640,
     &   3.56382751, 3.55366683, 3.54032469, 3.52361321, 3.50353217/
      data (wwi(i,1),i=1,mxwv)/
     &   0.99999893, 0.99999899, 0.99999905, 0.99999917, 0.99999917,
     &   0.99999928, 0.99999934, 0.99999934, 0.99999940, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999958, 0.99999964, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999994, 0.99999994, 0.99999994, 0.99999994, 0.99999994,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999976, 0.99999970, 0.99999964, 0.99999958,
     &   0.99999952, 0.99999946, 0.99999934, 0.99999928, 0.99999917,
     &   0.99999899, 0.99999875, 0.99999863, 0.99999845, 0.99999803,
     &   0.99999756, 0.99999690, 0.99999636, 0.99999601, 0.99999595,
     &   0.99999589, 0.99999487, 0.99999338, 0.99999100, 0.99998951,
     &   0.99998868, 0.99998760, 0.99998617, 0.99998116, 0.99997437,
     &   0.99996555, 0.99995404, 0.99995154, 0.99995726, 0.99996454,
     &   0.99996686, 0.99996716, 0.99996305, 0.99995965, 0.99993354,
     &   0.99990219, 0.99985403, 0.99981683, 0.99980050, 0.99980265,
     &   0.99981040, 0.99981564, 0.99981272, 0.99980217, 0.99975735,
     &   0.99927634, 0.99793428, 0.99523455, 0.99389344, 0.99461949,
     &   0.99556917, 0.99665117, 0.99732834, 0.99760228, 0.99796760,
     &   0.99828738, 0.99854553, 0.99868679, 0.99877495, 0.99887246,
     &   0.99888206, 0.99804980, 0.99478960, 0.98900956, 0.98652303,
     &   0.98732007, 0.98947984, 0.99293846, 0.99592048, 0.99739212,
     &   0.99805778, 0.99829984, 0.99786675, 0.99673408, 0.99558574,
     &   0.99478257, 0.99389750, 0.99280512, 0.99200344, 0.99055749,
     &   0.97978437, 0.94881326, 0.86067367, 0.60967690, 0.22692783,
     &   0.24402110, 0.34908041, 0.38435230, 0.42141807, 0.44705504,
     &   0.46337482, 0.50317681, 0.58499479, 0.69056219, 0.79392713,
     &   0.87180734, 0.91696227, 0.94493204, 0.95685589, 0.96005332,
     &   0.96240419, 0.96105951, 0.95565194, 0.94754022, 0.94055450,
     &   0.93296361, 0.92204070, 0.90845293, 0.88575310, 0.85709125,
     &   0.83630675, 0.82634318, 0.86181229, 0.89117855, 0.90125400,
     &   0.91037828, 0.91374570, 0.90981656, 0.90216404, 0.89276022,
     &   0.87364262, 0.84773886, 0.81365931, 0.76795143, 0.71263850,
     &   0.63880861, 0.60081506, 0.58054703, 0.58459353, 0.60079491,
     &   0.61934543, 0.63395274, 0.62827170, 0.61567205, 0.61401254,
     &   0.61566389, 0.61506987, 0.61347836, 0.61699241, 0.61041307,
     &   0.60451680, 0.60382456, 0.61171627, 0.62421691, 0.63684320,
     &   0.63682020, 0.61171287, 0.59207040, 0.57065815, 0.54597348,
     &   0.51754898, 0.49330077, 0.46709991, 0.43758005, 0.40136182,
     &   0.34683588, 0.26755103, 0.18720371, 0.12375581, 0.09360991,
     &   0.09752577, 0.11323323, 0.12963845, 0.14419124, 0.15664892,
     &   0.16863048, 0.18015088, 0.19176209, 0.20369312, 0.21541660,
     &   0.22789824, 0.24071628, 0.25377411, 0.26638499, 0.28071111,
     &   0.29709741, 0.31369233, 0.33045653, 0.34305000, 0.35353214,
     &   0.35761806, 0.35893121, 0.35832030, 0.35484597, 0.34928825,
     &   0.35131142, 0.36001039, 0.36978176, 0.37955168, 0.38935012,
     &   0.38836625, 0.38247770, 0.36575145, 0.35199243, 0.35570320,
     &   0.37472662, 0.40229750, 0.42721802, 0.43801203, 0.44849926,
     &   0.45877716, 0.46303931, 0.45217001, 0.43415219, 0.41211826,
     &   0.38715214, 0.36229724, 0.33568290, 0.30611241, 0.27683201,
     &   0.24706928, 0.21694480, 0.18947335, 0.16467519, 0.14249958,
     &   0.12286275, 0.10575227, 0.09075926, 0.07770611, 0.06640869,
     &   0.05607647, 0.04675244, 0.03896822, 0.03250998, 0.02668021,
     &   0.02141886, 0.01760549, 0.01495353, 0.01326762, 0.01214987,
     &   0.01142772, 0.01102241, 0.01084100, 0.01077711, 0.01078768,
     &   0.01084391, 0.01094757, 0.01109459, 0.01140854, 0.01210978,
     &   0.01268792, 0.01358829, 0.01501046, 0.01692639, 0.01800643,
     &   0.01782241, 0.01725059, 0.01681728, 0.01664759, 0.01671414,
     &   0.01688782, 0.01740030, 0.01839100, 0.01860896, 0.01805196,
     &   0.01428891, 0.01127047, 0.00966097, 0.00928079, 0.00968938,
     &   0.00989777, 0.01011711, 0.01042244, 0.01086094, 0.01125539,
     &   0.01149649, 0.01165929, 0.01174688, 0.01185908, 0.01191653,
     &   0.01199670, 0.01212029, 0.01229469, 0.01249737, 0.01256203,
     &   0.01260173, 0.01230579, 0.01192239, 0.01157933, 0.01120660,
     &   0.01084624, 0.01049369, 0.01015059, 0.00981894, 0.00945324,
     &   0.00909977, 0.00875362, 0.00839303, 0.00804719, 0.00771550,
     &   0.00739740, 0.00709233, 0.00679977, 0.00651921, 0.00625017,
     &   0.00599218, 0.00574478, 0.00554037, 0.00534911, 0.00516451,
     &   0.00498634, 0.00481438, 0.00464839, 0.00448818, 0.00433353,
     &   0.00418425, 0.00404015, 0.00390106, 0.00376678, 0.00364388,
     &   0.00354948, 0.00345762, 0.00336821, 0.00328119, 0.00319648,
     &   0.00312487, 0.00312572, 0.00312688, 0.00312833, 0.00313576,
     &   0.00314757, 0.00315967, 0.00317993, 0.00321868, 0.00325812,
     &   0.00329554, 0.00329372, 0.00329197, 0.00329030, 0.00329054,
     &   0.00329126, 0.00329203, 0.00328997, 0.00328539, 0.00328084,
     &   0.00327255, 0.00325503, 0.00323760, 0.00321988, 0.00317981,
     &   0.00314022, 0.00310113, 0.00304452, 0.00298321, 0.00292311,
     &   0.00284231, 0.00274328, 0.00264768, 0.00255143, 0.00244715,
     &   0.00234711, 0.00225115, 0.00215691, 0.00206657, 0.00198000/
      data (wwi(i,2),i=1,mxwv)/
     &   0.99999857, 0.99999857, 0.99999875, 0.99999881, 0.99999887,
     &   0.99999899, 0.99999911, 0.99999911, 0.99999917, 0.99999928,
     &   0.99999934, 0.99999940, 0.99999946, 0.99999946, 0.99999958,
     &   0.99999952, 0.99999970, 0.99999970, 0.99999976, 0.99999976,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999988, 0.99999988, 0.99999988, 0.99999988, 0.99999988,
     &   0.99999988, 0.99999988, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999964, 0.99999958, 0.99999952, 0.99999940,
     &   0.99999928, 0.99999917, 0.99999905, 0.99999893, 0.99999881,
     &   0.99999851, 0.99999821, 0.99999797, 0.99999768, 0.99999714,
     &   0.99999636, 0.99999547, 0.99999458, 0.99999404, 0.99999398,
     &   0.99999386, 0.99999243, 0.99999017, 0.99998653, 0.99998426,
     &   0.99998301, 0.99998128, 0.99997902, 0.99997139, 0.99996084,
     &   0.99994707, 0.99992895, 0.99992466, 0.99993318, 0.99994421,
     &   0.99994755, 0.99994767, 0.99994075, 0.99993479, 0.99989200,
     &   0.99983996, 0.99975973, 0.99969667, 0.99966770, 0.99966949,
     &   0.99968088, 0.99968839, 0.99968231, 0.99966341, 0.99958628,
     &   0.99876487, 0.99647784, 0.99190092, 0.98964989, 0.99089575,
     &   0.99252105, 0.99436688, 0.99552828, 0.99601245, 0.99664521,
     &   0.99719727, 0.99764299, 0.99789530, 0.99806041, 0.99823797,
     &   0.99827814, 0.99704534, 0.99225670, 0.98400348, 0.98073733,
     &   0.98216856, 0.98544854, 0.99039799, 0.99455929, 0.99659723,
     &   0.99752426, 0.99788558, 0.99741632, 0.99615049, 0.99494487,
     &   0.99420083, 0.99343139, 0.99251938, 0.99199682, 0.99095076,
     &   0.98160130, 0.95582223, 0.88522732, 0.67619228, 0.29152361,
     &   0.30551463, 0.40425956, 0.43248120, 0.46102765, 0.47438729,
     &   0.47632262, 0.48597237, 0.52856725, 0.61600667, 0.72851557,
     &   0.82838976, 0.89086848, 0.92989016, 0.94715494, 0.95280212,
     &   0.95711452, 0.95696872, 0.95239574, 0.94519669, 0.93932480,
     &   0.93305498, 0.92369884, 0.91207933, 0.89209098, 0.86665779,
     &   0.84825599, 0.83940762, 0.87309009, 0.90115023, 0.91163224,
     &   0.92098618, 0.92509627, 0.92278033, 0.91729969, 0.91045165,
     &   0.89556497, 0.87514788, 0.84795827, 0.81076449, 0.76478988,
     &   0.70064265, 0.66558307, 0.64644486, 0.64964908, 0.66472000,
     &   0.68246102, 0.69751531, 0.69420624, 0.68410349, 0.68372840,
     &   0.68652105, 0.68738174, 0.68729419, 0.69198036, 0.68763900,
     &   0.68373132, 0.68441045, 0.69291615, 0.70569664, 0.71876508,
     &   0.72094256, 0.70096648, 0.68542290, 0.66807580, 0.64745861,
     &   0.62288517, 0.60180908, 0.57857347, 0.55155116, 0.51703018,
     &   0.46143696, 0.37303039, 0.27424991, 0.18889360, 0.14545546,
     &   0.15027069, 0.17136745, 0.19276491, 0.21116352, 0.22661430,
     &   0.24120043, 0.25507790, 0.26893476, 0.28307718, 0.29716191,
     &   0.31212389, 0.32757983, 0.34371549, 0.35946730, 0.37727159,
     &   0.39775208, 0.41893044, 0.44068408, 0.45802507, 0.47314292,
     &   0.48132104, 0.48632586, 0.48909095, 0.48860648, 0.48541075,
     &   0.49035192, 0.50275111, 0.51659834, 0.53041017, 0.54406279,
     &   0.54632390, 0.54339027, 0.52848268, 0.51607549, 0.52268535,
     &   0.54595524, 0.57763642, 0.60558456, 0.61895287, 0.63170552,
     &   0.64394981, 0.65039802, 0.64278865, 0.62817365, 0.60910833,
     &   0.58625871, 0.56245852, 0.53553045, 0.50365341, 0.47014183,
     &   0.43380564, 0.39443886, 0.35618335, 0.31949779, 0.28477150,
     &   0.25235251, 0.22272122, 0.19557977, 0.17097858, 0.14889865,
     &   0.12798254, 0.10847671, 0.09172940, 0.07749474, 0.06433093,
     &   0.05217541, 0.04322435, 0.03692795, 0.03291101, 0.03024849,
     &   0.02853328, 0.02758407, 0.02718678, 0.02708058, 0.02715923,
     &   0.02735126, 0.02766037, 0.02807737, 0.02890508, 0.03069258,
     &   0.03221051, 0.03452437, 0.03815779, 0.04311199, 0.04606583,
     &   0.04584938, 0.04457941, 0.04362706, 0.04332679, 0.04362273,
     &   0.04418838, 0.04561748, 0.04829028, 0.04899029, 0.04769443,
     &   0.03803721, 0.03016800, 0.02594159, 0.02495557, 0.02608044,
     &   0.02667335, 0.02729511, 0.02814514, 0.02934947, 0.03043726,
     &   0.03111591, 0.03158483, 0.03185226, 0.03218313, 0.03236684,
     &   0.03260928, 0.03296650, 0.03345834, 0.03402412, 0.03422131,
     &   0.03435063, 0.03358199, 0.03257618, 0.03167496, 0.03069100,
     &   0.02973729, 0.02880177, 0.02788912, 0.02700499, 0.02602631,
     &   0.02507834, 0.02414792, 0.02317596, 0.02224196, 0.02134454,
     &   0.02048236, 0.01965409, 0.01885850, 0.01809435, 0.01736046,
     &   0.01665569, 0.01597895, 0.01541959, 0.01489571, 0.01438951,
     &   0.01390040, 0.01342782, 0.01297121, 0.01253005, 0.01210381,
     &   0.01169199, 0.01129412, 0.01090973, 0.01053837, 0.01019826,
     &   0.00993718, 0.00968292, 0.00943530, 0.00919412, 0.00895921,
     &   0.00876067, 0.00876403, 0.00876823, 0.00877319, 0.00879479,
     &   0.00882859, 0.00886317, 0.00892047, 0.00902933, 0.00914010,
     &   0.00924520, 0.00924082, 0.00923664, 0.00923263, 0.00923394,
     &   0.00923659, 0.00923934, 0.00923414, 0.00922192, 0.00920974,
     &   0.00918713, 0.00913872, 0.00909055, 0.00904153, 0.00893008,
     &   0.00881996, 0.00871115, 0.00855340, 0.00838242, 0.00821479,
     &   0.00798918, 0.00771251, 0.00744530, 0.00717614, 0.00688438,
     &   0.00660437, 0.00633565, 0.00607163, 0.00581846, 0.00557577/
      data (wwi(i,3),i=1,mxwv)/
     &   0.99999797, 0.99999809, 0.99999821, 0.99999803, 0.99999851,
     &   0.99999863, 0.99999875, 0.99999887, 0.99999887, 0.99999899,
     &   0.99999911, 0.99999911, 0.99999923, 0.99999928, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999958, 0.99999964, 0.99999964,
     &   0.99999970, 0.99999976, 0.99999976, 0.99999976, 0.99999982,
     &   0.99999976, 0.99999982, 0.99999982, 0.99999982, 0.99999982,
     &   0.99999982, 0.99999982, 0.99999976, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999964, 0.99999970, 0.99999970, 0.99999964,
     &   0.99999958, 0.99999946, 0.99999946, 0.99999928, 0.99999917,
     &   0.99999899, 0.99999875, 0.99999857, 0.99999845, 0.99999827,
     &   0.99999785, 0.99999738, 0.99999702, 0.99999648, 0.99999589,
     &   0.99999470, 0.99999338, 0.99999213, 0.99999136, 0.99999124,
     &   0.99999112, 0.99998885, 0.99998564, 0.99998027, 0.99997681,
     &   0.99997509, 0.99997258, 0.99996936, 0.99995804, 0.99994266,
     &   0.99992245, 0.99989593, 0.99988919, 0.99990171, 0.99991786,
     &   0.99992269, 0.99992281, 0.99991250, 0.99990356, 0.99983984,
     &   0.99976224, 0.99964201, 0.99954683, 0.99950188, 0.99950266,
     &   0.99951786, 0.99952716, 0.99951559, 0.99948412, 0.99936241,
     &   0.99808753, 0.99452692, 0.98739529, 0.98381990, 0.98565078,
     &   0.98810995, 0.99096310, 0.99276608, 0.99350077, 0.99449104,
     &   0.99536479, 0.99607617, 0.99647552, 0.99673522, 0.99702120,
     &   0.99707931, 0.99498278, 0.98689884, 0.97315419, 0.96781981,
     &   0.97025156, 0.97575086, 0.98401922, 0.99098343, 0.99440980,
     &   0.99598193, 0.99662071, 0.99594933, 0.99409890, 0.99245036,
     &   0.99158871, 0.99079233, 0.98992407, 0.98971587, 0.98902500,
     &   0.97933984, 0.95467716, 0.89368397, 0.71991342, 0.35597938,
     &   0.36294839, 0.44760695, 0.46896651, 0.49101466, 0.49698979,
     &   0.49195448, 0.48743597, 0.49699655, 0.54460973, 0.63827211,
     &   0.74900037, 0.83253992, 0.89060301, 0.91839516, 0.92872238,
     &   0.93694675, 0.93857855, 0.93416202, 0.92662853, 0.92106938,
     &   0.91534007, 0.90619451, 0.89491481, 0.87470204, 0.84903949,
     &   0.83077210, 0.82196420, 0.85876620, 0.89044076, 0.90366453,
     &   0.91534114, 0.92132443, 0.92058009, 0.91668010, 0.91159970,
     &   0.89901149, 0.88158673, 0.85832006, 0.82625198, 0.78655219,
     &   0.72958285, 0.69650561, 0.67804277, 0.67992616, 0.69364184,
     &   0.71069682, 0.72671425, 0.72591513, 0.71818978, 0.71909010,
     &   0.72309375, 0.72546870, 0.72690678, 0.73287809, 0.73078024,
     &   0.72883052, 0.73084658, 0.73999798, 0.75320184, 0.76695657,
     &   0.77141821, 0.75637853, 0.74496543, 0.73193198, 0.71595711,
     &   0.69626629, 0.67957819, 0.66102076, 0.63887697, 0.60961592,
     &   0.55929643, 0.47209683, 0.36490318, 0.26295909, 0.20659688,
     &   0.21099554, 0.23535700, 0.25928351, 0.27904636, 0.29518691,
     &   0.31003377, 0.32392046, 0.33758909, 0.35139573, 0.36531287,
     &   0.38007131, 0.39543122, 0.41194376, 0.42826164, 0.44675690,
     &   0.46836591, 0.49131596, 0.51538950, 0.53545219, 0.55346936,
     &   0.56479460, 0.57283747, 0.57852614, 0.58094192, 0.58024454,
     &   0.58708692, 0.60119194, 0.61704695, 0.63285345, 0.64830089,
     &   0.65328091, 0.65354687, 0.64248669, 0.63312858, 0.64125800,
     &   0.66474921, 0.69542569, 0.72215962, 0.73601115, 0.74902600,
     &   0.76131314, 0.76877147, 0.76512146, 0.75587291, 0.74292994,
     &   0.72666281, 0.70924449, 0.68871361, 0.66313326, 0.63502032,
     &   0.60288113, 0.56592357, 0.52793407, 0.48935369, 0.45066297,
     &   0.41241905, 0.37551603, 0.33983079, 0.30575699, 0.27362531,
     &   0.24161421, 0.21024558, 0.18209258, 0.15719207, 0.13318883,
     &   0.11010216, 0.09258837, 0.07999343, 0.07189626, 0.06652693,
     &   0.06308193, 0.06121970, 0.06053429, 0.06047411, 0.06081149,
     &   0.06139238, 0.06222116, 0.06327982, 0.06519528, 0.06912104,
     &   0.07256513, 0.07764237, 0.08548961, 0.09631079, 0.10336585,
     &   0.10392884, 0.10195810, 0.10053621, 0.10045928, 0.10167964,
     &   0.10350128, 0.10723408, 0.11390983, 0.11633753, 0.11433665,
     &   0.09292936, 0.07471169, 0.06471749, 0.06239028, 0.06528799,
     &   0.06691725, 0.06862637, 0.07089777, 0.07404611, 0.07693660,
     &   0.07883425, 0.08021669, 0.08110239, 0.08212784, 0.08278591,
     &   0.08357335, 0.08463445, 0.08601744, 0.08756842, 0.08821471,
     &   0.08868483, 0.08694308, 0.08459746, 0.08248746, 0.08015265,
     &   0.07787441, 0.07562384, 0.07341421, 0.07126124, 0.06885435,
     &   0.06651006, 0.06419604, 0.06176138, 0.05941038, 0.05714085,
     &   0.05495062, 0.05283751, 0.05079934, 0.04883397, 0.04693926,
     &   0.04511309, 0.04335339, 0.04189736, 0.04053020, 0.03920541,
     &   0.03792183, 0.03667833, 0.03547380, 0.03430715, 0.03317729,
     &   0.03208319, 0.03102382, 0.02999816, 0.02900523, 0.02809460,
     &   0.02739607, 0.02671467, 0.02604999, 0.02540161, 0.02476914,
     &   0.02423464, 0.02424949, 0.02426641, 0.02428515, 0.02434907,
     &   0.02444605, 0.02454499, 0.02470569, 0.02500676, 0.02531286,
     &   0.02560330, 0.02559537, 0.02558784, 0.02558063, 0.02558776,
     &   0.02559840, 0.02560920, 0.02559821, 0.02556794, 0.02553769,
     &   0.02547883, 0.02534947, 0.02522062, 0.02508932, 0.02478738,
     &   0.02448880, 0.02419355, 0.02376417, 0.02329820, 0.02284097,
     &   0.02222438, 0.02146698, 0.02073468, 0.01999615, 0.01919451,
     &   0.01842428, 0.01768429, 0.01695644, 0.01625776, 0.01558734/
      data (wwi(i,4),i=1,mxwv)/
     &   0.99999690, 0.99999720, 0.99999750, 0.99999779, 0.99999803,
     &   0.99999803, 0.99999827, 0.99999839, 0.99999851, 0.99999863,
     &   0.99999875, 0.99999845, 0.99999899, 0.99999911, 0.99999923,
     &   0.99999934, 0.99999940, 0.99999946, 0.99999952, 0.99999952,
     &   0.99999958, 0.99999958, 0.99999964, 0.99999970, 0.99999970,
     &   0.99999976, 0.99999976, 0.99999970, 0.99999976, 0.99999976,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999970, 0.99999940,
     &   0.99999958, 0.99999940, 0.99999958, 0.99999958, 0.99999940,
     &   0.99999940, 0.99999934, 0.99999923, 0.99999905, 0.99999881,
     &   0.99999863, 0.99999839, 0.99999809, 0.99999785, 0.99999696,
     &   0.99999678, 0.99999642, 0.99999577, 0.99999535, 0.99999410,
     &   0.99999255, 0.99998999, 0.99998891, 0.99998713, 0.99998754,
     &   0.99998719, 0.99998409, 0.99997950, 0.99997163, 0.99996680,
     &   0.99996382, 0.99995965, 0.99995548, 0.99993813, 0.99991578,
     &   0.99988824, 0.99985093, 0.99984014, 0.99985707, 0.99988145,
     &   0.99988663, 0.99988759, 0.99987274, 0.99985945, 0.99976832,
     &   0.99965429, 0.99948084, 0.99934053, 0.99927515, 0.99927640,
     &   0.99929756, 0.99931014, 0.99929345, 0.99924713, 0.99906850,
     &   0.99721551, 0.99208367, 0.98192412, 0.97684270, 0.97937059,
     &   0.98281169, 0.98685193, 0.98941767, 0.99045193, 0.99186695,
     &   0.99312198, 0.99414700, 0.99471527, 0.99507779, 0.99548221,
     &   0.99554276, 0.99230957, 0.97992903, 0.95904315, 0.95073581,
     &   0.95400792, 0.96202672, 0.97457939, 0.98543733, 0.99086004,
     &   0.99336326, 0.99436998, 0.99320835, 0.99007154, 0.98728120,
     &   0.98584831, 0.98460054, 0.98336112, 0.98338550, 0.98292482,
     &   0.97006178, 0.94111669, 0.88206238, 0.73682249, 0.41387951,
     &   0.41163549, 0.47926876, 0.49554500, 0.51364166, 0.51610518,
     &   0.50881821, 0.50125271, 0.49812537, 0.51700819, 0.57900298,
     &   0.67619991, 0.76468641, 0.83548468, 0.87218833, 0.88618559,
     &   0.89802551, 0.90050828, 0.89422029, 0.88380665, 0.87681055,
     &   0.87008601, 0.85910672, 0.84601504, 0.82221764, 0.79289371,
     &   0.77284414, 0.76324528, 0.80773807, 0.84883595, 0.86819023,
     &   0.88546151, 0.89544320, 0.89689618, 0.89454031, 0.89102691,
     &   0.87925756, 0.86280787, 0.84099406, 0.81106561, 0.77481395,
     &   0.72218657, 0.68974519, 0.67137045, 0.67147094, 0.68371469,
     &   0.70044667, 0.71844357, 0.72065336, 0.71530974, 0.71771610,
     &   0.72334802, 0.72764039, 0.73094314, 0.73880970, 0.73922181,
     &   0.73941463, 0.74306631, 0.75353950, 0.76819772, 0.78389245,
     &   0.79140508, 0.78081620, 0.77328527, 0.76433641, 0.75281018,
     &   0.73791057, 0.72579348, 0.71238667, 0.69596606, 0.67353213,
     &   0.63192511, 0.55345327, 0.44839990, 0.33894250, 0.27287275,
     &   0.27556446, 0.30030939, 0.32371524, 0.34214598, 0.35663515,
     &   0.36951265, 0.38123545, 0.39251095, 0.40369707, 0.41503990,
     &   0.42704034, 0.43965736, 0.45376852, 0.46803796, 0.48449513,
     &   0.50442481, 0.52665740, 0.55099648, 0.57250202, 0.59256524,
     &   0.60670698, 0.61759663, 0.62606800, 0.63132489, 0.63305616,
     &   0.64126319, 0.65623486, 0.67331511, 0.69038224, 0.70690686,
     &   0.71409214, 0.71709162, 0.70965403, 0.70306194, 0.71170419,
     &   0.73397452, 0.76244360, 0.78726590, 0.80096114, 0.81366360,
     &   0.82548577, 0.83327550, 0.83231491, 0.82695627, 0.81870323,
     &   0.80781567, 0.79599196, 0.78162980, 0.76304656, 0.74208182,
     &   0.71727872, 0.68759894, 0.65599281, 0.62264925, 0.58782929,
     &   0.55193180, 0.51584721, 0.47936654, 0.44293183, 0.40699750,
     &   0.36943814, 0.33074740, 0.29433805, 0.26066911, 0.22658072,
     &   0.19208945, 0.16487387, 0.14470139, 0.13158947, 0.12289254,
     &   0.11734834, 0.11445559, 0.11360842, 0.11385804, 0.11480498,
     &   0.11617858, 0.11797658, 0.12016739, 0.12377104, 0.13066253,
     &   0.13682570, 0.14553519, 0.15855931, 0.17631584, 0.18879677,
     &   0.19171400, 0.19014901, 0.18923990, 0.19041193, 0.19376323,
     &   0.19819941, 0.20589019, 0.21927404, 0.22626847, 0.22620827,
     &   0.19080292, 0.15767540, 0.13861781, 0.13399532, 0.14017294,
     &   0.14396426, 0.14797883, 0.15316658, 0.16021024, 0.16695239,
     &   0.17180833, 0.17567647, 0.17857942, 0.18172130, 0.18412407,
     &   0.18673681, 0.18989514, 0.19368844, 0.19775851, 0.20000853,
     &   0.20185278, 0.19916102, 0.19513771, 0.19148555, 0.18726942,
     &   0.18308161, 0.17886376, 0.17465101, 0.17048416, 0.16569348,
     &   0.16096187, 0.15622249, 0.15113990, 0.14617294, 0.14132275,
     &   0.13659006, 0.13197535, 0.12747873, 0.12310009, 0.11883899,
     &   0.11469480, 0.11066666, 0.10732517, 0.10416777, 0.10108624,
     &   0.09807993, 0.09514812, 0.09229001, 0.08950470, 0.08679126,
     &   0.08414867, 0.08157586, 0.07907175, 0.07663515, 0.07439242,
     &   0.07267463, 0.07099206, 0.06934420, 0.06773057, 0.06615065,
     &   0.06481542, 0.06488576, 0.06495991, 0.06503740, 0.06522968,
     &   0.06550568, 0.06578565, 0.06622311, 0.06701973, 0.06782790,
     &   0.06859459, 0.06859595, 0.06859744, 0.06859890, 0.06863630,
     &   0.06868194, 0.06872728, 0.06871604, 0.06865463, 0.06859259,
     &   0.06845646, 0.06813845, 0.06782097, 0.06749639, 0.06673151,
     &   0.06597377, 0.06522313, 0.06412415, 0.06292796, 0.06175176,
     &   0.06015848, 0.05819366, 0.05628844, 0.05436117, 0.05226212,
     &   0.05023926, 0.04829020, 0.04636759, 0.04451698, 0.04273654/
      data (wwi(i,5),i=1,mxwv)/
     &   0.99999571, 0.99999642, 0.99999648, 0.99999684, 0.99999720,
     &   0.99999738, 0.99999762, 0.99999779, 0.99999779, 0.99999803,
     &   0.99999821, 0.99999839, 0.99999857, 0.99999881, 0.99999887,
     &   0.99999911, 0.99999917, 0.99999928, 0.99999928, 0.99999934,
     &   0.99999946, 0.99999952, 0.99999952, 0.99999952, 0.99999964,
     &   0.99999970, 0.99999970, 0.99999970, 0.99999970, 0.99999964,
     &   0.99999964, 0.99999964, 0.99999958, 0.99999958, 0.99999952,
     &   0.99999946, 0.99999928, 0.99999940, 0.99999940, 0.99999934,
     &   0.99999923, 0.99999911, 0.99999881, 0.99999857, 0.99999839,
     &   0.99999809, 0.99999774, 0.99999738, 0.99999696, 0.99999666,
     &   0.99999583, 0.99999493, 0.99999434, 0.99999350, 0.99999171,
     &   0.99998987, 0.99998730, 0.99998468, 0.99998164, 0.99998093,
     &   0.99998200, 0.99997848, 0.99996895, 0.99996084, 0.99995434,
     &   0.99994832, 0.99994546, 0.99993604, 0.99991548, 0.99988425,
     &   0.99984282, 0.99978954, 0.99977040, 0.99980152, 0.99983352,
     &   0.99984425, 0.99983877, 0.99981791, 0.99980444, 0.99966788,
     &   0.99951112, 0.99926293, 0.99907678, 0.99897170, 0.99897444,
     &   0.99899608, 0.99902570, 0.99898279, 0.99892312, 0.99866891,
     &   0.99605852, 0.98890817, 0.97494942, 0.96801221, 0.97140336,
     &   0.97605962, 0.98157442, 0.98509455, 0.98650831, 0.98846227,
     &   0.99020374, 0.99163312, 0.99242485, 0.99292731, 0.99349242,
     &   0.99356771, 0.98894376, 0.97150588, 0.94273639, 0.93136054,
     &   0.93558359, 0.94632095, 0.96356243, 0.97882515, 0.98657000,
     &   0.99016881, 0.99159712, 0.98979926, 0.98499626, 0.98061687,
     &   0.97818506, 0.97593158, 0.97357845, 0.97312975, 0.97190785,
     &   0.95086074, 0.90789688, 0.83981323, 0.72004807, 0.45874456,
     &   0.44906506, 0.50140125, 0.51464319, 0.53070879, 0.53179508,
     &   0.52394485, 0.51614994, 0.50962806, 0.51396328, 0.54971373,
     &   0.62666029, 0.71087909, 0.78647399, 0.82764715, 0.84271610,
     &   0.85554725, 0.85658431, 0.84601110, 0.82989573, 0.81857991,
     &   0.80777073, 0.79167807, 0.77336162, 0.74265110, 0.70754492,
     &   0.68505913, 0.67468548, 0.72447717, 0.77499813, 0.80053198,
     &   0.82440317, 0.83908021, 0.84234589, 0.84070975, 0.83797604,
     &   0.82504177, 0.80733538, 0.78470600, 0.75474375, 0.72060037,
     &   0.67216825, 0.64120466, 0.62383342, 0.62203586, 0.63189358,
     &   0.64734155, 0.66685683, 0.67229587, 0.66956961, 0.67358196,
     &   0.68103147, 0.68755323, 0.69306958, 0.70341456, 0.70685804,
     &   0.70963466, 0.71543026, 0.72813290, 0.74570233, 0.76533687,
     &   0.77763504, 0.77170604, 0.76840264, 0.76379889, 0.75686824,
     &   0.74678254, 0.73950177, 0.73167652, 0.72153169, 0.70678067,
     &   0.67549229, 0.60986775, 0.51498902, 0.40781400, 0.33724284,
     &   0.33723295, 0.35952792, 0.37977231, 0.39490563, 0.40618727,
     &   0.41576651, 0.42408228, 0.43174544, 0.43905476, 0.44628635,
     &   0.45382908, 0.46176884, 0.47102141, 0.48070803, 0.49227959,
     &   0.50720692, 0.52532303, 0.54683006, 0.56768090, 0.58847594,
     &   0.60496897, 0.61862314, 0.63001370, 0.63836402, 0.64272326,
     &   0.65233964, 0.66831923, 0.68709874, 0.70611846, 0.72453362,
     &   0.73447275, 0.74055976, 0.73656994, 0.73248005, 0.74178505,
     &   0.76364505, 0.79143792, 0.81604803, 0.83054161, 0.84383142,
     &   0.85603762, 0.86458528, 0.86570859, 0.86310881, 0.85811287,
     &   0.85097188, 0.84311551, 0.83324993, 0.81998068, 0.80474705,
     &   0.78626466, 0.76351523, 0.73879498, 0.71211076, 0.68352199,
     &   0.65322232, 0.62195462, 0.58931011, 0.55558532, 0.52113807,
     &   0.48374462, 0.44360363, 0.40421996, 0.36629352, 0.32608670,
     &   0.28334799, 0.24818780, 0.22122918, 0.20343685, 0.19158533,
     &   0.18402475, 0.18014783, 0.17921408, 0.17987368, 0.18154389,
     &   0.18383326, 0.18671885, 0.19014981, 0.19542804, 0.20489724,
     &   0.21337272, 0.22487064, 0.24133372, 0.26297423, 0.27875537,
     &   0.28428361, 0.28419828, 0.28468490, 0.28755471, 0.29312277,
     &   0.30013442, 0.31116003, 0.33031642, 0.34328014, 0.34907261,
     &   0.30852255, 0.26479188, 0.23780739, 0.23057866, 0.24008158,
     &   0.24634682, 0.25304750, 0.26153952, 0.27294552, 0.28445965,
     &   0.29357123, 0.30147812, 0.30820724, 0.31530622, 0.32148260,
     &   0.32795537, 0.33537555, 0.34385279, 0.35266131, 0.35899734,
     &   0.36469665, 0.36362508, 0.36036927, 0.35740325, 0.35335019,
     &   0.34914801, 0.34468633, 0.34003326, 0.33526844, 0.32929918,
     &   0.32322305, 0.31692597, 0.30981618, 0.30269742, 0.29557979,
     &   0.28847313, 0.28138712, 0.27433103, 0.26731384, 0.26034418,
     &   0.25343025, 0.24657983, 0.24089096, 0.23544154, 0.23003358,
     &   0.22467196, 0.21936122, 0.21410558, 0.20890896, 0.20377502,
     &   0.19870704, 0.19370808, 0.18878087, 0.18392788, 0.17942446,
     &   0.17599539, 0.17260179, 0.16924505, 0.16592643, 0.16264707,
     &   0.15987764, 0.16021007, 0.16054167, 0.16087195, 0.16144976,
     &   0.16220631, 0.16296443, 0.16406555, 0.16595612, 0.16786337,
     &   0.16967043, 0.16978228, 0.16988923, 0.16999124, 0.17016809,
     &   0.17035893, 0.17054509, 0.17060238, 0.17054501, 0.17048278,
     &   0.17025302, 0.16961657, 0.16897723, 0.16831829, 0.16667424,
     &   0.16503865, 0.16341168, 0.16099398, 0.15834458, 0.15572689,
     &   0.15214501, 0.14768814, 0.14333655, 0.13890286, 0.13403505,
     &   0.12930973, 0.12472465, 0.12016995, 0.11575629, 0.11148240/
      data (wwi(i,6),i=1,mxwv)/
     &   0.99999470, 0.99999493, 0.99999529, 0.99999577, 0.99999601,
     &   0.99999636, 0.99999666, 0.99999690, 0.99999708, 0.99999720,
     &   0.99999726, 0.99999785, 0.99999809, 0.99999833, 0.99999857,
     &   0.99999869, 0.99999881, 0.99999893, 0.99999905, 0.99999899,
     &   0.99999923, 0.99999928, 0.99999934, 0.99999934, 0.99999946,
     &   0.99999952, 0.99999958, 0.99999952, 0.99999952, 0.99999952,
     &   0.99999952, 0.99999946, 0.99999940, 0.99999940, 0.99999934,
     &   0.99999928, 0.99999899, 0.99999923, 0.99999917, 0.99999911,
     &   0.99999881, 0.99999863, 0.99999845, 0.99999815, 0.99999750,
     &   0.99999738, 0.99999696, 0.99999636, 0.99999589, 0.99999535,
     &   0.99999434, 0.99999315, 0.99999124, 0.99999118, 0.99998873,
     &   0.99998593, 0.99997866, 0.99997890, 0.99997705, 0.99997544,
     &   0.99997610, 0.99996948, 0.99996066, 0.99994707, 0.99993730,
     &   0.99993157, 0.99991971, 0.99991482, 0.99988270, 0.99983984,
     &   0.99977517, 0.99971038, 0.99968499, 0.99972385, 0.99976933,
     &   0.99977845, 0.99977899, 0.99975592, 0.99972248, 0.99954194,
     &   0.99933481, 0.99897343, 0.99871051, 0.99859446, 0.99858415,
     &   0.99861479, 0.99864805, 0.99860424, 0.99852669, 0.99816507,
     &   0.99455482, 0.98479646, 0.96588796, 0.95657581, 0.96110421,
     &   0.96734828, 0.97477561, 0.97953320, 0.98143697, 0.98409063,
     &   0.98644549, 0.98839158, 0.98945570, 0.99013281, 0.99089706,
     &   0.99098766, 0.98457909, 0.96081322, 0.92256647, 0.90768814,
     &   0.91307044, 0.92700070, 0.94980031, 0.97040492, 0.98103279,
     &   0.98601687, 0.98799670, 0.98542607, 0.97862685, 0.97241724,
     &   0.96890450, 0.96558833, 0.96204346, 0.96104026, 0.95868099,
     &   0.92679322, 0.86155975, 0.76680946, 0.66494864, 0.48639616,
     &   0.47543660, 0.51664793, 0.52835000, 0.54348129, 0.54419661,
     &   0.53627580, 0.52858984, 0.52130926, 0.51886147, 0.53458047,
     &   0.58759916, 0.66168153, 0.73852605, 0.78354228, 0.80012691,
     &   0.81426316, 0.81454360, 0.80098510, 0.78090227, 0.76648808,
     &   0.75255865, 0.73265558, 0.71040064, 0.67549634, 0.63823324,
     &   0.61578643, 0.60597807, 0.65280223, 0.70466685, 0.73121166,
     &   0.75716645, 0.77285588, 0.77473134, 0.77062821, 0.76531208,
     &   0.74761462, 0.72512829, 0.69861865, 0.66676211, 0.63446152,
     &   0.59389639, 0.56960905, 0.55673379, 0.55453837, 0.56084377,
     &   0.57206208, 0.58816493, 0.59403747, 0.59299392, 0.59712553,
     &   0.60441583, 0.61135203, 0.61758089, 0.62852359, 0.63390082,
     &   0.63847178, 0.64560992, 0.65956461, 0.67935747, 0.70303953,
     &   0.72100908, 0.72032279, 0.72196585, 0.72247726, 0.72091335,
     &   0.71633983, 0.71476835, 0.71360534, 0.71092457, 0.70527965,
     &   0.68601847, 0.63612288, 0.55740422, 0.46161699, 0.39287212,
     &   0.39002362, 0.40812832, 0.42393029, 0.43520954, 0.44303995,
     &   0.44933280, 0.45436689, 0.45865315, 0.46240443, 0.46564230,
     &   0.46882913, 0.47201312, 0.47570330, 0.47973055, 0.48482683,
     &   0.49207604, 0.50211489, 0.51577210, 0.53105390, 0.54814214,
     &   0.56376076, 0.57796586, 0.59077215, 0.60134566, 0.60793483,
     &   0.61839724, 0.63483304, 0.65524292, 0.67672747, 0.69802028,
     &   0.71176857, 0.72195435, 0.72177136, 0.72028512, 0.73094249,
     &   0.75412762, 0.78407747, 0.81161261, 0.82913148, 0.84511274,
     &   0.85966516, 0.87041372, 0.87397343, 0.87389642, 0.87155199,
     &   0.86725479, 0.86238152, 0.85582477, 0.84639537, 0.83537102,
     &   0.82162863, 0.80426514, 0.78522706, 0.76444346, 0.74186999,
     &   0.71757591, 0.69220263, 0.66514659, 0.63653392, 0.60655469,
     &   0.57309783, 0.53601402, 0.49832359, 0.46069992, 0.41906238,
     &   0.37263796, 0.33271679, 0.30088645, 0.27937797, 0.26482558,
     &   0.25533783, 0.25028342, 0.24890468, 0.24953105, 0.25142017,
     &   0.25408113, 0.25745609, 0.26147342, 0.26750445, 0.27794579,
     &   0.28719562, 0.29935321, 0.31606746, 0.33703572, 0.35251474,
     &   0.35933667, 0.36060739, 0.36223540, 0.36603218, 0.37248543,
     &   0.38044843, 0.39232376, 0.41345140, 0.43048289, 0.44250435,
     &   0.40698865, 0.36182776, 0.33176336, 0.32241929, 0.33299115,
     &   0.34023628, 0.34804645, 0.35785949, 0.37104481, 0.38497403,
     &   0.39669532, 0.40740710, 0.41716969, 0.42739931, 0.43691245,
     &   0.44680214, 0.45804214, 0.47075862, 0.48385271, 0.49482587,
     &   0.50527024, 0.50847465, 0.50923514, 0.51010859, 0.50962573,
     &   0.50891387, 0.50777513, 0.50629973, 0.50459951, 0.50136667,
     &   0.49787045, 0.49393702, 0.48874405, 0.48336250, 0.47779551,
     &   0.47204712, 0.46612185, 0.46002507, 0.45376274, 0.44734144,
     &   0.44076836, 0.43405131, 0.42863321, 0.42334911, 0.41794226,
     &   0.41241917, 0.40678668, 0.40105185, 0.39522201, 0.38930461,
     &   0.38330725, 0.37723765, 0.37110358, 0.36491290, 0.35910130,
     &   0.35483652, 0.35051289, 0.34613588, 0.34171087, 0.33724323,
     &   0.33352023, 0.33489147, 0.33621588, 0.33749497, 0.33914840,
     &   0.34106094, 0.34293723, 0.34534496, 0.34901786, 0.35267323,
     &   0.35612217, 0.35677406, 0.35739139, 0.35797581, 0.35865891,
     &   0.35934314, 0.35999951, 0.36042687, 0.36064959, 0.36084765,
     &   0.36075661, 0.35998532, 0.35919341, 0.35835427, 0.35587651,
     &   0.35338816, 0.35089040, 0.34704405, 0.34276804, 0.33850265,
     &   0.33253762, 0.32497725, 0.31749469, 0.30976051, 0.30112940,
     &   0.29262686, 0.28425729, 0.27582169, 0.26753217, 0.25939491/
      data (wwi(i,7),i=1,mxwv)/
     &   0.99999255, 0.99999309, 0.99999279, 0.99999404, 0.99999464,
     &   0.99999493, 0.99999529, 0.99999571, 0.99999601, 0.99999624,
     &   0.99999660, 0.99999696, 0.99999732, 0.99999768, 0.99999803,
     &   0.99999815, 0.99999851, 0.99999857, 0.99999863, 0.99999881,
     &   0.99999893, 0.99999881, 0.99999911, 0.99999911, 0.99999923,
     &   0.99999934, 0.99999940, 0.99999940, 0.99999934, 0.99999934,
     &   0.99999934, 0.99999928, 0.99999917, 0.99999917, 0.99999881,
     &   0.99999899, 0.99999893, 0.99999893, 0.99999893, 0.99999809,
     &   0.99999851, 0.99999815, 0.99999791, 0.99999726, 0.99999696,
     &   0.99999636, 0.99999559, 0.99999505, 0.99999440, 0.99999344,
     &   0.99999213, 0.99998975, 0.99998909, 0.99998736, 0.99998432,
     &   0.99997973, 0.99997497, 0.99996978, 0.99996734, 0.99996513,
     &   0.99996489, 0.99995852, 0.99994528, 0.99992365, 0.99990976,
     &   0.99990642, 0.99989688, 0.99988323, 0.99983799, 0.99977738,
     &   0.99970424, 0.99960148, 0.99957687, 0.99962360, 0.99968183,
     &   0.99970078, 0.99969620, 0.99965364, 0.99961960, 0.99936873,
     &   0.99907732, 0.99860591, 0.99823815, 0.99805254, 0.99803793,
     &   0.99810493, 0.99812859, 0.99808371, 0.99796081, 0.99746168,
     &   0.99253482, 0.97918892, 0.95365870, 0.94124472, 0.94728035,
     &   0.95563769, 0.96564090, 0.97206491, 0.97465980, 0.97823447,
     &   0.98144251, 0.98408943, 0.98554575, 0.98646468, 0.98749626,
     &   0.98761821, 0.97887510, 0.94689888, 0.89689791, 0.87795103,
     &   0.88475353, 0.90256238, 0.93221515, 0.95957702, 0.97388685,
     &   0.98063713, 0.98331034, 0.97973824, 0.97038430, 0.96185732,
     &   0.95698440, 0.95235258, 0.94737762, 0.94578707, 0.94228941,
     &   0.89902043, 0.81365240, 0.69449663, 0.59222209, 0.49761581,
     &   0.49292842, 0.52717292, 0.53816378, 0.55289865, 0.55370462,
     &   0.54589027, 0.53834963, 0.53107214, 0.52626503, 0.52990526,
     &   0.55941999, 0.61633766, 0.68808460, 0.73487920, 0.75284362,
     &   0.76842284, 0.76854503, 0.75321376, 0.73104674, 0.71541166,
     &   0.70052803, 0.67986441, 0.65743524, 0.62418312, 0.59117740,
     &   0.57266456, 0.56503677, 0.60312712, 0.65007526, 0.67530072,
     &   0.70099223, 0.71648216, 0.71713477, 0.71113735, 0.70357615,
     &   0.68258715, 0.65694791, 0.62817121, 0.59623986, 0.56691623,
     &   0.53631592, 0.52160978, 0.51470423, 0.51382250, 0.51731914,
     &   0.52359128, 0.53273183, 0.53557217, 0.53428328, 0.53651512,
     &   0.54083371, 0.54500997, 0.54886252, 0.55618984, 0.55972993,
     &   0.56288838, 0.56812418, 0.57889402, 0.59516090, 0.61620593,
     &   0.63387561, 0.63516796, 0.63910967, 0.64265555, 0.64497447,
     &   0.64507926, 0.64855272, 0.65373802, 0.65876448, 0.66316700,
     &   0.65792406, 0.62723756, 0.57043421, 0.49463302, 0.43491712,
     &   0.43069193, 0.44469216, 0.45644018, 0.46465677, 0.46992609,
     &   0.47397962, 0.47688034, 0.47908401, 0.48074150, 0.48155433,
     &   0.48217729, 0.48255283, 0.48265144, 0.48285612, 0.48334455,
     &   0.48442167, 0.48663741, 0.49080411, 0.49677753, 0.50486189,
     &   0.51362807, 0.52269119, 0.53178519, 0.54018182, 0.54603112,
     &   0.55436724, 0.56756514, 0.58544016, 0.60578239, 0.62730610,
     &   0.64346081, 0.65706599, 0.66060275, 0.66171908, 0.67364097,
     &   0.69859481, 0.73251933, 0.76603180, 0.78947514, 0.81123781,
     &   0.83126754, 0.84689659, 0.85443175, 0.85773653, 0.85846585,
     &   0.85707623, 0.85513622, 0.85158849, 0.84532654, 0.83768040,
     &   0.82760674, 0.81431746, 0.79970288, 0.78367668, 0.76616323,
     &   0.74719232, 0.72741908, 0.70605963, 0.68313330, 0.65870547,
     &   0.63097107, 0.59955651, 0.56675798, 0.53306597, 0.49432448,
     &   0.44919133, 0.40868551, 0.37505969, 0.35174152, 0.33564109,
     &   0.32480785, 0.31867877, 0.31660613, 0.31675884, 0.31829870,
     &   0.32068288, 0.32381678, 0.32761428, 0.33332115, 0.34304449,
     &   0.35147858, 0.36233926, 0.37671486, 0.39353365, 0.40546077,
     &   0.41149923, 0.41327819, 0.41523635, 0.41882813, 0.42459813,
     &   0.43170053, 0.44200581, 0.46139804, 0.47989991, 0.49689630,
     &   0.47187611, 0.43189496, 0.40290672, 0.39200971, 0.40110400,
     &   0.40743762, 0.41437688, 0.42317769, 0.43525502, 0.44871286,
     &   0.46062449, 0.47195303, 0.48277017, 0.49408692, 0.50503623,
     &   0.51637489, 0.52929807, 0.54393607, 0.55893040, 0.57252097,
     &   0.58578253, 0.59237021, 0.59665626, 0.60090667, 0.60380071,
     &   0.60653293, 0.60883886, 0.61083460, 0.61266881, 0.61305010,
     &   0.61322320, 0.61297584, 0.61134511, 0.60955924, 0.60761452,
     &   0.60550702, 0.60323250, 0.60078704, 0.59816658, 0.59536707,
     &   0.59238464, 0.58921552, 0.58717108, 0.58519995, 0.58306015,
     &   0.58074868, 0.57826298, 0.57560080, 0.57276028, 0.56973982,
     &   0.56653851, 0.56315565, 0.55959117, 0.55584544, 0.55237466,
     &   0.55043787, 0.54833579, 0.54606974, 0.54364175, 0.54105383,
     &   0.53919441, 0.54299808, 0.54665148, 0.55015731, 0.55400401,
     &   0.55804962, 0.56195122, 0.56633908, 0.57200646, 0.57752925,
     &   0.58271080, 0.58482081, 0.58681488, 0.58869851, 0.59061348,
     &   0.59246039, 0.59421247, 0.59566271, 0.59684104, 0.59793931,
     &   0.59868443, 0.59867156, 0.59859180, 0.59841979, 0.59648716,
     &   0.59449530, 0.59244704, 0.58891404, 0.58485645, 0.58074236,
     &   0.57469159, 0.56674981, 0.55873573, 0.55026937, 0.54057080,
     &   0.53080964, 0.52099466, 0.51088047, 0.50072724, 0.49054807/
      data (wwi(i,8),i=1,mxwv)/
     &   0.99998969, 0.99999011, 0.99999064, 0.99999171, 0.99999255,
     &   0.99999321, 0.99999326, 0.99999398, 0.99999416, 0.99999475,
     &   0.99999523, 0.99999559, 0.99999630, 0.99999678, 0.99999726,
     &   0.99999756, 0.99999785, 0.99999797, 0.99999815, 0.99999833,
     &   0.99999851, 0.99999863, 0.99999863, 0.99999875, 0.99999899,
     &   0.99999911, 0.99999911, 0.99999911, 0.99999911, 0.99999911,
     &   0.99999899, 0.99999899, 0.99999893, 0.99999887, 0.99999875,
     &   0.99999863, 0.99999851, 0.99999851, 0.99999839, 0.99999809,
     &   0.99999791, 0.99999762, 0.99999702, 0.99999642, 0.99999547,
     &   0.99999434, 0.99999386, 0.99999291, 0.99999046, 0.99999112,
     &   0.99998903, 0.99998671, 0.99998438, 0.99998271, 0.99997705,
     &   0.99997044, 0.99996608, 0.99995780, 0.99995399, 0.99995506,
     &   0.99995351, 0.99994296, 0.99992490, 0.99989623, 0.99988037,
     &   0.99986863, 0.99985689, 0.99983799, 0.99977654, 0.99969500,
     &   0.99959052, 0.99944580, 0.99941140, 0.99947602, 0.99955958,
     &   0.99958712, 0.99958628, 0.99952239, 0.99947113, 0.99913001,
     &   0.99871629, 0.99807000, 0.99755913, 0.99731642, 0.99732220,
     &   0.99740362, 0.99743330, 0.99735701, 0.99718463, 0.99649984,
     &   0.98973489, 0.97147441, 0.93715960, 0.92077148, 0.92872870,
     &   0.93981159, 0.95318222, 0.96185178, 0.96535343, 0.97021794,
     &   0.97458816, 0.97817659, 0.98017842, 0.98142260, 0.98285961,
     &   0.98300296, 0.97111094, 0.92832935, 0.86407232, 0.84062225,
     &   0.84899884, 0.87119746, 0.90914834, 0.94515944, 0.96435314,
     &   0.97348422, 0.97710621, 0.97223204, 0.95956433, 0.94808847,
     &   0.94153589, 0.93530297, 0.92860019, 0.92631423, 0.92141187,
     &   0.86506021, 0.76146209, 0.63426930, 0.54093045, 0.49880630,
     &   0.50427961, 0.53445405, 0.54512352, 0.55971110, 0.56081378,
     &   0.55318195, 0.54579127, 0.53867877, 0.53338677, 0.53170085,
     &   0.54376727, 0.58018988, 0.63926029, 0.68360615, 0.70164955,
     &   0.71776003, 0.71779943, 0.70174044, 0.67921114, 0.66379941,
     &   0.64950556, 0.63034439, 0.61045742, 0.58285069, 0.55793810,
     &   0.54520673, 0.54032588, 0.56664348, 0.60397708, 0.62584710,
     &   0.64937234, 0.66401261, 0.66435808, 0.65818608, 0.65048957,
     &   0.63027310, 0.60662174, 0.58148426, 0.55559272, 0.53381413,
     &   0.51432484, 0.50710350, 0.50429356, 0.50439656, 0.50619602,
     &   0.50914401, 0.51322502, 0.51393235, 0.51264983, 0.51343077,
     &   0.51527661, 0.51698190, 0.51853931, 0.52208418, 0.52324367,
     &   0.52431196, 0.52681571, 0.53290975, 0.54281521, 0.55638176,
     &   0.56757390, 0.56618935, 0.56737971, 0.56857443, 0.56924862,
     &   0.56883585, 0.57166517, 0.57680058, 0.58326954, 0.59168983,
     &   0.59642875, 0.58529270, 0.55470306, 0.50597143, 0.46227494,
     &   0.45945811, 0.47085798, 0.47995150, 0.48642102, 0.49033925,
     &   0.49333930, 0.49531069, 0.49668834, 0.49759364, 0.49753919,
     &   0.49733704, 0.49686509, 0.49581558, 0.49480954, 0.49381322,
     &   0.49267083, 0.49157387, 0.49093556, 0.49108028, 0.49226460,
     &   0.49412802, 0.49659708, 0.49952355, 0.50251901, 0.50472307,
     &   0.50845939, 0.51507962, 0.52498704, 0.53747195, 0.55205041,
     &   0.56384617, 0.57457417, 0.57787728, 0.57927573, 0.58936965,
     &   0.61141610, 0.64414960, 0.67987311, 0.70743668, 0.73464185,
     &   0.76105893, 0.78313774, 0.79589939, 0.80359745, 0.80823928,
     &   0.81041080, 0.81208432, 0.81208408, 0.80922586, 0.80508190,
     &   0.79856831, 0.78897113, 0.77838892, 0.76673365, 0.75392240,
     &   0.73998976, 0.72571653, 0.71008360, 0.69305760, 0.67463416,
     &   0.65347159, 0.62910873, 0.60306448, 0.57563728, 0.54289728,
     &   0.50314498, 0.46602255, 0.43400630, 0.41130754, 0.39536557,
     &   0.38433525, 0.37778610, 0.37524134, 0.37491146, 0.37595445,
     &   0.37784126, 0.38045576, 0.38370645, 0.38860881, 0.39679497,
     &   0.40358913, 0.41232198, 0.42339751, 0.43466878, 0.44101697,
     &   0.44396722, 0.44505313, 0.44638655, 0.44876432, 0.45254219,
     &   0.45723107, 0.46396324, 0.47768304, 0.49364686, 0.51244569,
     &   0.50072682, 0.47095481, 0.44691303, 0.43603307, 0.44207382,
     &   0.44628677, 0.45101666, 0.45714170, 0.46586752, 0.47630486,
     &   0.48614299, 0.49601543, 0.50601554, 0.51664257, 0.52745193,
     &   0.53877378, 0.55195504, 0.56717187, 0.58288854, 0.59819233,
     &   0.61348552, 0.62289786, 0.63015574, 0.63715178, 0.64274883,
     &   0.64818233, 0.65312725, 0.65772676, 0.66216666, 0.66521841,
     &   0.66807687, 0.67050827, 0.67146361, 0.67229444, 0.67300117,
     &   0.67358410, 0.67404342, 0.67437911, 0.67459106, 0.67467892,
     &   0.67464215, 0.67447990, 0.67521620, 0.67603421, 0.67674661,
     &   0.67735207, 0.67784888, 0.67823535, 0.67850894, 0.67866784,
     &   0.67870933, 0.67863071, 0.67842895, 0.67810118, 0.67802787,
     &   0.67926645, 0.68037993, 0.68136537, 0.68221986, 0.68294042,
     &   0.68430334, 0.69060922, 0.69671112, 0.70260721, 0.70875770,
     &   0.71500599, 0.72101843, 0.72732460, 0.73456037, 0.74151003,
     &   0.74802428, 0.75202245, 0.75582021, 0.75942451, 0.76294303,
     &   0.76630437, 0.76949227, 0.77235961, 0.77493554, 0.77736491,
     &   0.77945501, 0.78092360, 0.78227395, 0.78349084, 0.78339756,
     &   0.78320640, 0.78292096, 0.78153336, 0.77972066, 0.77781349,
     &   0.77446717, 0.76968664, 0.76476043, 0.75940341, 0.75301617,
     &   0.74644816, 0.73970020, 0.73257220, 0.72525775, 0.71776128/
      data (wwi(i,9),i=1,mxwv)/
     &   0.99998528, 0.99998623, 0.99998713, 0.99998850, 0.99998945,
     &   0.99999034, 0.99999106, 0.99999154, 0.99999225, 0.99999273,
     &   0.99999332, 0.99999392, 0.99999475, 0.99999541, 0.99999607,
     &   0.99999666, 0.99999690, 0.99999720, 0.99999744, 0.99999768,
     &   0.99999791, 0.99999809, 0.99999821, 0.99999839, 0.99999857,
     &   0.99999881, 0.99999881, 0.99999881, 0.99999875, 0.99999875,
     &   0.99999869, 0.99999845, 0.99999845, 0.99999833, 0.99999827,
     &   0.99999803, 0.99999791, 0.99999785, 0.99999779, 0.99999756,
     &   0.99999708, 0.99999666, 0.99999601, 0.99999505, 0.99999404,
     &   0.99999285, 0.99999166, 0.99999022, 0.99998897, 0.99998772,
     &   0.99998343, 0.99998134, 0.99997872, 0.99997550, 0.99996990,
     &   0.99996233, 0.99995208, 0.99994361, 0.99993694, 0.99993598,
     &   0.99993479, 0.99991852, 0.99989659, 0.99985713, 0.99983293,
     &   0.99981987, 0.99980068, 0.99977589, 0.99969280, 0.99958175,
     &   0.99943036, 0.99923736, 0.99919081, 0.99927890, 0.99939525,
     &   0.99942529, 0.99942416, 0.99934632, 0.99927682, 0.99880105,
     &   0.99822730, 0.99734342, 0.99663258, 0.99629396, 0.99629128,
     &   0.99638420, 0.99644303, 0.99634665, 0.99609536, 0.99518204,
     &   0.98582214, 0.96089077, 0.91517615, 0.89391404, 0.90420872,
     &   0.91868699, 0.93636811, 0.94795632, 0.95265758, 0.95923680,
     &   0.96516001, 0.97007811, 0.97277099, 0.97448659, 0.97642940,
     &   0.97664094, 0.96047336, 0.90380538, 0.82338244, 0.79552865,
     &   0.80539334, 0.83203304, 0.87922555, 0.92584985, 0.95139802,
     &   0.96371382, 0.96862328, 0.96201414, 0.94497252, 0.92970431,
     &   0.92105329, 0.91286671, 0.90410668, 0.90108699, 0.89449662,
     &   0.82395452, 0.70604104, 0.58524776, 0.51620728, 0.49781516,
     &   0.51171857, 0.53945750, 0.54997706, 0.56453151, 0.56600684,
     &   0.55858439, 0.55133486, 0.54438412, 0.53912741, 0.53577876,
     &   0.53816414, 0.55648386, 0.59779572, 0.63486183, 0.65109849,
     &   0.66616404, 0.66606802, 0.65069473, 0.62999707, 0.61637908,
     &   0.60416484, 0.58852053, 0.57317257, 0.55359715, 0.53788561,
     &   0.53073823, 0.52827525, 0.54312158, 0.56818068, 0.58456367,
     &   0.60352045, 0.61590177, 0.61608911, 0.61062950, 0.60393089,
     &   0.58710158, 0.56853384, 0.55021173, 0.53306353, 0.52014452,
     &   0.51031178, 0.50773400, 0.50710100, 0.50758231, 0.50841516,
     &   0.50949550, 0.51075327, 0.51060838, 0.50977939, 0.50991970,
     &   0.51047319, 0.51093161, 0.51134914, 0.51267028, 0.51278007,
     &   0.51291645, 0.51385188, 0.51669586, 0.52173269, 0.52911967,
     &   0.53508842, 0.53271896, 0.53206283, 0.53125286, 0.52993733,
     &   0.52786297, 0.52779627, 0.52894735, 0.53097761, 0.53458345,
     &   0.53754485, 0.53501779, 0.52483916, 0.50270194, 0.47823194,
     &   0.47916222, 0.48956871, 0.49734437, 0.50305361, 0.50644571,
     &   0.50909686, 0.51080126, 0.51198131, 0.51274389, 0.51253891,
     &   0.51223952, 0.51170409, 0.51054972, 0.50945705, 0.50834548,
     &   0.50693423, 0.50525379, 0.50350702, 0.50198615, 0.50082761,
     &   0.49997735, 0.49951708, 0.49937809, 0.49937129, 0.49935013,
     &   0.50024652, 0.50251305, 0.50627947, 0.51167297, 0.51875657,
     &   0.52428406, 0.52920151, 0.52952576, 0.52924681, 0.53527355,
     &   0.55012923, 0.57428902, 0.60286838, 0.62596142, 0.65041780,
     &   0.67585903, 0.69823104, 0.71125799, 0.71932447, 0.72454697,
     &   0.72752237, 0.73055530, 0.73227745, 0.73134267, 0.72960651,
     &   0.72586036, 0.71940672, 0.71261394, 0.70534724, 0.69747263,
     &   0.68899733, 0.68083209, 0.67165434, 0.66136706, 0.64989781,
     &   0.63649172, 0.62066591, 0.60308564, 0.58388418, 0.55969369,
     &   0.52876365, 0.49857253, 0.47149432, 0.45195988, 0.43809170,
     &   0.42833143, 0.42242455, 0.42004350, 0.41964051, 0.42047086,
     &   0.42208445, 0.42437285, 0.42724746, 0.43150702, 0.43831259,
     &   0.44353333, 0.45043743, 0.45878077, 0.46534488, 0.46640950,
     &   0.46530178, 0.46483320, 0.46491799, 0.46567488, 0.46705875,
     &   0.46881548, 0.47142354, 0.47703764, 0.48545533, 0.49914953,
     &   0.49848372, 0.48186517, 0.46627069, 0.45836964, 0.46141663,
     &   0.46350458, 0.46585277, 0.46887192, 0.47319749, 0.47866747,
     &   0.48422068, 0.49024054, 0.49688739, 0.50429165, 0.51243365,
     &   0.52132571, 0.53225315, 0.54556453, 0.55988950, 0.57524037,
     &   0.59137559, 0.60309309, 0.61308551, 0.62271935, 0.63101763,
     &   0.63922161, 0.64691126, 0.65423989, 0.66142297, 0.66724592,
     &   0.67286640, 0.67800689, 0.68147016, 0.68479311, 0.68797535,
     &   0.69101685, 0.69391805, 0.69667941, 0.69930184, 0.70178622,
     &   0.70413381, 0.70634574, 0.70918399, 0.71204793, 0.71480072,
     &   0.71744329, 0.71997666, 0.72240198, 0.72472036, 0.72693300,
     &   0.72904098, 0.73104548, 0.73294759, 0.73474830, 0.73676270,
     &   0.73985249, 0.74284184, 0.74573141, 0.74852180, 0.75121373,
     &   0.75447100, 0.76191956, 0.76914936, 0.77615988, 0.78339410,
     &   0.79069138, 0.79772902, 0.80494958, 0.81286085, 0.82043546,
     &   0.82756376, 0.83272600, 0.83766627, 0.84238970, 0.84697002,
     &   0.85135752, 0.85554171, 0.85942346, 0.86302757, 0.86645490,
     &   0.86957848, 0.87221968, 0.87471884, 0.87706769, 0.87852120,
     &   0.87986785, 0.88111061, 0.88163042, 0.88185757, 0.88199747,
     &   0.88124061, 0.87960213, 0.87786275, 0.87584335, 0.87318820,
     &   0.87040406, 0.86748797, 0.86431175, 0.86099195, 0.85752696/
      data (wwi(i,10),i=1,mxwv)/
     &   0.99997979, 0.99998081, 0.99998200, 0.99998385, 0.99998540,
     &   0.99998665, 0.99998754, 0.99998826, 0.99998897, 0.99998981,
     &   0.99999064, 0.99999148, 0.99999267, 0.99999356, 0.99999458,
     &   0.99999535, 0.99999577, 0.99999613, 0.99999642, 0.99999672,
     &   0.99999696, 0.99999732, 0.99999744, 0.99999774, 0.99999797,
     &   0.99999833, 0.99999833, 0.99999833, 0.99999833, 0.99999821,
     &   0.99999815, 0.99999809, 0.99999785, 0.99999762, 0.99999762,
     &   0.99999732, 0.99999720, 0.99999708, 0.99999696, 0.99999666,
     &   0.99999601, 0.99999535, 0.99999440, 0.99999285, 0.99999166,
     &   0.99998999, 0.99998838, 0.99998665, 0.99998426, 0.99998248,
     &   0.99997866, 0.99997449, 0.99997175, 0.99996591, 0.99995840,
     &   0.99994767, 0.99993312, 0.99992096, 0.99991316, 0.99991220,
     &   0.99991053, 0.99988854, 0.99985403, 0.99980116, 0.99976766,
     &   0.99974769, 0.99972063, 0.99968898, 0.99957484, 0.99941993,
     &   0.99921381, 0.99894023, 0.99887574, 0.99899697, 0.99915886,
     &   0.99920529, 0.99920511, 0.99909341, 0.99899673, 0.99833751,
     &   0.99754190, 0.99630767, 0.99532640, 0.99485826, 0.99485046,
     &   0.99498266, 0.99506736, 0.99493062, 0.99459398, 0.99332082,
     &   0.98039597, 0.94648480, 0.88647074, 0.85956895, 0.87252933,
     &   0.89100826, 0.91397405, 0.92925662, 0.93551785, 0.94431257,
     &   0.95228577, 0.95891243, 0.96259665, 0.96492767, 0.96757382,
     &   0.96786588, 0.94601011, 0.87214041, 0.77523756, 0.74400938,
     &   0.75493312, 0.78520906, 0.84153700, 0.90039146, 0.93392253,
     &   0.95039684, 0.95700729, 0.94811273, 0.92540389, 0.90538114,
     &   0.89416790, 0.88364208, 0.87247258, 0.86861551, 0.86045718,
     &   0.77533758, 0.65086740, 0.55011916, 0.50729215, 0.49844530,
     &   0.51668364, 0.54283750, 0.55328339, 0.56785572, 0.56971020,
     &   0.56249642, 0.55537301, 0.54855192, 0.54340982, 0.53974509,
     &   0.53817362, 0.54451299, 0.56792945, 0.59421641, 0.60683686,
     &   0.61915570, 0.61886877, 0.60583919, 0.58917457, 0.57871747,
     &   0.56970781, 0.55881631, 0.54881632, 0.53731048, 0.52926302,
     &   0.52609700, 0.52522051, 0.53179425, 0.54541320, 0.55550212,
     &   0.56831735, 0.57719713, 0.57716703, 0.57293272, 0.56785601,
     &   0.55586672, 0.54355514, 0.53244436, 0.52314657, 0.51693672,
     &   0.51302701, 0.51269066, 0.51298720, 0.51360828, 0.51405078,
     &   0.51433623, 0.51432621, 0.51388806, 0.51341653, 0.51332086,
     &   0.51329970, 0.51321030, 0.51312876, 0.51332110, 0.51303160,
     &   0.51281762, 0.51296747, 0.51388633, 0.51577073, 0.51885265,
     &   0.52132320, 0.51941282, 0.51843601, 0.51730597, 0.51581818,
     &   0.51386160, 0.51283371, 0.51213920, 0.51160109, 0.51136726,
     &   0.51026142, 0.50726056, 0.50397617, 0.49726787, 0.48822245,
     &   0.49311444, 0.50330585, 0.51049882, 0.51590151, 0.51912713,
     &   0.52169764, 0.52336931, 0.52454996, 0.52533811, 0.52517587,
     &   0.52494043, 0.52448767, 0.52343047, 0.52244800, 0.52145058,
     &   0.52013749, 0.51850438, 0.51667726, 0.51490253, 0.51324290,
     &   0.51173204, 0.51048130, 0.50945008, 0.50854951, 0.50789654,
     &   0.50773656, 0.50807649, 0.50889862, 0.51054233, 0.51325184,
     &   0.51527297, 0.51699036, 0.51625848, 0.51549429, 0.51856667,
     &   0.52715546, 0.54253596, 0.56236959, 0.57911640, 0.59788871,
     &   0.61844605, 0.63688904, 0.64656454, 0.65143216, 0.65346485,
     &   0.65326172, 0.65315753, 0.65182024, 0.64801031, 0.64380878,
     &   0.63818669, 0.63069510, 0.62383783, 0.61756438, 0.61178017,
     &   0.60647941, 0.60248095, 0.59844089, 0.59414518, 0.58938283,
     &   0.58388346, 0.57717842, 0.56905729, 0.55940944, 0.54575658,
     &   0.52653712, 0.50638998, 0.48735294, 0.47339058, 0.46345055,
     &   0.45649853, 0.45247993, 0.45113575, 0.45132291, 0.45248431,
     &   0.45429754, 0.45669547, 0.45959944, 0.46368015, 0.46975887,
     &   0.47399038, 0.47986123, 0.48664033, 0.49041250, 0.48825750,
     &   0.48424318, 0.48243633, 0.48148799, 0.48112652, 0.48096168,
     &   0.48081622, 0.48088735, 0.48048279, 0.48063210, 0.48387235,
     &   0.48523891, 0.47947499, 0.47311258, 0.47063690, 0.47232452,
     &   0.47345170, 0.47453797, 0.47566760, 0.47683054, 0.47790617,
     &   0.47903854, 0.48048192, 0.48239219, 0.48487449, 0.48803639,
     &   0.49190289, 0.49723473, 0.50451297, 0.51315510, 0.52370137,
     &   0.53592092, 0.54622197, 0.55591154, 0.56566566, 0.57468730,
     &   0.58399343, 0.59310633, 0.60213166, 0.61126912, 0.61934441,
     &   0.62738383, 0.63504791, 0.64094871, 0.64676934, 0.65249854,
     &   0.65812606, 0.66364259, 0.66903991, 0.67431092, 0.67944926,
     &   0.68444973, 0.68930781, 0.69451922, 0.69969803, 0.70475137,
     &   0.70967674, 0.71447194, 0.71913552, 0.72366643, 0.72806406,
     &   0.73232841, 0.73645955, 0.74045801, 0.74432492, 0.74832726,
     &   0.75319374, 0.75792587, 0.76252466, 0.76699108, 0.77132654,
     &   0.77613777, 0.78471017, 0.79300189, 0.80101246, 0.80920565,
     &   0.81740695, 0.82527906, 0.83323425, 0.84173143, 0.84981781,
     &   0.85740888, 0.86327004, 0.86886144, 0.87419283, 0.87932557,
     &   0.88422656, 0.88889331, 0.89325452, 0.89733678, 0.90122277,
     &   0.90481985, 0.90800107, 0.91103196, 0.91390908, 0.91609412,
     &   0.91817641, 0.92015964, 0.92161363, 0.92285246, 0.92402178,
     &   0.92457885, 0.92455393, 0.92447621, 0.92423165, 0.92358857,
     &   0.92288566, 0.92212081, 0.92120576, 0.92022258, 0.91916978/
      data (wwi(i,11),i=1,mxwv)/
     &   0.99997157, 0.99997312, 0.99997473, 0.99997711, 0.99997920,
     &   0.99998111, 0.99998242, 0.99998355, 0.99998462, 0.99998569,
     &   0.99998695, 0.99998838, 0.99998957, 0.99999124, 0.99999243,
     &   0.99999338, 0.99999410, 0.99999452, 0.99999493, 0.99999547,
     &   0.99999589, 0.99999624, 0.99999654, 0.99999678, 0.99999714,
     &   0.99999750, 0.99999768, 0.99999768, 0.99999762, 0.99999756,
     &   0.99999744, 0.99999726, 0.99999708, 0.99999696, 0.99999666,
     &   0.99999630, 0.99999607, 0.99999583, 0.99999571, 0.99999523,
     &   0.99999440, 0.99999338, 0.99999225, 0.99999034, 0.99998814,
     &   0.99998623, 0.99998361, 0.99998111, 0.99997878, 0.99997592,
     &   0.99997032, 0.99996424, 0.99995893, 0.99995357, 0.99994200,
     &   0.99992681, 0.99990833, 0.99989063, 0.99987948, 0.99987596,
     &   0.99987954, 0.99984401, 0.99979836, 0.99972236, 0.99967605,
     &   0.99964833, 0.99961233, 0.99956578, 0.99940604, 0.99918777,
     &   0.99890107, 0.99852163, 0.99840873, 0.99859887, 0.99882597,
     &   0.99889129, 0.99888849, 0.99873692, 0.99860823, 0.99769086,
     &   0.99657488, 0.99485594, 0.99349159, 0.99284047, 0.99283320,
     &   0.99302882, 0.99314034, 0.99295044, 0.99246335, 0.99070680,
     &   0.97287041, 0.92710042, 0.85004085, 0.81722605, 0.83291352,
     &   0.85571396, 0.88474482, 0.90449214, 0.91268474, 0.92427528,
     &   0.93487251, 0.94375205, 0.94871175, 0.95186335, 0.95544517,
     &   0.95584655, 0.92656493, 0.83254957, 0.72177881, 0.68941015,
     &   0.70053363, 0.73247302, 0.79597992, 0.86761600, 0.91069239,
     &   0.93243086, 0.94127816, 0.92939585, 0.89959717, 0.87391114,
     &   0.85967451, 0.84664196, 0.83288068, 0.82813537, 0.81819493,
     &   0.72114515, 0.60157156, 0.52897120, 0.50617892, 0.50053662,
     &   0.52002406, 0.54505867, 0.55546647, 0.57007706, 0.57228178,
     &   0.56526077, 0.55824584, 0.55152339, 0.54648167, 0.54283589,
     &   0.54011321, 0.54057783, 0.55053782, 0.56556612, 0.57361239,
     &   0.58201760, 0.58155215, 0.57202649, 0.56062031, 0.55383313,
     &   0.54823780, 0.54193026, 0.53654355, 0.53103775, 0.52772653,
     &   0.52666354, 0.52656579, 0.52875811, 0.53447056, 0.53929436,
     &   0.54656243, 0.55135339, 0.55110633, 0.54826182, 0.54495907,
     &   0.53780890, 0.53103322, 0.52546042, 0.52128196, 0.51873124,
     &   0.51745594, 0.51794130, 0.51855987, 0.51925069, 0.51961184,
     &   0.51968455, 0.51928270, 0.51879501, 0.51851422, 0.51837891,
     &   0.51818991, 0.51793712, 0.51770318, 0.51746428, 0.51706398,
     &   0.51675540, 0.51659769, 0.51663071, 0.51690781, 0.51755542,
     &   0.51793975, 0.51657826, 0.51565576, 0.51464546, 0.51344901,
     &   0.51204389, 0.51095271, 0.50981796, 0.50856924, 0.50710398,
     &   0.50477469, 0.50145370, 0.49900305, 0.49702924, 0.49609333,
     &   0.50349849, 0.51354831, 0.52042884, 0.52567488, 0.52885407,
     &   0.53142518, 0.53313208, 0.53436613, 0.53521943, 0.53514135,
     &   0.53499466, 0.53463840, 0.53369242, 0.53282601, 0.53194726,
     &   0.53075820, 0.52924830, 0.52752441, 0.52581406, 0.52414441,
     &   0.52257067, 0.52120024, 0.51999944, 0.51892066, 0.51816899,
     &   0.51769751, 0.51730710, 0.51687032, 0.51671720, 0.51708072,
     &   0.51723403, 0.51730090, 0.51644379, 0.51579666, 0.51699334,
     &   0.52091825, 0.52891290, 0.54060149, 0.55139095, 0.56448668,
     &   0.57985961, 0.59433824, 0.60181195, 0.60516316, 0.60592896,
     &   0.60456902, 0.60306656, 0.60021472, 0.59493142, 0.58906841,
     &   0.58182269, 0.57295883, 0.56470776, 0.55714357, 0.55032670,
     &   0.54437524, 0.53989637, 0.53613216, 0.53300661, 0.53039527,
     &   0.52821720, 0.52627802, 0.52414280, 0.52145845, 0.51674145,
     &   0.50878149, 0.49926108, 0.48947683, 0.48213962, 0.47699189,
     &   0.47366509, 0.47227123, 0.47253346, 0.47376996, 0.47565252,
     &   0.47800115, 0.48080930, 0.48401690, 0.48825070, 0.49413782,
     &   0.49794725, 0.50340873, 0.50949800, 0.51200503, 0.50854445,
     &   0.50337601, 0.50103176, 0.49969807, 0.49898252, 0.49835375,
     &   0.49765018, 0.49705356, 0.49464273, 0.49099636, 0.48727807,
     &   0.48436791, 0.48265463, 0.48202023, 0.48423749, 0.48611850,
     &   0.48738393, 0.48841423, 0.48921287, 0.48945656, 0.48880860,
     &   0.48795328, 0.48708713, 0.48626670, 0.48571694, 0.48536608,
     &   0.48538685, 0.48588148, 0.48715982, 0.48935270, 0.49259436,
     &   0.49713695, 0.50138187, 0.50583017, 0.51080859, 0.51582873,
     &   0.52144140, 0.52736473, 0.53365147, 0.54044372, 0.54689884,
     &   0.55370945, 0.56059039, 0.56630397, 0.57220143, 0.57826692,
     &   0.58447951, 0.59082329, 0.59727764, 0.60382336, 0.61043817,
     &   0.61710382, 0.62379944, 0.63074780, 0.63775414, 0.64475906,
     &   0.65174603, 0.65869927, 0.66560394, 0.67244607, 0.67921263,
     &   0.68589181, 0.69247276, 0.69894564, 0.70530176, 0.71176392,
     &   0.71896023, 0.72602147, 0.73294014, 0.73970979, 0.74632525,
     &   0.75338960, 0.76420063, 0.77469277, 0.78485197, 0.79522532,
     &   0.80559063, 0.81553149, 0.82548410, 0.83592981, 0.84583443,
     &   0.85511583, 0.86260581, 0.86970729, 0.87643373, 0.88284355,
     &   0.88891393, 0.89464915, 0.89999115, 0.90497255, 0.90967542,
     &   0.91402203, 0.91790754, 0.92158151, 0.92504662, 0.92783815,
     &   0.93048704, 0.93300092, 0.93502152, 0.93682683, 0.93854415,
     &   0.93973470, 0.94044054, 0.94109994, 0.94162363, 0.94183302,
     &   0.94200522, 0.94214064, 0.94216591, 0.94215584, 0.94211054/
      data (wwi(i,12),i=1,mxwv)/
     &   0.99996018, 0.99996233, 0.99996448, 0.99996817, 0.99997133,
     &   0.99997342, 0.99997538, 0.99997675, 0.99997807, 0.99997973,
     &   0.99998176, 0.99998361, 0.99998558, 0.99998766, 0.99998951,
     &   0.99999064, 0.99999160, 0.99999231, 0.99999297, 0.99999362,
     &   0.99999416, 0.99999470, 0.99999511, 0.99999553, 0.99999607,
     &   0.99999660, 0.99999672, 0.99999672, 0.99999672, 0.99999654,
     &   0.99999636, 0.99999619, 0.99999583, 0.99999577, 0.99999535,
     &   0.99999475, 0.99999452, 0.99999428, 0.99999392, 0.99999332,
     &   0.99999213, 0.99999058, 0.99998903, 0.99998671, 0.99998349,
     &   0.99998057, 0.99997711, 0.99997354, 0.99997044, 0.99996555,
     &   0.99995840, 0.99994969, 0.99994284, 0.99993461, 0.99991959,
     &   0.99989706, 0.99987197, 0.99984741, 0.99983126, 0.99982804,
     &   0.99982464, 0.99978292, 0.99971628, 0.99961179, 0.99954647,
     &   0.99950945, 0.99945909, 0.99939084, 0.99916971, 0.99886501,
     &   0.99846148, 0.99793321, 0.99780399, 0.99804276, 0.99835807,
     &   0.99845052, 0.99844873, 0.99823725, 0.99805319, 0.99677461,
     &   0.99521673, 0.99281961, 0.99092633, 0.99001288, 0.99000132,
     &   0.99027729, 0.99042988, 0.99017227, 0.98951429, 0.98705631,
     &   0.96249783, 0.90147668, 0.80562341, 0.76759136, 0.78556985,
     &   0.81239784, 0.84772259, 0.87248057, 0.88292021, 0.89786154,
     &   0.91168851, 0.92339516, 0.92998582, 0.93419278, 0.93898928,
     &   0.93953145, 0.90087593, 0.78521699, 0.66726077, 0.63703525,
     &   0.64716989, 0.67767406, 0.74395090, 0.82682663, 0.88046443,
     &   0.90856194, 0.92018789, 0.90460217, 0.86638349, 0.83446580,
     &   0.81729609, 0.80162638, 0.78546542, 0.77992159, 0.76844603,
     &   0.66560441, 0.56355190, 0.51906425, 0.50767183, 0.50301433,
     &   0.52225459, 0.54645956, 0.55684704, 0.57150155, 0.57401156,
     &   0.56716001, 0.56023592, 0.55358654, 0.54862088, 0.54503930,
     &   0.54211062, 0.54048026, 0.54292977, 0.54926586, 0.55311370,
     &   0.55755568, 0.55692917, 0.55105895, 0.54456019, 0.54086870,
     &   0.53793663, 0.53485292, 0.53243893, 0.53022784, 0.52907652,
     &   0.52883989, 0.52903610, 0.52954900, 0.53125972, 0.53283924,
     &   0.53561121, 0.53779018, 0.53739613, 0.53571427, 0.53382987,
     &   0.53025913, 0.52711678, 0.52469963, 0.52299541, 0.52192694,
     &   0.52153081, 0.52225387, 0.52297437, 0.52370793, 0.52408701,
     &   0.52415287, 0.52370542, 0.52325958, 0.52308035, 0.52298439,
     &   0.52279472, 0.52254450, 0.52231479, 0.52199244, 0.52160782,
     &   0.52131760, 0.52110368, 0.52086896, 0.52057189, 0.52022207,
     &   0.51967138, 0.51863223, 0.51779628, 0.51691508, 0.51594788,
     &   0.51491398, 0.51392853, 0.51276636, 0.51143175, 0.50975293,
     &   0.50740790, 0.50447679, 0.50211191, 0.50089693, 0.50294864,
     &   0.51135242, 0.52113497, 0.52778858, 0.53292787, 0.53610033,
     &   0.53869790, 0.54046082, 0.54176360, 0.54269302, 0.54270738,
     &   0.54265195, 0.54238957, 0.54154837, 0.54078674, 0.54001302,
     &   0.53893507, 0.53754491, 0.53594339, 0.53435063, 0.53278720,
     &   0.53130585, 0.53000885, 0.52886289, 0.52783120, 0.52714020,
     &   0.52665639, 0.52610922, 0.52530855, 0.52452654, 0.52393568,
     &   0.52326858, 0.52257103, 0.52173513, 0.52122599, 0.52145249,
     &   0.52263433, 0.52559382, 0.53078043, 0.53618699, 0.54357463,
     &   0.55316329, 0.56289589, 0.56802881, 0.57024318, 0.57054341,
     &   0.56923133, 0.56777442, 0.56525278, 0.56087166, 0.55602914,
     &   0.55015647, 0.54315501, 0.53660017, 0.53051263, 0.52491403,
     &   0.51986563, 0.51565224, 0.51191837, 0.50868291, 0.50594914,
     &   0.50363564, 0.50180095, 0.50044602, 0.49939367, 0.49806443,
     &   0.49592346, 0.49312311, 0.49005634, 0.48781326, 0.48648864,
     &   0.48618436, 0.48705888, 0.48876095, 0.49096569, 0.49353313,
     &   0.49639556, 0.49960124, 0.50311077, 0.50754327, 0.51343137,
     &   0.51712030, 0.52244914, 0.52825660, 0.53025872, 0.52642530,
     &   0.52106160, 0.51869333, 0.51737064, 0.51667970, 0.51606399,
     &   0.51537216, 0.51480281, 0.51222259, 0.50795257, 0.50246644,
     &   0.49699733, 0.49548760, 0.49667239, 0.50100672, 0.50334775,
     &   0.50493896, 0.50621611, 0.50719780, 0.50750536, 0.50672090,
     &   0.50563037, 0.50441390, 0.50307375, 0.50188237, 0.50065070,
     &   0.49960119, 0.49854830, 0.49762186, 0.49714014, 0.49682167,
     &   0.49706176, 0.49725640, 0.49762997, 0.49841940, 0.49937105,
     &   0.50066727, 0.50224394, 0.50412381, 0.50637466, 0.50858408,
     &   0.51111382, 0.51385075, 0.51616699, 0.51871538, 0.52150369,
     &   0.52453887, 0.52782685, 0.53137934, 0.53517884, 0.53924811,
     &   0.54358041, 0.54817438, 0.55321187, 0.55853271, 0.56409591,
     &   0.56989306, 0.57591414, 0.58214766, 0.58858061, 0.59519851,
     &   0.60198587, 0.60892588, 0.61600077, 0.62319225, 0.63067621,
     &   0.63898182, 0.64736527, 0.65580344, 0.66427356, 0.67275327,
     &   0.68184304, 0.69498175, 0.70803058, 0.72093362, 0.73434746,
     &   0.74798304, 0.76127589, 0.77471340, 0.78886199, 0.80243373,
     &   0.81529701, 0.82612032, 0.83643210, 0.84623104, 0.85556477,
     &   0.86440313, 0.87274617, 0.88052720, 0.88778245, 0.89460337,
     &   0.90090507, 0.90657741, 0.91190583, 0.91689801, 0.92105675,
     &   0.92497307, 0.92866015, 0.93174976, 0.93454665, 0.93718916,
     &   0.93923986, 0.94076139, 0.94220144, 0.94347417, 0.94441032,
     &   0.94529015, 0.94611591, 0.94681209, 0.94746071, 0.94806463/
      data (wwi(i,13),i=1,mxwv)/
     &   0.99994397, 0.99994683, 0.99994993, 0.99995512, 0.99995947,
     &   0.99996275, 0.99996543, 0.99996746, 0.99996954, 0.99997163,
     &   0.99997431, 0.99997693, 0.99997962, 0.99998254, 0.99998498,
     &   0.99998695, 0.99998820, 0.99998915, 0.99999011, 0.99999100,
     &   0.99999183, 0.99999225, 0.99999315, 0.99999368, 0.99999446,
     &   0.99999529, 0.99999535, 0.99999547, 0.99999535, 0.99999517,
     &   0.99999487, 0.99999464, 0.99999428, 0.99999398, 0.99999338,
     &   0.99999267, 0.99999225, 0.99999195, 0.99999160, 0.99999064,
     &   0.99998891, 0.99998719, 0.99998462, 0.99998093, 0.99997705,
     &   0.99997270, 0.99996823, 0.99996316, 0.99995857, 0.99995267,
     &   0.99994135, 0.99992931, 0.99991935, 0.99990851, 0.99988705,
     &   0.99985594, 0.99981946, 0.99978524, 0.99976349, 0.99975967,
     &   0.99975348, 0.99969482, 0.99960482, 0.99945796, 0.99936438,
     &   0.99931228, 0.99924177, 0.99914789, 0.99883658, 0.99840736,
     &   0.99784654, 0.99710482, 0.99692428, 0.99726140, 0.99769688,
     &   0.99782658, 0.99782604, 0.99753135, 0.99727440, 0.99548447,
     &   0.99330676, 0.98996419, 0.98731917, 0.98605686, 0.98604143,
     &   0.98640394, 0.98664272, 0.98628390, 0.98536098, 0.98196095,
     &   0.94833440, 0.86839569, 0.75435889, 0.71325016, 0.73237950,
     &   0.76195765, 0.80271542, 0.83246469, 0.84529561, 0.86394382,
     &   0.88149941, 0.89658171, 0.90529615, 0.91067415, 0.91701174,
     &   0.91770720, 0.86778182, 0.73201197, 0.61752981, 0.59286350,
     &   0.60085982, 0.62645626, 0.68891060, 0.77840281, 0.84233302,
     &   0.87757158, 0.89249009, 0.87253189, 0.82509375, 0.78717029,
     &   0.76741540, 0.74978489, 0.73200721, 0.72595251, 0.71362531,
     &   0.61457199, 0.53908926, 0.51592219, 0.50949192, 0.50523412,
     &   0.52371132, 0.54728872, 0.55766535, 0.57236242, 0.57512861,
     &   0.56842053, 0.56157064, 0.55497426, 0.55006373, 0.54653651,
     &   0.54361850, 0.54146630, 0.54084122, 0.54224122, 0.54325140,
     &   0.54474169, 0.54400682, 0.54091889, 0.53778684, 0.53601962,
     &   0.53461945, 0.53327858, 0.53220618, 0.53132915, 0.53091210,
     &   0.53092873, 0.53122127, 0.53128290, 0.53152823, 0.53166479,
     &   0.53228545, 0.53276998, 0.53229702, 0.53134251, 0.53030682,
     &   0.52869260, 0.52731389, 0.52623224, 0.52542645, 0.52482742,
     &   0.52467722, 0.52546579, 0.52622551, 0.52698714, 0.52739853,
     &   0.52750003, 0.52709281, 0.52670491, 0.52659017, 0.52654386,
     &   0.52639896, 0.52619535, 0.52601236, 0.52571869, 0.52538681,
     &   0.52514803, 0.52496588, 0.52470845, 0.52429920, 0.52369070,
     &   0.52287382, 0.52200103, 0.52124661, 0.52046305, 0.51963097,
     &   0.51878256, 0.51789367, 0.51679677, 0.51552081, 0.51389050,
     &   0.51173609, 0.50919211, 0.50694948, 0.50577539, 0.50866687,
     &   0.51720780, 0.52664953, 0.53311723, 0.53817564, 0.54135650,
     &   0.54398936, 0.54581290, 0.54718673, 0.54819262, 0.54829651,
     &   0.54832739, 0.54815155, 0.54740387, 0.54673368, 0.54605061,
     &   0.54506814, 0.54378021, 0.54228652, 0.54080451, 0.53935182,
     &   0.53798103, 0.53678972, 0.53574413, 0.53481030, 0.53421420,
     &   0.53380454, 0.53329796, 0.53248525, 0.53160661, 0.53079224,
     &   0.52991778, 0.52901185, 0.52824795, 0.52785152, 0.52775455,
     &   0.52780771, 0.52828956, 0.52964276, 0.53136855, 0.53433359,
     &   0.53885669, 0.54394060, 0.54659522, 0.54757720, 0.54740548,
     &   0.54624057, 0.54498661, 0.54306680, 0.54001260, 0.53671253,
     &   0.53285903, 0.52844328, 0.52437431, 0.52063370, 0.51720387,
     &   0.51408297, 0.51133376, 0.50880611, 0.50650239, 0.50442791,
     &   0.50245237, 0.50062311, 0.49910966, 0.49789754, 0.49686700,
     &   0.49600849, 0.49543124, 0.49521044, 0.49540424, 0.49605757,
     &   0.49741662, 0.49958539, 0.50213015, 0.50490737, 0.50788093,
     &   0.51104528, 0.51448137, 0.51816458, 0.52270836, 0.52860308,
     &   0.53228211, 0.53757149, 0.54326755, 0.54509109, 0.54121184,
     &   0.53595024, 0.53371310, 0.53251505, 0.53193700, 0.53142744,
     &   0.53084552, 0.53038800, 0.52794439, 0.52381068, 0.51830536,
     &   0.51250786, 0.51093882, 0.51250571, 0.51741683, 0.51996905,
     &   0.52172792, 0.52315366, 0.52427340, 0.52471685, 0.52406836,
     &   0.52311176, 0.52201569, 0.52076840, 0.51963669, 0.51840764,
     &   0.51729429, 0.51603967, 0.51469135, 0.51355845, 0.51226097,
     &   0.51115566, 0.51010150, 0.50917864, 0.50852597, 0.50802070,
     &   0.50766909, 0.50748533, 0.50746679, 0.50762594, 0.50778931,
     &   0.50810194, 0.50853235, 0.50885302, 0.50927687, 0.50980818,
     &   0.51045394, 0.51122034, 0.51211447, 0.51314360, 0.51431477,
     &   0.51563585, 0.51711428, 0.51901877, 0.52115303, 0.52347654,
     &   0.52599531, 0.52871525, 0.53164202, 0.53478098, 0.53813708,
     &   0.54171467, 0.54551756, 0.54954910, 0.55381155, 0.55847943,
     &   0.56404990, 0.56988710, 0.57598573, 0.58233917, 0.58893931,
     &   0.59633428, 0.60773814, 0.61954051, 0.63168603, 0.64481789,
     &   0.65870613, 0.67278284, 0.68755698, 0.70369303, 0.71975100,
     &   0.73551220, 0.74931049, 0.76285470, 0.77608883, 0.78901452,
     &   0.80154508, 0.81363171, 0.82514286, 0.83608055, 0.84652323,
     &   0.85632426, 0.86530668, 0.87382573, 0.88187271, 0.88876885,
     &   0.89529455, 0.90145862, 0.90677464, 0.91164428, 0.91624749,
     &   0.92002666, 0.92307717, 0.92597508, 0.92861235, 0.93078244,
     &   0.93284434, 0.93480223, 0.93655455, 0.93821645, 0.93979287/
      data (ggi(i,1),i=1,mxwv)/
     &   0.83874619, 0.83875793, 0.83866620, 0.83844739, 0.83812904,
     &   0.83784223, 0.83761531, 0.83722752, 0.83697462, 0.83668435,
     &   0.83627301, 0.83598143, 0.83544493, 0.83492452, 0.83447802,
     &   0.83397424, 0.83342069, 0.83286768, 0.83227932, 0.83163297,
     &   0.83097708, 0.83025908, 0.82951319, 0.82873517, 0.82793087,
     &   0.82708627, 0.82618356, 0.82529402, 0.82433593, 0.82331395,
     &   0.82229072, 0.82122678, 0.82013166, 0.81899190, 0.81781930,
     &   0.81661677, 0.81538779, 0.81412470, 0.81283450, 0.81151956,
     &   0.81019109, 0.80883986, 0.80747682, 0.80610102, 0.80472231,
     &   0.80334157, 0.80196357, 0.80059606, 0.79924405, 0.79791546,
     &   0.79661429, 0.79534733, 0.79412585, 0.79295444, 0.79184258,
     &   0.79079598, 0.78981405, 0.78894639, 0.78815562, 0.78746992,
     &   0.78689855, 0.78644216, 0.78612000, 0.78593296, 0.78588909,
     &   0.78599322, 0.78625208, 0.78667647, 0.78725672, 0.78800917,
     &   0.78892136, 0.78999937, 0.79123855, 0.79263479, 0.79418236,
     &   0.79587364, 0.79770547, 0.79966879, 0.80175310, 0.80394322,
     &   0.80623436, 0.80860651, 0.81104773, 0.81353170, 0.81606686,
     &   0.81863874, 0.82123142, 0.82382888, 0.82645243, 0.82908064,
     &   0.83182973, 0.83467823, 0.83752465, 0.83997345, 0.84202760,
     &   0.84400803, 0.84602392, 0.84807688, 0.85009962, 0.85205173,
     &   0.85396874, 0.85584259, 0.85767365, 0.85944986, 0.86109680,
     &   0.86280072, 0.86460447, 0.86661416, 0.86852723, 0.86977708,
     &   0.87054926, 0.87123984, 0.87187123, 0.87273359, 0.87383097,
     &   0.87498325, 0.87617630, 0.87748677, 0.87874663, 0.88009584,
     &   0.88142580, 0.88291037, 0.88457304, 0.88648546, 0.88884646,
     &   0.89218503, 0.89597559, 0.90047467, 0.90469503, 0.90548348,
     &   0.89393139, 0.86898398, 0.85222101, 0.82893193, 0.81734544,
     &   0.81707454, 0.80532569, 0.78413570, 0.76797599, 0.75873268,
     &   0.75862682, 0.76342672, 0.76863849, 0.77396429, 0.77784407,
     &   0.78097796, 0.78311825, 0.78463280, 0.78572464, 0.78583872,
     &   0.78560168, 0.78513879, 0.78451830, 0.78396630, 0.78293478,
     &   0.78038275, 0.77640998, 0.77023315, 0.76498204, 0.76154208,
     &   0.75804943, 0.75485033, 0.75184029, 0.74874973, 0.74553114,
     &   0.74247533, 0.73932809, 0.73607588, 0.73272145, 0.72916502,
     &   0.72537106, 0.71990061, 0.71392894, 0.70733136, 0.70054513,
     &   0.69375968, 0.68718827, 0.68088764, 0.67444164, 0.66762286,
     &   0.66062671, 0.65351892, 0.64626241, 0.63877457, 0.63120866,
     &   0.62345958, 0.61551034, 0.60730892, 0.59882468, 0.58999389,
     &   0.58081663, 0.57148474, 0.56191128, 0.55204350, 0.54187435,
     &   0.53145057, 0.52073193, 0.50952369, 0.49787042, 0.48552960,
     &   0.47227919, 0.45816705, 0.44353256, 0.42811969, 0.41264573,
     &   0.39927053, 0.38807437, 0.37964272, 0.37176767, 0.36505762,
     &   0.35880020, 0.35334224, 0.34852433, 0.34433725, 0.34104297,
     &   0.33801129, 0.33507434, 0.33216941, 0.32855204, 0.32477379,
     &   0.32061231, 0.31533492, 0.30884263, 0.30096099, 0.29233268,
     &   0.28303388, 0.27376184, 0.26465940, 0.25572476, 0.24739619,
     &   0.23976064, 0.23226026, 0.22436742, 0.21644016, 0.20869276,
     &   0.20095909, 0.19339371, 0.18635289, 0.18005219, 0.17420787,
     &   0.16834669, 0.16226423, 0.15605254, 0.15000923, 0.14420974,
     &   0.13864677, 0.13329220, 0.12814108, 0.12324487, 0.11857629,
     &   0.11411741, 0.10983279, 0.10570837, 0.10174163, 0.09792793,
     &   0.09426469, 0.09074659, 0.08736110, 0.08410282, 0.08096659,
     &   0.07794465, 0.07501174, 0.07218920, 0.06947257, 0.06685756,
     &   0.06429966, 0.06179715, 0.05939137, 0.05707805, 0.05482369,
     &   0.05260702, 0.05050712, 0.04851654, 0.04663692, 0.04484820,
     &   0.04313698, 0.04149871, 0.03995646, 0.03848788, 0.03707949,
     &   0.03571666, 0.03439042, 0.03309963, 0.03184167, 0.03068038,
     &   0.02977655, 0.02888698, 0.02831453, 0.02878861, 0.02957804,
     &   0.02947856, 0.02874881, 0.02801049, 0.02737986, 0.02686523,
     &   0.02639063, 0.02603038, 0.02569732, 0.02488645, 0.02372128,
     &   0.02203722, 0.02069433, 0.01966441, 0.01900977, 0.01876168,
     &   0.01844225, 0.01814760, 0.01790375, 0.01770471, 0.01743343,
     &   0.01706745, 0.01665195, 0.01619747, 0.01574667, 0.01527589,
     &   0.01481480, 0.01434643, 0.01387076, 0.01340701, 0.01291592,
     &   0.01243109, 0.01195248, 0.01149363, 0.01106209, 0.01065247,
     &   0.01025789, 0.00988114, 0.00952000, 0.00917197, 0.00883853,
     &   0.00851722, 0.00820918, 0.00791995, 0.00764089, 0.00737166,
     &   0.00711190, 0.00686129, 0.00661949, 0.00638621, 0.00616113,
     &   0.00594398, 0.00573448, 0.00553612, 0.00534527, 0.00516098,
     &   0.00498304, 0.00481121, 0.00464530, 0.00448509, 0.00433040,
     &   0.00418103, 0.00403681, 0.00389755, 0.00376309, 0.00363360,
     &   0.00350983, 0.00339025, 0.00327474, 0.00316316, 0.00305536,
     &   0.00295119, 0.00285032, 0.00275286, 0.00265870, 0.00256687,
     &   0.00247758, 0.00239137, 0.00230783, 0.00222650, 0.00214802,
     &   0.00207226, 0.00199879, 0.00192793, 0.00185956, 0.00179367,
     &   0.00173012, 0.00166882, 0.00160967, 0.00155259, 0.00149754,
     &   0.00144445, 0.00139327, 0.00134390, 0.00129627, 0.00125042,
     &   0.00120619, 0.00116353, 0.00112242, 0.00108277, 0.00104453,
     &   0.00100769, 0.00097219, 0.00093795, 0.00090494, 0.00087317,
     &   0.00084252, 0.00081295, 0.00078452, 0.00075708, 0.00073060/
      data (ggi(i,2),i=1,mxwv)/
     &   0.84917140, 0.84926456, 0.84936696, 0.84919518, 0.84908676,
     &   0.84889764, 0.84881067, 0.84856558, 0.84849083, 0.84841669,
     &   0.84816706, 0.84799922, 0.84766787, 0.84742087, 0.84716910,
     &   0.84669441, 0.84648794, 0.84621769, 0.84593588, 0.84557152,
     &   0.84517574, 0.84470445, 0.84429395, 0.84385592, 0.84338874,
     &   0.84299600, 0.84242749, 0.84194636, 0.84143001, 0.84092623,
     &   0.84034145, 0.83968282, 0.83917385, 0.83852208, 0.83786619,
     &   0.83718598, 0.83651030, 0.83579898, 0.83501047, 0.83424908,
     &   0.83344120, 0.83264542, 0.83178341, 0.83089077, 0.82996136,
     &   0.82901311, 0.82803476, 0.82702070, 0.82598084, 0.82490480,
     &   0.82378823, 0.82265246, 0.82148319, 0.82028925, 0.81906003,
     &   0.81781399, 0.81654078, 0.81524265, 0.81392950, 0.81259841,
     &   0.81125748, 0.80990350, 0.80854684, 0.80718958, 0.80583549,
     &   0.80449080, 0.80316085, 0.80185473, 0.80057502, 0.79933429,
     &   0.79813409, 0.79698664, 0.79589570, 0.79487097, 0.79392213,
     &   0.79306108, 0.79230011, 0.79164577, 0.79111117, 0.79070884,
     &   0.79044741, 0.79033697, 0.79037809, 0.79056978, 0.79093295,
     &   0.79147685, 0.79220688, 0.79312509, 0.79426038, 0.79561424,
     &   0.79737663, 0.79962528, 0.80235118, 0.80466044, 0.80642235,
     &   0.80828089, 0.81037319, 0.81278962, 0.81544471, 0.81820953,
     &   0.82113928, 0.82421690, 0.82743859, 0.83076888, 0.83408225,
     &   0.83764076, 0.84156382, 0.84610569, 0.85076064, 0.85425222,
     &   0.85689265, 0.85942078, 0.86188591, 0.86482960, 0.86828721,
     &   0.87192333, 0.87568557, 0.87968713, 0.88364244, 0.88774800,
     &   0.89181852, 0.89612395, 0.90068567, 0.90559161, 0.91111666,
     &   0.91797650, 0.92525935, 0.93318057, 0.94029433, 0.94308221,
     &   0.93246245, 0.90762490, 0.89092463, 0.86711878, 0.85522097,
     &   0.85717922, 0.85129422, 0.82542664, 0.79088098, 0.76473993,
     &   0.75694960, 0.76204813, 0.77066338, 0.78112841, 0.79004753,
     &   0.79800308, 0.80457574, 0.81026953, 0.81534708, 0.81890559,
     &   0.82195395, 0.82469428, 0.82722872, 0.82989496, 0.83183336,
     &   0.83147663, 0.82899904, 0.82329947, 0.81911016, 0.81781822,
     &   0.81654161, 0.81581968, 0.81548071, 0.81511676, 0.81467724,
     &   0.81461549, 0.81455046, 0.81447977, 0.81441712, 0.81426167,
     &   0.81389618, 0.81053841, 0.80645800, 0.80118477, 0.79568779,
     &   0.79041481, 0.78602189, 0.78247619, 0.77872699, 0.77428758,
     &   0.76973355, 0.76524407, 0.76065886, 0.75589448, 0.75142258,
     &   0.74668378, 0.74158508, 0.73617482, 0.73067921, 0.72528714,
     &   0.72025174, 0.71548736, 0.71022648, 0.70471579, 0.69896567,
     &   0.69294691, 0.68654519, 0.67981690, 0.67269862, 0.66507792,
     &   0.65676397, 0.64742184, 0.63667083, 0.62384343, 0.60865778,
     &   0.59332752, 0.57937968, 0.56869352, 0.55878782, 0.55046654,
     &   0.54255033, 0.53547692, 0.52896595, 0.52300751, 0.51824027,
     &   0.51371914, 0.50946736, 0.50580972, 0.50166601, 0.49755046,
     &   0.49356103, 0.48906791, 0.48379770, 0.47720644, 0.46972069,
     &   0.46119544, 0.45218688, 0.44287264, 0.43326879, 0.42376509,
     &   0.41481608, 0.40595838, 0.39649197, 0.38664594, 0.37663409,
     &   0.36613703, 0.35544324, 0.34502843, 0.33545977, 0.32656282,
     &   0.31756175, 0.30795959, 0.29777062, 0.28750828, 0.27742225,
     &   0.26754534, 0.25785923, 0.24839330, 0.23930289, 0.23056434,
     &   0.22216430, 0.21404478, 0.20619054, 0.19860987, 0.19130133,
     &   0.18426859, 0.17750756, 0.17099294, 0.16471666, 0.15867044,
     &   0.15283981, 0.14717169, 0.14171445, 0.13645975, 0.13139947,
     &   0.12644352, 0.12159006, 0.11692380, 0.11243618, 0.10806518,
     &   0.10377247, 0.09971109, 0.09586556, 0.09222864, 0.08876160,
     &   0.08544204, 0.08226056, 0.07925844, 0.07639313, 0.07363933,
     &   0.07096922, 0.06836597, 0.06582751, 0.06334890, 0.06105047,
     &   0.05924718, 0.05746062, 0.05627456, 0.05715230, 0.05877275,
     &   0.05867526, 0.05726555, 0.05582485, 0.05459800, 0.05360858,
     &   0.05270187, 0.05203269, 0.05143032, 0.04981858, 0.04745170,
     &   0.04401491, 0.04130376, 0.03923849, 0.03793414, 0.03745477,
     &   0.03683023, 0.03625653, 0.03578776, 0.03541326, 0.03489154,
     &   0.03417274, 0.03334953, 0.03244369, 0.03154372, 0.03060090,
     &   0.02967687, 0.02873661, 0.02778024, 0.02684781, 0.02585855,
     &   0.02488202, 0.02391864, 0.02299587, 0.02212896, 0.02130678,
     &   0.02051512, 0.01975966, 0.01903583, 0.01833849, 0.01767066,
     &   0.01702731, 0.01641069, 0.01583209, 0.01527392, 0.01473545,
     &   0.01421598, 0.01371483, 0.01323136, 0.01276494, 0.01231497,
     &   0.01188087, 0.01146206, 0.01106562, 0.01068422, 0.01031594,
     &   0.00996032, 0.00961695, 0.00928539, 0.00896524, 0.00865611,
     &   0.00835762, 0.00806940, 0.00779111, 0.00752240, 0.00726364,
     &   0.00701630, 0.00677736, 0.00654652, 0.00632353, 0.00610812,
     &   0.00589994, 0.00569832, 0.00550354, 0.00531536, 0.00513181,
     &   0.00495333, 0.00478102, 0.00461404, 0.00445146, 0.00429457,
     &   0.00414313, 0.00399627, 0.00385461, 0.00371796, 0.00358624,
     &   0.00345920, 0.00333665, 0.00321841, 0.00310431, 0.00299426,
     &   0.00288812, 0.00278580, 0.00268710, 0.00259190, 0.00250024,
     &   0.00241182, 0.00232652, 0.00224433, 0.00216507, 0.00208861,
     &   0.00201496, 0.00194399, 0.00187553, 0.00180953, 0.00174603,
     &   0.00168475, 0.00162563, 0.00156877, 0.00151391, 0.00146096/
      data (ggi(i,3),i=1,mxwv)/
     &   0.85731214, 0.85751832, 0.85769582, 0.85763127, 0.85768604,
     &   0.85763150, 0.85752910, 0.85752320, 0.85745639, 0.85750520,
     &   0.85742235, 0.85738337, 0.85708910, 0.85693592, 0.85684413,
     &   0.85660189, 0.85646498, 0.85621679, 0.85595161, 0.85581541,
     &   0.85546273, 0.85530627, 0.85497016, 0.85468125, 0.85443079,
     &   0.85408998, 0.85377288, 0.85343581, 0.85305929, 0.85269463,
     &   0.85228992, 0.85192668, 0.85154629, 0.85099095, 0.85069889,
     &   0.85023099, 0.84971017, 0.84940726, 0.84888673, 0.84845698,
     &   0.84793580, 0.84745103, 0.84697056, 0.84644598, 0.84585786,
     &   0.84529459, 0.84478503, 0.84409571, 0.84356952, 0.84295863,
     &   0.84233254, 0.84167403, 0.84101832, 0.84032494, 0.83966959,
     &   0.83887935, 0.83813560, 0.83738202, 0.83658934, 0.83576691,
     &   0.83493620, 0.83403969, 0.83314896, 0.83222657, 0.83125597,
     &   0.83027440, 0.82925826, 0.82822347, 0.82715118, 0.82605857,
     &   0.82493895, 0.82379597, 0.82261276, 0.82141232, 0.82018697,
     &   0.81894636, 0.81769210, 0.81642693, 0.81515050, 0.81388038,
     &   0.81261504, 0.81136030, 0.81011015, 0.80886012, 0.80762446,
     &   0.80641752, 0.80524963, 0.80413151, 0.80308330, 0.80212706,
     &   0.80152977, 0.80151254, 0.80227512, 0.80231804, 0.80132455,
     &   0.80033767, 0.79948217, 0.79904205, 0.79899687, 0.79910892,
     &   0.79948801, 0.80015701, 0.80116028, 0.80247766, 0.80401713,
     &   0.80604094, 0.80891079, 0.81330240, 0.81870443, 0.82241613,
     &   0.82478410, 0.82702869, 0.82926118, 0.83249348, 0.83698964,
     &   0.84220809, 0.84800673, 0.85455728, 0.86140716, 0.86874408,
     &   0.87623596, 0.88429636, 0.89292371, 0.90218478, 0.91246170,
     &   0.92479515, 0.93736798, 0.95006543, 0.96054292, 0.96502995,
     &   0.95502400, 0.93037206, 0.91381347, 0.88978171, 0.87796861,
     &   0.88211477, 0.88510287, 0.87315297, 0.83507568, 0.78570020,
     &   0.75208068, 0.74180454, 0.74340969, 0.75362378, 0.76536810,
     &   0.77708125, 0.78805768, 0.79854274, 0.80850369, 0.81622535,
     &   0.82326323, 0.82998604, 0.83645207, 0.84328443, 0.84903896,
     &   0.85117799, 0.85002869, 0.84332681, 0.83905494, 0.83957446,
     &   0.84024423, 0.84191298, 0.84428531, 0.84667689, 0.84900165,
     &   0.85194093, 0.85488957, 0.85783505, 0.86078650, 0.86360306,
     &   0.86610591, 0.86431295, 0.86156297, 0.85709888, 0.85232991,
     &   0.84795195, 0.84499478, 0.84340358, 0.84156454, 0.83875245,
     &   0.83585775, 0.83315665, 0.83039367, 0.82746953, 0.82511652,
     &   0.82243043, 0.81926054, 0.81571627, 0.81220913, 0.80911303,
     &   0.80689496, 0.80533332, 0.80312854, 0.80077583, 0.79832697,
     &   0.79573309, 0.79282141, 0.78989345, 0.78687322, 0.78386104,
     &   0.78094399, 0.77781820, 0.77360320, 0.76741683, 0.75775671,
     &   0.74516881, 0.73197037, 0.72116786, 0.71116471, 0.70283335,
     &   0.69484293, 0.68759382, 0.68071568, 0.67414892, 0.66863853,
     &   0.66312528, 0.65776843, 0.65307921, 0.64809889, 0.64303142,
     &   0.63820076, 0.63349152, 0.62871897, 0.62365615, 0.61831939,
     &   0.61274010, 0.60681701, 0.60059381, 0.59411353, 0.58729154,
     &   0.58024347, 0.57309926, 0.56584638, 0.55838126, 0.55068594,
     &   0.54277521, 0.53458345, 0.52621317, 0.51783425, 0.50951731,
     &   0.50101513, 0.49203783, 0.48249784, 0.47260541, 0.46244609,
     &   0.45203879, 0.44136575, 0.43044001, 0.41943035, 0.40836319,
     &   0.39727059, 0.38614506, 0.37498906, 0.36385530, 0.35279247,
     &   0.34184843, 0.33106044, 0.32043698, 0.31000233, 0.29977682,
     &   0.28976548, 0.27989513, 0.27028051, 0.26092851, 0.25184318,
     &   0.24285835, 0.23398384, 0.22540589, 0.21711864, 0.20900692,
     &   0.20100541, 0.19343153, 0.18625684, 0.17946242, 0.17297190,
     &   0.16674344, 0.16075818, 0.15509589, 0.14967351, 0.14444257,
     &   0.13935012, 0.13436420, 0.12948166, 0.12468329, 0.12016890,
     &   0.11658015, 0.11294117, 0.11030856, 0.11156409, 0.11480656,
     &   0.11513623, 0.11266553, 0.11004021, 0.10780244, 0.10605422,
     &   0.10449169, 0.10344395, 0.10269877, 0.09976295, 0.09506969,
     &   0.08793950, 0.08237317, 0.07817542, 0.07553288, 0.07463518,
     &   0.07344513, 0.07236794, 0.07152173, 0.07089716, 0.06997933,
     &   0.06863035, 0.06704286, 0.06526260, 0.06348125, 0.06159545,
     &   0.05974023, 0.05784176, 0.05590114, 0.05400637, 0.05198567,
     &   0.04999034, 0.04802433, 0.04614521, 0.04438427, 0.04271778,
     &   0.04111495, 0.03958770, 0.03812613, 0.03671932, 0.03537368,
     &   0.03407831, 0.03283788, 0.03167582, 0.03055526, 0.02947469,
     &   0.02843264, 0.02742770, 0.02645853, 0.02552382, 0.02462232,
     &   0.02375283, 0.02291420, 0.02212080, 0.02135766, 0.02062088,
     &   0.01990955, 0.01922277, 0.01855971, 0.01791952, 0.01730142,
     &   0.01670465, 0.01612846, 0.01557215, 0.01503502, 0.01451783,
     &   0.01402353, 0.01354602, 0.01308474, 0.01263913, 0.01220867,
     &   0.01179267, 0.01138973, 0.01100046, 0.01062440, 0.01025755,
     &   0.00990084, 0.00955647, 0.00922273, 0.00889777, 0.00858420,
     &   0.00828153, 0.00798802, 0.00770489, 0.00743179, 0.00716854,
     &   0.00691466, 0.00666975, 0.00643344, 0.00620542, 0.00598548,
     &   0.00577337, 0.00556888, 0.00537163, 0.00518137, 0.00499819,
     &   0.00482147, 0.00465100, 0.00448674, 0.00432834, 0.00417553,
     &   0.00402831, 0.00388648, 0.00374964, 0.00361774, 0.00349081,
     &   0.00336834, 0.00325016, 0.00313652, 0.00302686, 0.00292103/
      data (ggi(i,4),i=1,mxwv)/
     &   0.86354595, 0.86387378, 0.86413205, 0.86419100, 0.86422288,
     &   0.86427242, 0.86433119, 0.86443490, 0.86452848, 0.86455113,
     &   0.86457717, 0.86457628, 0.86446059, 0.86437178, 0.86437982,
     &   0.86421132, 0.86415386, 0.86410737, 0.86397141, 0.86383533,
     &   0.86373389, 0.86348313, 0.86330944, 0.86324066, 0.86297172,
     &   0.86276370, 0.86253339, 0.86231703, 0.86193711, 0.86176825,
     &   0.86156946, 0.86123502, 0.86095035, 0.86073852, 0.86043340,
     &   0.86010987, 0.85968465, 0.85946202, 0.85907412, 0.85879147,
     &   0.85842693, 0.85807860, 0.85774779, 0.85727656, 0.85688710,
     &   0.85652345, 0.85615265, 0.85575706, 0.85528415, 0.85483712,
     &   0.85449290, 0.85404968, 0.85355610, 0.85312176, 0.85266221,
     &   0.85217464, 0.85172027, 0.85118085, 0.85069901, 0.85021353,
     &   0.84965587, 0.84921789, 0.84863478, 0.84808910, 0.84746087,
     &   0.84692204, 0.84639084, 0.84579170, 0.84516639, 0.84450972,
     &   0.84398657, 0.84330082, 0.84262276, 0.84194064, 0.84124148,
     &   0.84050077, 0.83976346, 0.83899713, 0.83821011, 0.83742809,
     &   0.83661664, 0.83580232, 0.83494693, 0.83405042, 0.83311909,
     &   0.83215487, 0.83116984, 0.83018994, 0.82916421, 0.82816952,
     &   0.82757306, 0.82770216, 0.82884902, 0.82878834, 0.82705367,
     &   0.82511985, 0.82308894, 0.82140428, 0.82006902, 0.81868273,
     &   0.81738156, 0.81619018, 0.81516963, 0.81427956, 0.81345153,
     &   0.81287485, 0.81317407, 0.81551582, 0.81953561, 0.82102489,
     &   0.82033187, 0.81902516, 0.81723422, 0.81641126, 0.81729722,
     &   0.81934178, 0.82237464, 0.82671618, 0.83210492, 0.83858246,
     &   0.84589791, 0.85473263, 0.86530066, 0.87780613, 0.89306289,
     &   0.91311407, 0.93446958, 0.95546496, 0.97123039, 0.97769755,
     &   0.96824318, 0.94388378, 0.92750490, 0.90339875, 0.89163810,
     &   0.89680916, 0.90399241, 0.90662426, 0.88911843, 0.84652519,
     &   0.79877818, 0.76724547, 0.74883264, 0.74460483, 0.74767816,
     &   0.75290531, 0.76069516, 0.77065963, 0.78178132, 0.79104638,
     &   0.80020177, 0.80996084, 0.82001293, 0.83169460, 0.84248418,
     &   0.84785253, 0.84818053, 0.83737630, 0.83020800, 0.83127660,
     &   0.83294308, 0.83665371, 0.84194493, 0.84751421, 0.85307115,
     &   0.85979772, 0.86658484, 0.87336135, 0.88012540, 0.88659400,
     &   0.89264452, 0.89250737, 0.89094067, 0.88682079, 0.88206333,
     &   0.87775147, 0.87540203, 0.87526834, 0.87495059, 0.87323362,
     &   0.87141740, 0.86994582, 0.86845738, 0.86677080, 0.86601371,
     &   0.86484480, 0.86303002, 0.86072308, 0.85854781, 0.85708266,
     &   0.85702944, 0.85800797, 0.85812747, 0.85812110, 0.85807353,
     &   0.85791397, 0.85742182, 0.85708213, 0.85679901, 0.85680407,
     &   0.85733366, 0.85806596, 0.85786474, 0.85587513, 0.85014510,
     &   0.84028894, 0.82874978, 0.81882548, 0.80963886, 0.80207026,
     &   0.79482174, 0.78824687, 0.78194654, 0.77581888, 0.77057147,
     &   0.76512247, 0.75965375, 0.75464767, 0.74935979, 0.74371111,
     &   0.73804140, 0.73263568, 0.72746491, 0.72283596, 0.71847183,
     &   0.71474427, 0.71100259, 0.70716035, 0.70332587, 0.69908053,
     &   0.69380438, 0.68804616, 0.68261850, 0.67729807, 0.67192739,
     &   0.66729426, 0.66284418, 0.65851980, 0.65340877, 0.64711559,
     &   0.64022440, 0.63332695, 0.62688315, 0.62094045, 0.61484838,
     &   0.60857582, 0.60221910, 0.59584361, 0.58918631, 0.58218908,
     &   0.57483935, 0.56710190, 0.55899405, 0.55052477, 0.54167122,
     &   0.53245729, 0.52289951, 0.51299149, 0.50276053, 0.49223804,
     &   0.48144868, 0.47034889, 0.45906255, 0.44763228, 0.43610087,
     &   0.42428482, 0.41219944, 0.40013352, 0.38813147, 0.37601596,
     &   0.36367574, 0.35172105, 0.34016061, 0.32910264, 0.31844157,
     &   0.30808532, 0.29801974, 0.28844926, 0.27922574, 0.27026030,
     &   0.26145995, 0.25276443, 0.24417309, 0.23559672, 0.22727811,
     &   0.22055119, 0.21342406, 0.20751449, 0.20776242, 0.21277854,
     &   0.21450683, 0.21097265, 0.20686291, 0.20325723, 0.20056030,
     &   0.19831036, 0.19712707, 0.19755411, 0.19415170, 0.18691573,
     &   0.17322907, 0.16198197, 0.15339760, 0.14774854, 0.14600536,
     &   0.14376703, 0.14182289, 0.14044149, 0.13971111, 0.13859823,
     &   0.13657883, 0.13400170, 0.13093664, 0.12778251, 0.12431413,
     &   0.12083100, 0.11719103, 0.11338978, 0.10961869, 0.10549580,
     &   0.10138208, 0.09729552, 0.09338834, 0.08973753, 0.08629080,
     &   0.08297966, 0.07983184, 0.07682534, 0.07393590, 0.07117860,
     &   0.06852850, 0.06599586, 0.06363205, 0.06135518, 0.05916191,
     &   0.05704901, 0.05501337, 0.05305203, 0.05116215, 0.04934097,
     &   0.04758589, 0.04589439, 0.04429648, 0.04276063, 0.04127862,
     &   0.03984851, 0.03846842, 0.03713657, 0.03585123, 0.03461072,
     &   0.03341345, 0.03225788, 0.03114253, 0.03006596, 0.02902969,
     &   0.02803972, 0.02708360, 0.02616015, 0.02526824, 0.02440680,
     &   0.02357438, 0.02276794, 0.02198900, 0.02123664, 0.02050266,
     &   0.01978899, 0.01910012, 0.01843258, 0.01778257, 0.01715545,
     &   0.01655023, 0.01596339, 0.01539737, 0.01485144, 0.01432528,
     &   0.01381786, 0.01332843, 0.01285618, 0.01240055, 0.01196107,
     &   0.01153727, 0.01112871, 0.01073461, 0.01035449, 0.00998852,
     &   0.00963548, 0.00929491, 0.00896676, 0.00865031, 0.00834502,
     &   0.00805093, 0.00776759, 0.00749421, 0.00723069, 0.00697712,
     &   0.00673243, 0.00649633, 0.00626929, 0.00605019, 0.00583875/
      data (ggi(i,5),i=1,mxwv)/
     &   0.86838096, 0.86875856, 0.86908442, 0.86922479, 0.86933666,
     &   0.86943299, 0.86961442, 0.86972928, 0.86985797, 0.86996901,
     &   0.87007201, 0.87014949, 0.87014484, 0.87012798, 0.87012559,
     &   0.87017876, 0.87011158, 0.87016016, 0.87010252, 0.87005627,
     &   0.87000841, 0.86992210, 0.86978942, 0.86971080, 0.86960274,
     &   0.86950409, 0.86931765, 0.86922896, 0.86904919, 0.86887044,
     &   0.86871040, 0.86852163, 0.86835635, 0.86814195, 0.86795092,
     &   0.86772966, 0.86752564, 0.86732262, 0.86705369, 0.86682278,
     &   0.86657232, 0.86627513, 0.86598843, 0.86579013, 0.86551493,
     &   0.86521584, 0.86494446, 0.86460012, 0.86430830, 0.86403984,
     &   0.86367214, 0.86339486, 0.86301410, 0.86268014, 0.86237687,
     &   0.86204690, 0.86167049, 0.86134934, 0.86091506, 0.86055094,
     &   0.86018711, 0.85983330, 0.85942614, 0.85913426, 0.85864145,
     &   0.85823268, 0.85784298, 0.85743535, 0.85708499, 0.85662627,
     &   0.85628480, 0.85584235, 0.85532409, 0.85493201, 0.85450584,
     &   0.85408115, 0.85353297, 0.85309374, 0.85271180, 0.85222512,
     &   0.85178685, 0.85136116, 0.85097802, 0.85041732, 0.84994894,
     &   0.84942216, 0.84892517, 0.84832358, 0.84786195, 0.84739792,
     &   0.84750330, 0.84861159, 0.85105985, 0.85187608, 0.85047966,
     &   0.84879333, 0.84692645, 0.84547734, 0.84446496, 0.84331936,
     &   0.84221393, 0.84117162, 0.84026653, 0.83942169, 0.83853674,
     &   0.83780444, 0.83813238, 0.84112406, 0.84615624, 0.84754169,
     &   0.84580934, 0.84287065, 0.83874601, 0.83521438, 0.83340871,
     &   0.83265251, 0.83260894, 0.83360553, 0.83550674, 0.83796716,
     &   0.84081024, 0.84482038, 0.85049701, 0.85833484, 0.87017781,
     &   0.89104879, 0.91909635, 0.95126897, 0.97576815, 0.98497438,
     &   0.97604096, 0.95203799, 0.93583339, 0.91169918, 0.89989978,
     &   0.90539831, 0.91353375, 0.92159641, 0.91971308, 0.89743727,
     &   0.85954458, 0.82665408, 0.80163753, 0.78942090, 0.78513712,
     &   0.78199524, 0.78234386, 0.78613597, 0.79192972, 0.79652435,
     &   0.80151880, 0.80843174, 0.81661129, 0.82880801, 0.84197265,
     &   0.84944063, 0.85108382, 0.83348817, 0.81915843, 0.81601423,
     &   0.81419849, 0.81609708, 0.82153934, 0.82835704, 0.83576417,
     &   0.84591317, 0.85681725, 0.86820453, 0.88001001, 0.89146018,
     &   0.90276057, 0.90520024, 0.90522403, 0.90135294, 0.89595729,
     &   0.89067113, 0.88753659, 0.88812786, 0.88899964, 0.88778406,
     &   0.88631666, 0.88539517, 0.88455653, 0.88332641, 0.88366002,
     &   0.88352442, 0.88248128, 0.88063532, 0.87893230, 0.87829667,
     &   0.87993032, 0.88337511, 0.88559616, 0.88768411, 0.88976085,
     &   0.89172405, 0.89324391, 0.89502597, 0.89694661, 0.89934087,
     &   0.90253925, 0.90608829, 0.90853316, 0.90901518, 0.90537524,
     &   0.89698350, 0.88647991, 0.87727243, 0.86880010, 0.86195111,
     &   0.85545212, 0.84965134, 0.84414417, 0.83881807, 0.83432990,
     &   0.82958782, 0.82473695, 0.82013643, 0.81513250, 0.80945992,
     &   0.80329686, 0.79705817, 0.79085726, 0.78564465, 0.78108543,
     &   0.77800393, 0.77534139, 0.77286035, 0.77071422, 0.76825815,
     &   0.76409483, 0.75902587, 0.75453782, 0.75040478, 0.74637717,
     &   0.74396956, 0.74215633, 0.74089068, 0.73829252, 0.73334700,
     &   0.72721839, 0.72127998, 0.71663326, 0.71340317, 0.71021879,
     &   0.70704192, 0.70416385, 0.70184112, 0.69941545, 0.69676477,
     &   0.69385946, 0.69065410, 0.68723279, 0.68362385, 0.67970580,
     &   0.67550164, 0.67100930, 0.66612589, 0.66083783, 0.65513271,
     &   0.64899886, 0.64241886, 0.63537943, 0.62787741, 0.61991292,
     &   0.61142838, 0.60235256, 0.59274513, 0.58263242, 0.57184297,
     &   0.56014174, 0.54799092, 0.53545177, 0.52293658, 0.51045066,
     &   0.49785882, 0.48519766, 0.47292006, 0.46087933, 0.44897196,
     &   0.43708691, 0.42513299, 0.41312411, 0.40079215, 0.38826188,
     &   0.37798098, 0.36652920, 0.35578287, 0.35208037, 0.35617396,
     &   0.35890427, 0.35470846, 0.34929320, 0.34410274, 0.34002972,
     &   0.33670491, 0.33484378, 0.33721039, 0.33651474, 0.33140498,
     &   0.31408641, 0.29631740, 0.28128579, 0.26981828, 0.26586258,
     &   0.26146874, 0.25774226, 0.25517273, 0.25425673, 0.25364563,
     &   0.25183871, 0.24921700, 0.24579599, 0.24210177, 0.23774494,
     &   0.23316461, 0.22822882, 0.22284484, 0.21719675, 0.21052060,
     &   0.20350261, 0.19604886, 0.18869008, 0.18169537, 0.17497513,
     &   0.16843066, 0.16214304, 0.15608481, 0.15021770, 0.14458449,
     &   0.13914658, 0.13393623, 0.12907989, 0.12439491, 0.11987738,
     &   0.11552306, 0.11132742, 0.10728579, 0.10339330, 0.09964504,
     &   0.09603606, 0.09256144, 0.08928916, 0.08614790, 0.08311960,
     &   0.08020022, 0.07738582, 0.07467259, 0.07205678, 0.06953481,
     &   0.06710318, 0.06475854, 0.06249764, 0.06031736, 0.05822078,
     &   0.05622032, 0.05428965, 0.05242627, 0.05062773, 0.04889172,
     &   0.04721497, 0.04559001, 0.04402156, 0.04250762, 0.04103084,
     &   0.03959535, 0.03821066, 0.03686936, 0.03556348, 0.03430438,
     &   0.03308989, 0.03191287, 0.03077812, 0.02968409, 0.02863011,
     &   0.02761402, 0.02663424, 0.02568915, 0.02477753, 0.02389844,
     &   0.02305088, 0.02223398, 0.02144614, 0.02068634, 0.01995495,
     &   0.01924949, 0.01856903, 0.01791344, 0.01728128, 0.01667147,
     &   0.01608405, 0.01551815, 0.01497217, 0.01444589, 0.01393951,
     &   0.01345088, 0.01297937, 0.01252600, 0.01208848, 0.01166624/
      data (ggi(i,6),i=1,mxwv)/
     &   0.87211460, 0.87250054, 0.87290180, 0.87308013, 0.87326646,
     &   0.87341970, 0.87361413, 0.87382364, 0.87396908, 0.87413549,
     &   0.87432647, 0.87444997, 0.87450641, 0.87457401, 0.87463570,
     &   0.87466305, 0.87472379, 0.87478226, 0.87484366, 0.87484998,
     &   0.87487835, 0.87483084, 0.87478560, 0.87476498, 0.87468469,
     &   0.87468803, 0.87460333, 0.87453461, 0.87444782, 0.87436163,
     &   0.87427795, 0.87414104, 0.87405950, 0.87393260, 0.87376136,
     &   0.87368381, 0.87353325, 0.87337106, 0.87320298, 0.87311167,
     &   0.87286401, 0.87272090, 0.87253100, 0.87235028, 0.87214327,
     &   0.87196612, 0.87184113, 0.87153298, 0.87133908, 0.87115264,
     &   0.87092948, 0.87066299, 0.87039733, 0.87019694, 0.86992651,
     &   0.86968297, 0.86932713, 0.86921030, 0.86894697, 0.86864859,
     &   0.86842763, 0.86811930, 0.86784267, 0.86755341, 0.86729580,
     &   0.86698449, 0.86662751, 0.86644572, 0.86605936, 0.86583889,
     &   0.86540848, 0.86521250, 0.86488497, 0.86455214, 0.86427909,
     &   0.86390573, 0.86357230, 0.86330497, 0.86288124, 0.86265224,
     &   0.86242467, 0.86211646, 0.86187357, 0.86157715, 0.86127752,
     &   0.86089891, 0.86059821, 0.86025608, 0.85976595, 0.85972464,
     &   0.86027038, 0.86210948, 0.86564142, 0.86713147, 0.86586392,
     &   0.86425000, 0.86241645, 0.86112815, 0.86041868, 0.85955983,
     &   0.85877168, 0.85808843, 0.85759044, 0.85720241, 0.85677582,
     &   0.85656017, 0.85777766, 0.86247283, 0.86962467, 0.87208539,
     &   0.87064701, 0.86762863, 0.86295080, 0.85875630, 0.85657376,
     &   0.85556304, 0.85523623, 0.85594505, 0.85749227, 0.85907441,
     &   0.86033303, 0.86195737, 0.86424363, 0.86721301, 0.87249482,
     &   0.88623512, 0.90951407, 0.94370192, 0.97657281, 0.98917103,
     &   0.98069805, 0.95706266, 0.94099015, 0.91681218, 0.90492719,
     &   0.91054976, 0.91885984, 0.92823571, 0.93279898, 0.92492247,
     &   0.90004659, 0.87198770, 0.84781528, 0.83580482, 0.83194387,
     &   0.82864749, 0.82871908, 0.83215129, 0.83719033, 0.84045821,
     &   0.84361374, 0.84855461, 0.85447776, 0.86481184, 0.87677634,
     &   0.88387138, 0.88597357, 0.86657989, 0.84795535, 0.83949620,
     &   0.83192277, 0.82816517, 0.82890147, 0.83179164, 0.83569789,
     &   0.84396309, 0.85431594, 0.86652118, 0.88070738, 0.89532304,
     &   0.91126752, 0.91691226, 0.91885489, 0.91596919, 0.91046572,
     &   0.90414584, 0.89879417, 0.89847970, 0.89956403, 0.89800155,
     &   0.89580387, 0.89425510, 0.89295840, 0.89078230, 0.89101493,
     &   0.89092451, 0.88964587, 0.88690472, 0.88396615, 0.88226908,
     &   0.88417518, 0.88977790, 0.89375889, 0.89769417, 0.90174347,
     &   0.90575397, 0.90909624, 0.91279995, 0.91673136, 0.92132956,
     &   0.92706025, 0.93325996, 0.93797249, 0.94030994, 0.93805128,
     &   0.93054593, 0.92065972, 0.91191608, 0.90392560, 0.89760011,
     &   0.89168519, 0.88654560, 0.88179141, 0.87732542, 0.87381166,
     &   0.87015879, 0.86649358, 0.86309236, 0.85923648, 0.85460222,
     &   0.84910703, 0.84282571, 0.83566451, 0.82895350, 0.82263029,
     &   0.81825483, 0.81479764, 0.81197995, 0.81001514, 0.80822921,
     &   0.80411553, 0.79847056, 0.79342633, 0.78900969, 0.78498912,
     &   0.78381354, 0.78382361, 0.78513134, 0.78462899, 0.78040040,
     &   0.77416027, 0.76818442, 0.76442868, 0.76321661, 0.76226634,
     &   0.76150346, 0.76140571, 0.76240152, 0.76339161, 0.76417524,
     &   0.76470971, 0.76493782, 0.76502168, 0.76503181, 0.76478058,
     &   0.76432908, 0.76370573, 0.76275116, 0.76146227, 0.75983441,
     &   0.75786787, 0.75559944, 0.75295484, 0.74992520, 0.74650025,
     &   0.74282122, 0.73880482, 0.73424256, 0.72912103, 0.72342581,
     &   0.71691245, 0.70940202, 0.70085424, 0.69167840, 0.68200642,
     &   0.67168874, 0.66075504, 0.64975744, 0.63865614, 0.62738830,
     &   0.61587459, 0.60400933, 0.59181446, 0.57879215, 0.56474072,
     &   0.55312717, 0.53957844, 0.52570742, 0.51755250, 0.51795936,
     &   0.51969588, 0.51537287, 0.50945234, 0.50308090, 0.49731916,
     &   0.49214229, 0.48782438, 0.48847565, 0.48954117, 0.48949090,
     &   0.47991022, 0.46390310, 0.44673562, 0.43007836, 0.42219958,
     &   0.41447735, 0.40761873, 0.40211445, 0.39889261, 0.39729971,
     &   0.39517438, 0.39266431, 0.38974142, 0.38661045, 0.38304085,
     &   0.37923121, 0.37536398, 0.37126458, 0.36670384, 0.36101365,
     &   0.35461843, 0.34689870, 0.33870435, 0.33049348, 0.32215154,
     &   0.31368065, 0.30518368, 0.29666618, 0.28810906, 0.27955130,
     &   0.27101874, 0.26258501, 0.25448290, 0.24647875, 0.23858696,
     &   0.23082100, 0.22319329, 0.21571513, 0.20839651, 0.20124617,
     &   0.19427152, 0.18747862, 0.18104693, 0.17482202, 0.16877306,
     &   0.16290158, 0.15720814, 0.15169252, 0.14635366, 0.14118978,
     &   0.13619852, 0.13137695, 0.12672170, 0.12222897, 0.11790897,
     &   0.11379270, 0.10982004, 0.10598692, 0.10228913, 0.09872243,
     &   0.09527965, 0.09194119, 0.08872246, 0.08561946, 0.08259300,
     &   0.07965332, 0.07682229, 0.07408309, 0.07141794, 0.06885269,
     &   0.06638243, 0.06399202, 0.06169080, 0.05947514, 0.05734333,
     &   0.05529058, 0.05331330, 0.05140788, 0.04957163, 0.04780236,
     &   0.04609793, 0.04445642, 0.04287431, 0.04134941, 0.03988253,
     &   0.03846831, 0.03710479, 0.03579175, 0.03452610, 0.03330557,
     &   0.03213032, 0.03099849, 0.02990673, 0.02885461, 0.02784255,
     &   0.02686610, 0.02592400, 0.02501832, 0.02414439, 0.02330104/
      data (ggi(i,7),i=1,mxwv)/
     &   0.87489736, 0.87536228, 0.87580419, 0.87601978, 0.87623501,
     &   0.87644792, 0.87665993, 0.87689215, 0.87713051, 0.87734789,
     &   0.87764251, 0.87774098, 0.87784582, 0.87795979, 0.87807024,
     &   0.87815166, 0.87831157, 0.87835789, 0.87845975, 0.87846845,
     &   0.87857407, 0.87858820, 0.87860405, 0.87861580, 0.87863743,
     &   0.87864137, 0.87864751, 0.87862623, 0.87860060, 0.87858325,
     &   0.87852675, 0.87849015, 0.87841409, 0.87838513, 0.87829834,
     &   0.87822765, 0.87812150, 0.87808865, 0.87809438, 0.87776786,
     &   0.87778407, 0.87766737, 0.87752956, 0.87743974, 0.87739658,
     &   0.87720048, 0.87703884, 0.87692267, 0.87676561, 0.87664968,
     &   0.87648410, 0.87628216, 0.87616801, 0.87598616, 0.87584031,
     &   0.87567508, 0.87547106, 0.87526476, 0.87506282, 0.87490451,
     &   0.87471068, 0.87456602, 0.87435168, 0.87417197, 0.87394166,
     &   0.87375855, 0.87356663, 0.87334198, 0.87316120, 0.87294936,
     &   0.87274522, 0.87257886, 0.87229490, 0.87211519, 0.87182957,
     &   0.87166476, 0.87139982, 0.87119293, 0.87101024, 0.87079394,
     &   0.87066066, 0.87049103, 0.87038189, 0.87018633, 0.86990988,
     &   0.86973697, 0.86951125, 0.86929208, 0.86912382, 0.86900789,
     &   0.86991549, 0.87243515, 0.87710911, 0.87919140, 0.87782729,
     &   0.87602842, 0.87392509, 0.87254107, 0.87187999, 0.87102556,
     &   0.87026834, 0.86964446, 0.86927855, 0.86905372, 0.86878699,
     &   0.86880833, 0.87063873, 0.87685049, 0.88602275, 0.88939589,
     &   0.88808012, 0.88485980, 0.87957078, 0.87481499, 0.87254775,
     &   0.87176722, 0.87187141, 0.87332505, 0.87585932, 0.87834668,
     &   0.88025701, 0.88232601, 0.88468713, 0.88688713, 0.89011818,
     &   0.90051132, 0.91849756, 0.94621319, 0.97738570, 0.99166113,
     &   0.98353404, 0.96022242, 0.94422466, 0.91996735, 0.90798366,
     &   0.91369164, 0.92208660, 0.93176490, 0.93848187, 0.93865818,
     &   0.92570221, 0.90386218, 0.88089544, 0.86870730, 0.86527020,
     &   0.86260271, 0.86380816, 0.86881191, 0.87544841, 0.87994230,
     &   0.88412040, 0.88986540, 0.89619434, 0.90621847, 0.91684711,
     &   0.92278701, 0.92461038, 0.90893048, 0.89198440, 0.88335657,
     &   0.87497205, 0.86984807, 0.86897194, 0.86998409, 0.87157649,
     &   0.87733382, 0.88502771, 0.89454907, 0.90629107, 0.91854757,
     &   0.93262810, 0.93836540, 0.94045883, 0.93885386, 0.93500090,
     &   0.93000871, 0.92455918, 0.92306060, 0.92336857, 0.92147356,
     &   0.91870147, 0.91629559, 0.91411424, 0.91060263, 0.90945196,
     &   0.90826494, 0.90590507, 0.90141857, 0.89575839, 0.89032817,
     &   0.88886052, 0.89370137, 0.89714098, 0.90097338, 0.90540570,
     &   0.91028595, 0.91436553, 0.91899598, 0.92416441, 0.93040222,
     &   0.93848342, 0.94747895, 0.95448643, 0.95845783, 0.95725816,
     &   0.95037115, 0.94089997, 0.93247509, 0.92481881, 0.91886896,
     &   0.91338760, 0.90876234, 0.90462297, 0.90089512, 0.89829606,
     &   0.89574659, 0.89340532, 0.89162701, 0.88958377, 0.88709831,
     &   0.88413751, 0.88041413, 0.87537229, 0.86956131, 0.86294156,
     &   0.85715687, 0.85191441, 0.84723741, 0.84357655, 0.84082448,
     &   0.83556044, 0.82788473, 0.81990546, 0.81233013, 0.80530614,
     &   0.80261242, 0.80202776, 0.80450666, 0.80521810, 0.80033422,
     &   0.79176861, 0.78295738, 0.77750510, 0.77663648, 0.77659875,
     &   0.77723813, 0.77931231, 0.78351158, 0.78797728, 0.79230440,
     &   0.79638225, 0.80008072, 0.80365342, 0.80720711, 0.81043726,
     &   0.81344217, 0.81626815, 0.81867486, 0.82067162, 0.82226688,
     &   0.82347870, 0.82439262, 0.82490426, 0.82501912, 0.82474071,
     &   0.82436675, 0.82382596, 0.82276607, 0.82118684, 0.81923968,
     &   0.81676531, 0.81319636, 0.80846018, 0.80287498, 0.79663545,
     &   0.78960156, 0.78176433, 0.77360624, 0.76516479, 0.75640935,
     &   0.74729240, 0.73770267, 0.72765034, 0.71648002, 0.70364016,
     &   0.69305831, 0.68020046, 0.66625619, 0.65638417, 0.65435702,
     &   0.65504020, 0.65146577, 0.64630198, 0.64015591, 0.63388097,
     &   0.62770528, 0.62131339, 0.61806768, 0.61758286, 0.61917257,
     &   0.62052071, 0.61365849, 0.60157758, 0.58548135, 0.57491457,
     &   0.56512362, 0.55591118, 0.54745370, 0.54058194, 0.53571296,
     &   0.53117913, 0.52682227, 0.52265739, 0.51838368, 0.51422042,
     &   0.50994319, 0.50600219, 0.50230789, 0.49837613, 0.49449265,
     &   0.49037835, 0.48558050, 0.48037100, 0.47487730, 0.46907040,
     &   0.46305850, 0.45682245, 0.45037875, 0.44373208, 0.43682143,
     &   0.42971131, 0.42241168, 0.41499060, 0.40742370, 0.39971462,
     &   0.39186892, 0.38389328, 0.37579608, 0.36758679, 0.35927647,
     &   0.35087746, 0.34240332, 0.33411080, 0.32582510, 0.31751582,
     &   0.30919680, 0.30088246, 0.29258755, 0.28432706, 0.27611610,
     &   0.26796952, 0.25990200, 0.25192770, 0.24406014, 0.23634697,
     &   0.22888875, 0.22156101, 0.21437353, 0.20733497, 0.20045301,
     &   0.19373281, 0.18716472, 0.18075943, 0.17452340, 0.16837765,
     &   0.16235748, 0.15652721, 0.15085737, 0.14531362, 0.13996680,
     &   0.13481236, 0.12982033, 0.12501794, 0.12039964, 0.11596362,
     &   0.11170006, 0.10760162, 0.10366062, 0.09987106, 0.09622794,
     &   0.09272630, 0.08936160, 0.08612531, 0.08301214, 0.08002397,
     &   0.07714792, 0.07437937, 0.07171776, 0.06915582, 0.06668823,
     &   0.06431539, 0.06203297, 0.05983338, 0.05771556, 0.05568038,
     &   0.05371813, 0.05182608, 0.05000858, 0.04825566, 0.04656487/
      data (ggi(i,8),i=1,mxwv)/
     &   0.87702900, 0.87750286, 0.87797087, 0.87823170, 0.87848151,
     &   0.87873423, 0.87918609, 0.87923157, 0.87950087, 0.87977129,
     &   0.88003421, 0.88023639, 0.88036680, 0.88052553, 0.88067180,
     &   0.88080782, 0.88095564, 0.88109183, 0.88121873, 0.88132393,
     &   0.88141263, 0.88146991, 0.88155776, 0.88160396, 0.88164806,
     &   0.88169473, 0.88173652, 0.88174814, 0.88177133, 0.88178670,
     &   0.88177919, 0.88179058, 0.88179141, 0.88177693, 0.88174957,
     &   0.88173079, 0.88169396, 0.88167149, 0.88161951, 0.88156986,
     &   0.88150948, 0.88146520, 0.88139462, 0.88134205, 0.88123500,
     &   0.88119894, 0.88110691, 0.88102382, 0.88094038, 0.88086313,
     &   0.88075858, 0.88066620, 0.88053167, 0.88045263, 0.88034385,
     &   0.88023865, 0.88012064, 0.88000244, 0.87989795, 0.87980473,
     &   0.87965238, 0.87953591, 0.87940776, 0.87927681, 0.87918669,
     &   0.87899941, 0.87889147, 0.87872767, 0.87860870, 0.87849033,
     &   0.87835717, 0.87824869, 0.87808877, 0.87795341, 0.87779617,
     &   0.87764204, 0.87747824, 0.87734622, 0.87714136, 0.87713099,
     &   0.87707418, 0.87704492, 0.87699425, 0.87693119, 0.87680399,
     &   0.87670964, 0.87650484, 0.87638730, 0.87632322, 0.87632942,
     &   0.87761837, 0.88097847, 0.88703430, 0.88978845, 0.88823950,
     &   0.88613898, 0.88365626, 0.88202226, 0.88132119, 0.88037926,
     &   0.87954634, 0.87886858, 0.87853670, 0.87834972, 0.87814313,
     &   0.87824601, 0.88066286, 0.88850969, 0.89980119, 0.90393406,
     &   0.90249646, 0.89875138, 0.89242822, 0.88661581, 0.88382643,
     &   0.88288945, 0.88306129, 0.88492942, 0.88818324, 0.89138317,
     &   0.89390159, 0.89665413, 0.89980876, 0.90275431, 0.90675056,
     &   0.91843033, 0.93579048, 0.95839447, 0.98161435, 0.99325639,
     &   0.98530501, 0.96223730, 0.94626170, 0.92189318, 0.90981185,
     &   0.91559756, 0.92407387, 0.93385553, 0.94116622, 0.94491476,
     &   0.94105834, 0.92730904, 0.90775907, 0.89568383, 0.89203686,
     &   0.88915932, 0.89056623, 0.89604878, 0.90325737, 0.90821546,
     &   0.91282392, 0.91883940, 0.92519528, 0.93424124, 0.94278073,
     &   0.94702858, 0.94815660, 0.93735641, 0.92395926, 0.91687906,
     &   0.90968078, 0.90542096, 0.90524334, 0.90681678, 0.90879345,
     &   0.91443926, 0.92136651, 0.92925239, 0.93816435, 0.94654000,
     &   0.95505744, 0.95787472, 0.95849162, 0.95731509, 0.95522135,
     &   0.95260626, 0.94966692, 0.94867682, 0.94865847, 0.94739848,
     &   0.94557375, 0.94386870, 0.94223994, 0.93958068, 0.93831581,
     &   0.93703747, 0.93492013, 0.93099838, 0.92550969, 0.91903788,
     &   0.91473961, 0.91643131, 0.91713780, 0.91831559, 0.92027193,
     &   0.92310011, 0.92520738, 0.92783654, 0.93134832, 0.93639177,
     &   0.94450635, 0.95508730, 0.96386683, 0.96907544, 0.96861637,
     &   0.96215349, 0.95297867, 0.94479007, 0.93737257, 0.93168330,
     &   0.92649525, 0.92221189, 0.91847128, 0.91521281, 0.91320026,
     &   0.91135025, 0.90985376, 0.90917945, 0.90843785, 0.90761822,
     &   0.90696359, 0.90629041, 0.90511256, 0.90318638, 0.90028328,
     &   0.89703238, 0.89341265, 0.88959008, 0.88600516, 0.88303906,
     &   0.87823004, 0.87100232, 0.86207819, 0.85224664, 0.84193170,
     &   0.83532202, 0.83073848, 0.83093196, 0.83064669, 0.82378435,
     &   0.81089908, 0.79541075, 0.78279436, 0.77687871, 0.77268893,
     &   0.77015036, 0.77058202, 0.77534741, 0.78157002, 0.78836071,
     &   0.79537594, 0.80218756, 0.80914634, 0.81638634, 0.82332432,
     &   0.83008820, 0.83671045, 0.84274369, 0.84819371, 0.85307109,
     &   0.85740441, 0.86131197, 0.86467367, 0.86750984, 0.86983991,
     &   0.87210381, 0.87422240, 0.87571764, 0.87661237, 0.87718785,
     &   0.87732846, 0.87623101, 0.87385130, 0.87049812, 0.86643070,
     &   0.86153626, 0.85580093, 0.84966624, 0.84321105, 0.83642268,
     &   0.82927221, 0.82164830, 0.81355280, 0.80425525, 0.79299402,
     &   0.78385818, 0.77237499, 0.75951463, 0.75006205, 0.74803442,
     &   0.74895465, 0.74660075, 0.74277550, 0.73779410, 0.73229510,
     &   0.72656864, 0.71997279, 0.71447879, 0.71187681, 0.71243256,
     &   0.72048289, 0.72161776, 0.71573585, 0.70331126, 0.69275677,
     &   0.68329298, 0.67407542, 0.66495985, 0.65636337, 0.64919442,
     &   0.64264446, 0.63639814, 0.63052559, 0.62444240, 0.61868644,
     &   0.61278033, 0.60712254, 0.60171598, 0.59619337, 0.59167331,
     &   0.58739585, 0.58383155, 0.58033121, 0.57650852, 0.57254881,
     &   0.56853402, 0.56437922, 0.56011420, 0.55577713, 0.55135763,
     &   0.54684478, 0.54219896, 0.53729630, 0.53229904, 0.52720213,
     &   0.52200025, 0.51668799, 0.51125973, 0.50570977, 0.50003225,
     &   0.49422151, 0.48827195, 0.48218122, 0.47597626, 0.46965465,
     &   0.46321201, 0.45664436, 0.44994792, 0.44312108, 0.43616062,
     &   0.42906573, 0.42183623, 0.41447309, 0.40697780, 0.39938506,
     &   0.39179289, 0.38409445, 0.37629509, 0.36840087, 0.36041933,
     &   0.35237446, 0.34434521, 0.33623254, 0.32804963, 0.31970280,
     &   0.31122857, 0.30272046, 0.29415566, 0.28549361, 0.27685335,
     &   0.26825371, 0.25964844, 0.25114310, 0.24275860, 0.23452401,
     &   0.22644874, 0.21854599, 0.21082552, 0.20329954, 0.19598044,
     &   0.18887758, 0.18199916, 0.17534190, 0.16890720, 0.16271180,
     &   0.15673484, 0.15097319, 0.14543256, 0.14010046, 0.13496809,
     &   0.13004008, 0.12530832, 0.12075571, 0.11638103, 0.11218712,
     &   0.10815117, 0.10426702, 0.10054480, 0.09696128, 0.09351071/
      data (ggi(i,9),i=1,mxwv)/
     &   0.87861216, 0.87912714, 0.87961048, 0.87989771, 0.88015419,
     &   0.88045174, 0.88072830, 0.88100988, 0.88131195, 0.88159961,
     &   0.88191193, 0.88211364, 0.88225728, 0.88245320, 0.88263482,
     &   0.88280320, 0.88297671, 0.88315302, 0.88330525, 0.88342798,
     &   0.88357282, 0.88366103, 0.88373339, 0.88384557, 0.88393360,
     &   0.88399094, 0.88407284, 0.88412911, 0.88418782, 0.88423204,
     &   0.88427269, 0.88429838, 0.88433433, 0.88435364, 0.88437861,
     &   0.88439059, 0.88438499, 0.88441080, 0.88438809, 0.88436329,
     &   0.88437539, 0.88435322, 0.88433629, 0.88432211, 0.88421333,
     &   0.88425434, 0.88420820, 0.88416022, 0.88414431, 0.88405228,
     &   0.88404387, 0.88398880, 0.88393426, 0.88387388, 0.88382107,
     &   0.88376039, 0.88370198, 0.88363826, 0.88354838, 0.88348168,
     &   0.88342309, 0.88335329, 0.88325542, 0.88320518, 0.88313800,
     &   0.88304681, 0.88296896, 0.88289082, 0.88281608, 0.88278264,
     &   0.88269329, 0.88265270, 0.88258392, 0.88247150, 0.88237703,
     &   0.88225985, 0.88217372, 0.88211381, 0.88202369, 0.88204759,
     &   0.88208342, 0.88218290, 0.88223863, 0.88224030, 0.88216460,
     &   0.88207263, 0.88198572, 0.88195962, 0.88192153, 0.88211077,
     &   0.88384169, 0.88824773, 0.89600974, 0.89953816, 0.89772618,
     &   0.89521414, 0.89219284, 0.89022022, 0.88942087, 0.88831621,
     &   0.88734370, 0.88657820, 0.88619983, 0.88603407, 0.88583034,
     &   0.88600457, 0.88909823, 0.89893109, 0.91256785, 0.91742581,
     &   0.91582614, 0.91149068, 0.90394008, 0.89676589, 0.89320081,
     &   0.89191365, 0.89198589, 0.89414859, 0.89801866, 0.90173972,
     &   0.90456200, 0.90759873, 0.91106200, 0.91421902, 0.91852558,
     &   0.93224186, 0.95138603, 0.97166264, 0.98712981, 0.99436945,
     &   0.98643923, 0.96352768, 0.94753546, 0.92303532, 0.91086286,
     &   0.91672194, 0.92527878, 0.93513221, 0.94259757, 0.94754589,
     &   0.94885170, 0.94281578, 0.92904377, 0.91857743, 0.91510749,
     &   0.91221404, 0.91349638, 0.91887504, 0.92572075, 0.93032610,
     &   0.93451089, 0.93975532, 0.94502383, 0.95185995, 0.95752108,
     &   0.95991820, 0.96029598, 0.95403159, 0.94481820, 0.93950540,
     &   0.93371493, 0.93027365, 0.93050712, 0.93235248, 0.93456882,
     &   0.93987775, 0.94588572, 0.95207924, 0.95822030, 0.96320325,
     &   0.96720278, 0.96763885, 0.96713793, 0.96598178, 0.96477473,
     &   0.96359318, 0.96261293, 0.96236354, 0.96231294, 0.96169502,
     &   0.96089458, 0.96016288, 0.95942676, 0.95826197, 0.95773053,
     &   0.95711833, 0.95604551, 0.95402884, 0.95104790, 0.94718951,
     &   0.94423604, 0.94497985, 0.94494987, 0.94507951, 0.94558775,
     &   0.94661868, 0.94698578, 0.94739658, 0.94822210, 0.94993538,
     &   0.95441270, 0.96255887, 0.97056401, 0.97573411, 0.97548461,
     &   0.96925932, 0.96029502, 0.95228016, 0.94502527, 0.93949974,
     &   0.93448442, 0.93039072, 0.92685962, 0.92383349, 0.92209685,
     &   0.92055541, 0.91941339, 0.91918170, 0.91895413, 0.91877759,
     &   0.91903824, 0.91970581, 0.92050463, 0.92105937, 0.92119813,
     &   0.92098552, 0.92030317, 0.91923994, 0.91800463, 0.91666961,
     &   0.91434044, 0.91074061, 0.90597856, 0.90001529, 0.89289916,
     &   0.88731909, 0.88245511, 0.88105732, 0.87982208, 0.87359488,
     &   0.86139870, 0.84443849, 0.82708508, 0.81474429, 0.80317056,
     &   0.79265922, 0.78519297, 0.78332424, 0.78440309, 0.78737509,
     &   0.79183400, 0.79694194, 0.80320889, 0.81091559, 0.81908399,
     &   0.82786471, 0.83720589, 0.84616733, 0.85464323, 0.86255389,
     &   0.86985534, 0.87659317, 0.88265854, 0.88805199, 0.89278567,
     &   0.89742756, 0.90187275, 0.90550929, 0.90838367, 0.91091967,
     &   0.91301948, 0.91366345, 0.91285694, 0.91091794, 0.90818310,
     &   0.90457714, 0.90009391, 0.89516687, 0.88990366, 0.88430488,
     &   0.87835324, 0.87193900, 0.86506182, 0.85695267, 0.84675574,
     &   0.83866948, 0.82823300, 0.81638521, 0.80800879, 0.80724674,
     &   0.80938607, 0.80833244, 0.80586404, 0.80228645, 0.79818672,
     &   0.79382873, 0.78858125, 0.78370869, 0.78042197, 0.77895218,
     &   0.78832126, 0.79356980, 0.79157293, 0.78230286, 0.77314705,
     &   0.76518047, 0.75736427, 0.74942994, 0.74151385, 0.73438507,
     &   0.72764087, 0.72093368, 0.71433550, 0.70726514, 0.70031285,
     &   0.69300628, 0.68547165, 0.67774314, 0.66982085, 0.66321880,
     &   0.65712357, 0.65319151, 0.64990377, 0.64639097, 0.64303654,
     &   0.63975614, 0.63645667, 0.63315028, 0.62987131, 0.62674826,
     &   0.62361676, 0.62042904, 0.61702919, 0.61358297, 0.61008787,
     &   0.60654104, 0.60293984, 0.59928191, 0.59556472, 0.59178591,
     &   0.58794308, 0.58403379, 0.57974291, 0.57535118, 0.57091630,
     &   0.56643617, 0.56190848, 0.55733067, 0.55269986, 0.54801291,
     &   0.54326642, 0.53845650, 0.53357917, 0.52862978, 0.52357388,
     &   0.51834142, 0.51305032, 0.50769496, 0.50226957, 0.49676803,
     &   0.49117112, 0.48541486, 0.47958785, 0.47368118, 0.46767673,
     &   0.46155202, 0.45529982, 0.44889754, 0.44231009, 0.43555477,
     &   0.42862305, 0.42148268, 0.41414553, 0.40661177, 0.39889035,
     &   0.39098251, 0.38289407, 0.37463105, 0.36620566, 0.35763535,
     &   0.34893969, 0.34014118, 0.33125597, 0.32230702, 0.31333917,
     &   0.30435616, 0.29538226, 0.28645572, 0.27759159, 0.26880825,
     &   0.26014522, 0.25162235, 0.24323891, 0.23502812, 0.22702031,
     &   0.21919525, 0.21156321, 0.20417121, 0.19698486, 0.19000819/
      data (ggi(i,10),i=1,mxwv)/
     &   0.87979376, 0.88032848, 0.88083392, 0.88113421, 0.88143677,
     &   0.88173145, 0.88203132, 0.88235635, 0.88264561, 0.88296133,
     &   0.88326079, 0.88352317, 0.88370222, 0.88391024, 0.88411182,
     &   0.88430208, 0.88449323, 0.88469124, 0.88486171, 0.88502699,
     &   0.88517451, 0.88529623, 0.88541865, 0.88552934, 0.88563472,
     &   0.88574493, 0.88583863, 0.88592499, 0.88600731, 0.88607621,
     &   0.88614625, 0.88620913, 0.88626486, 0.88631040, 0.88635892,
     &   0.88640118, 0.88644695, 0.88647515, 0.88650399, 0.88654381,
     &   0.88654196, 0.88654375, 0.88656640, 0.88657153, 0.88658404,
     &   0.88657010, 0.88657951, 0.88657016, 0.88657296, 0.88656485,
     &   0.88654953, 0.88654172, 0.88714188, 0.88648868, 0.88646346,
     &   0.88645452, 0.88641918, 0.88639730, 0.88634604, 0.88633174,
     &   0.88629454, 0.88626850, 0.88622606, 0.88619983, 0.88616955,
     &   0.88613683, 0.88609660, 0.88607973, 0.88605475, 0.88605869,
     &   0.88604754, 0.88604963, 0.88603532, 0.88595802, 0.88588345,
     &   0.88584608, 0.88580573, 0.88581634, 0.88577354, 0.88585353,
     &   0.88598418, 0.88617688, 0.88634092, 0.88640004, 0.88638735,
     &   0.88634479, 0.88632965, 0.88634443, 0.88643354, 0.88668120,
     &   0.88898635, 0.89472604, 0.90456676, 0.90896869, 0.90682262,
     &   0.90379131, 0.90007579, 0.89764601, 0.89668661, 0.89533091,
     &   0.89413399, 0.89317906, 0.89274049, 0.89254421, 0.89230603,
     &   0.89253563, 0.89646429, 0.90868044, 0.92476171, 0.93022448,
     &   0.92848057, 0.92359602, 0.91475803, 0.90598249, 0.90142864,
     &   0.89965779, 0.89956105, 0.90205628, 0.90665066, 0.91095221,
     &   0.91406715, 0.91733652, 0.92099941, 0.92413270, 0.92856389,
     &   0.94336253, 0.96317536, 0.98099470, 0.99096775, 0.99514866,
     &   0.98717970, 0.96434897, 0.94831508, 0.92367476, 0.91141993,
     &   0.91734707, 0.92598063, 0.93590057, 0.94342828, 0.94869637,
     &   0.95217532, 0.95145297, 0.94410676, 0.93665594, 0.93394721,
     &   0.93152255, 0.93276757, 0.93747848, 0.94327670, 0.94704199,
     &   0.95035034, 0.95425290, 0.95792890, 0.96218264, 0.96518654,
     &   0.96615005, 0.96599674, 0.96308529, 0.95794392, 0.95465624,
     &   0.95069665, 0.94822532, 0.94856626, 0.95017272, 0.95205963,
     &   0.95616853, 0.96049207, 0.96455652, 0.96811312, 0.97063696,
     &   0.97212183, 0.97148764, 0.97055143, 0.96940392, 0.96853417,
     &   0.96793222, 0.96781260, 0.96790284, 0.96782964, 0.96750915,
     &   0.96720397, 0.96696639, 0.96669918, 0.96635556, 0.96627432,
     &   0.96608490, 0.96566695, 0.96491885, 0.96382827, 0.96238154,
     &   0.96138304, 0.96219093, 0.96250582, 0.96287417, 0.96339291,
     &   0.96411461, 0.96444833, 0.96475303, 0.96514386, 0.96579385,
     &   0.96773350, 0.97197193, 0.97682559, 0.98040169, 0.97976696,
     &   0.97362471, 0.96480590, 0.95691127, 0.94975901, 0.94432712,
     &   0.93940437, 0.93540674, 0.93197727, 0.92905933, 0.92744422,
     &   0.92603165, 0.92502546, 0.92495328, 0.92489690, 0.92491436,
     &   0.92542505, 0.92645055, 0.92779893, 0.92912191, 0.93031877,
     &   0.93132830, 0.93198889, 0.93235546, 0.93252581, 0.93229401,
     &   0.93152797, 0.93037713, 0.92898679, 0.92701304, 0.92425054,
     &   0.92202544, 0.92000103, 0.91952801, 0.91887331, 0.91540736,
     &   0.90829754, 0.89733404, 0.88459027, 0.87427866, 0.86341333,
     &   0.85222799, 0.84256935, 0.83717424, 0.83405441, 0.83238333,
     &   0.83197147, 0.83196044, 0.83305508, 0.83589453, 0.83963567,
     &   0.84474093, 0.85138506, 0.85839379, 0.86563969, 0.87297726,
     &   0.88024527, 0.88723594, 0.89394855, 0.90026438, 0.90609854,
     &   0.91197270, 0.91774392, 0.92269945, 0.92684251, 0.93069410,
     &   0.93413043, 0.93593293, 0.93613827, 0.93506306, 0.93311942,
     &   0.93027020, 0.92651713, 0.92229575, 0.91773015, 0.91282851,
     &   0.90758020, 0.90187413, 0.89571095, 0.88830084, 0.87874442,
     &   0.87132609, 0.86155802, 0.85040843, 0.84297812, 0.84355748,
     &   0.84712535, 0.84725481, 0.84596300, 0.84365994, 0.84101284,
     &   0.83827382, 0.83499861, 0.83305395, 0.83225995, 0.83155298,
     &   0.83807749, 0.84286100, 0.84174609, 0.83395487, 0.82649809,
     &   0.82020181, 0.81414944, 0.80815220, 0.80242020, 0.79749197,
     &   0.79273629, 0.78779131, 0.78261924, 0.77675301, 0.77054012,
     &   0.76363128, 0.75582349, 0.74693936, 0.73723072, 0.72783619,
     &   0.71840322, 0.71183252, 0.70637661, 0.70097148, 0.69618779,
     &   0.69165581, 0.68738163, 0.68333447, 0.67950350, 0.67625403,
     &   0.67317420, 0.67024541, 0.66743612, 0.66467988, 0.66196764,
     &   0.65929055, 0.65663987, 0.65400732, 0.65138477, 0.64876449,
     &   0.64613914, 0.64350194, 0.64027697, 0.63694346, 0.63360316,
     &   0.63025236, 0.62688738, 0.62350488, 0.62010193, 0.61667567,
     &   0.61322343, 0.60974312, 0.60623258, 0.60268980, 0.59901553,
     &   0.59495693, 0.59089035, 0.58681422, 0.58272731, 0.57862824,
     &   0.57442844, 0.56967968, 0.56498164, 0.56033063, 0.55579168,
     &   0.55134094, 0.54692262, 0.54252720, 0.53815275, 0.53379744,
     &   0.52946311, 0.52530861, 0.52110183, 0.51683092, 0.51247317,
     &   0.50802636, 0.50348043, 0.49882972, 0.49405795, 0.48914522,
     &   0.48408338, 0.47886485, 0.47346637, 0.46787786, 0.46210799,
     &   0.45612791, 0.44993117, 0.44352534, 0.43689722, 0.43004137,
     &   0.42297319, 0.41569436, 0.40819597, 0.40049344, 0.39261612,
     &   0.38455194, 0.37631485, 0.36795920, 0.35946748, 0.35085794/
      data (ggi(i,11),i=1,mxwv)/
     &   0.88066339, 0.88120908, 0.88173294, 0.88205522, 0.88236547,
     &   0.88267845, 0.88299239, 0.88331431, 0.88364393, 0.88397080,
     &   0.88429195, 0.88461709, 0.88473970, 0.88498348, 0.88520390,
     &   0.88541657, 0.88562816, 0.88583797, 0.88603282, 0.88621503,
     &   0.88637757, 0.88652039, 0.88665456, 0.88679475, 0.88691747,
     &   0.88704973, 0.88715374, 0.88726604, 0.88736075, 0.88745922,
     &   0.88754660, 0.88762981, 0.88771355, 0.88778538, 0.88785505,
     &   0.88791400, 0.88797486, 0.88802952, 0.88807821, 0.88813043,
     &   0.88817018, 0.88819098, 0.88825345, 0.88827866, 0.88835782,
     &   0.88833857, 0.88836157, 0.88838464, 0.88840622, 0.88841337,
     &   0.88843554, 0.88845074, 0.88845468, 0.88846695, 0.88847953,
     &   0.88849545, 0.88847977, 0.88849252, 0.88848519, 0.88851607,
     &   0.88849652, 0.88849056, 0.88848889, 0.88849419, 0.88850379,
     &   0.88850605, 0.88850141, 0.88850945, 0.88852823, 0.88855696,
     &   0.88860232, 0.88866580, 0.88859779, 0.88861579, 0.88860649,
     &   0.88858974, 0.88859046, 0.88861984, 0.88866389, 0.88882238,
     &   0.88902396, 0.88933867, 0.88959175, 0.88972080, 0.88974708,
     &   0.88975030, 0.88976580, 0.88983673, 0.88995016, 0.89034355,
     &   0.89339954, 0.90084314, 0.91316867, 0.91850615, 0.91597867,
     &   0.91234374, 0.90779412, 0.90476966, 0.90359145, 0.90188682,
     &   0.90037119, 0.89915407, 0.89858395, 0.89832759, 0.89800942,
     &   0.89826846, 0.90323257, 0.91823560, 0.93656152, 0.94236481,
     &   0.94055212, 0.93530101, 0.92530113, 0.91476488, 0.90899682,
     &   0.90659070, 0.90625381, 0.90912789, 0.91458327, 0.91954356,
     &   0.92306191, 0.92646646, 0.93031567, 0.93338442, 0.93776107,
     &   0.95344085, 0.97247976, 0.98636436, 0.99275845, 0.99565822,
     &   0.98766810, 0.96486235, 0.94877279, 0.92399305, 0.91166526,
     &   0.91765422, 0.92635828, 0.93634081, 0.94391459, 0.94926751,
     &   0.95343864, 0.95540911, 0.95319784, 0.94931072, 0.94780892,
     &   0.94630069, 0.94742376, 0.95117557, 0.95546168, 0.95814556,
     &   0.96042442, 0.96290946, 0.96510208, 0.96731728, 0.96860933,
     &   0.96879452, 0.96837735, 0.96729606, 0.96512389, 0.96364540,
     &   0.96160340, 0.96019381, 0.96060711, 0.96184611, 0.96321714,
     &   0.96589863, 0.96850723, 0.97075891, 0.97252941, 0.97369009,
     &   0.97410107, 0.97312063, 0.97207224, 0.97094482, 0.97018766,
     &   0.96977717, 0.96995217, 0.97016162, 0.97008073, 0.96986955,
     &   0.96974742, 0.96969098, 0.96960610, 0.96959198, 0.96967834,
     &   0.96965051, 0.96950549, 0.96931767, 0.96913141, 0.96895415,
     &   0.96905959, 0.96984696, 0.97035998, 0.97090423, 0.97152799,
     &   0.97221911, 0.97278267, 0.97345287, 0.97421879, 0.97519898,
     &   0.97677088, 0.97919279, 0.98174733, 0.98368287, 0.98247862,
     &   0.97635037, 0.96763343, 0.95981675, 0.95272458, 0.94734240,
     &   0.94246662, 0.93851686, 0.93513787, 0.93227422, 0.93072075,
     &   0.92937338, 0.92843693, 0.92844021, 0.92846286, 0.92856294,
     &   0.92916483, 0.93030107, 0.93179739, 0.93332368, 0.93480086,
     &   0.93615466, 0.93721682, 0.93803525, 0.93867207, 0.93882686,
     &   0.93862557, 0.93840992, 0.93842572, 0.93828690, 0.93776536,
     &   0.93743682, 0.93718022, 0.93735957, 0.93711954, 0.93552172,
     &   0.93227226, 0.92685300, 0.91981798, 0.91370970, 0.90669924,
     &   0.89891648, 0.89190871, 0.88820475, 0.88620311, 0.88518304,
     &   0.88494724, 0.88469523, 0.88498515, 0.88634503, 0.88802856,
     &   0.89054918, 0.89409631, 0.89775276, 0.90153104, 0.90543002,
     &   0.90940338, 0.91321909, 0.91712129, 0.92105407, 0.92494911,
     &   0.92912775, 0.93351781, 0.93757397, 0.94120055, 0.94482338,
     &   0.94825530, 0.95023984, 0.95071483, 0.94994199, 0.94832754,
     &   0.94582945, 0.94244283, 0.93860263, 0.93442756, 0.92992455,
     &   0.92508125, 0.91978729, 0.91404158, 0.90704256, 0.89786971,
     &   0.89085722, 0.88148582, 0.87074673, 0.86389637, 0.86532712,
     &   0.86987442, 0.87081742, 0.87031996, 0.86885744, 0.86717349,
     &   0.86552829, 0.86358345, 0.86403656, 0.86649483, 0.86970085,
     &   0.87498027, 0.87718159, 0.87491196, 0.86705559, 0.86077291,
     &   0.85561657, 0.85082155, 0.84630245, 0.84248394, 0.83994228,
     &   0.83771849, 0.83548546, 0.83318025, 0.83038884, 0.82732356,
     &   0.82366312, 0.81932664, 0.81397164, 0.80746692, 0.80017656,
     &   0.79177207, 0.78460199, 0.77777672, 0.77067822, 0.76392931,
     &   0.75708234, 0.75034338, 0.74370748, 0.73714429, 0.73133099,
     &   0.72570044, 0.72037840, 0.71579009, 0.71135604, 0.70708501,
     &   0.70297903, 0.69904447, 0.69528079, 0.69168657, 0.68825865,
     &   0.68499178, 0.68187875, 0.67820239, 0.67452371, 0.67096698,
     &   0.66752595, 0.66419339, 0.66096187, 0.65782326, 0.65476936,
     &   0.65179163, 0.64888167, 0.64603102, 0.64323121, 0.64030689,
     &   0.63680935, 0.63337010, 0.62998265, 0.62664062, 0.62333786,
     &   0.61988097, 0.61529011, 0.61084288, 0.60653073, 0.60243827,
     &   0.59855002, 0.59479159, 0.59113210, 0.58756268, 0.58412963,
     &   0.58084953, 0.57811159, 0.57541138, 0.57273859, 0.57005566,
     &   0.56737715, 0.56470078, 0.56203973, 0.55937952, 0.55669236,
     &   0.55398345, 0.55125880, 0.54847819, 0.54563630, 0.54276937,
     &   0.53981775, 0.53677589, 0.53366989, 0.53046495, 0.52714401,
     &   0.52373302, 0.52021658, 0.51655453, 0.51274061, 0.50876880,
     &   0.50463271, 0.50032556, 0.49582809, 0.49114525, 0.48627853/
      data (ggi(i,12),i=1,mxwv)/
     &   0.88130456, 0.88185936, 0.88239551, 0.88272601, 0.88305002,
     &   0.88337380, 0.88370150, 0.88405013, 0.88436913, 0.88471287,
     &   0.88504660, 0.88533360, 0.88554657, 0.88577622, 0.88600510,
     &   0.88623244, 0.88645703, 0.88668329, 0.88689429, 0.88708818,
     &   0.88725978, 0.88742155, 0.88757086, 0.88771790, 0.88786697,
     &   0.88800770, 0.88813609, 0.88826048, 0.88837469, 0.88848609,
     &   0.88859314, 0.88868964, 0.88878590, 0.88888061, 0.88895899,
     &   0.88904518, 0.88912338, 0.88919657, 0.88924992, 0.88932472,
     &   0.88938785, 0.88944894, 0.88950509, 0.88951111, 0.88960493,
     &   0.88965160, 0.88969654, 0.88973826, 0.88977927, 0.88981545,
     &   0.88985711, 0.88987285, 0.88992161, 0.88995242, 0.88998097,
     &   0.89000374, 0.89003915, 0.89006901, 0.89009422, 0.89011389,
     &   0.89013702, 0.89016229, 0.89019102, 0.89023012, 0.89026618,
     &   0.89029294, 0.89031744, 0.89035165, 0.89040911, 0.89048624,
     &   0.89056647, 0.89068520, 0.89073426, 0.89071530, 0.89068800,
     &   0.89070189, 0.89073473, 0.89081204, 0.89086831, 0.89112365,
     &   0.89142704, 0.89186597, 0.89223427, 0.89243048, 0.89249152,
     &   0.89251173, 0.89255542, 0.89267302, 0.89287096, 0.89337265,
     &   0.89740676, 0.90701556, 0.92215866, 0.92839813, 0.92549962,
     &   0.92123675, 0.91575497, 0.91202831, 0.91055310, 0.90841180,
     &   0.90647882, 0.90490443, 0.90414399, 0.90377647, 0.90332532,
     &   0.90359950, 0.90984130, 0.92792749, 0.94782466, 0.95351326,
     &   0.95178097, 0.94653469, 0.93579859, 0.92353439, 0.91634929,
     &   0.91315466, 0.91247791, 0.91580611, 0.92225921, 0.92793381,
     &   0.93165380, 0.93535984, 0.93935156, 0.94230515, 0.94656360,
     &   0.96264893, 0.97952163, 0.98911625, 0.99343938, 0.99597210,
     &   0.98799092, 0.96517259, 0.94902152, 0.92410946, 0.91171813,
     &   0.91776186, 0.92652911, 0.93656898, 0.94418383, 0.94957358,
     &   0.95392293, 0.95690757, 0.95768845, 0.95679438, 0.95656013,
     &   0.95615399, 0.95723540, 0.95986187, 0.96263027, 0.96433794,
     &   0.96576613, 0.96718967, 0.96839797, 0.96945864, 0.96995634,
     &   0.96983933, 0.96933150, 0.96898603, 0.96836644, 0.96806073,
     &   0.96736467, 0.96695894, 0.96743304, 0.96831805, 0.96927601,
     &   0.97083521, 0.97226429, 0.97344714, 0.97433794, 0.97495061,
     &   0.97502232, 0.97396290, 0.97290039, 0.97178876, 0.97106767,
     &   0.97071308, 0.97097415, 0.97122532, 0.97115219, 0.97098136,
     &   0.97091985, 0.97092581, 0.97090358, 0.97099829, 0.97114587,
     &   0.97117788, 0.97112960, 0.97114319, 0.97130191, 0.97165209,
     &   0.97224277, 0.97298092, 0.97355002, 0.97413927, 0.97477359,
     &   0.97541934, 0.97606641, 0.97688955, 0.97785187, 0.97909343,
     &   0.98076165, 0.98275363, 0.98454392, 0.98573810, 0.98418641,
     &   0.97807467, 0.96942991, 0.96165949, 0.95459759, 0.94923800,
     &   0.94438255, 0.94045442, 0.93709958, 0.93426359, 0.93274486,
     &   0.93143547, 0.93054074, 0.93059105, 0.93066269, 0.93081337,
     &   0.93146878, 0.93266201, 0.93422228, 0.93582129, 0.93739104,
     &   0.93885148, 0.94003636, 0.94099349, 0.94177675, 0.94206232,
     &   0.94204611, 0.94213253, 0.94261688, 0.94314420, 0.94351035,
     &   0.94397092, 0.94446802, 0.94487554, 0.94477618, 0.94406718,
     &   0.94287169, 0.94086522, 0.93790489, 0.93505311, 0.93131614,
     &   0.92668945, 0.92227262, 0.92018056, 0.91937596, 0.91935235,
     &   0.91995215, 0.92055464, 0.92156821, 0.92333567, 0.92524946,
     &   0.92758209, 0.93042916, 0.93310940, 0.93563187, 0.93800712,
     &   0.94022739, 0.94217938, 0.94404846, 0.94584525, 0.94758236,
     &   0.94945365, 0.95148247, 0.95344043, 0.95529455, 0.95730406,
     &   0.95933199, 0.96045351, 0.96044856, 0.95949590, 0.95786464,
     &   0.95543927, 0.95218676, 0.94852966, 0.94456446, 0.94028664,
     &   0.93567878, 0.93062705, 0.92512846, 0.91837311, 0.90942484,
     &   0.90264648, 0.89349306, 0.88296258, 0.87640011, 0.87826210,
     &   0.88330489, 0.88468528, 0.88461578, 0.88358808, 0.88237321,
     &   0.88123709, 0.87985557, 0.88123685, 0.88518256, 0.89089638,
     &   0.89722735, 0.89835244, 0.89520329, 0.88698846, 0.88122356,
     &   0.87660325, 0.87240171, 0.86856341, 0.86561084, 0.86419696,
     &   0.86324978, 0.86247569, 0.86185753, 0.86098820, 0.86014062,
     &   0.85899180, 0.85778570, 0.85634351, 0.85425770, 0.85191923,
     &   0.84881622, 0.84575772, 0.84244359, 0.83858252, 0.83451009,
     &   0.83002120, 0.82521212, 0.82007962, 0.81457996, 0.80921400,
     &   0.80359459, 0.79786384, 0.79262978, 0.78726637, 0.78179133,
     &   0.77622414, 0.77058595, 0.76490498, 0.75918847, 0.75347745,
     &   0.74779141, 0.74215508, 0.73608863, 0.72998315, 0.72394931,
     &   0.71800786, 0.71217847, 0.70647895, 0.70092565, 0.69553268,
     &   0.69031209, 0.68527383, 0.68042523, 0.67577171, 0.67110068,
     &   0.66582286, 0.66074908, 0.65588075, 0.65121734, 0.64675641,
     &   0.64215529, 0.63562381, 0.62946069, 0.62366122, 0.61822706,
     &   0.61320245, 0.60857701, 0.60423726, 0.60010928, 0.59640563,
     &   0.59314781, 0.59100783, 0.58910364, 0.58740634, 0.58584231,
     &   0.58442330, 0.58313537, 0.58199441, 0.58096784, 0.57999909,
     &   0.57909560, 0.57826823, 0.57743424, 0.57658303, 0.57579553,
     &   0.57494873, 0.57403600, 0.57312047, 0.57213670, 0.57105720,
     &   0.56995142, 0.56879979, 0.56752706, 0.56613201, 0.56460977,
     &   0.56296575, 0.56120086, 0.55925572, 0.55719405, 0.55501711/
      data (ggi(i,13),i=1,mxwv)/
     &   0.88176787, 0.88233030, 0.88287544, 0.88321048, 0.88354802,
     &   0.88388175, 0.88421369, 0.88455391, 0.88497645, 0.88525784,
     &   0.88559914, 0.88589621, 0.88610947, 0.88635570, 0.88659734,
     &   0.88683420, 0.88706934, 0.88730079, 0.88752598, 0.88773096,
     &   0.88791746, 0.88808781, 0.88825196, 0.88840854, 0.88856405,
     &   0.88871366, 0.88885254, 0.88898760, 0.88911897, 0.88924497,
     &   0.88935965, 0.88947034, 0.88958246, 0.88968349, 0.88977987,
     &   0.88987684, 0.88996911, 0.89005482, 0.89013535, 0.89021611,
     &   0.89029014, 0.89036846, 0.89043260, 0.89050347, 0.89063716,
     &   0.89062822, 0.89068967, 0.89074582, 0.89080125, 0.89086139,
     &   0.89090741, 0.89096159, 0.89100915, 0.89105791, 0.89111030,
     &   0.89114273, 0.89119983, 0.89125282, 0.89129454, 0.89133686,
     &   0.89137602, 0.89142442, 0.89147878, 0.89154613, 0.89159930,
     &   0.89163822, 0.89169592, 0.89175153, 0.89184439, 0.89196080,
     &   0.89209944, 0.89225805, 0.89233547, 0.89230490, 0.89230239,
     &   0.89232647, 0.89238673, 0.89249080, 0.89259416, 0.89294720,
     &   0.89336276, 0.89396697, 0.89445990, 0.89473408, 0.89481878,
     &   0.89479709, 0.89491165, 0.89506918, 0.89532238, 0.89599049,
     &   0.90131444, 0.91349667, 0.93169469, 0.93863505, 0.93546116,
     &   0.93065953, 0.92426121, 0.91977048, 0.91795242, 0.91528738,
     &   0.91284060, 0.91081309, 0.90980172, 0.90927804, 0.90864760,
     &   0.90890193, 0.91668153, 0.93787712, 0.95802271, 0.96300292,
     &   0.96154940, 0.95685148, 0.94619191, 0.93257153, 0.92389596,
     &   0.91974276, 0.91864043, 0.92247760, 0.93004495, 0.93643171,
     &   0.94040060, 0.94424218, 0.94828641, 0.95106107, 0.95510209,
     &   0.97075701, 0.98420942, 0.99033076, 0.99371439, 0.99616367,
     &   0.98820323, 0.96534932, 0.94913638, 0.92410302, 0.91165680,
     &   0.91774821, 0.92657071, 0.93666196, 0.94431365, 0.94973052,
     &   0.95412838, 0.95740408, 0.95947427, 0.96038705, 0.96109498,
     &   0.96159834, 0.96267641, 0.96440351, 0.96607649, 0.96715319,
     &   0.96807623, 0.96892661, 0.96966469, 0.97025162, 0.97048295,
     &   0.97028315, 0.96975541, 0.96963030, 0.96961218, 0.96985805,
     &   0.96993291, 0.97010976, 0.97062290, 0.97128659, 0.97199202,
     &   0.97292328, 0.97376961, 0.97449893, 0.97507775, 0.97553355,
     &   0.97553086, 0.97446322, 0.97340572, 0.97230405, 0.97159654,
     &   0.97125995, 0.97154623, 0.97181469, 0.97175181, 0.97159773,
     &   0.97155780, 0.97158617, 0.97158653, 0.97171557, 0.97188687,
     &   0.97194237, 0.97192711, 0.97200227, 0.97226757, 0.97279227,
     &   0.97355753, 0.97428107, 0.97487253, 0.97547889, 0.97611600,
     &   0.97674596, 0.97741687, 0.97828656, 0.97930562, 0.98062414,
     &   0.98232734, 0.98423541, 0.98589283, 0.98691958, 0.98525810,
     &   0.97917867, 0.97058159, 0.96283317, 0.95578039, 0.95042574,
     &   0.94557405, 0.94165188, 0.93830580, 0.93548185, 0.93398291,
     &   0.93269598, 0.93182731, 0.93190914, 0.93201363, 0.93219817,
     &   0.93289006, 0.93412220, 0.93572336, 0.93736482, 0.93897969,
     &   0.94048822, 0.94172424, 0.94273597, 0.94357628, 0.94391644,
     &   0.94396669, 0.94414794, 0.94477522, 0.94550705, 0.94616675,
     &   0.94690812, 0.94768608, 0.94818455, 0.94814652, 0.94778836,
     &   0.94746476, 0.94713491, 0.94654983, 0.94586289, 0.94456959,
     &   0.94256669, 0.94042343, 0.93957496, 0.93947011, 0.93985379,
     &   0.94064212, 0.94145262, 0.94255424, 0.94415063, 0.94582981,
     &   0.94774300, 0.94988871, 0.95186234, 0.95366979, 0.95531738,
     &   0.95680910, 0.95814061, 0.95935321, 0.96045280, 0.96144563,
     &   0.96248215, 0.96356601, 0.96454626, 0.96542907, 0.96632856,
     &   0.96716005, 0.96739358, 0.96680117, 0.96556967, 0.96383041,
     &   0.96138871, 0.95818388, 0.95462382, 0.95078021, 0.94663745,
     &   0.94217253, 0.93726838, 0.93191928, 0.92530906, 0.91648728,
     &   0.90983582, 0.90079004, 0.89034688, 0.88391441, 0.88600159,
     &   0.89129645, 0.89290643, 0.89306104, 0.89225572, 0.89126897,
     &   0.89036882, 0.88922769, 0.89090604, 0.89528263, 0.90174878,
     &   0.90884250, 0.91003853, 0.90679759, 0.89854389, 0.89300901,
     &   0.88864732, 0.88472486, 0.88118607, 0.87856781, 0.87755728,
     &   0.87705725, 0.87678564, 0.87674576, 0.87653172, 0.87645632,
     &   0.87620217, 0.87614548, 0.87622708, 0.87600857, 0.87604618,
     &   0.87584239, 0.87550390, 0.87493777, 0.87397999, 0.87277144,
     &   0.87133700, 0.86964512, 0.86770469, 0.86550903, 0.86322612,
     &   0.86071503, 0.85800284, 0.85531050, 0.85243976, 0.84939176,
     &   0.84615868, 0.84273535, 0.83911794, 0.83530354, 0.83129001,
     &   0.82707661, 0.82266444, 0.81777358, 0.81262279, 0.80726683,
     &   0.80171233, 0.79596734, 0.79004180, 0.78394771, 0.77769876,
     &   0.77131057, 0.76480055, 0.75818777, 0.75149298, 0.74456006,
     &   0.73690206, 0.72919738, 0.72147554, 0.71376270, 0.70608771,
     &   0.69803602, 0.68719238, 0.67652953, 0.66610724, 0.65569562,
     &   0.64549351, 0.63576972, 0.62630212, 0.61687696, 0.60816962,
     &   0.60027361, 0.59420335, 0.58877230, 0.58396691, 0.57971168,
     &   0.57602555, 0.57289237, 0.57034045, 0.56831920, 0.56673527,
     &   0.56560093, 0.56493354, 0.56456232, 0.56445575, 0.56477553,
     &   0.56525010, 0.56584996, 0.56669158, 0.56762594, 0.56858295,
     &   0.56969118, 0.57089680, 0.57203317, 0.57309139, 0.57406372,
     &   0.57492793, 0.57567775, 0.57621098, 0.57662112, 0.57690614/
cccc
      if (icall .eq. 0) then
        wmin=log(wlmin)
        wmax=log(wlmax)
        wstep=(wmax-wmin)/(mxwv-1)
        icall=1
      endif

      fw=1+(log(wl)-wmin)/wstep
      fw=min(max(fw,one),float(mxwv)-eps)
      iw=int(fw)
      fw=fw-iw

      fr=1.+((log(abs(re)))/log(2.)-1.)*2
      fr=min(max(fr,one),float(mre)-eps)
      ir=int(fr)
      fr=fr-ir
      if(re .lt. 0.) then 
        qc=qqi(iw  ,ir  )*(1.-fw)*(1.-fr) + qqi(iw+1,ir  )*fw*(1.-fr)+
     &     qqi(iw  ,ir+1)*(1.-fw)*fr      + qqi(iw+1,ir+1)*fw*fr

        wc=wwi(iw  ,ir  )*(1.-fw)*(1.-fr) + wwi(iw+1,ir  )*fw*(1.-fr)+
     &     wwi(iw  ,ir+1)*(1.-fw)*fr      + wwi(iw+1,ir+1)*fw*fr

        gc=ggi(iw  ,ir  )*(1.-fw)*(1.-fr) + ggi(iw+1,ir  )*fw*(1.-fr)+
     &     ggi(iw  ,ir+1)*(1.-fw)*fr      + ggi(iw+1,ir+1)*fw*fr

      else
        qc=qq(iw  ,ir  )*(1.-fw)*(1.-fr) + qq(iw+1,ir  )*fw*(1.-fr)+
     &     qq(iw  ,ir+1)*(1.-fw)*fr      + qq(iw+1,ir+1)*fw*fr

        wc=ww(iw  ,ir  )*(1.-fw)*(1.-fr) + ww(iw+1,ir  )*fw*(1.-fr)+
     &     ww(iw  ,ir+1)*(1.-fw)*fr      + ww(iw+1,ir+1)*fw*fr

        gc=gg(iw  ,ir  )*(1.-fw)*(1.-fr) + gg(iw+1,ir  )*fw*(1.-fr)+
     &     gg(iw  ,ir+1)*(1.-fw)*fr      + gg(iw+1,ir+1)*fw*fr

      endif
      return
      end

c=======================================================================





