"use strict";
// Copyright (c) Martin Renou
// Distributed under the terms of the Modified BSD License.
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const base_1 = require("@jupyter-widgets/base");
const canvas_1 = require("roughjs/bin/canvas");
const version_1 = require("./version");
const utils_1 = require("./utils");
function getContext(canvas) {
    const context = canvas.getContext("2d");
    if (context === null) {
        throw 'Could not create 2d context.';
    }
    return context;
}
function serializeImageData(array) {
    return new DataView(array.buffer.slice(0));
}
function deserializeImageData(dataview) {
    if (dataview === null) {
        return null;
    }
    return new Uint8ClampedArray(dataview.buffer);
}
class Path2DModel extends base_1.WidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: Path2DModel.model_name, _model_module: Path2DModel.model_module, _model_module_version: Path2DModel.model_module_version, _view_name: Path2DModel.view_name, _view_module: Path2DModel.view_module, _view_module_version: Path2DModel.view_module_version, value: '' });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.value = new Path2D(this.get('value'));
    }
}
exports.Path2DModel = Path2DModel;
Path2DModel.model_name = 'Path2DModel';
Path2DModel.model_module = version_1.MODULE_NAME;
Path2DModel.model_module_version = version_1.MODULE_VERSION;
Path2DModel.view_name = 'Path2DView';
Path2DModel.view_module = version_1.MODULE_NAME;
Path2DModel.view_module_version = version_1.MODULE_VERSION;
class CanvasModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: CanvasModel.model_name, _model_module: CanvasModel.model_module, _model_module_version: CanvasModel.model_module_version, _view_name: CanvasModel.view_name, _view_module: CanvasModel.view_module, _view_module_version: CanvasModel.view_module_version, width: 700, height: 500, sync_image_data: false, image_data: null });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.canvas = document.createElement('canvas');
        this.ctx = getContext(this.canvas);
        this.resizeCanvas();
        this.drawImageData();
        this.on_some_change(['width', 'height'], this.resizeCanvas, this);
        this.on('change:sync_image_data', this.syncImageData.bind(this));
        this.on('msg:custom', this.onCommand.bind(this));
        this.send({ event: 'client_ready' }, {});
    }
    drawImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.get('image_data') !== null) {
                const img = yield utils_1.fromBytes(this.get('image_data'));
                this.ctx.drawImage(img, 0, 0);
                this.trigger('new-frame');
            }
        });
    }
    onCommand(command, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.processCommand(command, buffers);
            this.forEachView((view) => {
                view.updateCanvas();
            });
            this.trigger('new-frame');
            this.syncImageData();
        });
    }
    processCommand(command, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            if (command instanceof Array) {
                let remainingBuffers = buffers;
                for (const subcommand of command) {
                    let subbuffers = [];
                    if (subcommand.n_buffers) {
                        subbuffers = remainingBuffers.slice(0, subcommand.n_buffers);
                        remainingBuffers = remainingBuffers.slice(subcommand.n_buffers);
                    }
                    yield this.processCommand(subcommand, subbuffers);
                }
                return;
            }
            const args = command.args;
            switch (command.name) {
                case 'fillRect':
                    this.fillRect(args[0], args[1], args[2], args[3]);
                    break;
                case 'strokeRect':
                    this.strokeRect(args[0], args[1], args[2], args[3]);
                    break;
                case 'fillRects':
                    this.drawRects(command.args, buffers, this.fillRect.bind(this));
                    break;
                case 'strokeRects':
                    this.drawRects(command.args, buffers, this.strokeRect.bind(this));
                    break;
                case 'fillArc':
                    this.fillArc(args[0], args[1], args[2], args[3], args[4], args[5]);
                    break;
                case 'strokeArc':
                    this.strokeArc(args[0], args[1], args[2], args[3], args[4], args[5]);
                    break;
                case 'fillArcs':
                    this.drawArcs(command.args, buffers, this.fillArc.bind(this));
                    break;
                case 'strokeArcs':
                    this.drawArcs(command.args, buffers, this.strokeArc.bind(this));
                    break;
                case 'fillCircle':
                    this.fillCircle(args[0], args[1], args[2]);
                    break;
                case 'strokeCircle':
                    this.strokeCircle(args[0], args[1], args[2]);
                    break;
                case 'fillCircles':
                    this.drawCircles(command.args, buffers, this.fillCircle.bind(this));
                    break;
                case 'strokeCircles':
                    this.drawCircles(command.args, buffers, this.strokeCircle.bind(this));
                    break;
                case 'strokeLine':
                    this.strokeLine(command.args, buffers);
                    break;
                case 'fillPath':
                    yield this.fillPath(command.args, buffers);
                    break;
                case 'drawImage':
                    yield this.drawImage(command.args, buffers);
                    break;
                case 'putImageData':
                    this.putImageData(command.args, buffers);
                    break;
                case 'set':
                    this.setAttr(command.attr, command.value);
                    break;
                case 'clear':
                    this.clearCanvas();
                    break;
                default:
                    this.executeCommand(command.name, command.args);
                    break;
            }
        });
    }
    fillRect(x, y, width, height) {
        this.ctx.fillRect(x, y, width, height);
    }
    strokeRect(x, y, width, height) {
        this.ctx.strokeRect(x, y, width, height);
    }
    drawRects(args, buffers, callback) {
        const x = utils_1.getArg(args[0], buffers);
        const y = utils_1.getArg(args[1], buffers);
        const width = utils_1.getArg(args[2], buffers);
        const height = utils_1.getArg(args[3], buffers);
        const numberRects = Math.min(x.length, y.length, width.length, height.length);
        for (let idx = 0; idx < numberRects; ++idx) {
            callback(x.getItem(idx), y.getItem(idx), width.getItem(idx), height.getItem(idx));
        }
    }
    fillArc(x, y, radius, startAngle, endAngle, anticlockwise) {
        this.ctx.beginPath();
        this.ctx.moveTo(x, y); // Move to center
        this.ctx.lineTo(x + radius * Math.cos(startAngle), y + radius * Math.sin(startAngle)); // Line to beginning of the arc
        this.ctx.arc(x, y, radius, startAngle, endAngle, anticlockwise);
        this.ctx.lineTo(x, y); // Line to center
        this.ctx.fill();
        this.ctx.closePath();
    }
    strokeArc(x, y, radius, startAngle, endAngle, anticlockwise) {
        this.ctx.beginPath();
        this.ctx.arc(x, y, radius, startAngle, endAngle, anticlockwise);
        this.ctx.stroke();
        this.ctx.closePath();
    }
    drawArcs(args, buffers, callback) {
        const x = utils_1.getArg(args[0], buffers);
        const y = utils_1.getArg(args[1], buffers);
        const radius = utils_1.getArg(args[2], buffers);
        const startAngle = utils_1.getArg(args[3], buffers);
        const endAngle = utils_1.getArg(args[4], buffers);
        const anticlockwise = utils_1.getArg(args[5], buffers);
        const numberArcs = Math.min(x.length, y.length, radius.length, startAngle.length, endAngle.length);
        for (let idx = 0; idx < numberArcs; ++idx) {
            callback(x.getItem(idx), y.getItem(idx), radius.getItem(idx), startAngle.getItem(idx), endAngle.getItem(idx), anticlockwise.getItem(idx));
        }
    }
    fillCircle(x, y, radius) {
        this.ctx.beginPath();
        this.ctx.arc(x, y, radius, 0, 2 * Math.PI);
        this.ctx.fill();
        this.ctx.closePath();
    }
    strokeCircle(x, y, radius) {
        this.ctx.beginPath();
        this.ctx.arc(x, y, radius, 0, 2 * Math.PI);
        this.ctx.stroke();
        this.ctx.closePath();
    }
    drawCircles(args, buffers, callback) {
        const x = utils_1.getArg(args[0], buffers);
        const y = utils_1.getArg(args[1], buffers);
        const radius = utils_1.getArg(args[2], buffers);
        const numberCircles = Math.min(x.length, y.length, radius.length);
        for (let idx = 0; idx < numberCircles; ++idx) {
            callback(x.getItem(idx), y.getItem(idx), radius.getItem(idx));
        }
    }
    strokeLine(args, buffers) {
        this.ctx.beginPath();
        this.ctx.moveTo(args[0], args[1]);
        this.ctx.lineTo(args[2], args[3]);
        this.ctx.stroke();
        this.ctx.closePath();
    }
    fillPath(args, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            const [serializedPath] = args;
            const path = yield base_1.unpack_models(serializedPath, this.widget_manager);
            this.ctx.fill(path.value);
        });
    }
    drawImage(args, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            const [serializedImage, x, y, width, height] = args;
            const image = yield base_1.unpack_models(serializedImage, this.widget_manager);
            if (image instanceof CanvasModel || image instanceof MultiCanvasModel) {
                this._drawImage(image.canvas, x, y, width, height);
                return;
            }
            if (image.get('_model_name') == 'ImageModel') {
                // Create the image manually instead of creating an ImageView
                let url;
                const format = image.get('format');
                const value = image.get('value');
                if (format !== 'url') {
                    const blob = new Blob([value], { type: `image/${format}` });
                    url = URL.createObjectURL(blob);
                }
                else {
                    url = (new TextDecoder('utf-8')).decode(value.buffer);
                }
                const img = new Image();
                return new Promise((resolve) => {
                    img.onload = () => {
                        this._drawImage(img, x, y, width, height);
                        resolve();
                    };
                    img.src = url;
                });
            }
        });
    }
    _drawImage(image, x, y, width, height) {
        if (width === undefined || height === undefined) {
            this.ctx.drawImage(image, x, y);
        }
        else {
            this.ctx.drawImage(image, x, y, width, height);
        }
    }
    putImageData(args, buffers) {
        const [bufferMetadata, dx, dy] = args;
        const width = bufferMetadata.shape[1];
        const height = bufferMetadata.shape[0];
        const data = new Uint8ClampedArray(buffers[0].buffer);
        const imageData = new ImageData(data, width, height);
        // Draw on a temporary off-screen canvas. This is a workaround for `putImageData` to support transparency.
        const offscreenCanvas = document.createElement('canvas');
        offscreenCanvas.width = width;
        offscreenCanvas.height = height;
        getContext(offscreenCanvas).putImageData(imageData, 0, 0);
        this.ctx.drawImage(offscreenCanvas, dx, dy);
    }
    setAttr(attr, value) {
        this.ctx[attr] = value;
    }
    clearCanvas() {
        this.forEachView((view) => {
            view.clear();
        });
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }
    executeCommand(name, args = []) {
        this.ctx[name](...args);
    }
    forEachView(callback) {
        for (const view_id in this.views) {
            this.views[view_id].then((view) => {
                callback(view);
            });
        }
    }
    resizeCanvas() {
        this.canvas.setAttribute('width', this.get('width'));
        this.canvas.setAttribute('height', this.get('height'));
    }
    syncImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.get('sync_image_data')) {
                return;
            }
            const bytes = yield utils_1.toBytes(this.canvas);
            this.set('image_data', bytes);
            this.save_changes();
        });
    }
}
exports.CanvasModel = CanvasModel;
CanvasModel.serializers = Object.assign(Object.assign({}, base_1.DOMWidgetModel.serializers), { image_data: {
        serialize: serializeImageData,
        deserialize: deserializeImageData
    } });
CanvasModel.model_name = 'CanvasModel';
CanvasModel.model_module = version_1.MODULE_NAME;
CanvasModel.model_module_version = version_1.MODULE_VERSION;
CanvasModel.view_name = 'CanvasView';
CanvasModel.view_module = version_1.MODULE_NAME;
CanvasModel.view_module_version = version_1.MODULE_VERSION;
class RoughCanvasModel extends CanvasModel {
    constructor() {
        super(...arguments);
        this.roughFillStyle = 'hachure';
        this.roughness = 1.;
        this.bowing = 1.;
    }
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: RoughCanvasModel.model_name });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.roughCanvas = new canvas_1.RoughCanvas(this.canvas);
    }
    fillRect(x, y, width, height) {
        this.roughCanvas.rectangle(x, y, width, height, this.getRoughFillStyle());
    }
    strokeRect(x, y, width, height) {
        this.roughCanvas.rectangle(x, y, width, height, this.getRoughStrokeStyle());
    }
    fillCircle(x, y, radius) {
        this.roughCanvas.circle(x, y, radius, this.getRoughFillStyle());
    }
    strokeCircle(x, y, radius) {
        this.roughCanvas.circle(x, y, radius, this.getRoughStrokeStyle());
    }
    strokeLine(args, buffers) {
        this.roughCanvas.line(args[0], args[1], args[2], args[3], this.getRoughStrokeStyle());
    }
    fillPath(args, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            const [serializedPath] = args;
            const path = yield base_1.unpack_models(serializedPath, this.widget_manager);
            this.roughCanvas.path(path.get('value'), this.getRoughFillStyle());
        });
    }
    fillArc(x, y, radius, startAngle, endAngle, anticlockwise) {
        const ellipseSize = 2. * radius;
        // The following is needed because roughjs does not allow a clockwise draw
        const start = anticlockwise ? endAngle : startAngle;
        const end = anticlockwise ? startAngle + 2. * Math.PI : endAngle;
        this.roughCanvas.arc(x, y, ellipseSize, ellipseSize, start, end, true, this.getRoughFillStyle());
    }
    strokeArc(x, y, radius, startAngle, endAngle, anticlockwise) {
        const ellipseSize = 2. * radius;
        // The following is needed because roughjs does not allow a clockwise draw
        const start = anticlockwise ? endAngle : startAngle;
        const end = anticlockwise ? startAngle + 2. * Math.PI : endAngle;
        this.roughCanvas.arc(x, y, ellipseSize, ellipseSize, start, end, false, this.getRoughStrokeStyle());
    }
    setAttr(attr, value) {
        switch (attr) {
            case 'roughFillStyle':
                this.roughFillStyle = value;
                break;
            case 'roughness':
                this.roughness = value;
                break;
            case 'bowing':
                this.bowing = value;
                break;
            default:
                super.setAttr(attr, value);
                break;
        }
    }
    getRoughFillStyle() {
        const fill = this.ctx.fillStyle;
        const lineWidth = this.ctx.lineWidth;
        return {
            fill,
            fillStyle: this.roughFillStyle,
            fillWeight: lineWidth / 2.,
            hachureGap: lineWidth * 4.,
            curveStepCount: 18,
            strokeWidth: 0.001,
            roughness: this.roughness,
            bowing: this.bowing,
        };
    }
    getRoughStrokeStyle() {
        const stroke = this.ctx.strokeStyle;
        const lineWidth = this.ctx.lineWidth;
        return {
            stroke,
            strokeWidth: lineWidth,
            roughness: this.roughness,
            bowing: this.bowing,
            curveStepCount: 18,
        };
    }
}
exports.RoughCanvasModel = RoughCanvasModel;
RoughCanvasModel.model_name = 'RoughCanvasModel';
class CanvasView extends base_1.DOMWidgetView {
    render() {
        this.ctx = getContext(this.el);
        this.resizeCanvas();
        this.model.on_some_change(['width', 'height'], this.resizeCanvas, this);
        this.el.addEventListener('mousemove', { handleEvent: this.onMouseMove.bind(this) });
        this.el.addEventListener('mousedown', { handleEvent: this.onMouseDown.bind(this) });
        this.el.addEventListener('mouseup', { handleEvent: this.onMouseUp.bind(this) });
        this.el.addEventListener('mouseout', { handleEvent: this.onMouseOut.bind(this) });
        this.el.addEventListener('touchstart', { handleEvent: this.onTouchStart.bind(this) });
        this.el.addEventListener('touchend', { handleEvent: this.onTouchEnd.bind(this) });
        this.el.addEventListener('touchmove', { handleEvent: this.onTouchMove.bind(this) });
        this.el.addEventListener('touchcancel', { handleEvent: this.onTouchCancel.bind(this) });
        this.updateCanvas();
    }
    clear() {
        this.ctx.clearRect(0, 0, this.el.width, this.el.height);
    }
    updateCanvas() {
        this.clear();
        this.ctx.drawImage(this.model.canvas, 0, 0);
    }
    resizeCanvas() {
        this.el.setAttribute('width', this.model.get('width'));
        this.el.setAttribute('height', this.model.get('height'));
    }
    onMouseMove(event) {
        this.model.send(Object.assign({ event: 'mouse_move' }, this.getCoordinates(event)), {});
    }
    onMouseDown(event) {
        this.model.send(Object.assign({ event: 'mouse_down' }, this.getCoordinates(event)), {});
    }
    onMouseUp(event) {
        this.model.send(Object.assign({ event: 'mouse_up' }, this.getCoordinates(event)), {});
    }
    onMouseOut(event) {
        this.model.send(Object.assign({ event: 'mouse_out' }, this.getCoordinates(event)), {});
    }
    onTouchStart(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_start', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchEnd(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_end', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchMove(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_move', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchCancel(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_cancel', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    getCoordinates(event) {
        const rect = this.el.getBoundingClientRect();
        const x = this.el.width * (event.clientX - rect.left) / rect.width;
        const y = this.el.height * (event.clientY - rect.top) / rect.height;
        return { x, y };
    }
    get tagName() {
        return 'canvas';
    }
}
exports.CanvasView = CanvasView;
class MultiCanvasModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: MultiCanvasModel.model_name, _model_module: MultiCanvasModel.model_module, _model_module_version: MultiCanvasModel.model_module_version, _view_name: MultiCanvasModel.view_name, _view_module: MultiCanvasModel.view_module, _view_module_version: MultiCanvasModel.view_module_version, _canvases: [], sync_image_data: false, image_data: null, width: 700, height: 500 });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.canvas = document.createElement('canvas');
        this.ctx = getContext(this.canvas);
        this.resizeCanvas();
        this.on_some_change(['width', 'height'], this.resizeCanvas, this);
        this.on('change:_canvases', this.updateCanvasModels.bind(this));
        this.on('change:sync_image_data', this.syncImageData.bind(this));
        this.updateCanvasModels();
    }
    get canvasModels() {
        return this.get('_canvases');
    }
    updateCanvasModels() {
        // TODO: Remove old listeners
        for (const canvasModel of this.canvasModels) {
            canvasModel.on('new-frame', this.updateCanvas, this);
        }
        this.updateCanvas();
    }
    updateCanvas() {
        this.ctx.clearRect(0, 0, this.get('width'), this.get('height'));
        for (const canvasModel of this.canvasModels) {
            this.ctx.drawImage(canvasModel.canvas, 0, 0);
        }
        this.forEachView((view) => {
            view.updateCanvas();
        });
        this.syncImageData();
    }
    resizeCanvas() {
        this.canvas.setAttribute('width', this.get('width'));
        this.canvas.setAttribute('height', this.get('height'));
    }
    syncImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.get('sync_image_data')) {
                return;
            }
            const bytes = yield utils_1.toBytes(this.canvas);
            this.set('image_data', bytes);
            this.save_changes();
        });
    }
    forEachView(callback) {
        for (const view_id in this.views) {
            this.views[view_id].then((view) => {
                callback(view);
            });
        }
    }
}
exports.MultiCanvasModel = MultiCanvasModel;
MultiCanvasModel.serializers = Object.assign(Object.assign({}, base_1.DOMWidgetModel.serializers), { _canvases: { deserialize: base_1.unpack_models }, image_data: { serialize: (bytes) => {
            return new DataView(bytes.buffer.slice(0));
        } } });
MultiCanvasModel.model_name = 'MultiCanvasModel';
MultiCanvasModel.model_module = version_1.MODULE_NAME;
MultiCanvasModel.model_module_version = version_1.MODULE_VERSION;
MultiCanvasModel.view_name = 'MultiCanvasView';
MultiCanvasModel.view_module = version_1.MODULE_NAME;
MultiCanvasModel.view_module_version = version_1.MODULE_VERSION;
class MultiCanvasView extends CanvasView {
}
exports.MultiCanvasView = MultiCanvasView;
//# sourceMappingURL=widget.js.map