"""
Module to deal with malware samples

"""
from .configuration import (
    BASE_MALWARES_URL, BASE_MALWARES_UPLOAD_URL
)

from .core import BASEModel, BluelivRequest


class BluelivMalware(BASEModel):
    """
    Model to store Malware details.

    """

    # pylint: disable=too-many-instance-attributes
    # pylint: disable=too-few-public-methods
    # All attributes are neccessary.
    # Is an object without actions. Actions to be implemented in next versions

    malware_id = None
    file = dict()
    status = None
    upload_date = None
    malicious = None
    domains = list()
    hosts = list()
    detected_antivirus = None
    total_antivirus = None

    def __init__(self, **kwargs):
        if 'malware_id' in kwargs:
            self.malware_id = kwargs.get('malware_id', None)

        if 'status' in kwargs:
            self.status = kwargs.get('status', None)

        if 'upload_date' in kwargs:
            self.upload_date = kwargs.get('upload_date', None)

        if 'malicious' in kwargs:
            self.malicious = kwargs.get('malicious', None)

        if 'detected_antivirus' in kwargs:
            self.detected_antivirus = kwargs.get('detected_antivirus', None)

        if 'total_antivirus' in kwargs:
            self.total_antivirus = kwargs.get('total_antivirus', None)

        if 'file' in kwargs:
            self.file = kwargs.get('file', {})

        if 'domains' in kwargs:
            self.domains = kwargs.get('domains', [])

        if 'hosts' in kwargs:
            self.hosts = kwargs.get('hosts', [])

        super().__init__()


class BluelivUploadResponse(BASEModel):
    """
    Model to store the response we receive to a Malware upload request.
    """

    # pylint: disable=too-few-public-methods
    # Model to manage a remote response

    md5 = None
    sha256 = None
    platform_type = None
    file_size = 0
    upload_status = None
    fileName = None

    def __init__(self):
        self.md5 = None
        self.sha256 = None
        self.platform_type = None
        self.file_size = 0
        self.upload_status = None
        self.file_name = None

        super().__init__()


class MalwaresRequest(BluelivRequest):
    """
    Model to deal with the requests about Malware samples. Here you will be
    able to download, upload and see several details.

    """
    _malwares_upload_url = None
    page = 0
    page_size = 0

    def __init__(self, **kwargs):
        self._category = 'malwares'
        self._base_url = '/malwares'
        self._malwares_upload_url = '/upload'
        self.page = 0
        self.page_size = 0

        if 'token' in kwargs:
            self._custom_token = kwargs.get('token', None)

        if 'base_url' in kwargs:
            self._base_url = kwargs.get('base_url', '/malwares')
        else:
            self._base_url = BASE_MALWARES_URL

        if 'category' in kwargs:
            self._category = kwargs.get('category', 'malwares')

        if 'upload_url' in kwargs:
            self._malwares_upload_url = kwargs.get('upload_url', '/upload')
        else:
            self._malwares_upload_url = BASE_MALWARES_UPLOAD_URL

        if 'page' in kwargs:
            self.page = kwargs.get('page', 0)

        if 'page_size' in kwargs:
            self.page_size = kwargs.get('page_size', 0)

        if 'limit' in kwargs:
            self.limit = kwargs.get('limit', None)

        if 'since_id' in kwargs:
            self.is_text = kwargs.get('since_id', False)

        super().__init__(token=self._custom_token,
                         base_url=self._base_url,
                         category=self._category,
                         limit=self.limit,
                         since_id=self.since_id)

    def list(self, page: int = 0, page_size: int = 0):
        """
        List malware items with pagination.

        :param page: the page number we want to list.
        :param page_size: the maximum number of pages.
        :return: dict, list or JSON.

        """
        params = {}

        if page != 0:
            params['page'] = page
        else:
            if self.page != 0:
                params['page'] = self.page

        if page_size != 0:
            params['pageSize'] = page_size
        else:
            if self.page_size != 0:
                params['pageSize'] = self.page_size

        results = self.request(resource=self._base_url,
                               params=params)

        return results

    def show(self, malware_id: str):
        """
        Show details about an specific malware sample identified by the id.
        :param malware_id: the id for the sample.
        :return: dict, list or JSON.

        """
        resource = '%s/%s' % (self._base_url,
                              malware_id)
        results = self.request(resource=resource)

        return results

    def upload(self, filename: str):
        """
        A method to upload malware samples to the Community.

        :param filename: the filename with the malware sample.
        :return: dict, list or JSON.
        """
        resource = '%s%s' % (self._base_url,
                             self._malwares_upload_url)
        file_object = None
        try:
            file_object = open(filename, 'rb')
        except Exception as exception:
            raise Exception('Error with [%s]: [%s]' % (filename,
                                                       exception))

        file = {'file': (filename, file_object)}
        results = self.request(resource=resource,
                               files=file)

        return results
