"use strict";
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'add metrics to graphs on either axis'(test) {
        // WHEN
        const stack = new cdk_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' })
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' })
            ]
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'left', period: 300, stat: 'Average' }],
                        ['CDK', 'Tast', { yAxis: 'right', period: 300, stat: 'Average' }]
                    ],
                    annotations: { horizontal: [] },
                    yAxis: {}
                }
            }]);
        test.done();
    },
    'label and color are respected in constructor'(test) {
        // WHEN
        const stack = new cdk_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test', label: 'MyMetric', color: '000000' })],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'left', period: 300, stat: 'Average', label: 'MyMetric', color: '000000' }],
                    ],
                    annotations: { horizontal: [] },
                    yAxis: {}
                }
            }]);
        test.done();
    },
    'singlevalue widget'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric]
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'left', period: 300, stat: 'Average' }],
                    ]
                }
            }]);
        test.done();
    },
    'alarm widget'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const alarm = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }).createAlarm(stack, 'Alarm', {
            evaluationPeriods: 2,
            threshold: 1000
        });
        // WHEN
        const widget = new lib_1.AlarmWidget({
            alarm,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    annotations: {
                        alarms: [{ 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] }]
                    },
                    yAxis: {}
                }
            }]);
        test.done();
    },
    'add annotations to graph'(test) {
        // WHEN
        const stack = new cdk_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' })
            ],
            leftAnnotations: [{
                    value: 1000,
                    color: '667788',
                    fill: lib_1.Shading.Below,
                    label: 'this is the annotation',
                }]
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'left', period: 300, stat: 'Average' }],
                    ],
                    annotations: { horizontal: [{
                                yAxis: 'left',
                                value: 1000,
                                color: '667788',
                                fill: 'below',
                                label: 'this is the annotation',
                            }] },
                    yAxis: {}
                }
            }]);
        test.done();
    },
    'convert alarm to annotation'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        const alarm = metric.createAlarm(stack, 'Alarm', {
            evaluationPeriods: 2,
            threshold: 1000
        });
        // WHEN
        const widget = new lib_1.GraphWidget({
            right: [metric],
            rightAnnotations: [alarm.toAnnotation()]
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'right', period: 300, stat: 'Average' }],
                    ],
                    annotations: {
                        horizontal: [{
                                yAxis: 'right',
                                value: 1000,
                                label: 'Test >= 1000 for 2 datapoints within 10 minutes',
                            }]
                    },
                    yAxis: {}
                }
            }]);
        test.done();
    },
    'add yAxis to graph'(test) {
        // WHEN
        const stack = new cdk_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' })
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' })
            ],
            leftYAxis: ({
                label: "Left yAxis",
                max: 100
            }),
            rightYAxis: ({
                label: "Right yAxis",
                min: 10,
                showUnits: false
            })
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'left', period: 300, stat: 'Average' }],
                        ['CDK', 'Tast', { yAxis: 'right', period: 300, stat: 'Average' }]
                    ],
                    annotations: { horizontal: [] },
                    yAxis: {
                        left: { label: "Left yAxis", max: 100 },
                        right: { label: "Right yAxis", min: 10, showUnits: false }
                    }
                }
            }]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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