"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
/**
 * Returns a property predicate that checks that the given Dashboard has the indicated widgets
 */
function thatHasWidgets(widgets) {
    return (props) => {
        try {
            const actualWidgets = JSON.parse(props.DashboardBody).widgets;
            return assert_1.isSuperObject(actualWidgets, widgets);
        }
        catch (e) {
            // tslint:disable-next-line:no-console
            console.error('Error parsing', props);
            throw e;
        }
    };
}
module.exports = {
    'widgets in different adds are laid out underneath each other'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 10,
            height: 2,
            markdown: "first"
        }));
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 1,
            height: 4,
            markdown: "second"
        }));
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 4,
            height: 1,
            markdown: "third"
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', thatHasWidgets([
            { type: 'text', width: 10, height: 2, x: 0, y: 0, properties: { markdown: 'first' } },
            { type: 'text', width: 1, height: 4, x: 0, y: 2, properties: { markdown: 'second' } },
            { type: 'text', width: 4, height: 1, x: 0, y: 6, properties: { markdown: 'third' } },
        ])));
        test.done();
    },
    'widgets in same add are laid out next to each other'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 10,
            height: 2,
            markdown: "first"
        }), new lib_1.TextWidget({
            width: 1,
            height: 4,
            markdown: "second"
        }), new lib_1.TextWidget({
            width: 4,
            height: 1,
            markdown: "third"
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', thatHasWidgets([
            { type: 'text', width: 10, height: 2, x: 0, y: 0, properties: { markdown: 'first' } },
            { type: 'text', width: 1, height: 4, x: 10, y: 0, properties: { markdown: 'second' } },
            { type: 'text', width: 4, height: 1, x: 11, y: 0, properties: { markdown: 'third' } },
        ])));
        test.done();
    },
    'tokens in widgets are retained'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.GraphWidget({ width: 1, height: 1 }) // GraphWidget has internal reference to current region
        );
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardBody: { "Fn::Join": ["", [
                        "{\"widgets\":[{\"type\":\"metric\",\"width\":1,\"height\":1,\"x\":0,\"y\":0,\"properties\":{\"view\":\"timeSeries\",\"region\":\"",
                        { Ref: "AWS::Region" },
                        "\",\"metrics\":[],\"annotations\":{\"horizontal\":[]},\"yAxis\":{}}}]}"
                    ]] }
        }));
        test.done();
    },
    'dashboard body includes non-widget fields'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash', {
            start: '-9H',
            end: '2018-12-17T06:00:00.000Z',
            periodOverride: lib_1.PeriodOverride.Inherit
        });
        // WHEN
        dashboard.addWidgets(new lib_1.GraphWidget({ width: 1, height: 1 }) // GraphWidget has internal reference to current region
        );
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardBody: { "Fn::Join": ["", [
                        "{\"start\":\"-9H\",\"end\":\"2018-12-17T06:00:00.000Z\",\"periodOverride\":\"inherit\",\
\"widgets\":[{\"type\":\"metric\",\"width\":1,\"height\":1,\"x\":0,\"y\":0,\"properties\":{\"view\":\"timeSeries\",\"region\":\"",
                        { Ref: "AWS::Region" },
                        "\",\"metrics\":[],\"annotations\":{\"horizontal\":[]},\"yAxis\":{}}}]}"
                    ]] }
        }));
        test.done();
    },
    'DashboardName is set when provided'(test) {
        // GIVEN
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        // WHEN
        new lib_1.Dashboard(stack, 'MyDashboard', {
            dashboardName: 'MyCustomDashboardName'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardName: 'MyCustomDashboardName'
        }));
        test.done();
    },
    'DashboardName is not generated if not provided'(test) {
        // GIVEN
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        // WHEN
        new lib_1.Dashboard(stack, 'MyDashboard');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {}));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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