"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const testMetric = new lib_1.Metric({
    namespace: 'CDK/Test',
    metricName: 'Metric',
});
class TestAlarmAction {
    constructor(arn) {
        this.arn = arn;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.arn };
    }
}
module.exports = {
    'can make simple alarm'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: "GreaterThanOrEqualToThreshold",
            EvaluationPeriods: 3,
            MetricName: "Metric",
            Namespace: "CDK/Test",
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override metric period in Alarm'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            periodSec: 600,
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: "GreaterThanOrEqualToThreshold",
            EvaluationPeriods: 3,
            MetricName: "Metric",
            Namespace: "CDK/Test",
            Period: 600,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override statistic Alarm'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            statistic: 'max',
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: "GreaterThanOrEqualToThreshold",
            EvaluationPeriods: 3,
            MetricName: "Metric",
            Namespace: "CDK/Test",
            Period: 300,
            Statistic: 'Maximum',
            Threshold: 1000,
        }));
        test.done();
    },
    'can set DatapointsToAlarm'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
            datapointsToAlarm: 2,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: "GreaterThanOrEqualToThreshold",
            EvaluationPeriods: 3,
            DatapointsToAlarm: 2,
            MetricName: "Metric",
            Namespace: "CDK/Test",
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'can add actions to alarms'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const alarm = new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 2
        });
        alarm.addAlarmAction(new TestAlarmAction('A'));
        alarm.addInsufficientDataAction(new TestAlarmAction('B'));
        alarm.addOkAction(new TestAlarmAction('C'));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            AlarmActions: ['A'],
            InsufficientDataActions: ['B'],
            OKActions: ['C'],
        }));
        test.done();
    },
    'can make alarm directly from metric'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'min',
            periodSec: 10,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: "GreaterThanOrEqualToThreshold",
            EvaluationPeriods: 2,
            MetricName: "Metric",
            Namespace: "CDK/Test",
            Period: 10,
            Statistic: 'Minimum',
            Threshold: 1000,
        }));
        test.done();
    },
    'can use percentile string to make alarm'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'p99.9'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ExtendedStatistic: 'p99.9',
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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