"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metric_types_1 = require("./metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.Average,
        avg: metric_types_1.Statistic.Average,
        minimum: metric_types_1.Statistic.Minimum,
        min: metric_types_1.Statistic.Minimum,
        maximum: metric_types_1.Statistic.Maximum,
        max: metric_types_1.Statistic.Maximum,
        samplecount: metric_types_1.Statistic.SampleCount,
        n: metric_types_1.Statistic.SampleCount,
        sum: metric_types_1.Statistic.Sum,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat]
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1])
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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