"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const alarm_1 = require("./alarm");
const util_statistic_1 = require("./util.statistic");
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*']
        });
    }
    constructor(props) {
        if (props.periodSec !== undefined
            && props.periodSec !== 1 && props.periodSec !== 5 && props.periodSec !== 10 && props.periodSec !== 30
            && props.periodSec % 60 !== 0) {
            throw new Error("'periodSec' must be 1, 5, 10, 30, or a multiple of 60");
        }
        this.dimensions = props.dimensions;
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        this.periodSec = props.periodSec !== undefined ? props.periodSec : 300;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = util_statistic_1.normalizeStatistic(props.statistic || "Average");
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        return new Metric({
            dimensions: ifUndefined(props.dimensions, this.dimensions),
            namespace: this.namespace,
            metricName: this.metricName,
            periodSec: ifUndefined(props.periodSec, this.periodSec),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color)
        });
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                periodSec: props.periodSec,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toAlarmConfig() {
        const stat = util_statistic_1.parseStatistic(this.statistic);
        const dims = this.dimensionsAsList();
        return {
            dimensions: dims.length > 0 ? dims : undefined,
            namespace: this.namespace,
            metricName: this.metricName,
            period: this.periodSec,
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit
        };
    }
    toGraphConfig() {
        return {
            dimensions: this.dimensionsAsList(),
            namespace: this.namespace,
            metricName: this.metricName,
            period: this.periodSec,
            statistic: this.statistic,
            unit: this.unit,
            color: this.color,
            label: this.label,
        };
    }
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).map(key => ({ name: key, value: dims[key] }));
        return list;
    }
}
exports.Metric = Metric;
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
//# sourceMappingURL=data:application/json;base64,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