import iam = require('@aws-cdk/aws-iam');
import cdk = require('@aws-cdk/cdk');
import { Alarm, ComparisonOperator, TreatMissingData } from './alarm';
import { IMetric, MetricAlarmConfig, MetricGraphConfig, Unit } from './metric-types';
export declare type DimensionHash = {
    [dim: string]: any;
};
/**
 * Options shared by most methods accepting metric options
 */
export interface CommonMetricOptions {
    /**
     * The period over which the specified statistic is applied.
     *
     * Specify time in seconds, in multiples of 60.
     *
     * @default 300
     */
    readonly periodSec?: number;
    /**
     * What function to use for aggregating.
     *
     * Can be one of the following:
     *
     * - "Minimum" | "min"
     * - "Maximum" | "max"
     * - "Average" | "avg"
     * - "Sum" | "sum"
     * - "SampleCount | "n"
     * - "pNN.NN"
     *
     * @default Average
     */
    readonly statistic?: string;
    /**
     * Dimensions of the metric
     *
     * @default - No dimensions.
     */
    readonly dimensions?: DimensionHash;
    /**
     * Unit for the metric that is associated with the alarm
     */
    readonly unit?: Unit;
    /**
     * Label for this metric when added to a Graph in a Dashboard
     */
    readonly label?: string;
    /**
     * Color for this metric when added to a Graph in a Dashboard
     */
    readonly color?: string;
}
/**
 * Properties for a metric
 */
export interface MetricProps extends CommonMetricOptions {
    /**
     * Namespace of the metric.
     */
    readonly namespace: string;
    /**
     * Name of the metric.
     */
    readonly metricName: string;
}
/**
 * Properties of a metric that can be changed
 */
export interface MetricOptions extends CommonMetricOptions {
}
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
export declare class Metric implements IMetric {
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee: iam.IGrantable): iam.Grant;
    readonly dimensions?: DimensionHash;
    readonly namespace: string;
    readonly metricName: string;
    readonly periodSec: number;
    readonly statistic: string;
    readonly unit?: Unit;
    readonly label?: string;
    readonly color?: string;
    constructor(props: MetricProps);
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props: MetricOptions): Metric;
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope: cdk.Construct, id: string, props: CreateAlarmOptions): Alarm;
    toAlarmConfig(): MetricAlarmConfig;
    toGraphConfig(): MetricGraphConfig;
    toString(): string;
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    private dimensionsAsList;
}
/**
 * Properties needed to make an alarm from a metric
 */
export interface CreateAlarmOptions {
    /**
     * The period over which the specified statistic is applied.
     *
     * Specify time in seconds, in multiples of 60.
     *
     * @default 300
     */
    readonly periodSec?: number;
    /**
     * What function to use for aggregating.
     *
     * Can be one of the following:
     *
     * - "Minimum" | "min"
     * - "Maximum" | "max"
     * - "Average" | "avg"
     * - "Sum" | "sum"
     * - "SampleCount | "n"
     * - "pNN.NN"
     *
     * @default Average
     */
    readonly statistic?: string;
    /**
     * Name of the alarm
     *
     * @default Automatically generated name
     */
    readonly alarmName?: string;
    /**
     * Description for the alarm
     *
     * @default No description
     */
    readonly alarmDescription?: string;
    /**
     * Comparison to use to check if metric is breaching
     *
     * @default GreaterThanOrEqualToThreshold
     */
    readonly comparisonOperator?: ComparisonOperator;
    /**
     * The value against which the specified statistic is compared.
     */
    readonly threshold: number;
    /**
     * The number of periods over which data is compared to the specified threshold.
     */
    readonly evaluationPeriods: number;
    /**
     * Specifies whether to evaluate the data and potentially change the alarm state if there are too few data points to be statistically significant.
     *
     * Used only for alarms that are based on percentiles.
     *
     * @default - Not configured.
     */
    readonly evaluateLowSampleCountPercentile?: string;
    /**
     * Sets how this alarm is to handle missing data points.
     *
     * @default TreatMissingData.Missing
     */
    readonly treatMissingData?: TreatMissingData;
    /**
     * Whether the actions for this alarm are enabled
     *
     * @default true
     */
    readonly actionsEnabled?: boolean;
    /**
     * The number of datapoints that must be breaching to trigger the alarm. This is used only if you are setting an "M
     * out of N" alarm. In that case, this value is the M. For more information, see Evaluating an Alarm in the Amazon
     * CloudWatch User Guide.
     *
     * @default ``evaluationPeriods``
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html#alarm-evaluation
     */
    readonly datapointsToAlarm?: number;
}
