import { Alarm } from "./alarm";
import { IMetric } from "./metric-types";
import { ConcreteWidget } from "./widget";
/**
 * Basic properties for widgets that display metrics
 */
export interface MetricWidgetProps {
    /**
     * Title for the graph
     */
    readonly title?: string;
    /**
     * The region the metrics of this graph should be taken from
     *
     * @default Current region
     */
    readonly region?: string;
    /**
     * Width of the widget, in a grid of 24 units wide
     *
     * @default 6
     */
    readonly width?: number;
    /**
     * Height of the widget
     *
     * @default Depends on the type of widget
     */
    readonly height?: number;
}
/**
 * Properties for a Y-Axis
 */
export interface YAxisProps {
    /**
     * The min value
     *
     * @default 0
     */
    readonly min?: number;
    /**
     * The max value
     *
     * @default No maximum value
     */
    readonly max?: number;
    /**
     * The label
     *
     * @default No label
     */
    readonly label?: string;
    /**
     * Whether to show units
     *
     * @default true
     */
    readonly showUnits?: boolean;
}
/**
 * Properties for an AlarmWidget
 */
export interface AlarmWidgetProps extends MetricWidgetProps {
    /**
     * The alarm to show
     */
    readonly alarm: Alarm;
    /**
     * Left Y axis
     */
    readonly leftYAxis?: YAxisProps;
}
/**
 * Display the metric associated with an alarm, including the alarm line
 */
export declare class AlarmWidget extends ConcreteWidget {
    private readonly props;
    constructor(props: AlarmWidgetProps);
    toJson(): any[];
}
/**
 * Properties for a GraphWidget
 */
export interface GraphWidgetProps extends MetricWidgetProps {
    /**
     * Metrics to display on left Y axis
     */
    readonly left?: IMetric[];
    /**
     * Metrics to display on right Y axis
     */
    readonly right?: IMetric[];
    /**
     * Annotations for the left Y axis
     */
    readonly leftAnnotations?: HorizontalAnnotation[];
    /**
     * Annotations for the right Y axis
     */
    readonly rightAnnotations?: HorizontalAnnotation[];
    /**
     * Whether the graph should be shown as stacked lines
     */
    readonly stacked?: boolean;
    /**
     * Left Y axis
     */
    readonly leftYAxis?: YAxisProps;
    /**
     * Right Y axis
     */
    readonly rightYAxis?: YAxisProps;
}
/**
 * A dashboard widget that displays metrics
 */
export declare class GraphWidget extends ConcreteWidget {
    private readonly props;
    constructor(props: GraphWidgetProps);
    toJson(): any[];
}
/**
 * Properties for a SingleValueWidget
 */
export interface SingleValueWidgetProps extends MetricWidgetProps {
    /**
     * Metrics to display
     */
    readonly metrics: IMetric[];
}
/**
 * A dashboard widget that displays the most recent value for every metric
 */
export declare class SingleValueWidget extends ConcreteWidget {
    private readonly props;
    constructor(props: SingleValueWidgetProps);
    toJson(): any[];
}
/**
 * Horizontal annotation to be added to a graph
 */
export interface HorizontalAnnotation {
    /**
     * The value of the annotation
     */
    readonly value: number;
    /**
     * Label for the annotation
     *
     * @default No label
     */
    readonly label?: string;
    /**
     * Hex color code to be used for the annotation
     *
     * @default Automatic color
     */
    readonly color?: string;
    /**
     * Add shading above or below the annotation
     *
     * @default No shading
     */
    readonly fill?: Shading;
    /**
     * Whether the annotation is visible
     *
     * @default true
     */
    readonly visible?: boolean;
}
export declare enum Shading {
    /**
     * Don't add shading
     */
    None = "none",
    /**
     * Add shading above the annotation
     */
    Above = "above",
    /**
     * Add shading below the annotation
     */
    Below = "below"
}
