# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['InquirerPy',
 'InquirerPy.base',
 'InquirerPy.prompts',
 'InquirerPy.prompts.fuzzy']

package_data = \
{'': ['*']}

install_requires = \
['prompt-toolkit>=3.0.1,<4.0.0']

setup_kwargs = {
    'name': 'inquirerpy',
    'version': '0.2.3',
    'description': 'Python port of Inquirer.js (A collection of common interactive command-line user interfaces)',
    'long_description': '# InquirerPy\n\n[![Test](https://github.com/kazhala/InquirerPy/workflows/Test/badge.svg)](https://github.com/kazhala/InquirerPy/actions?query=workflow%3ATest)\n[![Lint](https://github.com/kazhala/InquirerPy/workflows/Lint/badge.svg)](https://github.com/kazhala/InquirerPy/actions?query=workflow%3ALint)\n[![Build](https://codebuild.ap-southeast-2.amazonaws.com/badges?uuid=eyJlbmNyeXB0ZWREYXRhIjoiUUYyRUIxOXBWZ0hKcUhrbXplQklMemRsTVBxbUk3bFlTdldnRGpxeEpQSXJidEtmVEVzbVNCTE1UR3VoRSt2N0NQV0VaUXlCUzNackFBNzRVUFBBS1FnPSIsIml2UGFyYW1ldGVyU3BlYyI6IloxREtFeWY4WkhxV0NFWU0iLCJtYXRlcmlhbFNldFNlcmlhbCI6MX0%3D&branch=master)](https://ap-southeast-2.console.aws.amazon.com/codesuite/codebuild/378756445655/projects/InquirerPy/history?region=ap-southeast-2&builds-meta=eyJmIjp7InRleHQiOiIifSwicyI6e30sIm4iOjIwLCJpIjowfQ)\n[![Coverage](https://img.shields.io/coveralls/github/kazhala/InquirerPy?logo=coveralls)](https://coveralls.io/github/kazhala/InquirerPy?branch=master)\n[![Version](https://img.shields.io/pypi/pyversions/InquirerPy)](https://pypi.org/project/InquirerPy/)\n[![PyPi](https://img.shields.io/pypi/v/InquirerPy)](https://pypi.org/project/InquirerPy/)\n[![License](https://img.shields.io/pypi/l/InquirerPy)](https://github.com/kazhala/InquirerPy/blob/master/LICENSE)\n\nDocumentation: https://github.com/kazhala/InquirerPy/wiki\n\n## Introduction\n\n`InquirerPy` is a Python port of the famous [Inquirer.js](https://github.com/SBoudrias/Inquirer.js/) (A collection of common interactive command line user interfaces).\nThis project is a re-implementation of the [PyInquirer](https://github.com/CITGuru/PyInquirer) project, with bug fixes of known issues, new prompts, backward compatible APIs\nas well as more customization options.\n\n↓↓↓ Simple AWS S3 uploader/downloader prompt.\n\n![Demo](https://github.com/kazhala/gif/blob/master/InquirerPy-demo.gif)\n\n<details>\n  <summary>Classic Syntax (PyInquirer)</summary>\n\n```python\nimport boto3\nfrom InquirerPy import prompt\nfrom InquirerPy.validator import PathValidator\n\nclient = boto3.client("s3")\n\ndef get_bucket(_):\n    return [bucket["Name"] for bucket in client.list_buckets()["Buckets"]]\n\ndef walk_s3_bucket(result):\n    response = []\n    paginator = client.get_paginator("list_objects")\n    for result in paginator.paginate(Bucket=result["bucket"]):\n        for file in result["Contents"]:\n            response.append(file["Key"])\n    return response\n\ndef is_upload(result):\n    return result[0] == "Upload"\n\nquestions = [\n    {\n        "message": "Select an S3 action:",\n        "type": "list",\n        "choices": ["Upload", "Download"],\n    },\n    {\n        "message": "Enter the filepath to upload:",\n        "type": "filepath",\n        "when": is_upload,\n        "validate": PathValidator(),\n        "only_files": True,\n    },\n    {\n        "message": "Select a bucket:",\n        "type": "fuzzy",\n        "choices": get_bucket,\n        "name": "bucket",\n    },\n    {\n        "message": "Select files to download:",\n        "type": "fuzzy",\n        "when": lambda _: not is_upload(_),\n        "choices": walk_s3_bucket,\n        "multiselect": True,\n    },\n    {\n        "message": "Enter destination folder:",\n        "type": "filepath",\n        "when": lambda _: not is_upload(_),\n        "only_directories": True,\n        "validate": PathValidator(),\n    },\n    {"message": "Confirm?", "type": "confirm", "default": False},\n]\n\nresult = prompt(questions, vi_mode=True)\n\n# Download or Upload the file based on result ...\n```\n\n</details>\n\n<details>\n  <summary>Alternative Syntax</summary>\n\n```python\nimport os\nimport boto3\nfrom InquirerPy import inquirer\nfrom InquirerPy.validator import PathValidator\n\nclient = boto3.client("s3")\nos.environ["INQUIRERPY_VI_MODE"] = \'true\'\n\ndef get_bucket(_):\n    return [bucket["Name"] for bucket in client.list_buckets()["Buckets"]]\n\ndef walk_s3_bucket(bucket):\n    response = []\n    paginator = client.get_paginator("list_objects")\n    for result in paginator.paginate(Bucket=bucket):\n        for file in result["Contents"]:\n            response.append(file["Key"])\n    return response\n\naction = inquirer.select(\n    message="Select an S3 action:", choices=["Upload", "Download"]\n).execute()\n\nif action == "Upload":\n    file_to_upload = inquirer.filepath(\n        message="Enter the filepath to upload:",\n        validate=PathValidator(),\n        only_files=True,\n    ).execute()\n    bucket = inquirer.fuzzy(message="Select a bucket:", choices=get_bucket).execute()\nelse:\n    bucket = inquirer.fuzzy(message="Select a bucket:", choices=get_bucket).execute()\n    file_to_download = inquirer.fuzzy(\n        message="Select files to download:",\n        choices=lambda _: walk_s3_bucket(bucket),\n        multiselect=True,\n    ).execute()\n    destination = inquirer.filepath(\n        message="Enter destination folder:",\n        only_directories=True,\n        validate=PathValidator(),\n    ).execute()\n\nconfirm = inquirer.confirm(message="Confirm?").execute()\n\n# Download or Upload the file based on result ...\n```\n\n</details>\n\n## Motivation\n\n[PyInquirer](https://github.com/CITGuru/PyInquirer) is a great Python port of [Inquirer.js](https://github.com/SBoudrias/Inquirer.js/), however, the project is slowly reaching\nto an unmaintained state with various issues left behind and no intention to implement more feature requests. I was heavily relying on this library for other projects but\ncould not proceed due to the limitations.\n\nSome noticeable ones that bother me the most:\n\n- hard limit on `prompt_toolkit` version 1.0.3 (current release)\n- color issues (unreleased)\n- cursor issues (unreleased)\n- No options for VI/Emacs navigation key bindings\n- Pagination options don\'t work\n\nThis project uses python3.7+ type hinting with focus on resolving above issues while providing greater customization options.\n\n## Requirements\n\n### OS\n\nLeveraging `prompt_toolkit`, `InquirerPy` works cross platform for all OS. Although Unix platform may have a better experience than Windows.\n\n### Python\n\n`InquirerPy` requires Python3.7+.\n\n## Getting Started\n\nCheckout full documentation **[here](https://github.com/kazhala/InquirerPy/wiki)**.\n\n### Install\n\n```sh\npip3 install InquirerPy\n```\n\n### Quick Start\n\n#### Classic Syntax (PyInquirer)\n\n```python\nfrom InquirerPy import prompt\n\nquestions = [\n    {"type": "input", "message": "What\'s your name:", "name": "name"},\n    {"type": "confirm", "message": "Confirm?", "name": "confirm"},\n]\nresult = prompt(questions)\nname = result["name"]\nconfirm = result["confirm"]\n```\n\n#### Alternate Syntax\n\n```python\nfrom InquirerPy import inquirer\n\nname = inquirer.text(message="What\'s your name:")\nconfirm = inquirer.confirm(message="Confirm?")\n```\n\n## Migrating from PyInquirer\n\nMost APIs from `PyInquirer` should be compatible with `InquirerPy`. If you discovered any more incompatible APIs, please\nlet me know via issues or directly update README via pull request.\n\n### EditorPrompt\n\n`InquirerPy` does not support `editor` prompt as of now.\n\n### CheckboxPrompt\n\nThe following table containing the mapping of the incompatible parameters.\n\n| PyInquirer      | InquirerPy      |\n| --------------- | --------------- |\n| pointer_sign    | pointer         |\n| selected_sign   | enabled_symbol  |\n| unselected_sign | disabled_symbol |\n\n### Style\n\nEvery style keys from `PyInquirer` is present in `InquirerPy` except the once in the following table.\n\n| PyInquirer | InquirerPy |\n| ---------- | ---------- |\n| selected   | pointer    |\n\nAlthough `InquirerPy` support all the keys from `PyInquirer`, the styling works slightly different. Please refer to the [Style](https://github.com/kazhala/InquirerPy/wiki/Style) documentation for detailed information.\n\n## Similar projects\n\n### questionary\n\nWhile developing this project, I\'ve discovered there\'s already another re-implementation of [PyInquirer](https://github.com/CITGuru/PyInquirer) called [questionary](https://github.com/tmbo/questionary).\nIt\'s a fantastic fork which supports `prompt_toolkit` 3.0.0+ with performance improvement and more customization options. It\'s already a well established and stable library.\n\n_Unfortunately, the discovery of this project is a little too late since I really believe this is a great project and brings just what I needed to solve my issues with `PyInquirer` and\nI may not proceed to develop `InquirerPy`._\n\nComparing with [questionary](https://github.com/tmbo/questionary), `InquirerPy` offers even more customization options in styles, UI as well as key bindings. `InquirerPy` also provides a new\nand powerful [fuzzy](https://github.com/kazhala/InquirerPy/wiki/FuzzyPrompt) prompt.\n\nIf you are already using questionary, I do not suggest using `InquirerPy` unless you require more customization or wanna try out the [fuzzy](https://github.com/kazhala/InquirerPy/wiki/FuzzyPrompt) prompt as both library is not API compatible.\n\n### python-inquirer\n\n[python-inquirer](https://github.com/magmax/python-inquirer) is another great Python port of [Inquirer.js](https://github.com/SBoudrias/Inquirer.js/). Instead of using `prompt_toolkit`, it\nleverages the library `blessed` to implement the UI.\n\nBefore implementing `InquirerPy`, this library came up as an alternative. It\'s a more stable library comparing to the original [PyInquirer](https://github.com/CITGuru/PyInquirer), however\nit has a rather limited customization options and an older UI which did not solve the issues I was facing described in the [Motivation](#Motivation) section.\n\nComparing with [python-inquirer](https://github.com/magmax/python-inquirer), `InquirerPy` offers a slightly better UI,\nmore customization options in key bindings and styles, providing pagination as well as more prompts.\n\n## Credit\n\nThis project is based on the great work done by the following projects & their authors.\n\n- [PyInquirer](https://github.com/CITGuru/PyInquirer)\n- [prompt_toolkit](https://github.com/prompt-toolkit/python-prompt-toolkit)\n- [sweep.py](https://github.com/aslpavel/sweep.py)\n\n## License\n\nThis project is licensed under [MIT](https://github.com/kazhala/InquirerPy/blob/master/LICENSE). Copyright (c) 2020 Kevin Zhuang\n\nThe [fuzzy search logic](https://github.com/kazhala/InquirerPy/blob/master/InquirerPy/prompts/fuzzy/fzy.py) is licensed under [MIT](https://github.com/aslpavel/sweep.py/blob/master/LICENSE). Copyright (c) 2018 Pavel Aslanov\n',
    'author': 'Kevin Zhuang',
    'author_email': 'kevin7441@gmail.com',
    'maintainer': 'Kevin Zhuang',
    'maintainer_email': 'kevin7441@gmail.com',
    'url': 'https://github.com/kazhala/InquirerPy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
