"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const path = require("path");
const s3deploy = require("../lib");
/* eslint-disable max-len */
test('deploy from local directory asset', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [{
                Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
            }],
        SourceObjectKeys: [{
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            }],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('deploy from local directory assets', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [
            s3deploy.Source.asset(path.join(__dirname, 'my-website')),
            s3deploy.Source.asset(path.join(__dirname, 'my-website-second')),
        ],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [
            {
                Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
            },
            {
                Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3Bucket99793559',
            },
        ],
        SourceObjectKeys: [
            {
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            },
            {
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            },
        ],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('fails if local asset is a non-zip file', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // THEN
    expect(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
        destinationBucket: bucket,
    })).toThrow(/Asset path must be either a \.zip file or a directory/);
});
test('deploy from a local .zip file', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
    });
});
test('honors passed asset options', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'), {
                exclude: ['*', '!index.html'],
            })],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [{
                Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3BucketB41AE64D',
            }],
        SourceObjectKeys: [{
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3VersionKeyF3CBA38F',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3VersionKeyF3CBA38F',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            }],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('retainOnDelete can be used to retain files when resource is deleted', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        retainOnDelete: true,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        RetainOnDelete: true,
    });
});
test('user metadata is correctly transformed', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        metadata: {
            A: '1',
            B: '2',
        },
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        UserMetadata: { 'x-amzn-meta-a': '1', 'x-amzn-meta-b': '2' },
    });
});
test('system metadata is correctly transformed', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const expiration = cdk.Expiration.after(cdk.Duration.hours(12));
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        contentType: 'text/html',
        contentLanguage: 'en',
        storageClass: s3deploy.StorageClass.INTELLIGENT_TIERING,
        contentDisposition: 'inline',
        serverSideEncryption: s3deploy.ServerSideEncryption.AWS_KMS,
        serverSideEncryptionAwsKmsKeyId: 'mykey',
        serverSideEncryptionCustomerAlgorithm: 'rot13',
        websiteRedirectLocation: 'example',
        cacheControl: [s3deploy.CacheControl.setPublic(), s3deploy.CacheControl.maxAge(cdk.Duration.hours(1))],
        expires: expiration,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        SystemMetadata: {
            'content-type': 'text/html',
            'content-language': 'en',
            'content-disposition': 'inline',
            'storage-class': 'INTELLIGENT_TIERING',
            'sse': 'aws:kms',
            'sse-kms-key-id': 'mykey',
            'cache-control': 'public, max-age=3600',
            'expires': expiration.date.toUTCString(),
            'sse-c-copy-source': 'rot13',
            'website-redirect': 'example',
        },
    });
});
test('expires type has correct values', () => {
    expect(cdk.Expiration.atDate(new Date('Sun, 26 Jan 2020 00:53:20 GMT')).date.toUTCString()).toEqual('Sun, 26 Jan 2020 00:53:20 GMT');
    expect(cdk.Expiration.atTimestamp(1580000000000).date.toUTCString()).toEqual('Sun, 26 Jan 2020 00:53:20 GMT');
    expect(Math.abs(new Date(cdk.Expiration.after(cdk.Duration.minutes(10)).date.toUTCString()).getTime() - (Date.now() + 600000)) < 15000).toBeTruthy();
    expect(cdk.Expiration.fromString('Tue, 04 Feb 2020 08:45:33 GMT').date.toUTCString()).toEqual('Tue, 04 Feb 2020 08:45:33 GMT');
});
test('cache control type has correct values', () => {
    expect(s3deploy.CacheControl.mustRevalidate().value).toEqual('must-revalidate');
    expect(s3deploy.CacheControl.noCache().value).toEqual('no-cache');
    expect(s3deploy.CacheControl.noTransform().value).toEqual('no-transform');
    expect(s3deploy.CacheControl.setPublic().value).toEqual('public');
    expect(s3deploy.CacheControl.setPrivate().value).toEqual('private');
    expect(s3deploy.CacheControl.proxyRevalidate().value).toEqual('proxy-revalidate');
    expect(s3deploy.CacheControl.maxAge(cdk.Duration.minutes(1)).value).toEqual('max-age=60');
    expect(s3deploy.CacheControl.sMaxAge(cdk.Duration.minutes(1)).value).toEqual('s-maxage=60');
    expect(s3deploy.CacheControl.fromString('only-if-cached').value).toEqual('only-if-cached');
});
test('storage class type has correct values', () => {
    expect(s3deploy.StorageClass.STANDARD).toEqual('STANDARD');
    expect(s3deploy.StorageClass.REDUCED_REDUNDANCY).toEqual('REDUCED_REDUNDANCY');
    expect(s3deploy.StorageClass.STANDARD_IA).toEqual('STANDARD_IA');
    expect(s3deploy.StorageClass.ONEZONE_IA).toEqual('ONEZONE_IA');
    expect(s3deploy.StorageClass.INTELLIGENT_TIERING).toEqual('INTELLIGENT_TIERING');
    expect(s3deploy.StorageClass.GLACIER).toEqual('GLACIER');
    expect(s3deploy.StorageClass.DEEP_ARCHIVE).toEqual('DEEP_ARCHIVE');
});
test('server side encryption type has correct values', () => {
    expect(s3deploy.ServerSideEncryption.AES_256).toEqual('AES256');
    expect(s3deploy.ServerSideEncryption.AWS_KMS).toEqual('aws:kms');
});
test('distribution can be used to provide a CloudFront distribution for invalidation', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    s3BucketSource: bucket,
                },
                behaviors: [{ isDefaultBehavior: true }],
            },
        ],
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        distribution,
        distributionPaths: ['/images/*'],
    });
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        DistributionId: {
            Ref: 'DistributionCFDistribution882A7313',
        },
        DistributionPaths: ['/images/*'],
    });
});
test('invalidation can happen without distributionPaths provided', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    s3BucketSource: bucket,
                },
                behaviors: [{ isDefaultBehavior: true }],
            },
        ],
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        distribution,
    });
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        DistributionId: {
            Ref: 'DistributionCFDistribution882A7313',
        },
    });
});
test('fails if distribution paths provided but not distribution ID', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // THEN
    expect(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
        destinationBucket: bucket,
        distributionPaths: ['/images/*'],
    })).toThrow(/Distribution must be specified if distribution paths are specified/);
});
test('lambda execution role gets permissions to read from the source bucket and read/write in destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const source = new s3.Bucket(stack, 'Source');
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.bucket(source, 'file.zip')],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'Source71E471F1',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'Source71E471F1',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                        's3:DeleteObject*',
                        's3:PutObject*',
                        's3:Abort*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'DestC383B82A',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'DestC383B82A',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRoleDefaultPolicy88902FDF',
        Roles: [
            {
                Ref: 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265',
            },
        ],
    });
});
test('memoryLimit can be used to specify the memory limit for the deployment resource handler', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    // we define 3 deployments with 2 different memory configurations
    new s3deploy.BucketDeployment(stack, 'Deploy256-1', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 256,
    });
    new s3deploy.BucketDeployment(stack, 'Deploy256-2', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 256,
    });
    new s3deploy.BucketDeployment(stack, 'Deploy1024', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 1024,
    });
    // THEN
    // we expect to find only two handlers, one for each configuration
    expect(stack).toCountResources('AWS::Lambda::Function', 2);
    expect(stack).toHaveResource('AWS::Lambda::Function', { MemorySize: 256 });
    expect(stack).toHaveResource('AWS::Lambda::Function', { MemorySize: 1024 });
});
test('deployment allows custom role to be supplied', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const existingRole = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('lambda.amazon.com'),
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'DeployWithRole', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        role: existingRole,
    });
    // THEN
    expect(stack).toCountResources('AWS::IAM::Role', 1);
    expect(stack).toCountResources('AWS::Lambda::Function', 1);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Role: {
            'Fn::GetAtt': [
                'Role1ABCC5F0',
                'Arn',
            ],
        },
    });
});
test('deploy without deleting missing files from destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        prune: false,
    });
    expect(stack).toHaveResourceLike('Custom::CDKBucketDeployment', {
        Prune: false,
    });
});
//# sourceMappingURL=data:application/json;base64,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