"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'with defaults set, it successfully creates with cloudwatch logs destination'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123455'),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'NetworkInterface',
            TrafficType: 'ALL',
            ResourceId: 'eni-123455',
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['FlowLogsIAMRoleF18F4209', 'Arn'],
            },
            LogGroupName: {
                Ref: 'FlowLogsLogGroup9853A85F',
            },
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::Logs::LogGroup', 1));
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 1));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::S3::Bucket'));
        test.done();
    },
    'with cloudwatch logs as the destination, allows use of existing resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toCloudWatchLogs(new logs.LogGroup(stack, 'TestLogGroup', {
                retention: logs.RetentionDays.FIVE_DAYS,
            }), new iam.Role(stack, 'TestRole', {
                roleName: 'TestName',
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            })),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 5,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            RoleName: 'TestName',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::S3::Bucket'));
        test.done();
    },
    'with s3 as the destination, allows use of existing resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toS3(new s3.Bucket(stack, 'TestBucket', {
                bucketName: 'testbucket',
            })),
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Logs::LogGroup'));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            BucketName: 'testbucket',
        }));
        test.done();
    },
    'with s3 as the destination and all the defaults set, it successfully creates all the resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toS3(),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'NetworkInterface',
            TrafficType: 'ALL',
            ResourceId: 'eni-123456',
            LogDestination: {
                'Fn::GetAtt': ['FlowLogsBucket87F67F60', 'Arn'],
            },
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Logs::LogGroup'));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.countResources('AWS::S3::Bucket', 1));
        test.done();
    },
    'create with vpc'(test) {
        const stack = getTestStack();
        new lib_1.Vpc(stack, 'VPC', {
            flowLogs: {
                flowLogs: {},
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'VPC',
            TrafficType: 'ALL',
            ResourceId: {
                Ref: 'VPCB9E5F0B4',
            },
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['VPCflowLogsIAMRole9D21E1A6', 'Arn'],
            },
            LogGroupName: {
                Ref: 'VPCflowLogsLogGroupE900F980',
            },
        }));
        test.done();
    },
    'add to vpc'(test) {
        const stack = getTestStack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        vpc.addFlowLog('FlowLogs');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'VPC',
            TrafficType: 'ALL',
            ResourceId: {
                Ref: 'VPCB9E5F0B4',
            },
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['VPCFlowLogsIAMRole55343234', 'Arn'],
            },
            LogGroupName: {
                Ref: 'VPCFlowLogsLogGroupF48E1B0A',
            },
        }));
        test.done();
    },
});
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', {
        env: { account: '123456789012', region: 'us-east-1' },
    });
}
//# sourceMappingURL=data:application/json;base64,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