"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC).
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. Internet traffic will
     * be routed via a NAT Gateway.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            availabilityZones: subnets.map(s => s.availabilityZone),
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        var _a;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_a = options.vpnRoutePropagation) !== null && _a !== void 0 ? _a : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            this.node.addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE;
            subnets = this.selectSubnetObjectsByType(type);
        }
        if (selection.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnets = retainByAZ(subnets, selection.availabilityZones);
        }
        if (!!selection.onePerAz && subnets.length > 0) { // Ensure one per AZ if specified
            subnets = retainOnePerAz(subnets);
        }
        return subnets;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            if (this.privateSubnets.length > 0) {
                return {
                    subnetType: SubnetType.PRIVATE,
                    onePerAz: placement.onePerAz,
                    availabilityZones: placement.availabilityZones
                };
            }
            if (this.isolatedSubnets.length > 0) {
                return {
                    subnetType: SubnetType.ISOLATED,
                    onePerAz: placement.onePerAz,
                    availabilityZones: placement.availabilityZones
                };
            }
            return {
                subnetType: SubnetType.PUBLIC,
                onePerAz: placement.onePerAz,
                availabilityZones: placement.availabilityZones
            };
        }
        return placement;
    }
}
function retainByAZ(subnets, azs) {
    return subnets.filter(s => azs.includes(s.availabilityZone));
}
function retainOnePerAz(subnets) {
    const azsSeen = new Set();
    return subnets.filter(subnet => {
        if (azsSeen.has(subnet.availabilityZone)) {
            return false;
        }
        azsSeen.add(subnet.availabilityZone);
        return true;
    });
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        const enableDnsHostnames = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        const enableDnsSupport = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames,
            enableDnsSupport,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs !== undefined ? props.maxAzs : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import an exported VPC
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaing = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaing);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            subnet.node.applyAspect(new core_1.Tag(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes }));
            subnet.node.applyAspect(new core_1.Tag(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes }));
        });
    }
}
exports.Vpc = Vpc;
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZone = props.availabilityZone;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.stringValue({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explic it ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatwayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
}
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            const ref = core_1.Token.isUnresolved(attrs.subnetId)
                ? `at '${scope.node.path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            scope.node.addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error("You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
        }
        return this._availabilityZone;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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