"""Test script to demonstrate root user functionality in llm-sandbox."""

import logging

from llm_sandbox import SandboxBackend, SandboxSession

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")

logger = logging.getLogger(__name__)


def cleanup_test_pods() -> None:
    """Clean up any existing test pods to avoid conflicts."""
    try:
        from kubernetes import client, config

        # Load kubernetes config
        try:
            config.load_incluster_config()
        except config.ConfigException:
            config.load_kube_config()

        v1 = client.CoreV1Api()

        # List pods with our test label
        pods = v1.list_namespaced_pod(namespace="default", label_selector="app=sandbox")

        # Delete test pods that start with our test names
        for pod in pods.items:
            if pod.metadata.name.startswith(("test-non-root", "sandbox-")):
                logger.info("Cleaning up existing pod: %s", pod.metadata.name)
                try:
                    v1.delete_namespaced_pod(name=pod.metadata.name, namespace="default")
                except Exception:
                    logger.exception("Failed to delete pod %s", pod.metadata.name)

    except Exception:
        logger.exception("Pod cleanup skipped")


def test_default_root_user() -> None:
    """Test that containers run as root by default."""
    logger.info("=== Testing Default Root User ===")

    with SandboxSession(lang="python") as session:
        # Test 1: Check if running as root
        result = session.run("import os; print(f'UID: {os.getuid()}, GID: {os.getgid()}')")
        logger.info("Result: %s", result.stdout.strip())

        # Test 2: Create directory in root filesystem
        result = session.run(
            """
import os
os.makedirs('/test_dir', exist_ok=True)
with open('/test_dir/test_file.txt', 'w') as f:
    f.write('Hello from root!')
print('Successfully created file in /test_dir/')
"""
        )
        logger.info("File creation result: %s", result.stdout.strip())

        # Test 3: Check if we can access system directories
        result = session.run("import os; print('Can access /etc:', os.path.exists('/etc'))")
        logger.info("System access: %s", result.stdout.strip())


def test_custom_user() -> None:
    """Test running with a custom user."""
    logger.info("=== Testing Custom User ===")

    # Note: This might fail in some containers that don't have user 1000
    try:
        with SandboxSession(
            lang="python",
            runtime_configs={"user": "1000:1000"},
            workdir="/tmp/sandbox",
        ) as session:
            result = session.run("import os; print(f'UID: {os.getuid()}, GID: {os.getgid()}')")
            logger.info("Custom user result: %s", result.stdout.strip())

            # Test if we can still write to /tmp
            result = session.run(
                """
import os
import tempfile
with tempfile.NamedTemporaryFile(mode='w', delete=False, dir='/tmp') as f:
    f.write('Hello from user 1000!')
    temp_file = f.name
print(f'Successfully created file: {temp_file}')
"""
            )
            logger.info("Temp file creation: %s", result.stdout.strip())

    except Exception:
        logger.exception(
            "Custom user test failed (this is expected if user 1000 doesn't exist)",
        )


def test_permission_comparison() -> None:
    """Compare what we can do as root vs non-root."""
    logger.info("=== Permission Comparison ===")

    logger.info("As root:")
    with SandboxSession(lang="python") as session:
        result = session.run(
            """
import os
try:
    # Try to write to /etc (should work as root)
    with open('/etc/test_root_file', 'w') as f:
        f.write('test')
    print("✓ Can write to /etc")
    os.remove('/etc/test_root_file')
except Exception as e:
    print(f"✗ Cannot write to /etc: {e}")

try:
    # Try to create directory in /
    os.makedirs('/root_test_dir', exist_ok=True)
    print("✓ Can create directory in /")
    os.rmdir('/root_test_dir')
except Exception as e:
    print(f"✗ Cannot create directory in /: {e}")
"""
        )
        logger.info(result.stdout.strip())


def test_kubernetes_root_user() -> None:
    """Test Kubernetes root user behavior (requires k8s cluster)."""
    logger.info("=== Testing Kubernetes Root User (requires cluster) ===")

    try:
        with SandboxSession(backend=SandboxBackend.KUBERNETES, lang="python") as session:
            result = session.run("import os; print(f'K8s UID: {os.getuid()}, GID: {os.getgid()}')")
            logger.info("Kubernetes root result: %s", result.stdout.strip())

            # Test creating directory in root filesystem
            result = session.run(
                """
import os
os.makedirs('/k8s_test_dir', exist_ok=True)
with open('/k8s_test_dir/test.txt', 'w') as f:
    f.write('Hello from K8s root!')
print('✓ Successfully created file in /k8s_test_dir/')
"""
            )
            logger.info("K8s file creation: %s", result.stdout.strip())

    except Exception:
        logger.exception("Kubernetes test skipped (cluster not available)")


def test_kubernetes_custom_security() -> None:
    """Test Kubernetes with custom security context."""
    logger.info("=== Testing Kubernetes Custom Security Context ===")

    try:
        import uuid

        # Generate unique pod name to avoid conflicts
        unique_suffix = str(uuid.uuid4())[:8]
        pod_name = f"test-non-root-{unique_suffix}"

        # Custom pod manifest with non-root user
        pod_manifest = {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {
                "name": pod_name,
                "labels": {"app": "sandbox"},
            },
            "spec": {
                "containers": [
                    {
                        "name": "sandbox-container",
                        "image": "ghcr.io/vndee/sandbox-python-311-bullseye",
                        "tty": True,
                        "securityContext": {
                            "runAsUser": 1000,
                            "runAsGroup": 1000,
                        },
                    }
                ],
                "securityContext": {
                    "runAsUser": 1000,
                    "runAsGroup": 1000,
                },
            },
        }

        with SandboxSession(
            backend=SandboxBackend.KUBERNETES,
            lang="python",
            pod_manifest=pod_manifest,
            workdir="/tmp/sandbox",  # Use writable directory for non-root
        ) as session:
            result = session.run("import os; print(f'Custom K8s UID: {os.getuid()}, GID: {os.getgid()}')")
            logger.info("Custom security context result: %s", result.stdout.strip())

            # Test file creation with non-root user
            result = session.run(
                """
import os
import tempfile
try:
    with tempfile.NamedTemporaryFile(mode='w', delete=False, dir='/tmp') as f:
        f.write('Hello from non-root K8s user!')
        temp_file = f.name
    print(f'✓ Successfully created file: {temp_file}')
except Exception as e:
    print(f'✗ Failed to create file: {e}')
"""
            )
            logger.info("Non-root file creation: %s", result.stdout.strip())

    except Exception as e:
        if "AlreadyExists" in str(e):
            logger.warning("Pod with similar name already exists, skipping custom security test")
        else:
            logger.exception("Kubernetes custom security test skipped")


def test_podman_root_user() -> None:
    """Test Podman root user behavior (requires podman)."""
    logger.info("=== Testing Podman Root User (requires podman) ===")
    from podman import PodmanClient

    client = PodmanClient(
        base_url="unix:///var/folders/lh/rjbzw60n1fv7xr9kffn7gr840000gn/T/podman/podman-machine-default-api.sock"
    )
    try:
        with SandboxSession(backend=SandboxBackend.PODMAN, lang="python", client=client) as session:
            result = session.run("import os; print(f'Podman UID: {os.getuid()}, GID: {os.getgid()}')")
            logger.info("Podman root result: %s", result.stdout.strip())

            # Test creating directory in root filesystem
            result = session.run(
                """
import os
os.makedirs('/podman_test_dir', exist_ok=True)
with open('/podman_test_dir/test.txt', 'w') as f:
    f.write('Hello from Podman root!')
print('✓ Successfully created file in /podman_test_dir/')
"""
            )
            logger.info("Podman file creation: %s", result.stdout.strip())

    except Exception:
        logger.exception("Podman test skipped (podman not available)")


def test_podman_custom_user() -> None:
    """Test Podman with custom user."""
    logger.info("=== Testing Podman Custom User ===")
    from podman import PodmanClient

    client = PodmanClient(
        base_url="unix:///var/folders/lh/rjbzw60n1fv7xr9kffn7gr840000gn/T/podman/podman-machine-default-api.sock"
    )
    try:
        with SandboxSession(
            backend=SandboxBackend.PODMAN,
            lang="python",
            runtime_configs={"user": "1000:1000"},
            workdir="/tmp/sandbox",
            client=client,
        ) as session:
            result = session.run("import os; print(f'Podman Custom UID: {os.getuid()}, GID: {os.getgid()}')")
            logger.info("Podman custom user result: %s", result.stdout.strip())

            # Test if we can still write to /tmp
            result = session.run(
                """
import os
import tempfile
with tempfile.NamedTemporaryFile(mode='w', delete=False, dir='/tmp') as f:
    f.write('Hello from Podman user 1000!')
    temp_file = f.name
print(f'Successfully created file: {temp_file}')
"""
            )
            logger.info("Podman temp file creation: %s", result.stdout.strip())

    except Exception:
        logger.exception("Podman custom user test skipped")


if __name__ == "__main__":
    # Clean up any existing test pods first
    cleanup_test_pods()

    test_default_root_user()
    test_custom_user()
    test_permission_comparison()
    test_kubernetes_root_user()
    test_kubernetes_custom_security()
    test_podman_root_user()
    test_podman_custom_user()
    logger.info("=== All tests completed ===")
