"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
class CodeDeployEcsDeployAction extends action_1.Action {
    constructor(props) {
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision']
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig']
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition']
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                }
            }
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: determineTaskDefinitionArtifact(this.actionProps).artifactName,
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: determineAppSpecArtifact(this.actionProps).artifactName,
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = imageInput.input.artifactName;
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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