"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BootstraplessStackSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const REGION_PLACEHOLDER = '${AWS::Region}';
const ERR_MSG_CALL_BIND_FIRST = 'You must call bind() first';
/**
 * A Bootstrapless stack synthesizer that is designated to generate templates that can be directly used by Cloudformation.
 *
 * @stability stable
 */
class BootstraplessStackSynthesizer extends core_1.StackSynthesizer {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super();
        this.files = {};
        this.dockerImages = {};
        const { BSS_FILE_ASSET_BUCKET_NAME, BSS_IMAGE_ASSET_REPOSITORY_NAME, BSS_FILE_ASSET_PUBLISHING_ROLE_ARN, BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN, BSS_FILE_ASSET_PREFIX, BSS_FILE_ASSET_REGION_SET, BSS_TEMPLATE_BUCKET_NAME, BSS_IMAGE_ASSET_TAG_PREFIX, BSS_IMAGE_ASSET_REGION_SET, BSS_IMAGE_ASSET_ACCOUNT_ID, } = process.env;
        this.bucketName = (_b = props.fileAssetBucketName) !== null && _b !== void 0 ? _b : BSS_FILE_ASSET_BUCKET_NAME;
        this.repositoryName = (_c = props.imageAssetRepositoryName) !== null && _c !== void 0 ? _c : BSS_IMAGE_ASSET_REPOSITORY_NAME;
        this.fileAssetPublishingRoleArn = (_d = props.fileAssetPublishingRoleArn) !== null && _d !== void 0 ? _d : BSS_FILE_ASSET_PUBLISHING_ROLE_ARN;
        this.imageAssetPublishingRoleArn = (_e = props.imageAssetPublishingRoleArn) !== null && _e !== void 0 ? _e : BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN;
        this.fileAssetPrefix = (_f = props.fileAssetPrefix) !== null && _f !== void 0 ? _f : BSS_FILE_ASSET_PREFIX;
        this.fileAssetRegionSet = (_g = props.fileAssetRegionSet) !== null && _g !== void 0 ? _g : commaSplit(BSS_FILE_ASSET_REGION_SET);
        this.templateBucketName = (_h = props.templateBucketName) !== null && _h !== void 0 ? _h : BSS_TEMPLATE_BUCKET_NAME;
        this.imageAssetTagPrefix = (_k = ((_j = props.imageAssetTagPrefix) !== null && _j !== void 0 ? _j : BSS_IMAGE_ASSET_TAG_PREFIX)) !== null && _k !== void 0 ? _k : '';
        this.imageAssetRegionSet = (_l = props.imageAssetRegionSet) !== null && _l !== void 0 ? _l : commaSplit(BSS_IMAGE_ASSET_REGION_SET);
        this.imageAssetAccountId = (_m = props.imageAssetAccountId) !== null && _m !== void 0 ? _m : BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        var _b;
        if (this._stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._stack = stack;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            if (s === undefined) {
                return undefined;
            }
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        /* eslint-disable max-len */
        this.bucketName = specialize(this.bucketName);
        this.repositoryName = specialize(this.repositoryName);
        this.fileAssetPublishingRoleArn = specialize(this.fileAssetPublishingRoleArn);
        this.imageAssetPublishingRoleArn = specialize(this.imageAssetPublishingRoleArn);
        this.fileAssetPrefix = specialize((_b = this.fileAssetPrefix) !== null && _b !== void 0 ? _b : '');
        /* eslint-enable max-len */
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        return this._addFileAsset(asset);
    }
    _addFileAsset(asset, overrideBucketname) {
        var _b;
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.bucketName, 'The bucketName is null');
        validateFileAssetSource(asset);
        const bucketName = overrideBucketname !== null && overrideBucketname !== void 0 ? overrideBucketname : this.bucketName;
        const objectKey = this.fileAssetPrefix + asset.sourceHash + (asset.packaging === core_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        const destinations = {};
        if (((_b = this.fileAssetRegionSet) === null || _b === void 0 ? void 0 : _b.length) && bucketName.includes(REGION_PLACEHOLDER)) {
            for (const region of this.fileAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    bucketName: replaceAll(bucketName, REGION_PLACEHOLDER, region),
                    objectKey,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                bucketName,
                objectKey,
                region: resolvedOr(this.stack.region, undefined),
                assumeRoleArn: this.fileAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations,
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
            s3Url: httpUrl,
        };
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        var _b, _c;
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.repositoryName, 'The repositoryName is null');
        validateDockerImageAssetSource(asset);
        const imageTag = this.imageAssetTagPrefix + asset.sourceHash;
        const destinations = {};
        if ((_b = this.imageAssetRegionSet) === null || _b === void 0 ? void 0 : _b.length) {
            for (const region of this.imageAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                repositoryName: this.repositoryName,
                imageTag,
                region: resolvedOr(this.stack.region, undefined),
                assumeRoleArn: this.imageAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations,
        };
        let { account, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        account = (_c = this.imageAssetAccountId) !== null && _c !== void 0 ? _c : account;
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${REGION_PLACEHOLDER}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    /**
     * Dumps current manifest into JSON format.
     *
     * @stability stable
     */
    dumps() {
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        return JSON.stringify(manifest, undefined, 2);
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        this.synthesizeStackTemplate(this.stack, session);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        this.emitStackArtifact(this.stack, session, {
            stackTemplateAssetObjectUrl: templateManifestUrl,
            additionalDependencies: [artifactId],
        });
    }
    /**
     * @stability stable
     */
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(_) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const sourceHash = this.stack.templateFile;
        this._addFileAsset({
            fileName: this.stack.templateFile,
            packaging: core_1.FileAssetPackaging.FILE,
            sourceHash,
        }, this.templateBucketName);
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        fs.writeFileSync(outPath, this.dumps());
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.BootstraplessStackSynthesizer = BootstraplessStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
BootstraplessStackSynthesizer[_a] = { fqn: "cdk-bootstrapless-synthesizer.BootstraplessStackSynthesizer", version: "0.9.1" };
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return core_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? core_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
// function range(startIncl: number, endExcl: number) {
//     const ret = new Array<number>();
//     for (let i = startIncl; i < endExcl; i++) {
//     ret.push(i);
//     }
//     return ret;
// }
function assertNotNull(x, msg = 'Null value error') {
    if (x === null || x === undefined) {
        throw new Error(msg);
    }
}
function commaSplit(v) {
    if (v) {
        return v.split(',');
    }
    return undefined;
}
function validateFileAssetSource(asset) {
    if (!!asset.executable === !!asset.fileName) {
        throw new Error(`Exactly one of 'fileName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    if (!!asset.packaging !== !!asset.fileName) {
        throw new Error(`'packaging' is expected in combination with 'fileName', got: ${JSON.stringify(asset)}`);
    }
}
function validateDockerImageAssetSource(asset) {
    if (!!asset.executable === !!asset.directoryName) {
        throw new Error(`Exactly one of 'directoryName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    check('dockerBuildArgs');
    check('dockerBuildTarget');
    check('dockerFile');
    function check(key) {
        if (asset[key] && !asset.directoryName) {
            throw new Error(`'${key}' is only allowed in combination with 'directoryName', got: ${JSON.stringify(asset)}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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